import { ServiceClientOptions } from "@azure/ms-rest-js";
import * as msRest from "@azure/ms-rest-js";
/**
 * The configuration metrics for Iot Hub devices and modules.
 */
export interface ConfigurationMetrics {
    /**
     * The results of the metrics collection queries.
     */
    results?: {
        [propertyName: string]: number;
    };
    /**
     * The key-value pairs with queries and their identifier.
     */
    queries?: {
        [propertyName: string]: string;
    };
}
/**
 * The configuration content for devices or modules on edge devices.
 */
export interface ConfigurationContent {
    /**
     * The device configuration content.
     */
    deviceContent?: {
        [propertyName: string]: any;
    };
    /**
     * The modules configuration content.
     */
    modulesContent?: {
        [propertyName: string]: {
            [propertyName: string]: any;
        };
    };
    /**
     * The module configuration content.
     */
    moduleContent?: {
        [propertyName: string]: any;
    };
}
/**
 * The configuration for Iot Hub device and module twins.
 */
export interface Configuration {
    /**
     * The unique identifier of the configuration.
     */
    id?: string;
    /**
     * The schema version of the configuration.
     */
    schemaVersion?: string;
    /**
     * The key-value pairs used to describe the configuration.
     */
    labels?: {
        [propertyName: string]: string;
    };
    /**
     * The content of the configuration.
     */
    content?: ConfigurationContent;
    /**
     * The query used to define the targeted devices or modules. The query is based on twin tags
     * and/or reported properties.
     */
    targetCondition?: string;
    /**
     * The creation date and time of the configuration.
     */
    createdTimeUtc?: Date;
    /**
     * The update date and time of the configuration.
     */
    lastUpdatedTimeUtc?: Date;
    /**
     * The priority number assigned to the configuration.
     */
    priority?: number;
    /**
     * The system metrics computed by the IoT Hub that cannot be customized.
     */
    systemMetrics?: ConfigurationMetrics;
    /**
     * The custom metrics specified by the developer as queries against twin reported properties.
     */
    metrics?: ConfigurationMetrics;
    /**
     * The ETag of the configuration.
     */
    etag?: string;
}
/**
 * An interface representing ConfigurationQueriesTestInput.
 */
export interface ConfigurationQueriesTestInput {
    /**
     * The query used to define targeted devices or modules. The query is based on twin tags and/or
     * reported properties.
     */
    targetCondition?: string;
    /**
     * The key-value pairs with queries and their identifier.
     */
    customMetricQueries?: {
        [propertyName: string]: string;
    };
}
/**
 * An interface representing ConfigurationQueriesTestResponse.
 */
export interface ConfigurationQueriesTestResponse {
    /**
     * The errors from running the target condition query.
     */
    targetConditionError?: string;
    /**
     * The errors from running the custom metric query.
     */
    customMetricQueryErrors?: {
        [propertyName: string]: string;
    };
}
/**
 * An interface representing RegistryStatistics.
 */
export interface RegistryStatistics {
    /**
     * The total number of devices registered for the IoT Hub.
     */
    totalDeviceCount?: number;
    /**
     * The number of currently enabled devices.
     */
    enabledDeviceCount?: number;
    /**
     * The number of currently disabled devices.
     */
    disabledDeviceCount?: number;
}
/**
 * An interface representing ServiceStatistics.
 */
export interface ServiceStatistics {
    /**
     * The number of currently connected devices.
     */
    connectedDeviceCount?: number;
}
/**
 * An interface representing SymmetricKey.
 */
export interface SymmetricKey {
    /**
     * The base64 encoded primary key of the device.
     */
    primaryKey?: string;
    /**
     * The base64 encoded secondary key of the device.
     */
    secondaryKey?: string;
}
/**
 * An interface representing X509Thumbprint.
 */
export interface X509Thumbprint {
    /**
     * The X509 client certificate primary thumbprint.
     */
    primaryThumbprint?: string;
    /**
     * The X509 client certificate secondary thumbprint.
     */
    secondaryThumbprint?: string;
}
/**
 * An interface representing AuthenticationMechanism.
 */
export interface AuthenticationMechanism {
    /**
     * The primary and secondary keys used for SAS based authentication.
     */
    symmetricKey?: SymmetricKey;
    /**
     * The primary and secondary x509 thumbprints used for x509 based authentication.
     */
    x509Thumbprint?: X509Thumbprint;
    /**
     * The type of authentication used to connect to the service. Possible values include: 'sas',
     * 'selfSigned', 'certificateAuthority', 'none'
     */
    type?: Type;
}
/**
 * The status of capabilities enabled on the device.
 */
export interface DeviceCapabilities {
    /**
     * The property that determines if the device is an edge device or not.
     */
    iotEdge?: boolean;
}
/**
 * An interface representing Device.
 */
export interface Device {
    /**
     * The unique identifier of the device.
     */
    deviceId?: string;
    /**
     * The IoT Hub-generated, case-sensitive string up to 128 characters long. This value is used to
     * distinguish devices with the same deviceId, when they have been deleted and re-created.
     */
    generationId?: string;
    /**
     * The string representing a weak ETag for the device identity, as per RFC7232.
     */
    etag?: string;
    /**
     * The state of the device. Possible values include: 'Disconnected', 'Connected'
     */
    connectionState?: ConnectionState;
    /**
     * The status of the device. If the status disabled, a device cannot connect to the service.
     * Possible values include: 'enabled', 'disabled'
     */
    status?: Status;
    /**
     * The 128 character-long string that stores the reason for the device identity status. All UTF-8
     * characters are allowed.
     */
    statusReason?: string;
    /**
     * The date and time the connection state was last updated.
     */
    connectionStateUpdatedTime?: Date;
    /**
     * The date and time when the status field was last updated.
     */
    statusUpdatedTime?: Date;
    /**
     * The date and last time the device last connected, received, or sent a message.
     */
    lastActivityTime?: Date;
    /**
     * The number of cloud-to-device messages currently queued to be sent to the device.
     */
    cloudToDeviceMessageCount?: number;
    /**
     * The authentication mechanism used by the device.
     */
    authentication?: AuthenticationMechanism;
    /**
     * The set of capabilities of the device. For example, if this device is an edge device or not.
     */
    capabilities?: DeviceCapabilities;
    /**
     * The scope of the device. Auto generated and immutable for edge devices and modifiable in leaf
     * devices to create child/parent relationship.
     */
    deviceScope?: string;
    /**
     * The scopes of the upper level edge devices if applicable. Only available for edge devices.
     */
    parentScopes?: string[];
}
/**
 * The desired and reported properties of the twin. The maximum depth of the object is 10.
 */
export interface PropertyContainer {
    /**
     * The collection of desired property key-value pairs. The keys are UTF-8 encoded, case-sensitive
     * and up-to 1KB in length. Allowed characters exclude UNICODE control characters (segments C0
     * and C1), '.', '$' and space. The desired porperty values are JSON objects, up-to 4KB in
     * length.
     */
    desired?: {
        [propertyName: string]: any;
    };
    /**
     * The collection of reported property key-value pairs. The keys are UTF-8 encoded,
     * case-sensitive and up-to 1KB in length. Allowed characters exclude UNICODE control characters
     * (segments C0 and C1), '.', '$' and space. The reported property values are JSON objects, up-to
     * 4KB in length.
     */
    reported?: {
        [propertyName: string]: any;
    };
}
/**
 * An interface representing ExportImportDevice.
 */
export interface ExportImportDevice {
    /**
     * The unique identifier of the device.
     */
    id?: string;
    /**
     * The unique identifier of the module, if applicable.
     */
    moduleId?: string;
    /**
     * The string representing a weak ETag for the device RFC7232. The value is only used if import
     * mode is updateIfMatchETag, in that case the import operation is performed only if this ETag
     * matches the value maintained by the server.
     */
    eTag?: string;
    /**
     * The type of registry operation and ETag preferences. Possible values include: 'create',
     * 'update', 'updateIfMatchETag', 'delete', 'deleteIfMatchETag', 'updateTwin',
     * 'updateTwinIfMatchETag'
     */
    importMode?: ImportMode;
    /**
     * The status of the module. If disabled, the module cannot connect to the service. Possible
     * values include: 'enabled', 'disabled'
     */
    status?: Status1;
    /**
     * The 128 character-long string that stores the reason for the device identity status. All UTF-8
     * characters are allowed.
     */
    statusReason?: string;
    /**
     * The authentication mechanism used by the module. This parameter is optional and defaults to
     * SAS if not provided. In that case, primary/secondary access keys are auto-generated.
     */
    authentication?: AuthenticationMechanism;
    /**
     * The string representing a weak ETag for the device twin RFC7232. The value is only used if
     * import mode is updateIfMatchETag, in that case the import operation is performed only if this
     * ETag matches the value maintained by the server.
     */
    twinETag?: string;
    /**
     * The JSON document read and written by the solution back end. The tags are not visible to
     * device apps.
     */
    tags?: {
        [propertyName: string]: any;
    };
    /**
     * The desired and reported properties for the device.
     */
    properties?: PropertyContainer;
    /**
     * The status of capabilities enabled on the device.
     */
    capabilities?: DeviceCapabilities;
    /**
     * The scope of the device.
     */
    deviceScope?: string;
    /**
     * The scopes of the upper level edge devices if applicable. Only available for edge devices.
     */
    parentScopes?: string[];
}
/**
 * The device registry operation error details.
 */
export interface DeviceRegistryOperationError {
    /**
     * The unique identifier of the device.
     */
    deviceId?: string;
    /**
     * The error code. Possible values include: 'InvalidErrorCode', 'GenericBadRequest',
     * 'InvalidProtocolVersion', 'DeviceInvalidResultCount', 'InvalidOperation', 'ArgumentInvalid',
     * 'ArgumentNull', 'IotHubFormatError', 'DeviceStorageEntitySerializationError',
     * 'BlobContainerValidationError', 'ImportWarningExistsError', 'InvalidSchemaVersion',
     * 'DeviceDefinedMultipleTimes', 'DeserializationError', 'BulkRegistryOperationFailure',
     * 'DefaultStorageEndpointNotConfigured', 'InvalidFileUploadCorrelationId',
     * 'ExpiredFileUploadCorrelationId', 'InvalidStorageEndpoint', 'InvalidMessagingEndpoint',
     * 'InvalidFileUploadCompletionStatus', 'InvalidStorageEndpointOrBlob', 'RequestCanceled',
     * 'InvalidStorageEndpointProperty', 'EtagDoesNotMatch', 'RequestTimedOut',
     * 'UnsupportedOperationOnReplica', 'NullMessage', 'ConnectionForcefullyClosedOnNewConnection',
     * 'InvalidDeviceScope', 'ConnectionForcefullyClosedOnFaultInjection',
     * 'ConnectionRejectedOnFaultInjection', 'InvalidEndpointAuthenticationType',
     * 'ManagedIdentityNotEnabled', 'InvalidRouteTestInput', 'InvalidSourceOnRoute',
     * 'RoutingNotEnabled', 'InvalidContentEncodingOrType', 'InvalidEndorsementKey',
     * 'InvalidRegistrationId', 'InvalidStorageRootKey', 'InvalidEnrollmentGroupId',
     * 'TooManyEnrollments', 'RegistrationIdDefinedMultipleTimes', 'CustomAllocationFailed',
     * 'CustomAllocationIotHubNotSpecified', 'CustomAllocationUnauthorizedAccess',
     * 'CannotRegisterModuleToModule', 'TenantHubRoutingNotEnabled',
     * 'InvalidConfigurationTargetCondition', 'InvalidConfigurationContent',
     * 'CannotModifyImmutableConfigurationContent', 'InvalidConfigurationCustomMetricsQuery',
     * 'InvalidPnPInterfaceDefinition', 'InvalidPnPDesiredProperties',
     * 'InvalidPnPReportedProperties', 'InvalidPnPWritableReportedProperties',
     * 'InvalidDigitalTwinJsonPatch', 'InvalidDigitalTwinPayload', 'InvalidDigitalTwinPatch',
     * 'InvalidDigitalTwinPatchPath', 'GenericUnauthorized', 'IotHubNotFound',
     * 'IotHubUnauthorizedAccess', 'IotHubUnauthorized', 'ElasticPoolNotFound',
     * 'SystemModuleModifyUnauthorizedAccess', 'GenericForbidden', 'IotHubSuspended',
     * 'IotHubQuotaExceeded', 'JobQuotaExceeded', 'DeviceMaximumQueueDepthExceeded',
     * 'IotHubMaxCbsTokenExceeded', 'DeviceMaximumActiveFileUploadLimitExceeded',
     * 'DeviceMaximumQueueSizeExceeded', 'RoutingEndpointResponseForbidden',
     * 'InvalidMessageExpiryTime', 'OperationNotAvailableInCurrentTier', 'KeyEncryptionKeyRevoked',
     * 'DeviceModelMaxPropertiesExceeded', 'DeviceModelMaxIndexablePropertiesExceeded',
     * 'IotDpsSuspended', 'IotDpsSuspending', 'GenericNotFound', 'DeviceNotFound', 'JobNotFound',
     * 'QuotaMetricNotFound', 'SystemPropertyNotFound', 'AmqpAddressNotFound',
     * 'RoutingEndpointResponseNotFound', 'CertificateNotFound', 'ElasticPoolTenantHubNotFound',
     * 'ModuleNotFound', 'AzureTableStoreNotFound', 'IotHubFailingOver', 'FeatureNotSupported',
     * 'DigitalTwinInterfaceNotFound', 'QueryStoreClusterNotFound', 'DeviceNotOnline',
     * 'DeviceConnectionClosedRemotely', 'EnrollmentNotFound', 'DeviceRegistrationNotFound',
     * 'AsyncOperationNotFound', 'EnrollmentGroupNotFound', 'DeviceRecordNotFound',
     * 'GroupRecordNotFound', 'DeviceGroupNotFound', 'ProvisioningSettingsNotFound',
     * 'ProvisioningRecordNotFound', 'LinkedHubNotFound', 'CertificateAuthorityNotFound',
     * 'ConfigurationNotFound', 'GroupNotFound', 'DigitalTwinModelNotFound',
     * 'InterfaceNameModelNotFound', 'GenericMethodNotAllowed', 'OperationNotAllowedInCurrentState',
     * 'ImportDevicesNotSupported', 'BulkAddDevicesNotSupported', 'GenericConflict',
     * 'DeviceAlreadyExists', 'LinkCreationConflict', 'CallbackSubscriptionConflict',
     * 'ModelAlreadyExists', 'DeviceLocked', 'DeviceJobAlreadyExists', 'JobAlreadyExists',
     * 'EnrollmentConflict', 'EnrollmentGroupConflict', 'RegistrationStatusConflict',
     * 'DeviceRecordConflict', 'GroupRecordConflict', 'DeviceGroupConflict',
     * 'ProvisioningSettingsConflict', 'ProvisioningRecordConflict', 'LinkedHubConflict',
     * 'CertificateAuthorityConflict', 'ModuleAlreadyExistsOnDevice', 'ConfigurationAlreadyExists',
     * 'ApplyConfigurationAlreadyInProgressOnDevice', 'DigitalTwinModelAlreadyExists',
     * 'DigitalTwinModelExistsWithOtherModelType', 'InterfaceNameModelAlreadyExists',
     * 'GenericPreconditionFailed', 'PreconditionFailed', 'DeviceMessageLockLost',
     * 'JobRunPreconditionFailed', 'InflightMessagesInLink', 'GenericRequestEntityTooLarge',
     * 'MessageTooLarge', 'TooManyDevices', 'TooManyModulesOnDevice',
     * 'ConfigurationCountLimitExceeded', 'DigitalTwinModelCountLimitExceeded',
     * 'InterfaceNameCompressionModelCountLimitExceeded', 'GenericUnsupportedMediaType',
     * 'IncompatibleDataType', 'GenericTooManyRequests', 'ThrottlingException',
     * 'ThrottleBacklogLimitExceeded', 'ThrottlingBacklogTimeout',
     * 'ThrottlingMaxActiveJobCountExceeded', 'DeviceThrottlingLimitExceeded', 'ClientClosedRequest',
     * 'GenericServerError', 'ServerError', 'JobCancelled', 'StatisticsRetrievalError',
     * 'ConnectionForcefullyClosed', 'InvalidBlobState', 'BackupTimedOut', 'AzureStorageTimeout',
     * 'GenericTimeout', 'InvalidThrottleParameter', 'EventHubLinkAlreadyClosed',
     * 'ReliableBlobStoreError', 'RetryAttemptsExhausted', 'AzureTableStoreError',
     * 'CheckpointStoreNotFound', 'DocumentDbInvalidReturnValue', 'ReliableDocDbStoreStoreError',
     * 'ReliableBlobStoreTimeoutError', 'ConfigReadFailed', 'InvalidContainerReceiveLink',
     * 'InvalidPartitionEpoch', 'RestoreTimedOut', 'StreamReservationFailure', 'SerializationError',
     * 'UnexpectedPropertyValue', 'OrchestrationOperationFailed', 'ModelRepoEndpointError',
     * 'ResolutionError', 'UnableToFetchCredentials', 'UnableToFetchTenantInfo',
     * 'UnableToShareIdentity', 'UnableToExpandDiscoveryInfo', 'UnableToExpandComponentInfo',
     * 'UnableToCompressComponentInfo', 'UnableToCompressDiscoveryInfo', 'OrphanDiscoveryDocument',
     * 'GenericBadGateway', 'InvalidResponseWhileProxying', 'GenericServiceUnavailable',
     * 'ServiceUnavailable', 'PartitionNotFound', 'IotHubActivationFailed', 'ServerBusy',
     * 'IotHubRestoring', 'ReceiveLinkOpensThrottled', 'ConnectionUnavailable', 'DeviceUnavailable',
     * 'ConfigurationNotAvailable', 'GroupNotAvailable', 'HostingServiceNotAvailable',
     * 'GenericGatewayTimeout', 'GatewayTimeout'
     */
    errorCode?: ErrorCode;
    /**
     * The details of the error.
     */
    errorStatus?: string;
    /**
     * The unique identifier of the module, if applicable.
     */
    moduleId?: string;
    /**
     * The type of the operation that failed.
     */
    operation?: string;
}
/**
 * The device registry operation warning details.
 */
export interface DeviceRegistryOperationWarning {
    /**
     * The unique identifier of the device.
     */
    deviceId?: string;
    /**
     * The warning code. Possible values include: 'DeviceRegisteredWithoutTwin'
     */
    warningCode?: WarningCode;
    /**
     * The details of the warning.
     */
    warningStatus?: string;
}
/**
 * The result of the bulk registry operation.
 */
export interface BulkRegistryOperationResult {
    /**
     * The operation result.
     */
    isSuccessful?: boolean;
    /**
     * The device registry operation errors.
     */
    errors?: DeviceRegistryOperationError[];
    /**
     * The device registry operation warnings.
     */
    warnings?: DeviceRegistryOperationWarning[];
}
/**
 * The Json query request.
 */
export interface QuerySpecification {
    /**
     * The query string.
     */
    query?: string;
}
/**
 * The desired and reported properties of the twin. The maximum depth of the object is 10.
 */
export interface TwinProperties {
    /**
     * The collection of desired property key-value pairs. The keys are UTF-8 encoded, case-sensitive
     * and up-to 1KB in length. Allowed characters exclude UNICODE control characters (segments C0
     * and C1), '.', '$' and space. The desired porperty values are JSON objects, up-to 4KB in
     * length.
     */
    desired?: {
        [propertyName: string]: any;
    };
    /**
     * The collection of reported property key-value pairs. The keys are UTF-8 encoded,
     * case-sensitive and up-to 1KB in length. Allowed characters exclude UNICODE control characters
     * (segments C0 and C1), '.', '$' and space. The reported property values are JSON objects, up-to
     * 4KB in length.
     */
    reported?: {
        [propertyName: string]: any;
    };
}
/**
 * The state information for a device or module. This is implicitly created and deleted when the
 * corresponding device/ module identity is created or deleted in the IoT Hub.
 */
export interface Twin {
    /**
     * The unique identifier of the device in the identity registry of the IoT Hub. It is a
     * case-sensitive string (up to 128 char long) of ASCII 7-bit alphanumeric chars, and the
     * following special characters {'-', ':', '.', '+', '%', '_', '#', '*', '?', '!', '(', ')', ',',
     * '=', '@', ';', '$', '''}.
     */
    deviceId?: string;
    /**
     * The unique identifier of the module in the identity registry of the IoT Hub. It is a
     * case-sensitive string (up to 128 char long) of ASCII 7-bit alphanumeric chars, and the
     * following special characters {'-', ':', '.', '+', '%', '_', '#', '*', '?', '!', '(', ')', ',',
     * '=', '@', ';', '$', '''}.
     */
    moduleId?: string;
    /**
     * The collection of key-value pairs read and written by the solution back end. They are not
     * visible to device apps. They keys are UTF-8 encoded, case-sensitive and up-to 1KB in length.
     * Allowed characters exclude UNICODE control characters (segments C0 and C1), '.', '$' and
     * space. The values are JSON objects, up-to 4KB in length.
     */
    tags?: {
        [propertyName: string]: any;
    };
    /**
     * The desired and reported properties of the twin.
     */
    properties?: TwinProperties;
    /**
     * The string representing a ETag for the device twin, as per RFC7232.
     */
    etag?: string;
    /**
     * The version for the device twin including tags and desired properties
     */
    version?: number;
    /**
     * The string representing a ETag for the device, as per RFC7232.
     */
    deviceEtag?: string;
    /**
     * The enabled status of the device. If disabled, the device cannot connect to the service.
     * Possible values include: 'enabled', 'disabled'
     */
    status?: Status2;
    /**
     * The reason for the current status of the device, if any.
     */
    statusReason?: string;
    /**
     * The date and time when the status of the device was last updated.
     */
    statusUpdateTime?: Date;
    /**
     * The connection state of the device. Possible values include: 'Disconnected', 'Connected'
     */
    connectionState?: ConnectionState1;
    /**
     * The date and time when the device last connected or received or sent a message. The date and
     * time is sepecified in ISO8601 datetime format in UTC, for example, 2015-01-28T16:24:48.789Z.
     * This value is not updated if the device uses the HTTP/1 protocol to perform messaging
     * operations.
     */
    lastActivityTime?: Date;
    /**
     * The number of cloud-to-device messages sent.
     */
    cloudToDeviceMessageCount?: number;
    /**
     * The authentication type used by the device. Possible values include: 'sas', 'selfSigned',
     * 'certificateAuthority', 'none'
     */
    authenticationType?: AuthenticationType;
    /**
     * The X509 thumbprint of the device.
     */
    x509Thumbprint?: X509Thumbprint;
    capabilities?: DeviceCapabilities;
    /**
     * The scope of the device.
     */
    deviceScope?: string;
    /**
     * The scopes of the upper level edge devices if applicable. Only available for edge devices.
     */
    parentScopes?: string[];
}
/**
 * An interface representing JobProperties.
 */
export interface JobProperties {
    /**
     * The unique identifier of the job.
     */
    jobId?: string;
    /**
     * System generated. Ignored at creation. The start date and time of the job in UTC.
     */
    startTimeUtc?: Date;
    /**
     * System generated. Ignored at creation. The end date and time of the job in UTC.
     */
    endTimeUtc?: Date;
    /**
     * The job type. Possible values include: 'unknown', 'export', 'import', 'backup',
     * 'readDeviceProperties', 'writeDeviceProperties', 'updateDeviceConfiguration', 'rebootDevice',
     * 'factoryResetDevice', 'firmwareUpdate', 'scheduleDeviceMethod', 'scheduleUpdateTwin',
     * 'restoreFromBackup', 'failoverDataCopy'
     */
    type?: Type1;
    /**
     * System generated. Ignored at creation. The status of the job. Possible values include:
     * 'unknown', 'enqueued', 'running', 'completed', 'failed', 'cancelled', 'scheduled', 'queued'
     */
    status?: Status3;
    /**
     * System generated. Ignored at creation. The percentage of job completion.
     */
    progress?: number;
    /**
     * The URI containing SAS token to a blob container that contains registry data to sync.
     */
    inputBlobContainerUri?: string;
    /**
     * The blob name to use when importing from the input blob container.
     */
    inputBlobName?: string;
    /**
     * The SAS token to access the blob container. This is used to output the status and results of
     * the job.
     */
    outputBlobContainerUri?: string;
    /**
     * The blob name that will be created in the output blob container. This blob will contain the
     * exported device registry information for the IoT Hub.
     */
    outputBlobName?: string;
    /**
     * Optional for export jobs; ignored for other jobs. If not specified, the service defaults to
     * false. If false, authorization keys are included in export output. Keys are exported as null
     * otherwise.
     */
    excludeKeysInExport?: boolean;
    /**
     * The authentication type used for connecting to the storage account. Possible values include:
     * 'keyBased', 'identityBased'
     */
    storageAuthenticationType?: StorageAuthenticationType;
    /**
     * System genereated.  Ignored at creation. The reason for failure, if a failure occurred.
     */
    failureReason?: string;
    /**
     * Defaults to false. If true, then configurations are included in the data export/import.
     */
    includeConfigurations?: boolean;
    /**
     * Defaults to configurations.txt. Specifies the name of the blob to use when exporting/importing
     * configurations.
     */
    configurationsBlobName?: string;
}
/**
 * The result of a device message queue purge operation.
 */
export interface PurgeMessageQueueResult {
    totalMessagesPurged?: number;
    /**
     * The unique identifier of the device.
     */
    deviceId?: string;
    /**
     * The unique identifier of the module.
     */
    moduleId?: string;
}
/**
 * The parameters to execute a direct method on the device.
 */
export interface CloudToDeviceMethod {
    /**
     * The name of the method to execute.
     */
    methodName?: string;
    /**
     * The JSON-formatted direct method payload, up to 128kb in size.
     */
    payload?: any;
    responseTimeoutInSeconds?: number;
    connectTimeoutInSeconds?: number;
}
/**
 * An interface representing JobRequest.
 */
export interface JobRequest {
    /**
     * The unique identifier of the job.
     */
    jobId?: string;
    /**
     * The job type. Possible values include: 'unknown', 'export', 'import', 'backup',
     * 'readDeviceProperties', 'writeDeviceProperties', 'updateDeviceConfiguration', 'rebootDevice',
     * 'factoryResetDevice', 'firmwareUpdate', 'scheduleDeviceMethod', 'scheduleUpdateTwin',
     * 'restoreFromBackup', 'failoverDataCopy'
     */
    type?: Type2;
    /**
     * The method type and parameters. This is required if the job type is cloudToDeviceMethod.
     */
    cloudToDeviceMethod?: CloudToDeviceMethod;
    updateTwin?: Twin;
    /**
     * The condition for devices to execute the job. This is required if the job type is updateTwin
     * or cloudToDeviceMethod.
     */
    queryCondition?: string;
    /**
     * The start date and time of the job in ISO 8601 standard.
     */
    startTime?: Date;
    /**
     * The maximum execution time in secounds.
     */
    maxExecutionTimeInSeconds?: number;
}
/**
 * The job statistics regarding execution status.
 */
export interface DeviceJobStatistics {
    /**
     * The number of devices targeted by the job.
     */
    deviceCount?: number;
    /**
     * The number of failed jobs.
     */
    failedCount?: number;
    /**
     * The number of succeeded jobs.
     */
    succeededCount?: number;
    /**
     * The number of running jobs.
     */
    runningCount?: number;
    /**
     * The number of pending (scheduled) jobs.
     */
    pendingCount?: number;
}
/**
 * An interface representing JobResponse.
 */
export interface JobResponse {
    /**
     * System generated.  Ignored at creation. The unique identifier of the job.
     */
    jobId?: string;
    /**
     * The device query condition.
     */
    queryCondition?: string;
    /**
     * System generated.  Ignored at creation. The creation date and time of the job.
     */
    createdTime?: Date;
    /**
     * The start date and time of the scheduled job in UTC.
     */
    startTime?: Date;
    /**
     * System generated.  Ignored at creation. The end date and time of the job in UTC.
     */
    endTime?: Date;
    /**
     * The maximum execution time in secounds.
     */
    maxExecutionTimeInSeconds?: number;
    /**
     * The job type. Possible values include: 'unknown', 'export', 'import', 'backup',
     * 'readDeviceProperties', 'writeDeviceProperties', 'updateDeviceConfiguration', 'rebootDevice',
     * 'factoryResetDevice', 'firmwareUpdate', 'scheduleDeviceMethod', 'scheduleUpdateTwin',
     * 'restoreFromBackup', 'failoverDataCopy'
     */
    type?: Type3;
    /**
     * The method type and parameters. This is required if job type is cloudToDeviceMethod.
     */
    cloudToDeviceMethod?: CloudToDeviceMethod;
    updateTwin?: Twin;
    /**
     * System generated.  Ignored at creation. The status of the job. Possible values include:
     * 'unknown', 'enqueued', 'running', 'completed', 'failed', 'cancelled', 'scheduled', 'queued'
     */
    status?: Status4;
    /**
     * The reason for the failure, if a failure occurred.
     */
    failureReason?: string;
    /**
     * The status message of the job.
     */
    statusMessage?: string;
    /**
     * The details regarding job execution status.
     */
    deviceJobStatistics?: DeviceJobStatistics;
}
/**
 * The query result.
 */
export interface QueryResult {
    /**
     * The query result type. Possible values include: 'unknown', 'twin', 'deviceJob', 'jobResponse',
     * 'raw', 'enrollment', 'enrollmentGroup', 'deviceRegistration'
     */
    type?: Type4;
    /**
     * The query result items, as a collection.
     */
    items?: any[];
    /**
     * The continuation token.
     */
    continuationToken?: string;
}
/**
 * The module identity on a device.
 */
export interface Module {
    /**
     * The unique identifier of the module.
     */
    moduleId?: string;
    /**
     * Identifies who manages this module. For instance, this value is \"IotEdge\" if the edge
     * runtime owns this module.
     */
    managedBy?: string;
    /**
     * The unique identifier of the device.
     */
    deviceId?: string;
    /**
     * The IoT Hub generated, case-sensitive string up to 128 characters long. This value is used to
     * distinguish modules with the same moduleId, when they have been deleted and re-created.
     */
    generationId?: string;
    /**
     * The string representing a weak ETag for the module identity, as per RFC7232.
     */
    etag?: string;
    /**
     * The connection state of the device. Possible values include: 'Disconnected', 'Connected'
     */
    connectionState?: ConnectionState2;
    /**
     * The date and time the connection state was last updated.
     */
    connectionStateUpdatedTime?: Date;
    /**
     * The date and time the device last connected, received, or sent a message.
     */
    lastActivityTime?: Date;
    /**
     * The number of cloud-to-module messages currently queued to be sent to the module.
     */
    cloudToDeviceMessageCount?: number;
    /**
     * The authentication mechanism used by the module when connecting to the service and edge hub.
     */
    authentication?: AuthenticationMechanism;
}
/**
 * Represents the Device Method Invocation Results.
 */
export interface CloudToDeviceMethodResult {
    /**
     * Method invocation result status.
     */
    status?: number;
    /**
     * Method invocation result payload.
     */
    payload?: any;
}
/**
 * An interface representing IotHubGatewayServiceAPIsOptions.
 */
export interface IotHubGatewayServiceAPIsOptions extends ServiceClientOptions {
    /**
     * Version of the Api. Default value: '2020-09-30'.
     */
    apiVersion?: string;
    baseUri?: string;
}
/**
 * Optional Parameters.
 */
export interface ConfigurationCreateOrUpdateOptionalParams extends msRest.RequestOptionsBase {
    /**
     * The string representing a weak ETag for the configuration, as per RFC7232. This should not be
     * set when creating a configuration, but may be set when updating a configuration.
     */
    ifMatch?: string;
}
/**
 * Optional Parameters.
 */
export interface ConfigurationDeleteMethodOptionalParams extends msRest.RequestOptionsBase {
    /**
     * The string representing a weak ETag for the configuration, as per RFC7232. The delete
     * operation is performed only if this ETag matches the value maintained by the server,
     * indicating that the configuration has not been modified since it was last retrieved. To force
     * an unconditional delete, set If-Match to the wildcard character (*).
     */
    ifMatch?: string;
}
/**
 * Optional Parameters.
 */
export interface ConfigurationGetConfigurationsOptionalParams extends msRest.RequestOptionsBase {
    /**
     * The number of configurations to retrieve. Value will be overridden if greater than the maximum
     * deployment count for the IoT Hub.
     */
    top?: number;
}
/**
 * Optional Parameters.
 */
export interface DevicesGetDevicesOptionalParams extends msRest.RequestOptionsBase {
    /**
     * The maximum number of device identities returned by the query. Any value outside the range of
     * 1-1000 is considered to be 1000.
     */
    top?: number;
}
/**
 * Optional Parameters.
 */
export interface DevicesCreateOrUpdateIdentityOptionalParams extends msRest.RequestOptionsBase {
    /**
     * The string representing a weak ETag for the device identity, as per RFC7232. This should not
     * be set when creating a device, but may be set when updating a device.
     */
    ifMatch?: string;
}
/**
 * Optional Parameters.
 */
export interface DevicesDeleteIdentityOptionalParams extends msRest.RequestOptionsBase {
    /**
     * The string representing a weak ETag for the device identity, as per RFC7232. The delete
     * operation is performed only if this ETag matches the value maintained by the server,
     * indicating that the device identity has not been modified since it was last retrieved. To
     * force an unconditional delete, set If-Match to the wildcard character (*).
     */
    ifMatch?: string;
}
/**
 * Optional Parameters.
 */
export interface DevicesReplaceTwinOptionalParams extends msRest.RequestOptionsBase {
    /**
     * The string representing a weak ETag for the device twin, as per RFC7232. It determines if the
     * replace operation should be carried out.
     */
    ifMatch?: string;
}
/**
 * Optional Parameters.
 */
export interface DevicesUpdateTwinOptionalParams extends msRest.RequestOptionsBase {
    /**
     * The string representing a weak ETag for the device twin, as per RFC7232. It determines if the
     * update operation should be carried out.
     */
    ifMatch?: string;
}
/**
 * Optional Parameters.
 */
export interface QueryGetTwinsOptionalParams extends msRest.RequestOptionsBase {
    /**
     * The continuation token used to get the next page of results.
     */
    xMsContinuation?: string;
    /**
     * The maximum number of items returned per page. The service may use a different value if the
     * value specified is not acceptable.
     */
    xMsMaxItemCount?: string;
}
/**
 * Optional Parameters.
 */
export interface JobsQueryScheduledJobsOptionalParams extends msRest.RequestOptionsBase {
    /**
     * The job type. See
     * https://docs.microsoft.com/en-us/azure/iot-hub/iot-hub-devguide-jobs#querying-for-progress-on-jobs
     * for a list of possible job types.
     */
    jobType?: string;
    /**
     * The job status. See
     * https://docs.microsoft.com/en-us/azure/iot-hub/iot-hub-devguide-jobs#querying-for-progress-on-jobs
     * for a list of possible statuses.
     */
    jobStatus?: string;
}
/**
 * Optional Parameters.
 */
export interface ModulesReplaceTwinOptionalParams extends msRest.RequestOptionsBase {
    /**
     * The string representing a weak ETag for the device twin, as per RFC7232. It determines if the
     * replace operation should be carried out.
     */
    ifMatch?: string;
}
/**
 * Optional Parameters.
 */
export interface ModulesUpdateTwinOptionalParams extends msRest.RequestOptionsBase {
    /**
     * The string representing a weak ETag for the device twin, as per RFC7232. It determines if the
     * update operation should be carried out.
     */
    ifMatch?: string;
}
/**
 * Optional Parameters.
 */
export interface ModulesCreateOrUpdateIdentityOptionalParams extends msRest.RequestOptionsBase {
    /**
     * The string representing a weak ETag for the module, as per RFC7232. This should not be set
     * when creating a module, but may be set when updating a module.
     */
    ifMatch?: string;
}
/**
 * Optional Parameters.
 */
export interface ModulesDeleteIdentityOptionalParams extends msRest.RequestOptionsBase {
    /**
     * The string representing a weak ETag for the module, as per RFC7232. The delete operation is
     * performed only if this ETag matches the value maintained by the server, indicating that the
     * module has not been modified since it was last retrieved. To force an unconditional delete,
     * set If-Match to the wildcard character (*).
     */
    ifMatch?: string;
}
/**
 * Optional Parameters.
 */
export interface DigitalTwinUpdateDigitalTwinOptionalParams extends msRest.RequestOptionsBase {
    ifMatch?: string;
}
/**
 * Optional Parameters.
 */
export interface DigitalTwinInvokeRootLevelCommandOptionalParams extends msRest.RequestOptionsBase {
    /**
     * Maximum interval of time, in seconds, that the digital twin command will wait for the answer.
     */
    connectTimeoutInSeconds?: number;
    /**
     * Maximum interval of time, in seconds, that the digital twin command will wait for the answer.
     */
    responseTimeoutInSeconds?: number;
}
/**
 * Optional Parameters.
 */
export interface DigitalTwinInvokeComponentCommandOptionalParams extends msRest.RequestOptionsBase {
    /**
     * Maximum interval of time, in seconds, that the digital twin command will wait for the answer.
     */
    connectTimeoutInSeconds?: number;
    /**
     * Maximum interval of time, in seconds, that the digital twin command will wait for the answer.
     */
    responseTimeoutInSeconds?: number;
}
/**
 * Defines headers for GetTwins operation.
 */
export interface QueryGetTwinsHeaders {
    /**
     * The type of list of items.
     */
    xMsItemType: string;
    /**
     * The continuation token used to get the next page of results.
     */
    xMsContinuation: string;
}
/**
 * Defines headers for GetDigitalTwin operation.
 */
export interface DigitalTwinGetDigitalTwinHeaders {
    /**
     * Weak Etag
     */
    eTag: string;
}
/**
 * Defines headers for UpdateDigitalTwin operation.
 */
export interface DigitalTwinUpdateDigitalTwinHeaders {
    /**
     * Weak Etag of the modified resource
     */
    eTag: string;
    /**
     * URI of the digital twin
     */
    location: string;
}
/**
 * Defines headers for InvokeRootLevelCommand operation.
 */
export interface DigitalTwinInvokeRootLevelCommandHeaders {
    /**
     * Device Generated Status Code for this Operation
     */
    xMsCommandStatuscode: number;
    /**
     * Server Generated Request Id (GUID), to uniquely identify this request in the service
     */
    xMsRequestId: string;
}
/**
 * Defines headers for InvokeComponentCommand operation.
 */
export interface DigitalTwinInvokeComponentCommandHeaders {
    /**
     * Device Generated Status Code for this Operation
     */
    xMsCommandStatuscode: number;
    /**
     * Server Generated Request Id (GUID), to uniquely identify this request in the service
     */
    xMsRequestId: string;
}
/**
 * Defines values for Type.
 * Possible values include: 'sas', 'selfSigned', 'certificateAuthority', 'none'
 * @readonly
 * @enum {string}
 */
export declare type Type = 'sas' | 'selfSigned' | 'certificateAuthority' | 'none';
/**
 * Defines values for ConnectionState.
 * Possible values include: 'Disconnected', 'Connected'
 * @readonly
 * @enum {string}
 */
export declare type ConnectionState = 'Disconnected' | 'Connected';
/**
 * Defines values for Status.
 * Possible values include: 'enabled', 'disabled'
 * @readonly
 * @enum {string}
 */
export declare type Status = 'enabled' | 'disabled';
/**
 * Defines values for ImportMode.
 * Possible values include: 'create', 'update', 'updateIfMatchETag', 'delete', 'deleteIfMatchETag',
 * 'updateTwin', 'updateTwinIfMatchETag'
 * @readonly
 * @enum {string}
 */
export declare type ImportMode = 'create' | 'update' | 'updateIfMatchETag' | 'delete' | 'deleteIfMatchETag' | 'updateTwin' | 'updateTwinIfMatchETag';
/**
 * Defines values for Status1.
 * Possible values include: 'enabled', 'disabled'
 * @readonly
 * @enum {string}
 */
export declare type Status1 = 'enabled' | 'disabled';
/**
 * Defines values for ErrorCode.
 * Possible values include: 'InvalidErrorCode', 'GenericBadRequest', 'InvalidProtocolVersion',
 * 'DeviceInvalidResultCount', 'InvalidOperation', 'ArgumentInvalid', 'ArgumentNull',
 * 'IotHubFormatError', 'DeviceStorageEntitySerializationError', 'BlobContainerValidationError',
 * 'ImportWarningExistsError', 'InvalidSchemaVersion', 'DeviceDefinedMultipleTimes',
 * 'DeserializationError', 'BulkRegistryOperationFailure', 'DefaultStorageEndpointNotConfigured',
 * 'InvalidFileUploadCorrelationId', 'ExpiredFileUploadCorrelationId', 'InvalidStorageEndpoint',
 * 'InvalidMessagingEndpoint', 'InvalidFileUploadCompletionStatus', 'InvalidStorageEndpointOrBlob',
 * 'RequestCanceled', 'InvalidStorageEndpointProperty', 'EtagDoesNotMatch', 'RequestTimedOut',
 * 'UnsupportedOperationOnReplica', 'NullMessage', 'ConnectionForcefullyClosedOnNewConnection',
 * 'InvalidDeviceScope', 'ConnectionForcefullyClosedOnFaultInjection',
 * 'ConnectionRejectedOnFaultInjection', 'InvalidEndpointAuthenticationType',
 * 'ManagedIdentityNotEnabled', 'InvalidRouteTestInput', 'InvalidSourceOnRoute',
 * 'RoutingNotEnabled', 'InvalidContentEncodingOrType', 'InvalidEndorsementKey',
 * 'InvalidRegistrationId', 'InvalidStorageRootKey', 'InvalidEnrollmentGroupId',
 * 'TooManyEnrollments', 'RegistrationIdDefinedMultipleTimes', 'CustomAllocationFailed',
 * 'CustomAllocationIotHubNotSpecified', 'CustomAllocationUnauthorizedAccess',
 * 'CannotRegisterModuleToModule', 'TenantHubRoutingNotEnabled',
 * 'InvalidConfigurationTargetCondition', 'InvalidConfigurationContent',
 * 'CannotModifyImmutableConfigurationContent', 'InvalidConfigurationCustomMetricsQuery',
 * 'InvalidPnPInterfaceDefinition', 'InvalidPnPDesiredProperties', 'InvalidPnPReportedProperties',
 * 'InvalidPnPWritableReportedProperties', 'InvalidDigitalTwinJsonPatch',
 * 'InvalidDigitalTwinPayload', 'InvalidDigitalTwinPatch', 'InvalidDigitalTwinPatchPath',
 * 'GenericUnauthorized', 'IotHubNotFound', 'IotHubUnauthorizedAccess', 'IotHubUnauthorized',
 * 'ElasticPoolNotFound', 'SystemModuleModifyUnauthorizedAccess', 'GenericForbidden',
 * 'IotHubSuspended', 'IotHubQuotaExceeded', 'JobQuotaExceeded', 'DeviceMaximumQueueDepthExceeded',
 * 'IotHubMaxCbsTokenExceeded', 'DeviceMaximumActiveFileUploadLimitExceeded',
 * 'DeviceMaximumQueueSizeExceeded', 'RoutingEndpointResponseForbidden',
 * 'InvalidMessageExpiryTime', 'OperationNotAvailableInCurrentTier', 'KeyEncryptionKeyRevoked',
 * 'DeviceModelMaxPropertiesExceeded', 'DeviceModelMaxIndexablePropertiesExceeded',
 * 'IotDpsSuspended', 'IotDpsSuspending', 'GenericNotFound', 'DeviceNotFound', 'JobNotFound',
 * 'QuotaMetricNotFound', 'SystemPropertyNotFound', 'AmqpAddressNotFound',
 * 'RoutingEndpointResponseNotFound', 'CertificateNotFound', 'ElasticPoolTenantHubNotFound',
 * 'ModuleNotFound', 'AzureTableStoreNotFound', 'IotHubFailingOver', 'FeatureNotSupported',
 * 'DigitalTwinInterfaceNotFound', 'QueryStoreClusterNotFound', 'DeviceNotOnline',
 * 'DeviceConnectionClosedRemotely', 'EnrollmentNotFound', 'DeviceRegistrationNotFound',
 * 'AsyncOperationNotFound', 'EnrollmentGroupNotFound', 'DeviceRecordNotFound',
 * 'GroupRecordNotFound', 'DeviceGroupNotFound', 'ProvisioningSettingsNotFound',
 * 'ProvisioningRecordNotFound', 'LinkedHubNotFound', 'CertificateAuthorityNotFound',
 * 'ConfigurationNotFound', 'GroupNotFound', 'DigitalTwinModelNotFound',
 * 'InterfaceNameModelNotFound', 'GenericMethodNotAllowed', 'OperationNotAllowedInCurrentState',
 * 'ImportDevicesNotSupported', 'BulkAddDevicesNotSupported', 'GenericConflict',
 * 'DeviceAlreadyExists', 'LinkCreationConflict', 'CallbackSubscriptionConflict',
 * 'ModelAlreadyExists', 'DeviceLocked', 'DeviceJobAlreadyExists', 'JobAlreadyExists',
 * 'EnrollmentConflict', 'EnrollmentGroupConflict', 'RegistrationStatusConflict',
 * 'DeviceRecordConflict', 'GroupRecordConflict', 'DeviceGroupConflict',
 * 'ProvisioningSettingsConflict', 'ProvisioningRecordConflict', 'LinkedHubConflict',
 * 'CertificateAuthorityConflict', 'ModuleAlreadyExistsOnDevice', 'ConfigurationAlreadyExists',
 * 'ApplyConfigurationAlreadyInProgressOnDevice', 'DigitalTwinModelAlreadyExists',
 * 'DigitalTwinModelExistsWithOtherModelType', 'InterfaceNameModelAlreadyExists',
 * 'GenericPreconditionFailed', 'PreconditionFailed', 'DeviceMessageLockLost',
 * 'JobRunPreconditionFailed', 'InflightMessagesInLink', 'GenericRequestEntityTooLarge',
 * 'MessageTooLarge', 'TooManyDevices', 'TooManyModulesOnDevice',
 * 'ConfigurationCountLimitExceeded', 'DigitalTwinModelCountLimitExceeded',
 * 'InterfaceNameCompressionModelCountLimitExceeded', 'GenericUnsupportedMediaType',
 * 'IncompatibleDataType', 'GenericTooManyRequests', 'ThrottlingException',
 * 'ThrottleBacklogLimitExceeded', 'ThrottlingBacklogTimeout',
 * 'ThrottlingMaxActiveJobCountExceeded', 'DeviceThrottlingLimitExceeded', 'ClientClosedRequest',
 * 'GenericServerError', 'ServerError', 'JobCancelled', 'StatisticsRetrievalError',
 * 'ConnectionForcefullyClosed', 'InvalidBlobState', 'BackupTimedOut', 'AzureStorageTimeout',
 * 'GenericTimeout', 'InvalidThrottleParameter', 'EventHubLinkAlreadyClosed',
 * 'ReliableBlobStoreError', 'RetryAttemptsExhausted', 'AzureTableStoreError',
 * 'CheckpointStoreNotFound', 'DocumentDbInvalidReturnValue', 'ReliableDocDbStoreStoreError',
 * 'ReliableBlobStoreTimeoutError', 'ConfigReadFailed', 'InvalidContainerReceiveLink',
 * 'InvalidPartitionEpoch', 'RestoreTimedOut', 'StreamReservationFailure', 'SerializationError',
 * 'UnexpectedPropertyValue', 'OrchestrationOperationFailed', 'ModelRepoEndpointError',
 * 'ResolutionError', 'UnableToFetchCredentials', 'UnableToFetchTenantInfo',
 * 'UnableToShareIdentity', 'UnableToExpandDiscoveryInfo', 'UnableToExpandComponentInfo',
 * 'UnableToCompressComponentInfo', 'UnableToCompressDiscoveryInfo', 'OrphanDiscoveryDocument',
 * 'GenericBadGateway', 'InvalidResponseWhileProxying', 'GenericServiceUnavailable',
 * 'ServiceUnavailable', 'PartitionNotFound', 'IotHubActivationFailed', 'ServerBusy',
 * 'IotHubRestoring', 'ReceiveLinkOpensThrottled', 'ConnectionUnavailable', 'DeviceUnavailable',
 * 'ConfigurationNotAvailable', 'GroupNotAvailable', 'HostingServiceNotAvailable',
 * 'GenericGatewayTimeout', 'GatewayTimeout'
 * @readonly
 * @enum {string}
 */
export declare type ErrorCode = 'InvalidErrorCode' | 'GenericBadRequest' | 'InvalidProtocolVersion' | 'DeviceInvalidResultCount' | 'InvalidOperation' | 'ArgumentInvalid' | 'ArgumentNull' | 'IotHubFormatError' | 'DeviceStorageEntitySerializationError' | 'BlobContainerValidationError' | 'ImportWarningExistsError' | 'InvalidSchemaVersion' | 'DeviceDefinedMultipleTimes' | 'DeserializationError' | 'BulkRegistryOperationFailure' | 'DefaultStorageEndpointNotConfigured' | 'InvalidFileUploadCorrelationId' | 'ExpiredFileUploadCorrelationId' | 'InvalidStorageEndpoint' | 'InvalidMessagingEndpoint' | 'InvalidFileUploadCompletionStatus' | 'InvalidStorageEndpointOrBlob' | 'RequestCanceled' | 'InvalidStorageEndpointProperty' | 'EtagDoesNotMatch' | 'RequestTimedOut' | 'UnsupportedOperationOnReplica' | 'NullMessage' | 'ConnectionForcefullyClosedOnNewConnection' | 'InvalidDeviceScope' | 'ConnectionForcefullyClosedOnFaultInjection' | 'ConnectionRejectedOnFaultInjection' | 'InvalidEndpointAuthenticationType' | 'ManagedIdentityNotEnabled' | 'InvalidRouteTestInput' | 'InvalidSourceOnRoute' | 'RoutingNotEnabled' | 'InvalidContentEncodingOrType' | 'InvalidEndorsementKey' | 'InvalidRegistrationId' | 'InvalidStorageRootKey' | 'InvalidEnrollmentGroupId' | 'TooManyEnrollments' | 'RegistrationIdDefinedMultipleTimes' | 'CustomAllocationFailed' | 'CustomAllocationIotHubNotSpecified' | 'CustomAllocationUnauthorizedAccess' | 'CannotRegisterModuleToModule' | 'TenantHubRoutingNotEnabled' | 'InvalidConfigurationTargetCondition' | 'InvalidConfigurationContent' | 'CannotModifyImmutableConfigurationContent' | 'InvalidConfigurationCustomMetricsQuery' | 'InvalidPnPInterfaceDefinition' | 'InvalidPnPDesiredProperties' | 'InvalidPnPReportedProperties' | 'InvalidPnPWritableReportedProperties' | 'InvalidDigitalTwinJsonPatch' | 'InvalidDigitalTwinPayload' | 'InvalidDigitalTwinPatch' | 'InvalidDigitalTwinPatchPath' | 'GenericUnauthorized' | 'IotHubNotFound' | 'IotHubUnauthorizedAccess' | 'IotHubUnauthorized' | 'ElasticPoolNotFound' | 'SystemModuleModifyUnauthorizedAccess' | 'GenericForbidden' | 'IotHubSuspended' | 'IotHubQuotaExceeded' | 'JobQuotaExceeded' | 'DeviceMaximumQueueDepthExceeded' | 'IotHubMaxCbsTokenExceeded' | 'DeviceMaximumActiveFileUploadLimitExceeded' | 'DeviceMaximumQueueSizeExceeded' | 'RoutingEndpointResponseForbidden' | 'InvalidMessageExpiryTime' | 'OperationNotAvailableInCurrentTier' | 'KeyEncryptionKeyRevoked' | 'DeviceModelMaxPropertiesExceeded' | 'DeviceModelMaxIndexablePropertiesExceeded' | 'IotDpsSuspended' | 'IotDpsSuspending' | 'GenericNotFound' | 'DeviceNotFound' | 'JobNotFound' | 'QuotaMetricNotFound' | 'SystemPropertyNotFound' | 'AmqpAddressNotFound' | 'RoutingEndpointResponseNotFound' | 'CertificateNotFound' | 'ElasticPoolTenantHubNotFound' | 'ModuleNotFound' | 'AzureTableStoreNotFound' | 'IotHubFailingOver' | 'FeatureNotSupported' | 'DigitalTwinInterfaceNotFound' | 'QueryStoreClusterNotFound' | 'DeviceNotOnline' | 'DeviceConnectionClosedRemotely' | 'EnrollmentNotFound' | 'DeviceRegistrationNotFound' | 'AsyncOperationNotFound' | 'EnrollmentGroupNotFound' | 'DeviceRecordNotFound' | 'GroupRecordNotFound' | 'DeviceGroupNotFound' | 'ProvisioningSettingsNotFound' | 'ProvisioningRecordNotFound' | 'LinkedHubNotFound' | 'CertificateAuthorityNotFound' | 'ConfigurationNotFound' | 'GroupNotFound' | 'DigitalTwinModelNotFound' | 'InterfaceNameModelNotFound' | 'GenericMethodNotAllowed' | 'OperationNotAllowedInCurrentState' | 'ImportDevicesNotSupported' | 'BulkAddDevicesNotSupported' | 'GenericConflict' | 'DeviceAlreadyExists' | 'LinkCreationConflict' | 'CallbackSubscriptionConflict' | 'ModelAlreadyExists' | 'DeviceLocked' | 'DeviceJobAlreadyExists' | 'JobAlreadyExists' | 'EnrollmentConflict' | 'EnrollmentGroupConflict' | 'RegistrationStatusConflict' | 'DeviceRecordConflict' | 'GroupRecordConflict' | 'DeviceGroupConflict' | 'ProvisioningSettingsConflict' | 'ProvisioningRecordConflict' | 'LinkedHubConflict' | 'CertificateAuthorityConflict' | 'ModuleAlreadyExistsOnDevice' | 'ConfigurationAlreadyExists' | 'ApplyConfigurationAlreadyInProgressOnDevice' | 'DigitalTwinModelAlreadyExists' | 'DigitalTwinModelExistsWithOtherModelType' | 'InterfaceNameModelAlreadyExists' | 'GenericPreconditionFailed' | 'PreconditionFailed' | 'DeviceMessageLockLost' | 'JobRunPreconditionFailed' | 'InflightMessagesInLink' | 'GenericRequestEntityTooLarge' | 'MessageTooLarge' | 'TooManyDevices' | 'TooManyModulesOnDevice' | 'ConfigurationCountLimitExceeded' | 'DigitalTwinModelCountLimitExceeded' | 'InterfaceNameCompressionModelCountLimitExceeded' | 'GenericUnsupportedMediaType' | 'IncompatibleDataType' | 'GenericTooManyRequests' | 'ThrottlingException' | 'ThrottleBacklogLimitExceeded' | 'ThrottlingBacklogTimeout' | 'ThrottlingMaxActiveJobCountExceeded' | 'DeviceThrottlingLimitExceeded' | 'ClientClosedRequest' | 'GenericServerError' | 'ServerError' | 'JobCancelled' | 'StatisticsRetrievalError' | 'ConnectionForcefullyClosed' | 'InvalidBlobState' | 'BackupTimedOut' | 'AzureStorageTimeout' | 'GenericTimeout' | 'InvalidThrottleParameter' | 'EventHubLinkAlreadyClosed' | 'ReliableBlobStoreError' | 'RetryAttemptsExhausted' | 'AzureTableStoreError' | 'CheckpointStoreNotFound' | 'DocumentDbInvalidReturnValue' | 'ReliableDocDbStoreStoreError' | 'ReliableBlobStoreTimeoutError' | 'ConfigReadFailed' | 'InvalidContainerReceiveLink' | 'InvalidPartitionEpoch' | 'RestoreTimedOut' | 'StreamReservationFailure' | 'SerializationError' | 'UnexpectedPropertyValue' | 'OrchestrationOperationFailed' | 'ModelRepoEndpointError' | 'ResolutionError' | 'UnableToFetchCredentials' | 'UnableToFetchTenantInfo' | 'UnableToShareIdentity' | 'UnableToExpandDiscoveryInfo' | 'UnableToExpandComponentInfo' | 'UnableToCompressComponentInfo' | 'UnableToCompressDiscoveryInfo' | 'OrphanDiscoveryDocument' | 'GenericBadGateway' | 'InvalidResponseWhileProxying' | 'GenericServiceUnavailable' | 'ServiceUnavailable' | 'PartitionNotFound' | 'IotHubActivationFailed' | 'ServerBusy' | 'IotHubRestoring' | 'ReceiveLinkOpensThrottled' | 'ConnectionUnavailable' | 'DeviceUnavailable' | 'ConfigurationNotAvailable' | 'GroupNotAvailable' | 'HostingServiceNotAvailable' | 'GenericGatewayTimeout' | 'GatewayTimeout';
/**
 * Defines values for WarningCode.
 * Possible values include: 'DeviceRegisteredWithoutTwin'
 * @readonly
 * @enum {string}
 */
export declare type WarningCode = 'DeviceRegisteredWithoutTwin';
/**
 * Defines values for Status2.
 * Possible values include: 'enabled', 'disabled'
 * @readonly
 * @enum {string}
 */
export declare type Status2 = 'enabled' | 'disabled';
/**
 * Defines values for ConnectionState1.
 * Possible values include: 'Disconnected', 'Connected'
 * @readonly
 * @enum {string}
 */
export declare type ConnectionState1 = 'Disconnected' | 'Connected';
/**
 * Defines values for AuthenticationType.
 * Possible values include: 'sas', 'selfSigned', 'certificateAuthority', 'none'
 * @readonly
 * @enum {string}
 */
export declare type AuthenticationType = 'sas' | 'selfSigned' | 'certificateAuthority' | 'none';
/**
 * Defines values for Type1.
 * Possible values include: 'unknown', 'export', 'import', 'backup', 'readDeviceProperties',
 * 'writeDeviceProperties', 'updateDeviceConfiguration', 'rebootDevice', 'factoryResetDevice',
 * 'firmwareUpdate', 'scheduleDeviceMethod', 'scheduleUpdateTwin', 'restoreFromBackup',
 * 'failoverDataCopy'
 * @readonly
 * @enum {string}
 */
export declare type Type1 = 'unknown' | 'export' | 'import' | 'backup' | 'readDeviceProperties' | 'writeDeviceProperties' | 'updateDeviceConfiguration' | 'rebootDevice' | 'factoryResetDevice' | 'firmwareUpdate' | 'scheduleDeviceMethod' | 'scheduleUpdateTwin' | 'restoreFromBackup' | 'failoverDataCopy';
/**
 * Defines values for Status3.
 * Possible values include: 'unknown', 'enqueued', 'running', 'completed', 'failed', 'cancelled',
 * 'scheduled', 'queued'
 * @readonly
 * @enum {string}
 */
export declare type Status3 = 'unknown' | 'enqueued' | 'running' | 'completed' | 'failed' | 'cancelled' | 'scheduled' | 'queued';
/**
 * Defines values for StorageAuthenticationType.
 * Possible values include: 'keyBased', 'identityBased'
 * @readonly
 * @enum {string}
 */
export declare type StorageAuthenticationType = 'keyBased' | 'identityBased';
/**
 * Defines values for Type2.
 * Possible values include: 'unknown', 'export', 'import', 'backup', 'readDeviceProperties',
 * 'writeDeviceProperties', 'updateDeviceConfiguration', 'rebootDevice', 'factoryResetDevice',
 * 'firmwareUpdate', 'scheduleDeviceMethod', 'scheduleUpdateTwin', 'restoreFromBackup',
 * 'failoverDataCopy'
 * @readonly
 * @enum {string}
 */
export declare type Type2 = 'unknown' | 'export' | 'import' | 'backup' | 'readDeviceProperties' | 'writeDeviceProperties' | 'updateDeviceConfiguration' | 'rebootDevice' | 'factoryResetDevice' | 'firmwareUpdate' | 'scheduleDeviceMethod' | 'scheduleUpdateTwin' | 'restoreFromBackup' | 'failoverDataCopy';
/**
 * Defines values for Type3.
 * Possible values include: 'unknown', 'export', 'import', 'backup', 'readDeviceProperties',
 * 'writeDeviceProperties', 'updateDeviceConfiguration', 'rebootDevice', 'factoryResetDevice',
 * 'firmwareUpdate', 'scheduleDeviceMethod', 'scheduleUpdateTwin', 'restoreFromBackup',
 * 'failoverDataCopy'
 * @readonly
 * @enum {string}
 */
export declare type Type3 = 'unknown' | 'export' | 'import' | 'backup' | 'readDeviceProperties' | 'writeDeviceProperties' | 'updateDeviceConfiguration' | 'rebootDevice' | 'factoryResetDevice' | 'firmwareUpdate' | 'scheduleDeviceMethod' | 'scheduleUpdateTwin' | 'restoreFromBackup' | 'failoverDataCopy';
/**
 * Defines values for Status4.
 * Possible values include: 'unknown', 'enqueued', 'running', 'completed', 'failed', 'cancelled',
 * 'scheduled', 'queued'
 * @readonly
 * @enum {string}
 */
export declare type Status4 = 'unknown' | 'enqueued' | 'running' | 'completed' | 'failed' | 'cancelled' | 'scheduled' | 'queued';
/**
 * Defines values for Type4.
 * Possible values include: 'unknown', 'twin', 'deviceJob', 'jobResponse', 'raw', 'enrollment',
 * 'enrollmentGroup', 'deviceRegistration'
 * @readonly
 * @enum {string}
 */
export declare type Type4 = 'unknown' | 'twin' | 'deviceJob' | 'jobResponse' | 'raw' | 'enrollment' | 'enrollmentGroup' | 'deviceRegistration';
/**
 * Defines values for ConnectionState2.
 * Possible values include: 'Disconnected', 'Connected'
 * @readonly
 * @enum {string}
 */
export declare type ConnectionState2 = 'Disconnected' | 'Connected';
/**
 * Contains response data for the get operation.
 */
export declare type ConfigurationGetResponse = Configuration & {
    /**
     * The underlying HTTP response.
     */
    _response: msRest.HttpResponse & {
        /**
         * The response body as text (string format)
         */
        bodyAsText: string;
        /**
         * The response body as parsed JSON or XML
         */
        parsedBody: Configuration;
    };
};
/**
 * Contains response data for the createOrUpdate operation.
 */
export declare type ConfigurationCreateOrUpdateResponse = Configuration & {
    /**
     * The underlying HTTP response.
     */
    _response: msRest.HttpResponse & {
        /**
         * The response body as text (string format)
         */
        bodyAsText: string;
        /**
         * The response body as parsed JSON or XML
         */
        parsedBody: Configuration;
    };
};
/**
 * Contains response data for the getConfigurations operation.
 */
export declare type ConfigurationGetConfigurationsResponse = Array<Configuration> & {
    /**
     * The underlying HTTP response.
     */
    _response: msRest.HttpResponse & {
        /**
         * The response body as text (string format)
         */
        bodyAsText: string;
        /**
         * The response body as parsed JSON or XML
         */
        parsedBody: Configuration[];
    };
};
/**
 * Contains response data for the testQueries operation.
 */
export declare type ConfigurationTestQueriesResponse = ConfigurationQueriesTestResponse & {
    /**
     * The underlying HTTP response.
     */
    _response: msRest.HttpResponse & {
        /**
         * The response body as text (string format)
         */
        bodyAsText: string;
        /**
         * The response body as parsed JSON or XML
         */
        parsedBody: ConfigurationQueriesTestResponse;
    };
};
/**
 * Contains response data for the getDeviceStatistics operation.
 */
export declare type StatisticsGetDeviceStatisticsResponse = RegistryStatistics & {
    /**
     * The underlying HTTP response.
     */
    _response: msRest.HttpResponse & {
        /**
         * The response body as text (string format)
         */
        bodyAsText: string;
        /**
         * The response body as parsed JSON or XML
         */
        parsedBody: RegistryStatistics;
    };
};
/**
 * Contains response data for the getServiceStatistics operation.
 */
export declare type StatisticsGetServiceStatisticsResponse = ServiceStatistics & {
    /**
     * The underlying HTTP response.
     */
    _response: msRest.HttpResponse & {
        /**
         * The response body as text (string format)
         */
        bodyAsText: string;
        /**
         * The response body as parsed JSON or XML
         */
        parsedBody: ServiceStatistics;
    };
};
/**
 * Contains response data for the getDevices operation.
 */
export declare type DevicesGetDevicesResponse = Array<Device> & {
    /**
     * The underlying HTTP response.
     */
    _response: msRest.HttpResponse & {
        /**
         * The response body as text (string format)
         */
        bodyAsText: string;
        /**
         * The response body as parsed JSON or XML
         */
        parsedBody: Device[];
    };
};
/**
 * Contains response data for the getIdentity operation.
 */
export declare type DevicesGetIdentityResponse = Device & {
    /**
     * The underlying HTTP response.
     */
    _response: msRest.HttpResponse & {
        /**
         * The response body as text (string format)
         */
        bodyAsText: string;
        /**
         * The response body as parsed JSON or XML
         */
        parsedBody: Device;
    };
};
/**
 * Contains response data for the createOrUpdateIdentity operation.
 */
export declare type DevicesCreateOrUpdateIdentityResponse = Device & {
    /**
     * The underlying HTTP response.
     */
    _response: msRest.HttpResponse & {
        /**
         * The response body as text (string format)
         */
        bodyAsText: string;
        /**
         * The response body as parsed JSON or XML
         */
        parsedBody: Device;
    };
};
/**
 * Contains response data for the getTwin operation.
 */
export declare type DevicesGetTwinResponse = Twin & {
    /**
     * The underlying HTTP response.
     */
    _response: msRest.HttpResponse & {
        /**
         * The response body as text (string format)
         */
        bodyAsText: string;
        /**
         * The response body as parsed JSON or XML
         */
        parsedBody: Twin;
    };
};
/**
 * Contains response data for the replaceTwin operation.
 */
export declare type DevicesReplaceTwinResponse = Twin & {
    /**
     * The underlying HTTP response.
     */
    _response: msRest.HttpResponse & {
        /**
         * The response body as text (string format)
         */
        bodyAsText: string;
        /**
         * The response body as parsed JSON or XML
         */
        parsedBody: Twin;
    };
};
/**
 * Contains response data for the updateTwin operation.
 */
export declare type DevicesUpdateTwinResponse = Twin & {
    /**
     * The underlying HTTP response.
     */
    _response: msRest.HttpResponse & {
        /**
         * The response body as text (string format)
         */
        bodyAsText: string;
        /**
         * The response body as parsed JSON or XML
         */
        parsedBody: Twin;
    };
};
/**
 * Contains response data for the invokeMethod operation.
 */
export declare type DevicesInvokeMethodResponse = CloudToDeviceMethodResult & {
    /**
     * The underlying HTTP response.
     */
    _response: msRest.HttpResponse & {
        /**
         * The response body as text (string format)
         */
        bodyAsText: string;
        /**
         * The response body as parsed JSON or XML
         */
        parsedBody: CloudToDeviceMethodResult;
    };
};
/**
 * Contains response data for the updateRegistry operation.
 */
export declare type BulkRegistryUpdateRegistryResponse = BulkRegistryOperationResult & {
    /**
     * The underlying HTTP response.
     */
    _response: msRest.HttpResponse & {
        /**
         * The response body as text (string format)
         */
        bodyAsText: string;
        /**
         * The response body as parsed JSON or XML
         */
        parsedBody: BulkRegistryOperationResult;
    };
};
/**
 * Contains response data for the getTwins operation.
 */
export declare type QueryGetTwinsResponse = Array<Twin> & QueryGetTwinsHeaders & {
    /**
     * The underlying HTTP response.
     */
    _response: msRest.HttpResponse & {
        /**
         * The parsed HTTP response headers.
         */
        parsedHeaders: QueryGetTwinsHeaders;
        /**
         * The response body as text (string format)
         */
        bodyAsText: string;
        /**
         * The response body as parsed JSON or XML
         */
        parsedBody: Twin[];
    };
};
/**
 * Contains response data for the createImportExportJob operation.
 */
export declare type JobsCreateImportExportJobResponse = JobProperties & {
    /**
     * The underlying HTTP response.
     */
    _response: msRest.HttpResponse & {
        /**
         * The response body as text (string format)
         */
        bodyAsText: string;
        /**
         * The response body as parsed JSON or XML
         */
        parsedBody: JobProperties;
    };
};
/**
 * Contains response data for the getImportExportJobs operation.
 */
export declare type JobsGetImportExportJobsResponse = Array<JobProperties> & {
    /**
     * The underlying HTTP response.
     */
    _response: msRest.HttpResponse & {
        /**
         * The response body as text (string format)
         */
        bodyAsText: string;
        /**
         * The response body as parsed JSON or XML
         */
        parsedBody: JobProperties[];
    };
};
/**
 * Contains response data for the getImportExportJob operation.
 */
export declare type JobsGetImportExportJobResponse = JobProperties & {
    /**
     * The underlying HTTP response.
     */
    _response: msRest.HttpResponse & {
        /**
         * The response body as text (string format)
         */
        bodyAsText: string;
        /**
         * The response body as parsed JSON or XML
         */
        parsedBody: JobProperties;
    };
};
/**
 * Contains response data for the cancelImportExportJob operation.
 */
export declare type JobsCancelImportExportJobResponse = {
    /**
     * The parsed response body.
     */
    body: any;
    /**
     * The underlying HTTP response.
     */
    _response: msRest.HttpResponse & {
        /**
         * The response body as text (string format)
         */
        bodyAsText: string;
        /**
         * The response body as parsed JSON or XML
         */
        parsedBody: any;
    };
};
/**
 * Contains response data for the getScheduledJob operation.
 */
export declare type JobsGetScheduledJobResponse = JobResponse & {
    /**
     * The underlying HTTP response.
     */
    _response: msRest.HttpResponse & {
        /**
         * The response body as text (string format)
         */
        bodyAsText: string;
        /**
         * The response body as parsed JSON or XML
         */
        parsedBody: JobResponse;
    };
};
/**
 * Contains response data for the createScheduledJob operation.
 */
export declare type JobsCreateScheduledJobResponse = JobResponse & {
    /**
     * The underlying HTTP response.
     */
    _response: msRest.HttpResponse & {
        /**
         * The response body as text (string format)
         */
        bodyAsText: string;
        /**
         * The response body as parsed JSON or XML
         */
        parsedBody: JobResponse;
    };
};
/**
 * Contains response data for the cancelScheduledJob operation.
 */
export declare type JobsCancelScheduledJobResponse = JobResponse & {
    /**
     * The underlying HTTP response.
     */
    _response: msRest.HttpResponse & {
        /**
         * The response body as text (string format)
         */
        bodyAsText: string;
        /**
         * The response body as parsed JSON or XML
         */
        parsedBody: JobResponse;
    };
};
/**
 * Contains response data for the queryScheduledJobs operation.
 */
export declare type JobsQueryScheduledJobsResponse = QueryResult & {
    /**
     * The underlying HTTP response.
     */
    _response: msRest.HttpResponse & {
        /**
         * The response body as text (string format)
         */
        bodyAsText: string;
        /**
         * The response body as parsed JSON or XML
         */
        parsedBody: QueryResult;
    };
};
/**
 * Contains response data for the purgeCloudToDeviceMessageQueue operation.
 */
export declare type CloudToDeviceMessagesPurgeCloudToDeviceMessageQueueResponse = PurgeMessageQueueResult & {
    /**
     * The underlying HTTP response.
     */
    _response: msRest.HttpResponse & {
        /**
         * The response body as text (string format)
         */
        bodyAsText: string;
        /**
         * The response body as parsed JSON or XML
         */
        parsedBody: PurgeMessageQueueResult;
    };
};
/**
 * Contains response data for the getTwin operation.
 */
export declare type ModulesGetTwinResponse = Twin & {
    /**
     * The underlying HTTP response.
     */
    _response: msRest.HttpResponse & {
        /**
         * The response body as text (string format)
         */
        bodyAsText: string;
        /**
         * The response body as parsed JSON or XML
         */
        parsedBody: Twin;
    };
};
/**
 * Contains response data for the replaceTwin operation.
 */
export declare type ModulesReplaceTwinResponse = Twin & {
    /**
     * The underlying HTTP response.
     */
    _response: msRest.HttpResponse & {
        /**
         * The response body as text (string format)
         */
        bodyAsText: string;
        /**
         * The response body as parsed JSON or XML
         */
        parsedBody: Twin;
    };
};
/**
 * Contains response data for the updateTwin operation.
 */
export declare type ModulesUpdateTwinResponse = Twin & {
    /**
     * The underlying HTTP response.
     */
    _response: msRest.HttpResponse & {
        /**
         * The response body as text (string format)
         */
        bodyAsText: string;
        /**
         * The response body as parsed JSON or XML
         */
        parsedBody: Twin;
    };
};
/**
 * Contains response data for the getModulesOnDevice operation.
 */
export declare type ModulesGetModulesOnDeviceResponse = Array<Module> & {
    /**
     * The underlying HTTP response.
     */
    _response: msRest.HttpResponse & {
        /**
         * The response body as text (string format)
         */
        bodyAsText: string;
        /**
         * The response body as parsed JSON or XML
         */
        parsedBody: Module[];
    };
};
/**
 * Contains response data for the getIdentity operation.
 */
export declare type ModulesGetIdentityResponse = Module & {
    /**
     * The underlying HTTP response.
     */
    _response: msRest.HttpResponse & {
        /**
         * The response body as text (string format)
         */
        bodyAsText: string;
        /**
         * The response body as parsed JSON or XML
         */
        parsedBody: Module;
    };
};
/**
 * Contains response data for the createOrUpdateIdentity operation.
 */
export declare type ModulesCreateOrUpdateIdentityResponse = Module & {
    /**
     * The underlying HTTP response.
     */
    _response: msRest.HttpResponse & {
        /**
         * The response body as text (string format)
         */
        bodyAsText: string;
        /**
         * The response body as parsed JSON or XML
         */
        parsedBody: Module;
    };
};
/**
 * Contains response data for the invokeMethod operation.
 */
export declare type ModulesInvokeMethodResponse = CloudToDeviceMethodResult & {
    /**
     * The underlying HTTP response.
     */
    _response: msRest.HttpResponse & {
        /**
         * The response body as text (string format)
         */
        bodyAsText: string;
        /**
         * The response body as parsed JSON or XML
         */
        parsedBody: CloudToDeviceMethodResult;
    };
};
/**
 * Contains response data for the getDigitalTwin operation.
 */
export declare type DigitalTwinGetDigitalTwinResponse = DigitalTwinGetDigitalTwinHeaders & {
    /**
     * The parsed response body.
     */
    body: any;
    /**
     * The underlying HTTP response.
     */
    _response: msRest.HttpResponse & {
        /**
         * The parsed HTTP response headers.
         */
        parsedHeaders: DigitalTwinGetDigitalTwinHeaders;
        /**
         * The response body as text (string format)
         */
        bodyAsText: string;
        /**
         * The response body as parsed JSON or XML
         */
        parsedBody: any;
    };
};
/**
 * Contains response data for the updateDigitalTwin operation.
 */
export declare type DigitalTwinUpdateDigitalTwinResponse = DigitalTwinUpdateDigitalTwinHeaders & {
    /**
     * The underlying HTTP response.
     */
    _response: msRest.HttpResponse & {
        /**
         * The parsed HTTP response headers.
         */
        parsedHeaders: DigitalTwinUpdateDigitalTwinHeaders;
    };
};
/**
 * Contains response data for the invokeRootLevelCommand operation.
 */
export declare type DigitalTwinInvokeRootLevelCommandResponse = DigitalTwinInvokeRootLevelCommandHeaders & {
    /**
     * The parsed response body.
     */
    body: any;
    /**
     * The underlying HTTP response.
     */
    _response: msRest.HttpResponse & {
        /**
         * The parsed HTTP response headers.
         */
        parsedHeaders: DigitalTwinInvokeRootLevelCommandHeaders;
        /**
         * The response body as text (string format)
         */
        bodyAsText: string;
        /**
         * The response body as parsed JSON or XML
         */
        parsedBody: any;
    };
};
/**
 * Contains response data for the invokeComponentCommand operation.
 */
export declare type DigitalTwinInvokeComponentCommandResponse = DigitalTwinInvokeComponentCommandHeaders & {
    /**
     * The parsed response body.
     */
    body: any;
    /**
     * The underlying HTTP response.
     */
    _response: msRest.HttpResponse & {
        /**
         * The parsed HTTP response headers.
         */
        parsedHeaders: DigitalTwinInvokeComponentCommandHeaders;
        /**
         * The response body as text (string format)
         */
        bodyAsText: string;
        /**
         * The response body as parsed JSON or XML
         */
        parsedBody: any;
    };
};
