"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const modbus_1 = require("./modbus");
const core_1 = require("@node-wot/core");
const utils_1 = require("./utils");
const modbus_connection_1 = require("./modbus-connection");
const stream_1 = require("stream");
const Subscription_1 = require("rxjs/Subscription");
const DEFAULT_PORT = 805;
const DEFAULT_TIMEOUT = 1000;
const DEFAULT_POLLING = 2000;
class ModbusSubscription {
    constructor(form, client, next, error, complete) {
        if (!complete) {
            complete = () => {
            };
        }
        this.interval = global.setInterval(() => __awaiter(this, void 0, void 0, function* () {
            try {
                const result = yield client.readResource(form);
                next(result);
            }
            catch (e) {
                if (error) {
                    error(e);
                }
                clearInterval(this.interval);
            }
        }), form["modbus:pollingTime"]);
        this.complete = complete;
    }
    unsubscribe() {
        clearInterval(this.interval);
        this.complete();
    }
}
class ModbusClient {
    constructor() {
        this._subscriptions = new Map();
        this._connections = new Map();
    }
    readResource(form) {
        return this.performOperation(form);
    }
    writeResource(form, content) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.performOperation(form, content);
        });
    }
    invokeResource(form, content) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.performOperation(form, content);
            return { type: core_1.ContentSerdes.DEFAULT, body: stream_1.Readable.from("") };
        });
    }
    unlinkResource(form) {
        form = this.validateAndFillDefaultForm(form, 0);
        const id = `${form.href}/${form["modbus:unitID"]}#${form["modbus:function"]}?${form["modbus:address"]}&${form["modbus:quantity"]}`;
        this._subscriptions.get(id).unsubscribe();
        this._subscriptions.delete(id);
        return Promise.resolve();
    }
    subscribeResource(form, next, error, complete) {
        return new Promise((resolve, reject) => {
            form = this.validateAndFillDefaultForm(form, 0);
            const id = `${form.href}/${form["modbus:unitID"]}#${form["modbus:function"]}?${form["modbus:address"]}&${form["modbus:quantity"]}`;
            if (this._subscriptions.has(id)) {
                reject(new Error("Already subscribed for " + id + ". Multiple subscriptions are not supported"));
            }
            const subscription = new ModbusSubscription(form, this, next, error, complete);
            this._subscriptions.set(id, subscription);
            resolve(new Subscription_1.Subscription(() => {
                subscription.unsubscribe();
            }));
        });
    }
    start() {
        return __awaiter(this, void 0, void 0, function* () {
        });
    }
    stop() {
        return __awaiter(this, void 0, void 0, function* () {
            this._connections.forEach((connection) => {
                connection.close();
            });
        });
    }
    setSecurity(metadata, credentials) {
        return false;
    }
    performOperation(form, content) {
        return __awaiter(this, void 0, void 0, function* () {
            const parsed = new URL(form.href);
            const port = parsed.port ? parseInt(parsed.port, 10) : DEFAULT_PORT;
            let body;
            if (content) {
                body = yield core_1.ProtocolHelpers.readStreamFully(content.body);
            }
            form = this.validateAndFillDefaultForm(form, body === null || body === void 0 ? void 0 : body.byteLength);
            const endianness = this.validateEndianness(form);
            const host = parsed.hostname;
            const hostAndPort = host + ":" + port;
            this.overrideFormFromURLPath(form);
            if (body) {
                this.validateBufferLength(form, body);
            }
            let connection = this._connections.get(hostAndPort);
            if (!connection) {
                console.debug("[binding-modbus]", "Creating new ModbusConnection for ", hostAndPort);
                this._connections.set(hostAndPort, new modbus_connection_1.ModbusConnection(host, port, { connectionTimeout: form["modbus:timeout"] || DEFAULT_TIMEOUT }));
                connection = this._connections.get(hostAndPort);
            }
            else {
                console.debug("[binding-modbus]", "Reusing ModbusConnection for ", hostAndPort);
            }
            const operation = new modbus_connection_1.PropertyOperation(form, endianness, body);
            connection.enqueue(operation);
            return operation.execute();
        });
    }
    validateEndianness(form) {
        var _a;
        let endianness = modbus_1.ModbusEndianness.BIG_ENDIAN;
        if (form.contentType) {
            const contentValues = (_a = form.contentType.split(";")) !== null && _a !== void 0 ? _a : [];
            const byteSeq = contentValues.find((value) => /^byteSeq=/.test(value));
            if (byteSeq) {
                const guessEndianness = modbus_1.ModbusEndianness[byteSeq.split("=")[1]];
                if (guessEndianness) {
                    endianness = guessEndianness;
                }
                else {
                    throw new Error("Malformed form: Content Type endianness is not valid");
                }
            }
        }
        return endianness;
    }
    overrideFormFromURLPath(input) {
        const parsed = new URL(input.href);
        const pathComp = parsed.pathname.split("/");
        const query = parsed.searchParams;
        input["modbus:unitID"] = parseInt(pathComp[1], 10) || input["modbus:unitID"];
        input["modbus:address"] = parseInt(query.get("address"), 10) || input["modbus:address"];
        input["modbus:quantity"] = parseInt(query.get("quantity"), 10) || input["modbus:quantity"];
    }
    validateBufferLength(form, buffer) {
        const mpy = form["modbus:entity"] === "InputRegister" || form["modbus:entity"] === "HoldingRegister" ? 2 : 1;
        const quantity = form["modbus:quantity"];
        if (buffer && buffer.length !== mpy * quantity) {
            throw new Error("Content length does not match register / coil count, got " +
                buffer.length +
                " bytes for " +
                quantity +
                ` ${mpy === 2 ? "registers" : "coils"}`);
        }
    }
    validateAndFillDefaultForm(form, contentLength = 0) {
        const result = Object.assign({}, form);
        const mode = contentLength > 0 ? "w" : "r";
        if (!form["modbus:function"] && !form["modbus:entity"]) {
            throw new Error("Malformed form: modbus:function or modbus:entity must be defined");
        }
        if (form["modbus:function"]) {
            if (typeof form["modbus:function"] === "string") {
                result["modbus:function"] = modbus_1.ModbusFunction[form["modbus:function"]];
            }
            if (!Object.keys(modbus_1.ModbusFunction).includes(result["modbus:function"].toString())) {
                throw new Error("Undefined function number or name: " + form["modbus:function"]);
            }
        }
        if (form["modbus:entity"]) {
            switch (form["modbus:entity"]) {
                case "Coil":
                    result["modbus:function"] =
                        mode === "r"
                            ? modbus_1.ModbusFunction.readCoil
                            : contentLength > 1
                                ? modbus_1.ModbusFunction.writeMultipleCoils
                                : modbus_1.ModbusFunction.writeSingleCoil;
                    break;
                case "HoldingRegister":
                    result["modbus:function"] =
                        mode === "r"
                            ? modbus_1.ModbusFunction.readHoldingRegisters
                            : contentLength / 2 > 1
                                ? modbus_1.ModbusFunction.writeMultipleHoldingRegisters
                                : modbus_1.ModbusFunction.writeSingleHoldingRegister;
                    break;
                case "InputRegister":
                    result["modbus:function"] = modbus_1.ModbusFunction.readInputRegister;
                    break;
                case "DiscreteInput":
                    result["modbus:function"] = modbus_1.ModbusFunction.readDiscreteInput;
                    break;
                default:
                    throw new Error("Unknown modbus entity: " + form["modbus:entity"]);
            }
        }
        else {
            result["modbus:entity"] = (0, utils_1.modbusFunctionToEntity)(result["modbus:function"]);
        }
        if (form["modbus:address"] === undefined || form["modbus:address"] === null) {
            throw new Error("Malformed form: address must be defined");
        }
        if (!form["modbus:quantity"] && contentLength === 0) {
            result["modbus:quantity"] = 1;
        }
        else if (!form["modbus:quantity"] && contentLength > 0) {
            const regSize = result["modbus:entity"] === "InputRegister" || result["modbus:entity"] === "HoldingRegister" ? 2 : 1;
            result["modbus:quantity"] = contentLength / regSize;
        }
        result["modbus:pollingTime"] = form["modbus:pollingTime"] ? form["modbus:pollingTime"] : DEFAULT_POLLING;
        result["modbus:timeout"] = form["modbus:timeout"] ? form["modbus:timeout"] : DEFAULT_TIMEOUT;
        return result;
    }
}
exports.default = ModbusClient;
//# sourceMappingURL=modbus-client.js.map