/// <reference types="node" />
import { AbortSignalLike } from '@azure/abort-controller';
import { AmqpAnnotatedMessage } from '@azure/core-amqp';
import { AzureLogger } from '@azure/logger';
import { MessagingError } from '@azure/core-amqp';
import { NamedKeyCredential } from '@azure/core-auth';
import { OperationTracingOptions } from '@azure/core-tracing';
import { RetryMode } from '@azure/core-amqp';
import { RetryOptions } from '@azure/core-amqp';
import { SASCredential } from '@azure/core-auth';
import { TokenCredential } from '@azure/core-auth';
import { WebSocketImpl } from 'rhea-promise';
import { WebSocketOptions } from '@azure/core-amqp';
/**
 * Options to configure the `close` method on the `EventHubBufferedProducerClient`.
 */
export declare interface BufferedCloseOptions extends OperationOptions {
    /**
     * When `true`, all buffered events that are pending should be sent before closing.
     * When `false`, abandon all buffered events and close immediately.
     * Defaults to `true`.
     */
    flush?: boolean;
}
/**
 * Options to configure the `flush` method on the `EventHubBufferedProducerClient`.
 */
export declare interface BufferedFlushOptions extends OperationOptions {
}
/**
 * A checkpoint is meant to represent the last successfully processed event by the user from a particular
 * partition of a consumer group in an Event Hub instance.
 *
 * When the `updateCheckpoint()` method on the `PartitionProcessor` class is called by the user, a
 * `Checkpoint` is created internally. It is then stored in the storage solution implemented by the
 * `CheckpointManager` chosen by the user when creating an `EventProcessor`.
 *
 * Users are never expected to interact with `Checkpoint` directly. This interface exists to support the
 * internal workings of `EventProcessor` and `CheckpointManager`.
 **/
export declare interface Checkpoint {
    /**
     * The fully qualified Event Hubs namespace. This is likely to be similar to
     * <yournamespace>.servicebus.windows.net
     */
    fullyQualifiedNamespace: string;
    /**
     * The event hub name
     */
    eventHubName: string;
    /**
     * The consumer group name
     */
    consumerGroup: string;
    /**
     * The identifier of the Event Hub partition
     */
    partitionId: string;
    /**
     * The sequence number of the event
     */
    sequenceNumber: number;
    /**
     * The offset of the event.
     */
    offset: number;
}
/**
 * A checkpoint store stores and retrieves partition ownership information and checkpoint details
 * for each partition in a given consumer group of an event hub instance.
 *
 * Users are not meant to implement an `CheckpointStore`.
 * Users are expected to choose existing implementations of this interface, instantiate it, and pass
 * it to the `EventHubConsumerClient` class constructor when instantiating a client.
 * Users are not expected to use any of the methods on a checkpoint store, these are used internally by
 * the client.
 *
 * Implementations of `CheckpointStore` can be found on npm by searching for packages with the prefix &commat;azure/eventhub-checkpointstore-.
 */
export declare interface CheckpointStore {
    /**
     * Called to get the list of all existing partition ownership from the underlying data store. Could return empty
     * results if there are is no existing ownership information.
     *
     * @param fullyQualifiedNamespace - The fully qualified Event Hubs namespace. This is likely to be similar to
     * <yournamespace>.servicebus.windows.net.
     * @param eventHubName - The event hub name.
     * @param consumerGroup - The consumer group name.
     * @param options - A set of options that can be specified to influence the behavior of this method.
     *  - `abortSignal`: A signal used to request operation cancellation.
     *  - `tracingOptions`: Options for configuring tracing.
     * @returns A list of partition ownership details of all the partitions that have/had an owner.
     */
    listOwnership(fullyQualifiedNamespace: string, eventHubName: string, consumerGroup: string, options?: OperationOptions): Promise<PartitionOwnership[]>;
    /**
     * Called to claim ownership of a list of partitions. This will return the list of partitions that were owned
     * successfully.
     *
     * @param partitionOwnership - The list of partition ownership this instance is claiming to own.
     * @param options - A set of options that can be specified to influence the behavior of this method.
     *  - `abortSignal`: A signal used to request operation cancellation.
     *  - `tracingOptions`: Options for configuring tracing.
     * @returns A list of partitions this instance successfully claimed ownership.
     */
    claimOwnership(partitionOwnership: PartitionOwnership[], options?: OperationOptions): Promise<PartitionOwnership[]>;
    /**
     * Updates the checkpoint in the data store for a partition.
     *
     * @param checkpoint - The checkpoint.
     * @param options - A set of options that can be specified to influence the behavior of this method.
     *  - `abortSignal`: A signal used to request operation cancellation.
     *  - `tracingOptions`: Options for configuring tracing.
     */
    updateCheckpoint(checkpoint: Checkpoint, options?: OperationOptions): Promise<void>;
    /**
     * Lists all the checkpoints in a data store for a given namespace, eventhub and consumer group.
     *
     * @param fullyQualifiedNamespace - The fully qualified Event Hubs namespace. This is likely to be similar to
     * <yournamespace>.servicebus.windows.net.
     * @param eventHubName - The event hub name.
     * @param consumerGroup - The consumer group name.
     * @param options - A set of options that can be specified to influence the behavior of this method.
     *  - `abortSignal`: A signal used to request operation cancellation.
     *  - `tracingOptions`: Options for configuring tracing.
     * @returns A list of checkpoints for a given namespace, eventhub, and consumer group.
     */
    listCheckpoints(fullyQualifiedNamespace: string, eventHubName: string, consumerGroup: string, options?: OperationOptions): Promise<Checkpoint[]>;
}
/**
 * An enum representing the different reasons for an `EventHubConsumerClient` to stop processing
 * events from a partition in a consumer group of an Event Hub.
 */
export declare enum CloseReason {
    /**
     * Ownership of the partition was lost or transitioned to a new processor instance.
     */
    OwnershipLost = "OwnershipLost",
    /**
     * The EventProcessor was shutdown.
     */
    Shutdown = "Shutdown"
}
/**
 * Options to configure the `createBatch` method on the `EventHubProducerClient`.
 * - `partitionKey`  : A value that is hashed to produce a partition assignment.
 * - `maxSizeInBytes`: The upper limit for the size of batch.
 * - `abortSignal`   : A signal the request to cancel the send operation.
 *
 * Example usage:
 * ```js
 * {
 *     partitionKey: 'foo',
 *     maxSizeInBytes: 1024 * 1024 // 1 MB
 * }
 * ```
 */
export declare interface CreateBatchOptions extends OperationOptions {
    /**
     * A value that is hashed to produce a partition assignment. It guarantees that messages
     * with the same partitionKey end up in the same partition.
     * If this value is set then partitionId can not be set.
     */
    partitionKey?: string;
    /**
     * The partition this batch will be sent to.
     * If this value is set then partitionKey can not be set.
     */
    partitionId?: string;
    /**
     * The upper limit for the size of batch. The `tryAdd` function will return `false` after this limit is reached.
     */
    maxSizeInBytes?: number;
}
/**
 * A function that constructs an event data adapter. That adapter can be used
 * with `@azure/schema-registry-avro` to encode and decode body in event data.
 *
 * @param params - parameters to create the event data
 * @returns An event data adapter that can produce and consume event data
 */
export declare function createEventDataAdapter(params?: EventDataAdapterParameters): MessageAdapter<EventData>;
/**
 * Gets the `EventPosition` corresponding to the location of the the first event present in the partition.
 * Pass this position to the `EventHubConsumerClient.subscribe()` method to begin receiving events from the
 * first event in the partition which has not expired due to the retention policy.
 */
export declare const earliestEventPosition: EventPosition;
/**
 * Options to configure the `enqueueEvents` method on the `EventHubBufferedProcuerClient`.
 */
export declare interface EnqueueEventOptions extends SendBatchOptions {
}
/**
 * The interface that describes the data to be sent to Event Hub.
 * Use this as a reference when creating the object to be sent when using the `EventHubProducerClient`.
 * For example, `{ body: "your-data" }` or
 * ```
 * {
 *    body: "your-data",
 *    properties: {
 *       propertyName: "property value"
 *    }
 * }
 * ```
 */
export declare interface EventData {
    /**
     * The message body that needs to be sent.
     * If the application reading the events is not using this SDK,
     * convert your body payload to a byte array or Buffer for better
     * cross-language compatibility.
     */
    body: any;
    /**
     * The content type of the message. Optionally describes
     * the payload of the message, with a descriptor following the format of RFC2045, Section 5, for
     * example "application/json".
     */
    contentType?: string;
    /**
     * The correlation identifier that allows an
     * application to specify a context for the message for the purposes of correlation, for example
     * reflecting the MessageId of a message that is being replied to.
     */
    correlationId?: string | number | Buffer;
    /**
     * The message identifier is an
     * application-defined value that uniquely identifies the message and its payload.
     *
     * Note: Numbers that are not whole integers are not allowed.
     */
    messageId?: string | number | Buffer;
    /**
     * Set of key value pairs that can be used to set properties specific to user application.
     */
    properties?: {
        [key: string]: any;
    };
}
/**
 * Parameters to the `createEventDataAdapter` function that creates an event data adapter.
 */
export declare interface EventDataAdapterParameters {
    /**
     * The correlation identifier that allows an
     * application to specify a context for the message for the purposes of correlation, for example
     * reflecting the MessageId of a message that is being replied to.
     */
    correlationId?: string | number | Buffer;
    /**
     * The message identifier is an
     * application-defined value that uniquely identifies the message and its payload.
     *
     * Note: Numbers that are not whole integers are not allowed.
     */
    messageId?: string | number | Buffer;
    /**
     * Set of key value pairs that can be used to set properties specific to user application.
     */
    properties?: {
        [key: string]: any;
    };
}
/**
 * An interface representing a batch of events which can be used to send events to Event Hub.
 *
 * To create the batch, use the `createBatch()` method on the `EventHubProducerClient`.
 * To send the batch, use the `sendBatch()` method on the same client.
 * To fill the batch, use the `tryAdd()` method on the batch itself.
 *
 */
export declare interface EventDataBatch {
    /* Excluded from this release type: partitionKey */
    /* Excluded from this release type: partitionId */
    /**
     * Size of the batch in bytes after the events added to it have been encoded into a single AMQP
     * message.
     * @readonly
     */
    readonly sizeInBytes: number;
    /**
     * Number of events added to the batch.
     * @readonly
     */
    readonly count: number;
    /**
     * The maximum size of the batch, in bytes. The `tryAdd` function on the batch will return `false`
     * if the event being added causes the size of the batch to exceed this limit. Use the `createBatch()` method on
     * the `EventHubProducerClient` to set the maxSizeInBytes.
     * @readonly
     */
    readonly maxSizeInBytes: number;
    /**
     * Adds an event to the batch if permitted by the batch's size limit.
     * **NOTE**: Always remember to check the return value of this method, before calling it again
     * for the next event.
     *
     * @param eventData -  An individual event data object or AmqpAnnotatedMessage.
     * @returns A boolean value indicating if the event data has been added to the batch or not.
     */
    tryAdd(eventData: EventData | AmqpAnnotatedMessage, options?: TryAddOptions): boolean;
}
/**
 * The `EventHubBufferedProducerClient`is used to publish events to a specific Event Hub.
 *
 * The `EventHubBufferedProducerClient` does not publish events immediately.
 * Instead, events are buffered so they can be efficiently batched and published
 * when the batch is full or the `maxWaitTimeInMs` has elapsed with no new events
 * enqueued.
 *
 * Depending on the options specified when events are enqueued, they may be
 * automatically assigned to a partition, grouped according to the specified partition key,
 * or assigned a specifically requested partition.
 *
 * This model is intended to shift the burden of batch management from callers, at the cost of
 * non-deterministic timing, for when events will be published. There are additional trade-offs
 * to consider, as well:
 * - If the application crashes, events in the buffer will not have been published. To prevent
 *   data loss, callers are encouraged to track publishing progress using the
 *   `onSendEventsSuccessHandler` and `onSendEventsErrorHandler` handlers.
 * - Events specifying a partition key may be assigned a different partition than those using
 *   the same key with other producers.
 * - In the unlikely event that a partition becomes temporarily unavailable, the
 *   `EventHubBufferedProducerClient` may take longer to recover than other producers.
 *
 * In scenarios where it is important to have events published immediately with a deterministic
 * outcome, ensure that partition keys are assigned to a partition consistent with other
 * publishers, or where maximizing availability is a requirement, using the
 * `EventHubProducerClient` is recommended.
 */
export declare class EventHubBufferedProducerClient {
    /**
     * Controls the `abortSignal` passed to each `BatchingPartitionChannel`.
     * Used to signal when a channel should stop waiting for events.
     */
    private _abortController;
    /**
     * Indicates whether the client has been explicitly closed.
     */
    private _isClosed;
    /**
     * Handles assigning partitions.
     */
    private _partitionAssigner;
    /**
     * The known partitionIds that will be used when assigning events to partitions.
     */
    private _partitionIds;
    /**
     * The EventHubProducerClient to use when creating and sending batches to the Event Hub.
     */
    private _producer;
    /**
     * Mapping of partitionIds to `BatchingPartitionChannels`.
     * Each `BatchingPartitionChannel` handles buffering events and backpressure independently.
     */
    private _partitionChannels;
    /**
     * The options passed by the user when creating the EventHubBufferedProducerClient instance.
     */
    private _clientOptions;
    /**
     * The interval at which the background management operation should run.
     */
    private _backgroundManagementInterval;
    /**
     * Indicates whether the background management loop is running.
     */
    private _isBackgroundManagementRunning;
    /*
    * @readonly
    * The name of the Event Hub instance for which this client is created.
    */
    readonly eventHubName: string;
    /*
    * @readonly
    * The fully qualified namespace of the Event Hub instance for which this client is created.
    * This is likely to be similar to <yournamespace>.servicebus.windows.net.
    */
    readonly fullyQualifiedNamespace: string;
    /**
     * The `EventHubBufferedProducerClient` class is used to send events to an Event Hub.
     * Use the `options` parmeter to configure retry policy or proxy settings.
     * @param connectionString - The connection string to use for connecting to the Event Hub instance.
     * It is expected that the shared key properties and the Event Hub path are contained in this connection string.
     * e.g. 'Endpoint=sb://my-servicebus-namespace.servicebus.windows.net/;SharedAccessKeyName=my-SA-name;SharedAccessKey=my-SA-key;EntityPath=my-event-hub-name'.
     * @param options - A set of options to apply when configuring the client.
     * - `retryOptions`   : Configures the retry policy for all the operations on the client.
     * For example, `{ "maxRetries": 4 }` or `{ "maxRetries": 4, "retryDelayInMs": 30000 }`.
     * - `webSocketOptions`: Configures the channelling of the AMQP connection over Web Sockets.
     * - `userAgent`      : A string to append to the built in user agent string that is passed to the service.
     */
    constructor(connectionString: string, options: EventHubBufferedProducerClientOptions);
    /**
     * The `EventHubBufferedProducerClient` class is used to send events to an Event Hub.
     * Use the `options` parmeter to configure retry policy or proxy settings.
     * @param connectionString - The connection string to use for connecting to the Event Hubs namespace.
     * It is expected that the shared key properties are contained in this connection string, but not the Event Hub path,
     * e.g. 'Endpoint=sb://my-servicebus-namespace.servicebus.windows.net/;SharedAccessKeyName=my-SA-name;SharedAccessKey=my-SA-key;'.
     * @param eventHubName - The name of the specific Event Hub to connect the client to.
     * @param options - A set of options to apply when configuring the client.
     * - `retryOptions`   : Configures the retry policy for all the operations on the client.
     * For example, `{ "maxRetries": 4 }` or `{ "maxRetries": 4, "retryDelayInMs": 30000 }`.
     * - `webSocketOptions`: Configures the channelling of the AMQP connection over Web Sockets.
     * - `userAgent`      : A string to append to the built in user agent string that is passed to the service.
     */
    constructor(connectionString: string, eventHubName: string, options: EventHubBufferedProducerClientOptions);
    /**
     * The `EventHubBufferedProducerClient` class is used to send events to an Event Hub.
     * Use the `options` parmeter to configure retry policy or proxy settings.
     * @param fullyQualifiedNamespace - The full namespace which is likely to be similar to
     * <yournamespace>.servicebus.windows.net
     * @param eventHubName - The name of the specific Event Hub to connect the client to.
     * @param credential - An credential object used by the client to get the token to authenticate the connection
     * with the Azure Event Hubs service.
     * See &commat;azure/identity for creating credentials that support AAD auth.
     * Use the `AzureNamedKeyCredential` from &commat;azure/core-auth if you want to pass in a `SharedAccessKeyName`
     * and `SharedAccessKey` without using a connection string. These fields map to the `name` and `key` field respectively
     * in `AzureNamedKeyCredential`.
     * Use the `AzureSASCredential` from &commat;azure/core-auth if you want to pass in a `SharedAccessSignature`
     * without using a connection string. This field maps to `signature` in `AzureSASCredential`.
     * @param options - A set of options to apply when configuring the client.
     * - `retryOptions`   : Configures the retry policy for all the operations on the client.
     * For example, `{ "maxRetries": 4 }` or `{ "maxRetries": 4, "retryDelayInMs": 30000 }`.
     * - `webSocketOptions`: Configures the channelling of the AMQP connection over Web Sockets.
     * - `userAgent`      : A string to append to the built in user agent string that is passed to the service.
     */
    constructor(fullyQualifiedNamespace: string, eventHubName: string, credential: TokenCredential | NamedKeyCredential | SASCredential, options: EventHubBufferedProducerClientOptions);
    /**
     * Closes the AMQP connection to the Event Hub instance,
     * returning a promise that will be resolved when disconnection is completed.
     *
     * This will wait for enqueued events to be flushed to the service before closing
     * the connection.
     * To close without flushing, set the `flush` option to `false`.
     *
     * @param options - The set of options to apply to the operation call.
     * @returns Promise<void>
     * @throws Error if the underlying connection encounters an error while closing.
     */
    close(options?: BufferedCloseOptions): Promise<void>;
    /**
     * Enqueues an event into the buffer to be published to the Event Hub.
     * If there is no capacity in the buffer when this method is invoked,
     * it will wait for space to become available and ensure that the event
     * has been enqueued.
     *
     * When this call returns, the event has been accepted into the buffer,
     * but it may not have been published yet.
     * Publishing will take place at a nondeterministic point in the future as the buffer is processed.
     *
     * @param events - An {@link EventData} or `AmqpAnnotatedMessage`.
     * @param options - A set of options that can be specified to influence the way in which
     * the event is sent to the associated Event Hub.
     * - `abortSignal`  : A signal used to cancel the enqueueEvent operation.
     * - `partitionId`  : The partition this set of events will be sent to. If set, `partitionKey` can not be set.
     * - `partitionKey` : A value that is hashed to produce a partition assignment. If set, `partitionId` can not be set.
     * @returns The total number of events that are currently buffered and waiting to be published, across all partitions.
     */
    enqueueEvent(event: EventData | AmqpAnnotatedMessage, options?: EnqueueEventOptions): Promise<number>;
    /**
     * Enqueues events into the buffer to be published to the Event Hub.
     * If there is no capacity in the buffer when this method is invoked,
     * it will wait for space to become available and ensure that the events
     * have been enqueued.
     *
     * When this call returns, the events have been accepted into the buffer,
     * but it may not have been published yet.
     * Publishing will take place at a nondeterministic point in the future as the buffer is processed.
     *
     * @param events - An array of {@link EventData} or `AmqpAnnotatedMessage`.
     * @param options - A set of options that can be specified to influence the way in which
     * events are sent to the associated Event Hub.
     * - `abortSignal`  : A signal used to cancel the enqueueEvents operation.
     * - `partitionId`  : The partition this set of events will be sent to. If set, `partitionKey` can not be set.
     * - `partitionKey` : A value that is hashed to produce a partition assignment. If set, `partitionId` can not be set.
     * @returns The total number of events that are currently buffered and waiting to be published, across all partitions.
     */
    enqueueEvents(events: EventData[] | AmqpAnnotatedMessage[], options?: EnqueueEventOptions): Promise<number>;
    /**
     * Attempts to publish all events in the buffer immediately.
     * This may result in multiple batches being published,
     * the outcome of each of which will be individually reported by
     * the `onSendEventsSuccessHandler` and `onSendEventsErrorHandler` handlers.
     *
     * @param options - The set of options to apply to the operation call.
     */
    flush(options?: BufferedFlushOptions): Promise<void>;
    /**
     * Provides the Event Hub runtime information.
     * @param options - The set of options to apply to the operation call.
     * @returns A promise that resolves with information about the Event Hub instance.
     * @throws Error if the underlying connection has been closed, create a new EventHubBufferedProducerClient.
     * @throws AbortError if the operation is cancelled via the abortSignal.
     */
    getEventHubProperties(options?: GetEventHubPropertiesOptions): Promise<EventHubProperties>;
    /**
     * Provides the id for each partition associated with the Event Hub.
     * @param options - The set of options to apply to the operation call.
     * @returns A promise that resolves with an Array of strings representing the id for
     * each partition associated with the Event Hub.
     * @throws Error if the underlying connection has been closed, create a new EventHubBufferedProducerClient.
     * @throws AbortError if the operation is cancelled via the abortSignal.
     */
    getPartitionIds(options?: GetPartitionIdsOptions): Promise<Array<string>>;
    /**
     * Provides information about the state of the specified partition.
     * @param partitionId - The id of the partition for which information is required.
     * @param options - The set of options to apply to the operation call.
     * @returns A promise that resolves with information about the state of the partition .
     * @throws Error if the underlying connection has been closed, create a new EventHubBufferedProducerClient.
     * @throws AbortError if the operation is cancelled via the abortSignal.
     */
    getPartitionProperties(partitionId: string, options?: GetPartitionPropertiesOptions): Promise<PartitionProperties>;
    /**
     * Gets the `BatchingPartitionChannel` associated with the partitionId.
     *
     * If one does not exist, it is created.
     */
    private _getPartitionChannel;
    /**
     * Returns the total number of buffered events across all partitions.
     */
    private _getTotalBufferedEventsCount;
    private _updatePartitionIds;
    private _startPartitionIdsUpdateLoop;
}
/**
 * Describes the options that can be provided while creating the `EventHubBufferedProducerClient`.
 */
export declare interface EventHubBufferedProducerClientOptions extends EventHubClientOptions {
    /**
     * The total number of events that can be buffered for publishing at a given time for a given partition.
     *
     * Default: 1500
     */
    maxEventBufferLengthPerPartition?: number;
    /**
     * The amount of time to wait for a new event to be enqueued in the buffer before publishing a partially full batch.
     *
     * Default: 1 second.
     */
    maxWaitTimeInMs?: number;
    /**
     * The handler to call once a batch has successfully published.
     */
    onSendEventsSuccessHandler?: (ctx: OnSendEventsSuccessContext) => void;
    /**
     * The handler to call when a batch fails to publish.
     */
    onSendEventsErrorHandler: (ctx: OnSendEventsErrorContext) => void;
    /**
     * Indicates whether or not the EventHubProducerClient should enable idempotent publishing to Event Hub partitions.
     * If enabled, the producer will only be able to publish directly to partitions;
     * it will not be able to publish to the Event Hubs gateway for automatic partition routing
     * nor will it be able to use a partition key.
     * Default: false
     */
    enableIdempotentRetries?: boolean;
}
/**
 * Describes the options that can be provided while creating the EventHubClient.
 * - `userAgent`        : A string to append to the built in user agent string that is passed as a connection property
 * to the service.
 * - `webSocketOptions` : Options to configure the channelling of the AMQP connection over Web Sockets.
 *    - `websocket`     : The WebSocket constructor used to create an AMQP connection if you choose to make the connection
 * over a WebSocket.
 *    - `webSocketConstructorOptions` : Options to pass to the Websocket constructor when you choose to make the connection
 * over a WebSocket.
 * - `retryOptions`     : The retry options for all the operations on the client/producer/consumer.
 *    - `maxRetries` : The number of times the operation can be retried in case of a retryable error.
 *    - `maxRetryDelayInMs`: The maximum delay between retries. Applicable only when performing exponential retries.
 *    - `mode`: Which retry mode to apply, specified by the `RetryMode` enum. Options are `Exponential` and `Fixed`. Defaults to `Fixed`.
 *    - `retryDelayInMs`: Amount of time to wait in milliseconds before making the next attempt. When `mode` is set to `Exponential`,
 *       this is used to compute the exponentially increasing delays between retries. Default: 30000 milliseconds.
 *    - `timeoutInMs`: Amount of time in milliseconds to wait before the operation times out. This will trigger a retry if there are any
 *       retry attempts remaining. Minimum value: 60000 milliseconds.
 *
 * A simple usage can be `{ "maxRetries": 4 }`.
 *
 * Example usage:
 * ```js
 * {
 *     retryOptions: {
 *         maxRetries: 4
 *     }
 * }
 * ```
 */
export declare interface EventHubClientOptions {
    /**
     * A custom endpoint to use when connecting to the Event Hubs service.
     * This can be useful when your network does not allow connecting to the
     * standard Azure Event Hubs endpoint address, but does allow connecting
     * through an intermediary.
     *
     * Example: "https://my.custom.endpoint:100/"
     */
    customEndpointAddress?: string;
    /**
     * Options to configure the retry policy for all the operations on the client.
     * For example, `{ "maxRetries": 4 }` or `{ "maxRetries": 4, "retryDelayInMs": 30000 }`.
     *
     */
    retryOptions?: RetryOptions;
    /**
     * Options to configure the channelling of the AMQP connection over Web Sockets.
     */
    webSocketOptions?: WebSocketOptions;
    /**
     * Value that is appended to the built in user agent string that is passed to the Event Hubs service.
     */
    userAgent?: string;
}
/**
 * The set of properties that comprise an Event Hub connection string.
 */
export declare interface EventHubConnectionStringProperties {
    /**
     * The fully qualified Event Hub namespace extracted from the "Endpoint" in the
     * connection string. This is likely to be similar to `{yournamespace}.servicebus.windows.net`.
     * This is typically used to construct an EventHubProducerClient or an EventHubConsumerClient.
     */
    fullyQualifiedNamespace: string;
    /**
     * The value for "Endpoint" in the connection string.
     */
    endpoint: string;
    /**
     * The value for "EntityPath" in the connection string which would be the name of the event hub instance associated with the connection string.
     * Connection string from a Shared Access Policy created at the namespace level
     * will not have the EntityPath in it.
     */
    eventHubName?: string;
    /**
     * The value for "SharedAccessKey" in the connection string. This along with the "SharedAccessKeyName"
     * in the connection string is used to generate a SharedAccessSignature which can be used authorize
     * the connection to the service.
     */
    sharedAccessKey?: string;
    /**
     * The value for "SharedAccessKeyName" in the connection string. This along with the "SharedAccessKey"
     * in the connection string is used to generate a SharedAccessSignature which can be used authorize
     * the connection to the service.
     */
    sharedAccessKeyName?: string;
    /**
     * The value for "SharedAccessSignature" in the connection string. This is typically not present in the
     * connection string generated for a Shared Access Policy. It is instead generated by the
     * user and appended to the connection string for ease of use.
     */
    sharedAccessSignature?: string;
}
/**
 * The `EventHubConsumerClient` class is used to consume events from an Event Hub.
 *
 * There are multiple ways to create an `EventHubConsumerClient`
 * - Use the connection string from the SAS policy created for your Event Hub instance.
 * - Use the connection string from the SAS policy created for your Event Hub namespace,
 * and the name of the Event Hub instance
 * - Use the full namespace like `<yournamespace>.servicebus.windows.net`, and a credentials object.
 *
 * Optionally, you can also pass:
 * - An options bag to configure the retry policy or proxy settings.
 * - A checkpoint store that is used by the client to read checkpoints to determine the position from where it should
 * resume receiving events when your application gets restarted. The checkpoint store is also used by the client
 * to load balance multiple instances of your application.
 */
export declare class EventHubConsumerClient {
    private _consumerGroup;
    /**
     * Describes the amqp connection context for the client.
     */
    private _context;
    /**
     * The options passed by the user when creating the EventHubClient instance.
     */
    private _clientOptions;
    private _partitionGate;
    private _id;
    /**
     * The Subscriptions that were spawned by calling `subscribe()`.
     * Subscriptions that have been stopped by the user will not
     * be present in this set.
     */
    private _subscriptions;
    /**
     * The name of the default consumer group in the Event Hubs service.
     */
    static defaultConsumerGroupName: string;
    private _checkpointStore;
    private _userChoseCheckpointStore;
    /**
     * Options for configuring load balancing.
     */
    private readonly _loadBalancingOptions;
    /*
    * @readonly
    * The name of the Event Hub instance for which this client is created.
    */
    readonly eventHubName: string;
    /*
    * @readonly
    * The fully qualified namespace of the Event Hub instance for which this client is created.
    * This is likely to be similar to <yournamespace>.servicebus.windows.net.
    */
    readonly fullyQualifiedNamespace: string;
    /**
     * The `EventHubConsumerClient` class is used to consume events from an Event Hub.
     * Use the `options` parmeter to configure retry policy or proxy settings.
     * @param consumerGroup - The name of the consumer group from which you want to process events.
     * @param connectionString - The connection string to use for connecting to the Event Hub instance.
     * It is expected that the shared key properties and the Event Hub path are contained in this connection string.
     * e.g. 'Endpoint=sb://my-servicebus-namespace.servicebus.windows.net/;SharedAccessKeyName=my-SA-name;SharedAccessKey=my-SA-key;EntityPath=my-event-hub-name'.
     * @param options - A set of options to apply when configuring the client.
     * - `retryOptions`   : Configures the retry policy for all the operations on the client.
     * For example, `{ "maxRetries": 4 }` or `{ "maxRetries": 4, "retryDelayInMs": 30000 }`.
     * - `webSocketOptions`: Configures the channelling of the AMQP connection over Web Sockets.
     * - `userAgent`      : A string to append to the built in user agent string that is passed to the service.
     */
    constructor(consumerGroup: string, connectionString: string, options?: EventHubConsumerClientOptions);
    /**
     * The `EventHubConsumerClient` class is used to consume events from an Event Hub.
     * Use the `options` parmeter to configure retry policy or proxy settings.
     * @param consumerGroup - The name of the consumer group from which you want to process events.
     * @param connectionString - The connection string to use for connecting to the Event Hub instance.
     * It is expected that the shared key properties and the Event Hub path are contained in this connection string.
     * e.g. 'Endpoint=sb://my-servicebus-namespace.servicebus.windows.net/;SharedAccessKeyName=my-SA-name;SharedAccessKey=my-SA-key;EntityPath=my-event-hub-name'.
     * @param checkpointStore - A checkpoint store that is used by the client to read checkpoints to determine
     * the position from where it should resume receiving events when your application gets restarted.
     * It is also used by the client to load balance multiple instances of your application.
     * @param options - A set of options to apply when configuring the client.
     * - `retryOptions`   : Configures the retry policy for all the operations on the client.
     * For example, `{ "maxRetries": 4 }` or `{ "maxRetries": 4, "retryDelayInMs": 30000 }`.
     * - `webSocketOptions`: Configures the channelling of the AMQP connection over Web Sockets.
     * - `userAgent`      : A string to append to the built in user agent string that is passed to the service.
     */
    constructor(consumerGroup: string, connectionString: string, checkpointStore: CheckpointStore, options?: EventHubConsumerClientOptions);
    /**
     * The `EventHubConsumerClient` class is used to consume events from an Event Hub.
     * Use the `options` parmeter to configure retry policy or proxy settings.
     * @param consumerGroup - The name of the consumer group from which you want to process events.
     * @param connectionString - The connection string to use for connecting to the Event Hubs namespace.
     * It is expected that the shared key properties are contained in this connection string, but not the Event Hub path,
     * e.g. 'Endpoint=sb://my-servicebus-namespace.servicebus.windows.net/;SharedAccessKeyName=my-SA-name;SharedAccessKey=my-SA-key;'.
     * @param eventHubName - The name of the specific Event Hub to connect the client to.
     * @param options - A set of options to apply when configuring the client.
     * - `retryOptions`   : Configures the retry policy for all the operations on the client.
     * For example, `{ "maxRetries": 4 }` or `{ "maxRetries": 4, "retryDelayInMs": 30000 }`.
     * - `webSocketOptions`: Configures the channelling of the AMQP connection over Web Sockets.
     * - `userAgent`      : A string to append to the built in user agent string that is passed to the service.
     */
    constructor(consumerGroup: string, connectionString: string, eventHubName: string, options?: EventHubConsumerClientOptions);
    /**
     * The `EventHubConsumerClient` class is used to consume events from an Event Hub.
     * Use the `options` parmeter to configure retry policy or proxy settings.
     * @param consumerGroup - The name of the consumer group from which you want to process events.
     * @param connectionString - The connection string to use for connecting to the Event Hubs namespace.
     * It is expected that the shared key properties are contained in this connection string, but not the Event Hub path,
     * e.g. 'Endpoint=sb://my-servicebus-namespace.servicebus.windows.net/;SharedAccessKeyName=my-SA-name;SharedAccessKey=my-SA-key;'.
     * @param eventHubName - The name of the specific Event Hub to connect the client to.
     * @param checkpointStore - A checkpoint store that is used by the client to read checkpoints to determine
     * the position from where it should resume receiving events when your application gets restarted.
     * It is also used by the client to load balance multiple instances of your application.
     * @param options - A set of options to apply when configuring the client.
     * - `retryOptions`   : Configures the retry policy for all the operations on the client.
     * For example, `{ "maxRetries": 4 }` or `{ "maxRetries": 4, "retryDelayInMs": 30000 }`.
     * - `webSocketOptions`: Configures the channelling of the AMQP connection over Web Sockets.
     * - `userAgent`      : A string to append to the built in user agent string that is passed to the service.
     */
    constructor(consumerGroup: string, connectionString: string, eventHubName: string, checkpointStore: CheckpointStore, options?: EventHubConsumerClientOptions);
    /**
     * The `EventHubConsumerClient` class is used to consume events from an Event Hub.
     * Use the `options` parmeter to configure retry policy or proxy settings.
     * @param consumerGroup - The name of the consumer group from which you want to process events.
     * @param fullyQualifiedNamespace - The full namespace which is likely to be similar to
     * <yournamespace>.servicebus.windows.net
     * @param eventHubName - The name of the specific Event Hub to connect the client to.
     * @param credential - An credential object used by the client to get the token to authenticate the connection
     * with the Azure Event Hubs service.
     * See &commat;azure/identity for creating credentials that support AAD auth.
     * Use the `AzureNamedKeyCredential` from &commat;azure/core-auth if you want to pass in a `SharedAccessKeyName`
     * and `SharedAccessKey` without using a connection string. These fields map to the `name` and `key` field respectively
     * in `AzureNamedKeyCredential`.
     * Use the `AzureSASCredential` from &commat;azure/core-auth if you want to pass in a `SharedAccessSignature`
     * without using a connection string. This field maps to `signature` in `AzureSASCredential`.
     * @param options - A set of options to apply when configuring the client.
     * - `retryOptions`   : Configures the retry policy for all the operations on the client.
     * For example, `{ "maxRetries": 4 }` or `{ "maxRetries": 4, "retryDelayInMs": 30000 }`.
     * - `webSocketOptions`: Configures the channelling of the AMQP connection over Web Sockets.
     * - `userAgent`      : A string to append to the built in user agent string that is passed to the service.
     */
    constructor(consumerGroup: string, fullyQualifiedNamespace: string, eventHubName: string, credential: TokenCredential | NamedKeyCredential | SASCredential, options?: EventHubConsumerClientOptions);
    /**
     * The `EventHubConsumerClient` class is used to consume events from an Event Hub.
     * Use the `options` parmeter to configure retry policy or proxy settings.
     * @param consumerGroup - The name of the consumer group from which you want to process events.
     * @param fullyQualifiedNamespace - The full namespace which is likely to be similar to
     * <yournamespace>.servicebus.windows.net
     * @param eventHubName - The name of the specific Event Hub to connect the client to.
     * @param credential - An credential object used by the client to get the token to authenticate the connection
     * with the Azure Event Hubs service.
     * See &commat;azure/identity for creating credentials that support AAD auth.
     * Use the `AzureNamedKeyCredential` from &commat;azure/core-auth if you want to pass in a `SharedAccessKeyName`
     * and `SharedAccessKey` without using a connection string. These fields map to the `name` and `key` field respectively
     * in `AzureNamedKeyCredential`.
     * Use the `AzureSASCredential` from &commat;azure/core-auth if you want to pass in a `SharedAccessSignature`
     * without using a connection string. This field maps to `signature` in `AzureSASCredential`.
     * @param checkpointStore - A checkpoint store that is used by the client to read checkpoints to determine
     * the position from where it should resume receiving events when your application gets restarted.
     * It is also used by the client to load balance multiple instances of your application.
     * @param options - A set of options to apply when configuring the client.
     * - `retryOptions`   : Configures the retry policy for all the operations on the client.
     * For example, `{ "maxRetries": 4 }` or `{ "maxRetries": 4, "retryDelayInMs": 30000 }`.
     * - `webSocketOptions`: Configures the channelling of the AMQP connection over Web Sockets.
     * - `userAgent`      : A string to append to the built in user agent string that is passed to the service.
     */
    constructor(consumerGroup: string, fullyQualifiedNamespace: string, eventHubName: string, credential: TokenCredential | NamedKeyCredential | SASCredential, checkpointStore: CheckpointStore, options?: EventHubConsumerClientOptions);
    /**
     * Closes the AMQP connection to the Event Hub instance,
     * returning a promise that will be resolved when disconnection is completed.
     * @returns Promise<void>
     * @throws Error if the underlying connection encounters an error while closing.
     */
    close(): Promise<void>;
    /**
     * Provides the id for each partition associated with the Event Hub.
     * @param options - The set of options to apply to the operation call.
     * @returns A promise that resolves with an Array of strings representing the id for
     * each partition associated with the Event Hub.
     * @throws Error if the underlying connection has been closed, create a new EventHubConsumerClient.
     * @throws AbortError if the operation is cancelled via the abortSignal.
     */
    getPartitionIds(options?: GetPartitionIdsOptions): Promise<Array<string>>;
    /**
     * Provides information about the state of the specified partition.
     * @param partitionId - The id of the partition for which information is required.
     * @param options - The set of options to apply to the operation call.
     * @returns A promise that resolves with information about the state of the partition .
     * @throws Error if the underlying connection has been closed, create a new EventHubConsumerClient.
     * @throws AbortError if the operation is cancelled via the abortSignal.
     */
    getPartitionProperties(partitionId: string, options?: GetPartitionPropertiesOptions): Promise<PartitionProperties>;
    /**
     * Provides the Event Hub runtime information.
     * @param options - The set of options to apply to the operation call.
     * @returns A promise that resolves with information about the Event Hub instance.
     * @throws Error if the underlying connection has been closed, create a new EventHubConsumerClient.
     * @throws AbortError if the operation is cancelled via the abortSignal.
     */
    getEventHubProperties(options?: GetEventHubPropertiesOptions): Promise<EventHubProperties>;
    /**
     * Subscribe to events from all partitions.
     *
     * If checkpoint store is provided to the `EventHubConsumerClient` and there are multiple
     * instances of your application, then each instance will subscribe to a subset of the
     * partitions such that the load is balanced amongst them.
     *
     * Call close() on the returned object to stop receiving events.
     *
     * Example usage:
     * ```ts
     * const client = new EventHubConsumerClient(consumerGroup, connectionString, eventHubName);
     * const subscription = client.subscribe(
     *  {
     *    processEvents: (events, context) => { console.log("Received event count: ", events.length) },
     *    processError: (err, context) => { console.log("Error: ", err) }
     *  },
     *  { startPosition: earliestEventPosition }
     * );
     * ```
     *
     * @param handlers - Handlers for the lifecycle of the subscription - subscription initialization
     *                 per partition, receiving events, handling errors and the closing
     *                 of a subscription per partition.
     * @param options - Configures the way events are received.
     * Most common are `maxBatchSize` and `maxWaitTimeInSeconds` that control the flow of
     * events to the handler provided to receive events as well as the start position. For example,
     * `{ maxBatchSize: 20, maxWaitTimeInSeconds: 120, startPosition: { sequenceNumber: 123 } }`
     */
    subscribe(handlers: SubscriptionEventHandlers, options?: SubscribeOptions): Subscription;
    /**
     * Subscribe to events from a single partition.
     * Call close() on the returned object to stop receiving events.
     *
     * Example usage:
     * ```ts
     * const client = new EventHubConsumerClient(consumerGroup, connectionString, eventHubName);
     * const subscription = client.subscribe(
     *  partitionId,
     *  {
     *    processEvents: (events, context) => { console.log("Received event count: ", events.length) },
     *    processError: (err, context) => { console.log("Error: ", err) }
     *  },
     *  { startPosition: earliestEventPosition }
     * );
     * ```
     *
     * @param partitionId - The id of the partition to subscribe to.
     * @param handlers - Handlers for the lifecycle of the subscription - subscription initialization
     *                 of the partition, receiving events, handling errors and the closing
     *                 of a subscription to the partition.
     * @param options - Configures the way events are received.
     * Most common are `maxBatchSize` and `maxWaitTimeInSeconds` that control the flow of
     * events to the handler provided to receive events as well as the start position. For example,
     * `{ maxBatchSize: 20, maxWaitTimeInSeconds: 120, startPosition: { sequenceNumber: 123 } }`
     */
    subscribe(partitionId: string, handlers: SubscriptionEventHandlers, options?: SubscribeOptions): Subscription;
    /**
     * Gets the LoadBalancing strategy that should be used based on what the user provided.
     */
    private _getLoadBalancingStrategy;
    private createEventProcessorForAllPartitions;
    private createEventProcessorForSinglePartition;
    private _createEventProcessor;
}
/**
 * Describes the options that can be provided while creating the EventHubConsumerClient.
 * - `loadBalancingOptions`: Options to tune how the EventHubConsumerClient claims partitions.
 * - `userAgent`        : A string to append to the built in user agent string that is passed as a connection property
 * to the service.
 * - `webSocketOptions` : Options to configure the channelling of the AMQP connection over Web Sockets.
 *    - `websocket`     : The WebSocket constructor used to create an AMQP connection if you choose to make the connection
 * over a WebSocket.
 *    - `webSocketConstructorOptions` : Options to pass to the Websocket constructor when you choose to make the connection
 * over a WebSocket.
 * - `retryOptions`     : The retry options for all the operations on the client/producer/consumer.
 *    - `maxRetries` : The number of times the operation can be retried in case of a retryable error.
 *    - `maxRetryDelayInMs`: The maximum delay between retries. Applicable only when performing exponential retries.
 *    - `mode`: Which retry mode to apply, specified by the `RetryMode` enum. Options are `Exponential` and `Fixed`. Defaults to `Fixed`.
 *    - `retryDelayInMs`: Amount of time to wait in milliseconds before making the next attempt. When `mode` is set to `Exponential`,
 *       this is used to compute the exponentially increasing delays between retries. Default: 30000 milliseconds.
 *    - `timeoutInMs`: Amount of time in milliseconds to wait before the operation times out. This will trigger a retry if there are any
 *       retry attempts remaining. Minimum value: 60000 milliseconds.
 *
 * A simple usage can be `{ "maxRetries": 4 }`.
 *
 * Example usage:
 * ```js
 * {
 *     retryOptions: {
 *         maxRetries: 4
 *     }
 * }
 * ```
 */
export declare interface EventHubConsumerClientOptions extends EventHubClientOptions {
    /**
     * Options to tune how the EventHubConsumerClient claims partitions.
     */
    loadBalancingOptions?: LoadBalancingOptions;
}
/**
 * The `EventHubProducerClient` class is used to send events to an Event Hub.
 *
 * There are multiple ways to create an `EventHubProducerClient`
 * - Use the connection string from the SAS policy created for your Event Hub instance.
 * - Use the connection string from the SAS policy created for your Event Hub namespace,
 * and the name of the Event Hub instance
 * - Use the full namespace like `<yournamespace>.servicebus.windows.net`, and a credentials object.
 *
 * Optionally, you can also pass an options bag to configure the retry policy or proxy settings.
 *
 */
export declare class EventHubProducerClient {
    /**
     * Describes the amqp connection context for the client.
     */
    private _context;
    /**
     * The options passed by the user when creating the EventHubClient instance.
     */
    private _clientOptions;
    /**
     * Map of partitionId to senders
     */
    private _sendersMap;
    /**
     * Indicates whether or not the EventHubProducerClient should enable idempotent publishing to Event Hub partitions.
     * If enabled, the producer will only be able to publish directly to partitions;
     * it will not be able to publish to the Event Hubs gateway for automatic partition routing
     * nor will it be able to use a partition key.
     * Default: false
     */
    private _enableIdempotentRetries?;
    /**
     * The set of options that can be specified to influence publishing behavior specific to the configured Event Hub partition.
     * These options are not necessary in the majority of scenarios and are intended for use with specialized scenarios,
     * such as when recovering the state used for idempotent publishing.
     */
    private _partitionOptions?;
    /*
    * @readonly
    * The name of the Event Hub instance for which this client is created.
    */
    readonly eventHubName: string;
    /*
    * @readonly
    * The fully qualified namespace of the Event Hub instance for which this client is created.
    * This is likely to be similar to <yournamespace>.servicebus.windows.net.
    */
    readonly fullyQualifiedNamespace: string;
    /**
     * The `EventHubProducerClient` class is used to send events to an Event Hub.
     * Use the `options` parmeter to configure retry policy or proxy settings.
     * @param connectionString - The connection string to use for connecting to the Event Hub instance.
     * It is expected that the shared key properties and the Event Hub path are contained in this connection string.
     * e.g. 'Endpoint=sb://my-servicebus-namespace.servicebus.windows.net/;SharedAccessKeyName=my-SA-name;SharedAccessKey=my-SA-key;EntityPath=my-event-hub-name'.
     * @param options - A set of options to apply when configuring the client.
     * - `retryOptions`   : Configures the retry policy for all the operations on the client.
     * For example, `{ "maxRetries": 4 }` or `{ "maxRetries": 4, "retryDelayInMs": 30000 }`.
     * - `webSocketOptions`: Configures the channelling of the AMQP connection over Web Sockets.
     * - `userAgent`      : A string to append to the built in user agent string that is passed to the service.
     */
    constructor(connectionString: string, options?: EventHubClientOptions);
    /**
     * The `EventHubProducerClient` class is used to send events to an Event Hub.
     * Use the `options` parmeter to configure retry policy or proxy settings.
     * @param connectionString - The connection string to use for connecting to the Event Hubs namespace.
     * It is expected that the shared key properties are contained in this connection string, but not the Event Hub path,
     * e.g. 'Endpoint=sb://my-servicebus-namespace.servicebus.windows.net/;SharedAccessKeyName=my-SA-name;SharedAccessKey=my-SA-key;'.
     * @param eventHubName - The name of the specific Event Hub to connect the client to.
     * @param options - A set of options to apply when configuring the client.
     * - `retryOptions`   : Configures the retry policy for all the operations on the client.
     * For example, `{ "maxRetries": 4 }` or `{ "maxRetries": 4, "retryDelayInMs": 30000 }`.
     * - `webSocketOptions`: Configures the channelling of the AMQP connection over Web Sockets.
     * - `userAgent`      : A string to append to the built in user agent string that is passed to the service.
     */
    constructor(connectionString: string, eventHubName: string, options?: EventHubClientOptions);
    /**
     * The `EventHubProducerClient` class is used to send events to an Event Hub.
     * Use the `options` parmeter to configure retry policy or proxy settings.
     * @param fullyQualifiedNamespace - The full namespace which is likely to be similar to
     * <yournamespace>.servicebus.windows.net
     * @param eventHubName - The name of the specific Event Hub to connect the client to.
     * @param credential - An credential object used by the client to get the token to authenticate the connection
     * with the Azure Event Hubs service.
     * See &commat;azure/identity for creating credentials that support AAD auth.
     * Use the `AzureNamedKeyCredential` from &commat;azure/core-auth if you want to pass in a `SharedAccessKeyName`
     * and `SharedAccessKey` without using a connection string. These fields map to the `name` and `key` field respectively
     * in `AzureNamedKeyCredential`.
     * Use the `AzureSASCredential` from &commat;azure/core-auth if you want to pass in a `SharedAccessSignature`
     * without using a connection string. This field maps to `signature` in `AzureSASCredential`.
     * @param options - A set of options to apply when configuring the client.
     * - `retryOptions`   : Configures the retry policy for all the operations on the client.
     * For example, `{ "maxRetries": 4 }` or `{ "maxRetries": 4, "retryDelayInMs": 30000 }`.
     * - `webSocketOptions`: Configures the channelling of the AMQP connection over Web Sockets.
     * - `userAgent`      : A string to append to the built in user agent string that is passed to the service.
     */
    constructor(fullyQualifiedNamespace: string, eventHubName: string, credential: TokenCredential | NamedKeyCredential | SASCredential, options?: EventHubClientOptions);
    /**
     * Creates an instance of `EventDataBatch` to which one can add events until the maximum supported size is reached.
     * The batch can be passed to the {@link sendBatch} method of the `EventHubProducerClient` to be sent to Azure Event Hubs.
     *
     * Events with different values for partitionKey or partitionId will need to be put into different batches.
     * To simplify such batch management across partitions or to have the client automatically batch events
     * and send them in specific intervals, use `EventHubBufferedProducerClient` instead.
     *
     * The below example assumes you have an array of events at hand to be batched safely.
     * If you have events coming in one by one, `EventHubBufferedProducerClient` is recommended instead
     * for effecient management of batches.
     *
     * Example usage:
     * ```ts
     * const client = new EventHubProducerClient(connectionString);
     * let batch = await client.createBatch();
     * for (let i = 0; i < messages.length; i++) {
     *  if (!batch.tryAdd(messages[i])) {
     *    await client.sendBatch(batch);
     *    batch = await client.createBatch();
     *    if (!batch.tryAdd(messages[i])) {
     *      throw new Error("Message too big to fit")
     *    }
     *    if (i === messages.length - 1) {
     *      await client.sendBatch(batch);
     *    }
     *   }
     * }
     * ```
     *
     * @param options -  Configures the behavior of the batch.
     * - `partitionKey`  : A value that is hashed and used by the Azure Event Hubs service to determine the partition to which
     * the events need to be sent.
     * - `partitionId`   : Id of the partition to which the batch of events need to be sent.
     * - `maxSizeInBytes`: The upper limit for the size of batch. The `tryAdd` function will return `false` after this limit is reached.
     * - `abortSignal`   : A signal the request to cancel the operation.
     * @returns Promise<EventDataBatch>
     * @throws Error if both `partitionId` and `partitionKey` are set in the options.
     * @throws Error if the underlying connection has been closed, create a new EventHubProducerClient.
     * @throws AbortError if the operation is cancelled via the abortSignal in the options.
     */
    createBatch(options?: CreateBatchOptions): Promise<EventDataBatch>;
    /**
     * Get the information about the state of publishing for a partition as observed by the `EventHubProducerClient`.
     * This data can always be read, but will only be populated with information relevant to the active features
     * for the producer client.
     *
     * @param partitionId - Id of the partition from which to retrieve publishing properties.
     * @param options - The set of options to apply to the operation call.
     * - `abortSignal`  : A signal the request to cancel the send operation.
     * @returns Promise<void>
     * @throws AbortError if the operation is cancelled via the abortSignal.
     */
    private getPartitionPublishingProperties;
    /**
     * Sends an array of events as a batch to the associated Event Hub.
     *
     * Azure Event Hubs has a limit on the size of the batch that can be sent which if exceeded
     * will result in an error with code `MessageTooLargeError`.
     * To safely send within batch size limits, use `EventHubProducerClient.createBatch()` or
     * `EventHubBufferedProducerClient` instead.
     *
     * Example usage:
     * ```ts
     * const client = new EventHubProducerClient(connectionString);
     * await client.sendBatch(messages);
     * ```
     *
     * @param batch - An array of {@link EventData} or `AmqpAnnotatedMessage`.
     * @param options - A set of options that can be specified to influence the way in which
     * events are sent to the associated Event Hub.
     * - `abortSignal`  : A signal the request to cancel the send operation.
     * - `partitionId`  : The partition this batch will be sent to. If set, `partitionKey` can not be set.
     * - `partitionKey` : A value that is hashed to produce a partition assignment. If set, `partitionId` can not be set.
     *
     * @returns Promise<void>
     * @throws MessageTooLargeError if all the events in the input array cannot be fit into a batch.
     * @throws AbortError if the operation is cancelled via the abortSignal.
     * @throws MessagingError if an error is encountered while sending a message.
     * @throws Error if the underlying connection or sender has been closed.
     */
    sendBatch(batch: EventData[] | AmqpAnnotatedMessage[], options?: SendBatchOptions): Promise<void>;
    /**
     * Sends a batch of events created using `EventHubProducerClient.createBatch()` to the associated Event Hub.
     *
     * Events with different values for partitionKey or partitionId will need to be put into different batches.
     * To simplify such batch management across partitions or to have the client automatically batch events
     * and send them in specific intervals, use `EventHubBufferedProducerClient` instead.
     *
     * The below example assumes you have an array of events at hand to be batched safely.
     * If you have events coming in one by one, `EventHubBufferedProducerClient` is recommended instead
     * for effecient management of batches.
     *
     * Example usage:
     * ```ts
     * const client = new EventHubProducerClient(connectionString);
     * let batch = await client.createBatch();
     * for (let i = 0; i < messages.length; i++) {
     *  if (!batch.tryAdd(messages[i])) {
     *    await client.sendBatch(batch);
     *    batch = await client.createBatch();
     *    if (!batch.tryAdd(messages[i])) {
     *      throw new Error("Message too big to fit")
     *    }
     *    if (i === messages.length - 1) {
     *      await client.sendBatch(batch);
     *    }
     *   }
     * }
     * ```
     * @param batch - A batch of events that you can create using the {@link createBatch} method.
     * @param options - A set of options that can be specified to influence the way in which
     * events are sent to the associated Event Hub.
     * - `abortSignal`  : A signal the request to cancel the send operation.
     *
     * @returns Promise<void>
     * @throws AbortError if the operation is cancelled via the abortSignal.
     * @throws MessagingError if an error is encountered while sending a message.
     * @throws Error if the underlying connection or sender has been closed.
     */
    sendBatch(batch: EventDataBatch, options?: OperationOptions): Promise<void>;
    /**
     * Closes the AMQP connection to the Event Hub instance,
     * returning a promise that will be resolved when disconnection is completed.
     * @returns Promise<void>
     * @throws Error if the underlying connection encounters an error while closing.
     */
    close(): Promise<void>;
    /**
     * Provides the Event Hub runtime information.
     * @param options - The set of options to apply to the operation call.
     * @returns A promise that resolves with information about the Event Hub instance.
     * @throws Error if the underlying connection has been closed, create a new EventHubProducerClient.
     * @throws AbortError if the operation is cancelled via the abortSignal.
     */
    getEventHubProperties(options?: GetEventHubPropertiesOptions): Promise<EventHubProperties>;
    /**
     * Provides the id for each partition associated with the Event Hub.
     * @param options - The set of options to apply to the operation call.
     * @returns A promise that resolves with an Array of strings representing the id for
     * each partition associated with the Event Hub.
     * @throws Error if the underlying connection has been closed, create a new EventHubProducerClient.
     * @throws AbortError if the operation is cancelled via the abortSignal.
     */
    getPartitionIds(options?: GetPartitionIdsOptions): Promise<Array<string>>;
    /**
     * Provides information about the state of the specified partition.
     * @param partitionId - The id of the partition for which information is required.
     * @param options - The set of options to apply to the operation call.
     * @returns A promise that resolves with information about the state of the partition .
     * @throws Error if the underlying connection has been closed, create a new EventHubProducerClient.
     * @throws AbortError if the operation is cancelled via the abortSignal.
     */
    getPartitionProperties(partitionId: string, options?: GetPartitionPropertiesOptions): Promise<PartitionProperties>;
}
/**
 * Describes the runtime information of an Event Hub.
 */
export declare interface EventHubProperties {
    /**
     * The name of the event hub.
     */
    name: string;
    /**
     * The date and time the hub was created in UTC.
     */
    createdOn: Date;
    /**
     * The slice of string partition identifiers.
     */
    partitionIds: string[];
}
/**
 * Represents the position of an event in an Event Hub partition, typically used when calling the `subscribe()`
 * method on an `EventHubConsumerClient` to specify the position in the partition to begin receiving events from.
 *
 * To get an EventPosition representing the start or end of the stream, use the constants
 * `earliestEventPosition` and `latestEventPosition` respectively.
 *
 */
export declare interface EventPosition {
    /**
     * The offset of the event identified by this position.
     * Expected to be undefined if the position is just created from a sequence number or an enqueued time.
     *
     * The offset is the relative position for an event in the context of the partition.
     * The offset should not be considered a stable value.
     * The same offset may refer to a different event as events reach the age limit for
     * retention and are no longer visible within the partition.
     */
    offset?: number | "@latest";
    /**
     * Indicates if the specified offset is inclusive of the event which it identifies.
     * This information is only relevent if the event position was identified by an offset or sequence number.
     * Default value: `false`.
     */
    isInclusive?: boolean;
    /**
     * The enqueued time in UTC of the event identified by this position.
     * When provided as a number this value is the number of milliseconds since the Unix Epoch.
     * Expected to be undefined if the position is just created from a sequence number or an offset.
     */
    enqueuedOn?: Date | number;
    /**
     * The sequence number of the event identified by this position.
     * Expected to be undefined if the position is just created from an offset or enqueued time.
     */
    sequenceNumber?: number;
}
/**
 * The set of options to configure the behavior of `getEventHubProperties`.
 * - `abortSignal`  : An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
 * - `parentSpan` : The `Span` or `SpanContext` to use as the `parent` of the span created while calling this operation.
 */
export declare interface GetEventHubPropertiesOptions extends OperationOptions {
}
/**
 * The set of options to configure the behavior of `getPartitionIds`.
 * - `abortSignal`  : An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
 * - `parentSpan` : The `Span` or `SpanContext` to use as the `parent` of the span created while calling this operation.
 */
export declare interface GetPartitionIdsOptions extends OperationOptions {
}
/**
 * The set of options to configure the behavior of `getPartitionProperties`.
 * - `abortSignal`  : An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
 * - `parentSpan` : The `Span` or `SpanContext` to use as the `parent` of the span created while calling this operation.
 */
export declare interface GetPartitionPropertiesOptions extends OperationOptions {
}
/**
 * A set of information about the last enqueued event of a partition, as observed by the consumer as
 * events are received from the Event Hubs service
 */
export declare interface LastEnqueuedEventProperties {
    /**
     * The sequence number of the event that was last enqueued into the Event Hub partition from which
     * this event was received.
     */
    sequenceNumber?: number;
    /**
     * The date and time, in UTC, that the last event was enqueued into the Event Hub partition from
     * which this event was received.
     */
    enqueuedOn?: Date;
    /**
     * The offset of the event that was last enqueued into the Event Hub partition from which
     * this event was received.
     */
    offset?: string;
    /**
     * The date and time, in UTC, that the last event was retrieved from the Event Hub partition.
     */
    retrievedOn?: Date;
}
/**
 * Gets the `EventPosition` corresponding to the end of the partition.
 * Pass this position to the `EventHubConsumerClient.subscribe()` method to begin receiving events from the
 * event that is enqueued right after the method call.
 * @returns EventPosition
 */
export declare const latestEventPosition: EventPosition;
/**
 * An options bag to configure load balancing settings.
 */
export declare interface LoadBalancingOptions {
    /**
     * Whether to apply a greedy or a more balanced approach when
     * claiming partitions.
     *
     * - balanced: The `EventHubConsumerClient` will take a measured approach to
     * requesting partition ownership when balancing work with other clients,
     * slowly claiming partitions until a stabilized distribution is achieved.
     *
     * - greedy: The `EventHubConsumerClient` will attempt to claim ownership
     * of its fair share of partitions aggressively when balancing work with
     * other clients.
     *
     * This option is ignored when either:
     *   - `CheckpointStore` is __not__ provided to the `EventHubConsumerClient`.
     *   - `subscribe()` is called for a single partition.
     * Default: balanced
     */
    strategy?: "balanced" | "greedy";
    /**
     * The length of time between attempts to claim partitions.
     * Default: 10000
     */
    updateIntervalInMs?: number;
    /**
     * The length of time a partition claim is valid.
     * Default: 60000
     */
    partitionOwnershipExpirationIntervalInMs?: number;
}
/**
 * The `@azure/logger` configuration for this package.
 * This will output logs using the `azure:event-hubs` namespace prefix.
 */
export declare const logger: AzureLogger;
/**
 * A message adapter interface that specifies methods for producing and consuming
 * messages with payloads and content type fields.
 *
 * This interface is hidden because it is already exported by `@azure/schema-registry-avro`
 *
 * @hidden
 */
export declare interface MessageAdapter<MessageT> {
    /**
     * defines how to create a message from a payload and a content type
     */
    produce: (MessageContent: MessageContent) => MessageT;
    /**
     * defines how to access the payload and the content type of a message
     */
    consume: (message: MessageT) => MessageContent;
}
/**
 * A message with payload and content type fields
 *
 * This interface is hidden because it is already exported by `@azure/schema-registry-avro`
 *
 * @hidden
 */
export declare interface MessageContent {
    /**
     * The message's binary data
     */
    data: Uint8Array;
    /**
     * The message's content type
     */
    contentType: string;
}
export { MessagingError };
/**
 * Contains the events that were not successfully sent to the Event Hub,
 * the partition they were assigned to, and the error that was encountered while sending.
 */
export declare interface OnSendEventsErrorContext {
    /**
     * The partition each event was assigned.
     */
    partitionId: string;
    /**
     * The array of {@link EventData} and/or `AmqpAnnotatedMessage` that were not successfully sent to the Event Hub.
     */
    events: Array<EventData | AmqpAnnotatedMessage>;
    /**
     * The error that occurred when sending the associated events to the Event Hub.
     */
    error: Error;
}
/**
 * Contains the events that were successfully sent to the Event Hub,
 * and the partition they were assigned to.
 */
export declare interface OnSendEventsSuccessContext {
    /**
     * The partition each event was assigned.
     */
    partitionId: string;
    /**
     * The array of {@link EventData} and/or `AmqpAnnotatedMessage` that were successfully sent to the Event Hub.
     */
    events: Array<EventData | AmqpAnnotatedMessage>;
}
/**
 * Options for configuring tracing and the abortSignal.
 */
export declare interface OperationOptions {
    /**
     * The signal which can be used to abort requests.
     */
    abortSignal?: AbortSignalLike;
    /**
     * Options for configuring tracing.
     */
    tracingOptions?: OperationTracingOptions;
}
/**
 * Parses given connection string into the different properties applicable to Azure Event Hubs.
 * The properties are useful to then construct an EventHubProducerClient or an EventHubConsumerClient.
 * @param connectionString - The connection string associated with the Shared Access Policy created
 * for the Event Hubs namespace.
 */
export declare function parseEventHubConnectionString(connectionString: string): Readonly<EventHubConnectionStringProperties>;
/**
 * Interface that describes the context passed to each of the functions that are a part
 * of the `SubscriptionEventHandlers`. When implementing any of these functions, use
 * the context object to get information about the partition as well as the
 * ability to checkpoint.
 */
export declare interface PartitionContext {
    /**
     * The fully qualified Event Hubs namespace. This is likely to be similar to
     * <yournamespace>.servicebus.windows.net
     */
    readonly fullyQualifiedNamespace: string;
    /**
     * The event hub name.
     */
    readonly eventHubName: string;
    /**
     * The consumer group name.
     */
    readonly consumerGroup: string;
    /**
     * The identifier of the Event Hub partition.
     */
    readonly partitionId: string;
    /**
     * Information on the last enqueued event in the partition that is being processed.
     * This property is only updated if the `trackLastEnqueuedEventProperties` option is set to true
     * when creating an instance of EventProcessor.
     * @readonly
     */
    readonly lastEnqueuedEventProperties?: LastEnqueuedEventProperties;
    /**
     * Updates the checkpoint using the event data.
     *
     * A checkpoint is meant to represent the last successfully processed event by the user from a particular
     * partition of a consumer group in an Event Hub instance.
     *
     * @param eventData - The event that you want to update the checkpoint with.
     */
    updateCheckpoint(eventData: ReceivedEventData): Promise<void>;
}
/**
 * An interface representing the details on which instance of a `EventProcessor` owns processing
 * of a given partition from a consumer group of an Event Hub instance.
 *
 * **Note**: This is used internally by the `EventProcessor` and user never has to create it directly.
 */
export declare interface PartitionOwnership {
    /**
     * The fully qualified Event Hubs namespace. This is likely to be similar to
     * <yournamespace>.servicebus.windows.net
     */
    fullyQualifiedNamespace: string;
    /**
     * The event hub name
     */
    eventHubName: string;
    /**
     * The consumer group name
     */
    consumerGroup: string;
    /**
     * The identifier of the Event Hub partition.
     */
    partitionId: string;
    /**
     * The unique identifier of the event processor.
     */
    ownerId: string;
    /**
     * The last modified time.
     */
    lastModifiedTimeInMs?: number;
    /**
     * The unique identifier for the operation.
     */
    etag?: string;
}
/**
 * Describes the runtime information of an EventHub Partition.
 */
export declare interface PartitionProperties {
    /**
     * The name of the Event Hub.
     */
    eventHubName: string;
    /**
     * Identifier of the partition within the Event Hub.
     */
    partitionId: string;
    /**
     * The starting sequence number of the partition's message log.
     */
    beginningSequenceNumber: number;
    /**
     * The last sequence number of the partition's message log.
     */
    lastEnqueuedSequenceNumber: number;
    /**
     * The offset of the last enqueued message in the partition's message log.
     */
    lastEnqueuedOffset: number;
    /**
     * The time of the last enqueued message in the partition's message log in UTC.
     */
    lastEnqueuedOnUtc: Date;
    /**
     * Indicates whether the partition is empty.
     */
    isEmpty: boolean;
}
/**
 * Signature of the user provided function invoked by `EventHubConsumerClient` just after stopping to receive
 * events from a partition.
 */
export declare type ProcessCloseHandler = (reason: CloseReason, context: PartitionContext) => Promise<void>;
/**
 * Signature of the user provided function invoked by `EventHubConsumerClient` for errors that occur when
 * receiving events or when executing any of the user provided functions passed to the `subscribe()` method.
 */
export declare type ProcessErrorHandler = (error: Error | MessagingError, context: PartitionContext) => Promise<void>;
/**
 * Signature of the user provided function invoked by `EventHubConsumerClient` when a set of events is received.
 */
export declare type ProcessEventsHandler = (events: ReceivedEventData[], context: PartitionContext) => Promise<void>;
/**
 * Signature of the user provided function invoked by `EventHubConsumerClient` just before starting to receive
 * events from a partition.
 */
export declare type ProcessInitializeHandler = (context: PartitionContext) => Promise<void>;
/**
 * The interface that describes the structure of the event received from Event Hub.
 * Use this as a reference when creating the `processEvents` function to process the events
 * recieved from an Event Hub when using the `EventHubConsumerClient`.
 */
export declare interface ReceivedEventData {
    /**
     * The message body that needs to be sent or is received.
     */
    body: any;
    /**
     * The application specific properties.
     */
    properties?: {
        [key: string]: any;
    };
    /**
     * The enqueued time of the event.
     */
    enqueuedTimeUtc: Date;
    /**
     * When specified Event Hub will hash this to a partitionId.
     * It guarantees that messages end up in a specific partition on the event hub.
     */
    partitionKey: string | null;
    /**
     * The offset of the event.
     */
    offset: number;
    /**
     * The sequence number of the event.
     */
    sequenceNumber: number;
    /**
     * The properties set by the service.
     */
    systemProperties?: {
        [key: string]: any;
    };
    /**
     * The content type of the message. Optionally describes
     * the payload of the message, with a descriptor following the format of RFC2045, Section 5, for
     * example "application/json".
     */
    contentType?: string;
    /**
     * The correlation identifier that allows an
     * application to specify a context for the message for the purposes of correlation, for example
     * reflecting the MessageId of a message that is being replied to.
     */
    correlationId?: string | number | Buffer;
    /**
     * The message identifier is an
     * application-defined value that uniquely identifies the message and its payload.
     */
    messageId?: string | number | Buffer;
    /**
     * Returns the underlying raw amqp message.
     */
    getRawAmqpMessage(): AmqpAnnotatedMessage;
}
export { RetryMode };
export { RetryOptions };
/**
 * Options to configure the `sendBatch` method on the `EventHubProducerClient`
 * when sending an array of events.
 * If `partitionId` is set, `partitionKey` must not be set and vice versa.
 *
 * - `partitionId`  : The partition this batch will be sent to.
 * - `partitionKey` : A value that is hashed to produce a partition assignment.
 * - `abortSignal`  : A signal used to cancel the send operation.
 */
export declare interface SendBatchOptions extends OperationOptions {
    /**
     * The partition this batch will be sent to.
     * If this value is set then partitionKey can not be set.
     */
    partitionId?: string;
    /**
     * A value that is hashed to produce a partition assignment.
     * It guarantees that messages with the same partitionKey end up in the same partition.
     * Specifying this will throw an error if the producer was created using a `paritionId`.
     */
    partitionKey?: string;
}
/**
 * Options to configure the `subscribe` method on the `EventHubConsumerClient`.
 * For example, `{ maxBatchSize: 20, maxWaitTimeInSeconds: 120, startPosition: { sequenceNumber: 123 } }`
 */
export declare interface SubscribeOptions {
    /**
     * The number of events to request per batch
     */
    maxBatchSize?: number;
    /**
     * The maximum amount of time to wait to build up the requested message count before
     * passing the data to user code for processing. If not provided, it defaults to 60 seconds.
     */
    maxWaitTimeInSeconds?: number;
    /**
     * The event position in a partition to start receiving events from if no checkpoint is found.
     * Pass a map of partition id to position if you would like to use different starting
     * position for each partition.
     */
    startPosition?: EventPosition | {
        [partitionId: string]: EventPosition;
    };
    /**
     * Indicates whether or not the consumer should request information on the last enqueued event on its
     * associated partition, and track that information as events are received.

     * When information about the partition's last enqueued event is being tracked, each event received
     * from the Event Hubs service will carry metadata about the partition that it otherwise would not. This results in a small amount of
     * additional network bandwidth consumption that is generally a favorable trade-off when considered
     * against periodically making requests for partition properties using the Event Hub client.
     */
    trackLastEnqueuedEventProperties?: boolean;
    /**
     * The owner level to use as this subscription subscribes to partitions.
     */
    ownerLevel?: number;
    /**
     * Options for configuring tracing.
     */
    tracingOptions?: OperationTracingOptions;
    /**
     * Option to disable the client from running JSON.parse() on the message body when receiving the message.
     * Not applicable if the message was sent with AMQP body type value or sequence. Use this option when you
     * prefer to work directly with the bytes present in the message body than have the client attempt to parse it.
     */
    skipParsingBodyAsJson?: boolean;
}
/**
 * Interface that describes the object returned by the `subscribe()` method on the `EventHubConsumerClient`.
 */
export declare interface Subscription {
    /**
     * Stops the subscription from receiving more messages.
     *
     * If a checkpoint store has been configured this will also mark this subscription's
     * partitions as abandoned, freeing them up to be read by other consumers.
     *
     * @returns Promise<void>
     * @throws Error if the underlying connection encounters an error while closing.
     */
    close(): Promise<void>;
    /**
     * Indicates whether the receiver is running.
     * `true` - is running; `false` otherwise.
     * @readonly
     */
    isRunning: boolean;
}
/**
 * Interface that describes the functions to be implemented by the user which are invoked by
 * the `EventHubConsumerClient` when the `subscribe()` method is called to receive events
 * from Event Hub.
 */
export declare interface SubscriptionEventHandlers {
    /**
     * The function invoked by `EventHubConsumerClient` when a set of events is received. The
     * `PartitionContext` passed to this function can be used to determine which partition is being read from.
     *
     * The `updateCheckpoint()` method on the context can be used to update checkpoints in the `CheckpointStore`
     * (if one was provided to the client). Use this in frequent intervals to mark events that have been processed
     * so that the client can restart from such checkpoints in the event of a restart or error recovery.
     *
     * Note: It is possible for received events to be an empty array.
     * This can happen if there are no new events to receive
     * in the `maxWaitTimeInSeconds`, which is defaulted to 60 seconds.
     * The `maxWaitTimeInSeconds` can be changed by setting
     * it in the `options` passed to `subscribe()`.
     */
    processEvents: ProcessEventsHandler;
    /**
     * The function invoked by `EventHubConsumerClient` for errors that occur when receiving events
     * or when executing any of the user provided functions passed to the `subscribe()` method.
     *
     * The `PartitionContext` passed to this function will indicate the partition that was being processed
     * when the error was thrown. In cases where an error is thrown outside of processing events from a
     * partition(e.g. failure to perform load balancing), the `partitionId` on the context will be an empty string.
     *
     * After the client completes executing this function, the `partitionClose` function is invoked.
     */
    processError: ProcessErrorHandler;
    /**
     * The function invoked by `EventHubConsumerClient` each time the subscription is about to begin
     * reading from a partition. The `PartitionContext` passed to this function can be used to determine
     * which partition is about to be read from.
     *
     * The client will start receiving events for the partition only after completing the execution of
     * this function (if provided). Therefore, use this function to carry out any setup work including
     * async tasks.
     */
    processInitialize?: ProcessInitializeHandler;
    /**
     * The function invoked by `EventHubConsumerClient` each time the subscription stops reading events from
     * a partition. The information on this partition will be available on the `PartitionContext` passed to the
     * function `processClose`.
     *
     * If the `CloseReason` passed to this function is `OwnershipLost`, then another subscription has taken over
     * reading from the same partition using the same consumer group. This is expected if you have multiple
     * instances of your application running and have passed the `CheckpointStore` to the client to load balance.
     *
     * If the `CloseReason` is `Shutdown`, this indicates that either `subscription.close()` was called, or an
     * error occured. Unless the subscription was explicitly closed via `subscription.close()`, the subscription
     * will attempt to resume reading events from the last checkpoint for the partition.
     */
    processClose?: ProcessCloseHandler;
}
export { TokenCredential };
/**
 * Options to configure the behavior of the `tryAdd` method on the `EventDataBatch` class.
 */
export declare interface TryAddOptions {
    /**
     * The options to use when creating Spans for tracing.
     */
    tracingOptions?: OperationTracingOptions;
}
export { WebSocketImpl };
export { WebSocketOptions };
export {};
