"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BasicSensorCreator = void 0;
const z2mModels_1 = require("../z2mModels");
const monitor_1 = require("./monitor");
const helpers_1 = require("../helpers");
const hap_1 = require("../hap");
class BasicSensorMapping {
    constructor(name, type, implementation) {
        this.name = name;
        this.type = type;
        this.implementation = implementation;
    }
}
class BasicSensorHandler {
    constructor(accessory, sensorExpose, otherExposes, identifierGen, service, additionalSubType) {
        this.sensorExpose = sensorExpose;
        this.monitors = [];
        this.identifier = '';
        const endpoint = sensorExpose.endpoint;
        let sub = endpoint;
        if (additionalSubType !== undefined) {
            if (sub === undefined) {
                sub = additionalSubType;
            }
            else {
                sub += ' ' + additionalSubType;
            }
        }
        this.serviceName = accessory.getDefaultServiceDisplayName(sub);
        this.identifier = identifierGen(endpoint);
        this.service = accessory.getOrAddService(service(this.serviceName, sub));
        this.tryCreateLowBattery(otherExposes, this.service);
        this.tryCreateTamper(otherExposes, this.service);
    }
    get getableKeys() {
        const keys = [];
        if ((0, z2mModels_1.exposesCanBeGet)(this.sensorExpose)) {
            keys.push(this.sensorExpose.property);
        }
        if (this.tamperExpose !== undefined && (0, z2mModels_1.exposesCanBeGet)(this.tamperExpose)) {
            keys.push(this.tamperExpose.property);
        }
        if (this.lowBatteryExpose !== undefined && (0, z2mModels_1.exposesCanBeGet)(this.lowBatteryExpose)) {
            keys.push(this.lowBatteryExpose.property);
        }
        return keys;
    }
    createOptionalGenericCharacteristics(exposes, service) {
        this.tryCreateTamper(exposes, service);
        this.tryCreateLowBattery(exposes, service);
    }
    tryCreateTamper(exposes, service) {
        this.tamperExpose = exposes.find(e => e.name === 'tamper' && (0, z2mModels_1.exposesIsPublished)(e));
        if (this.tamperExpose !== undefined) {
            (0, helpers_1.getOrAddCharacteristic)(service, hap_1.hap.Characteristic.StatusTampered);
            const mapping = new Map();
            mapping.set(this.tamperExpose.value_on, hap_1.hap.Characteristic.StatusTampered.TAMPERED);
            mapping.set(this.tamperExpose.value_off, hap_1.hap.Characteristic.StatusTampered.NOT_TAMPERED);
            this.monitors.push(new monitor_1.MappingCharacteristicMonitor(this.tamperExpose.property, service, hap_1.hap.Characteristic.StatusTampered, mapping));
        }
    }
    tryCreateLowBattery(exposes, service) {
        this.lowBatteryExpose = exposes.find(e => e.name === 'battery_low' && (0, z2mModels_1.exposesIsPublished)(e));
        if (this.lowBatteryExpose !== undefined) {
            (0, helpers_1.getOrAddCharacteristic)(service, hap_1.hap.Characteristic.StatusLowBattery);
            const mapping = new Map();
            mapping.set(this.lowBatteryExpose.value_on, hap_1.hap.Characteristic.StatusLowBattery.BATTERY_LEVEL_LOW);
            mapping.set(this.lowBatteryExpose.value_off, hap_1.hap.Characteristic.StatusLowBattery.BATTERY_LEVEL_NORMAL);
            this.monitors.push(new monitor_1.MappingCharacteristicMonitor(this.lowBatteryExpose.property, service, hap_1.hap.Characteristic.StatusLowBattery, mapping));
        }
    }
    updateState(state) {
        this.monitors.forEach(m => m.callback(state));
    }
}
class HumiditySensorHandler extends BasicSensorHandler {
    constructor(expose, allExposes, accessory) {
        super(accessory, expose, allExposes, HumiditySensorHandler.generateIdentifier, (n, t) => new hap_1.hap.Service.HumiditySensor(n, t));
        accessory.log.debug(`Configuring HumiditySensor for ${this.serviceName}`);
        const characteristic = (0, helpers_1.getOrAddCharacteristic)(this.service, hap_1.hap.Characteristic.CurrentRelativeHumidity);
        (0, helpers_1.copyExposesRangeToCharacteristic)(expose, characteristic);
        this.monitors.push(new monitor_1.PassthroughCharacteristicMonitor(expose.property, this.service, hap_1.hap.Characteristic.CurrentRelativeHumidity));
    }
    static generateIdentifier(endpoint) {
        let identifier = hap_1.hap.Service.HumiditySensor.UUID;
        if (endpoint !== undefined) {
            identifier += '_' + endpoint.trim();
        }
        return identifier;
    }
}
class AirPressureSensorHandler extends BasicSensorHandler {
    constructor(expose, allExposes, accessory) {
        super(accessory, expose, allExposes, AirPressureSensorHandler.generateIdentifier, (n, t) => AirPressureSensorHandler.AirPressureSensor(n, t));
        accessory.log.debug(`Configuring AirPressureSensor for ${this.serviceName}`);
        this.monitors.push(new monitor_1.PassthroughCharacteristicMonitor(expose.property, this.service, AirPressureSensorHandler.CharacteristicName));
    }
    static AirPressureSensor(displayName, subtype) {
        const service = new hap_1.hap.Service(displayName, AirPressureSensorHandler.ServiceUUID, subtype);
        service.addCharacteristic(AirPressureSensorHandler.AirPressure);
        return service;
    }
    static get AirPressure() {
        const characteristic = new hap_1.hap.Characteristic(AirPressureSensorHandler.CharacteristicName, AirPressureSensorHandler.CharacteristicUUID, {
            format: "uint16" /* UINT16 */,
            perms: ["pr" /* PAIRED_READ */, "ev" /* NOTIFY */],
            minValue: 700,
            maxValue: 1100,
            minStep: 1,
        });
        characteristic.value = 1013;
        return characteristic;
    }
    static generateIdentifier(endpoint) {
        let identifier = AirPressureSensorHandler.ServiceUUID;
        if (endpoint !== undefined) {
            identifier += '_' + endpoint.trim();
        }
        return identifier;
    }
}
AirPressureSensorHandler.ServiceUUID = 'E863F00A-079E-48FF-8F27-9C2605A29F52';
AirPressureSensorHandler.CharacteristicUUID = 'E863F10F-079E-48FF-8F27-9C2605A29F52';
AirPressureSensorHandler.CharacteristicName = 'Air Pressure';
class TemperatureSensorHandler extends BasicSensorHandler {
    constructor(expose, allExposes, accessory) {
        super(accessory, expose, allExposes, TemperatureSensorHandler.generateIdentifier, (n, t) => new hap_1.hap.Service.TemperatureSensor(n, t));
        accessory.log.debug(`Configuring TemperatureSensor for ${this.serviceName}`);
        const characteristic = (0, helpers_1.getOrAddCharacteristic)(this.service, hap_1.hap.Characteristic.CurrentTemperature);
        if (!(0, helpers_1.copyExposesRangeToCharacteristic)(expose, characteristic)) {
            // Cannot take over range from exposes entry -> Set default range
            characteristic.setProps({
                minValue: -100,
                maxValue: 100,
            });
        }
        this.monitors.push(new monitor_1.PassthroughCharacteristicMonitor(expose.property, this.service, hap_1.hap.Characteristic.CurrentTemperature));
    }
    static generateIdentifier(endpoint) {
        let identifier = hap_1.hap.Service.TemperatureSensor.UUID;
        if (endpoint !== undefined) {
            identifier += '_' + endpoint.trim();
        }
        return identifier;
    }
}
class LightSensorHandler extends BasicSensorHandler {
    constructor(expose, allExposes, accessory) {
        super(accessory, expose, allExposes, LightSensorHandler.generateIdentifier, (n, t) => new hap_1.hap.Service.LightSensor(n, t));
        accessory.log.debug(`Configuring LightSensor for ${this.serviceName}`);
        const characteristic = (0, helpers_1.getOrAddCharacteristic)(this.service, hap_1.hap.Characteristic.CurrentAmbientLightLevel);
        if (!(0, helpers_1.copyExposesRangeToCharacteristic)(expose, characteristic)) {
            // Cannot take over range from exposes entry -> Set default props
            characteristic.setProps({
                minValue: 0,
            });
        }
        this.monitors.push(new monitor_1.PassthroughCharacteristicMonitor(expose.property, this.service, hap_1.hap.Characteristic.CurrentAmbientLightLevel));
    }
    static generateIdentifier(endpoint) {
        let identifier = hap_1.hap.Service.LightSensor.UUID;
        if (endpoint !== undefined) {
            identifier += '_' + endpoint.trim();
        }
        return identifier;
    }
}
class BinarySensorHandler extends BasicSensorHandler {
    constructor(accessory, expose, otherExposes, identifierGen, logName, service, characteristic, hapOnValue, hapOffValue, additionalSubType) {
        super(accessory, expose, otherExposes, identifierGen, service, additionalSubType);
        accessory.log.debug(`Configuring ${logName} for ${this.serviceName}`);
        (0, helpers_1.getOrAddCharacteristic)(this.service, characteristic);
        const mapping = new Map();
        mapping.set(expose.value_on, hapOnValue);
        mapping.set(expose.value_off, hapOffValue);
        this.monitors.push(new monitor_1.MappingCharacteristicMonitor(expose.property, this.service, characteristic, mapping));
    }
}
class ContactSensorHandler extends BinarySensorHandler {
    constructor(expose, otherExposes, accessory) {
        super(accessory, expose, otherExposes, ContactSensorHandler.generateIdentifier, 'ContactSensor', (n, t) => new hap_1.hap.Service.ContactSensor(n, t), hap_1.hap.Characteristic.ContactSensorState, hap_1.hap.Characteristic.ContactSensorState.CONTACT_NOT_DETECTED, hap_1.hap.Characteristic.ContactSensorState.CONTACT_DETECTED);
    }
    static generateIdentifier(endpoint) {
        let identifier = hap_1.hap.Service.ContactSensor.UUID;
        if (endpoint !== undefined) {
            identifier += '_' + endpoint.trim();
        }
        return identifier;
    }
}
class OccupancySensorHandler extends BinarySensorHandler {
    constructor(expose, otherExposes, accessory) {
        super(accessory, expose, otherExposes, OccupancySensorHandler.generateIdentifier, 'OccupancySensor', (n, t) => new hap_1.hap.Service.OccupancySensor(n, t), hap_1.hap.Characteristic.OccupancyDetected, hap_1.hap.Characteristic.OccupancyDetected.OCCUPANCY_DETECTED, hap_1.hap.Characteristic.OccupancyDetected.OCCUPANCY_NOT_DETECTED);
    }
    static generateIdentifier(endpoint) {
        let identifier = hap_1.hap.Service.OccupancySensor.UUID;
        if (endpoint !== undefined) {
            identifier += '_' + endpoint.trim();
        }
        return identifier;
    }
}
class PresenceSensorHandler extends BinarySensorHandler {
    constructor(expose, otherExposes, accessory) {
        super(accessory, expose, otherExposes, PresenceSensorHandler.generateIdentifier, 'Occupancy Sensor (presence)', (n, t) => new hap_1.hap.Service.OccupancySensor(n, (PresenceSensorHandler.NAME + ' ' + (t !== null && t !== void 0 ? t : '')).trim()), hap_1.hap.Characteristic.OccupancyDetected, hap_1.hap.Characteristic.OccupancyDetected.OCCUPANCY_DETECTED, hap_1.hap.Characteristic.OccupancyDetected.OCCUPANCY_NOT_DETECTED);
    }
    static generateIdentifier(endpoint) {
        let identifier = PresenceSensorHandler.NAME + '_' + hap_1.hap.Service.OccupancySensor.UUID;
        if (endpoint !== undefined) {
            identifier += '_' + endpoint.trim();
        }
        return identifier;
    }
}
PresenceSensorHandler.NAME = 'presence';
class VibrationSensorHandler extends BinarySensorHandler {
    constructor(expose, otherExposes, accessory) {
        super(accessory, expose, otherExposes, VibrationSensorHandler.generateIdentifier, 'Motion Sensor (vibration)', (n, t) => new hap_1.hap.Service.MotionSensor(n, (VibrationSensorHandler.NAME + ' ' + (t !== null && t !== void 0 ? t : '')).trim()), hap_1.hap.Characteristic.MotionDetected, true, false);
    }
    static generateIdentifier(endpoint) {
        let identifier = VibrationSensorHandler.NAME + '_' + hap_1.hap.Service.MotionSensor.UUID;
        if (endpoint !== undefined) {
            identifier += '_' + endpoint.trim();
        }
        return identifier;
    }
}
VibrationSensorHandler.NAME = 'vibration';
class SmokeSensorHandler extends BinarySensorHandler {
    constructor(expose, otherExposes, accessory) {
        super(accessory, expose, otherExposes, SmokeSensorHandler.generateIdentifier, 'SmokeSensor', (n, t) => new hap_1.hap.Service.SmokeSensor(n, t), hap_1.hap.Characteristic.SmokeDetected, hap_1.hap.Characteristic.SmokeDetected.SMOKE_DETECTED, hap_1.hap.Characteristic.SmokeDetected.SMOKE_NOT_DETECTED);
    }
    static generateIdentifier(endpoint) {
        let identifier = hap_1.hap.Service.SmokeSensor.UUID;
        if (endpoint !== undefined) {
            identifier += '_' + endpoint.trim();
        }
        return identifier;
    }
}
class CarbonMonoxideSensorHandler extends BinarySensorHandler {
    constructor(expose, otherExposes, accessory) {
        super(accessory, expose, otherExposes, CarbonMonoxideSensorHandler.generateIdentifier, 'CarbonMonoxideSensor', (n, t) => new hap_1.hap.Service.CarbonMonoxideSensor(n, t), hap_1.hap.Characteristic.CarbonMonoxideDetected, hap_1.hap.Characteristic.CarbonMonoxideDetected.CO_LEVELS_ABNORMAL, hap_1.hap.Characteristic.CarbonMonoxideDetected.CO_LEVELS_NORMAL);
    }
    static generateIdentifier(endpoint) {
        let identifier = hap_1.hap.Service.CarbonMonoxideSensor.UUID;
        if (endpoint !== undefined) {
            identifier += '_' + endpoint.trim();
        }
        return identifier;
    }
}
class LeakSensorHandler extends BinarySensorHandler {
    constructor(subType, identifierGen, expose, otherExposes, accessory) {
        super(accessory, expose, otherExposes, identifierGen, subType + ' LeakSensor', (n, t) => new hap_1.hap.Service.LeakSensor(n, t), hap_1.hap.Characteristic.LeakDetected, hap_1.hap.Characteristic.LeakDetected.LEAK_DETECTED, hap_1.hap.Characteristic.LeakDetected.LEAK_NOT_DETECTED, subType);
    }
    static generateIdentifier(endpoint, additionalSubType) {
        let identifier = `${additionalSubType}_${hap_1.hap.Service.LeakSensor.UUID}`;
        if (endpoint !== undefined) {
            identifier += '_' + endpoint.trim();
        }
        return identifier;
    }
}
class WaterLeakSensorHandler extends LeakSensorHandler {
    constructor(expose, otherExposes, accessory) {
        super(WaterLeakSensorHandler.SUBTYPE, WaterLeakSensorHandler.generateIdentifier, expose, otherExposes, accessory);
    }
    static generateIdentifier(endpoint) {
        return LeakSensorHandler.generateIdentifier(endpoint, WaterLeakSensorHandler.SUBTYPE);
    }
}
WaterLeakSensorHandler.SUBTYPE = 'water';
class GasLeakSensorHandler extends LeakSensorHandler {
    constructor(expose, otherExposes, accessory) {
        super(GasLeakSensorHandler.SUBTYPE, GasLeakSensorHandler.generateIdentifier, expose, otherExposes, accessory);
    }
    static generateIdentifier(endpoint) {
        return LeakSensorHandler.generateIdentifier(endpoint, GasLeakSensorHandler.SUBTYPE);
    }
}
GasLeakSensorHandler.SUBTYPE = 'gas';
class BasicSensorCreator {
    createServicesFromExposes(accessory, exposes) {
        const endpointMap = (0, helpers_1.groupByEndpoint)(exposes.filter(e => (0, z2mModels_1.exposesHasProperty)(e) && !accessory.isPropertyExcluded(e.property)
            && (0, z2mModels_1.exposesIsPublished)(e)).map(e => e));
        endpointMap.forEach((value, key) => {
            const optionalProperties = value.filter(e => (0, z2mModels_1.exposesHasBinaryProperty)(e) && (e.name === 'battery_low' || e.name === 'tamper'))
                .map(e => e);
            BasicSensorCreator.mapping.forEach(m => {
                if (!accessory.isServiceHandlerIdKnown(m.implementation.generateIdentifier(key))) {
                    value.filter(e => e.name === m.name && e.type === m.type)
                        .forEach(e => this.createService(accessory, e, (x) => new m.implementation(x, optionalProperties, accessory)));
                }
            });
        });
    }
    createService(accessory, expose, creator) {
        try {
            const handler = creator(expose);
            accessory.registerServiceHandler(handler);
        }
        catch (error) {
            accessory.log.warn(`Failed to setup sensor for accessory ${accessory.displayName} from expose "${JSON.stringify(expose)}": ${error}`);
        }
    }
}
exports.BasicSensorCreator = BasicSensorCreator;
BasicSensorCreator.mapping = [
    new BasicSensorMapping('humidity', z2mModels_1.ExposesKnownTypes.NUMERIC, HumiditySensorHandler),
    new BasicSensorMapping('temperature', z2mModels_1.ExposesKnownTypes.NUMERIC, TemperatureSensorHandler),
    new BasicSensorMapping('illuminance_lux', z2mModels_1.ExposesKnownTypes.NUMERIC, LightSensorHandler),
    new BasicSensorMapping('pressure', z2mModels_1.ExposesKnownTypes.NUMERIC, AirPressureSensorHandler),
    new BasicSensorMapping('contact', z2mModels_1.ExposesKnownTypes.BINARY, ContactSensorHandler),
    new BasicSensorMapping('occupancy', z2mModels_1.ExposesKnownTypes.BINARY, OccupancySensorHandler),
    new BasicSensorMapping(PresenceSensorHandler.NAME, z2mModels_1.ExposesKnownTypes.BINARY, PresenceSensorHandler),
    new BasicSensorMapping(VibrationSensorHandler.NAME, z2mModels_1.ExposesKnownTypes.BINARY, VibrationSensorHandler),
    new BasicSensorMapping('smoke', z2mModels_1.ExposesKnownTypes.BINARY, SmokeSensorHandler),
    new BasicSensorMapping('carbon_monoxide', z2mModels_1.ExposesKnownTypes.BINARY, CarbonMonoxideSensorHandler),
    new BasicSensorMapping('water_leak', z2mModels_1.ExposesKnownTypes.BINARY, WaterLeakSensorHandler),
    new BasicSensorMapping('gas', z2mModels_1.ExposesKnownTypes.BINARY, GasLeakSensorHandler),
];
//# sourceMappingURL=basic_sensors.js.map