/*
 * Soloscli - Command line interface to Solos multi-port ADSL2+ PCI card
 * Author : Nathan Williams <nathan@traverse.com.au>
 * Last Mod: 2009-06-16
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include "soloscli.h"

void usage(void);

struct options{
	int action; /* 1 for get, 2 for set, 0 undefined */
	int port;   /* port number [0..3] */
	char *parameter;
	char *value;
};

int get_parm(struct options *opts);
int set_parm(struct options *opts);
int read_file(char *filename, char *buf);
int write_file(char *filename, char *buf);
int receive_reply(char *path, char *buf);
int check_ready(char *path, char *buf);

int main (int argc, char **argv)
{
	struct options opts;
	int err = 0;

	opts.action = 0;
	opts.port = 0;
	opts.parameter = NULL;
	opts.value = NULL;
	if (argc == 1) {
		usage();
		return 0;
	}

	if (strcmp(argv[1],"-s") == 0) {
		opts.action = SET;
	} else if (strcmp(argv[1],"--set") == 0) {
		opts.action = SET;
	} else if (strcmp(argv[1],"-g") == 0) {
		opts.action = GET;
	} else if (strcmp(argv[1],"--get") == 0) {
		opts.action = GET;
	} else if (strcmp(argv[1],"-h") == 0) {
		usage();
		return 0;
	} else if (strcmp(argv[1],"-?") == 0) {
		usage();
		return 0;
	} else if (strcmp(argv[1],"--help") == 0) {
		usage();
		return 0;
	} else {
		printf("Invalid first argument\n");
		usage();
		return 1;
	}

	if (argc < 4 || argc > 5) {
		printf("Invalid number of arguments\n");
		usage();
		return 1;
	}

	if (strcmp(argv[2],"0") == 0) {
		opts.port = 0;
	} else if (strcmp(argv[2],"1") == 0) {
		opts.port = 1;
	} else if (strcmp(argv[2],"2") == 0) {
		opts.port = 2;
	} else if (strcmp(argv[2],"3") == 0) {
		opts.port = 3;
	} else {
		printf("Invalid port number\n");
		usage();
		return 1;
	}

	if (opts.action == SET && argc != 5) {
		printf("Missing value for parameter\n");
		usage();
		return 1;
	}
	if (opts.action == GET && argc != 4) {
		printf("Too many parameters\n");
		usage();
		return 1;
	}

	opts.parameter = argv[3];
	if (opts.action == GET) {
		err = get_parm(&opts);
	} else {
		opts.value = argv[4];
		err = set_parm(&opts);
	}

	return(err);
}

void usage(void)
{
	printf("soloscli v3.04\n");
	printf("Usage: soloscli -g port parameter\n   or: soloscli -s port parameter value\n");
	return;
}

int get_parm(struct options *opts)
{
	char path[256];
	char buf[BUFSIZE];
	char pid[10];
	int err;
	sprintf(path, "/sys/class/atm/solos-pci%d/console", opts->port);
	err = check_ready(path, buf);
	if (err != 0) {
		return err;
	}
	sprintf(pid,"x%d",getpid());
	sprintf(buf,"%s\n%s\n",pid,opts->parameter);
	err = write_file(path, buf);
	if (err != 0) {
		return err;
	}
	err = receive_reply(path, buf);

	return err;
}

int set_parm(struct options *opts)
{
	char path[256];
	char buf[BUFSIZE];
	char pid[10];
	int err;
	sprintf(path, "/sys/class/atm/solos-pci%d/console", opts->port);
	err = check_ready(path, buf);
	if (err != 0) {
		return err;
	}
	sprintf(pid,"x%d",getpid());
	sprintf(buf,"%s\n%s\n%s\n",pid,opts->parameter,opts->value);
	err = write_file(path, buf);
	if (err != 0) {
		return err;
	}
	err = receive_reply(path, buf);

	return err;
}

int read_file(char *filename, char *buf)
{
	FILE *fp;
	int i;
	int c;
	fp = fopen(filename,"rb+");
	if (fp == NULL) {
		printf("Error: Failed to open file '%s'\n",filename);
		return 1;
	};
	i = 0;
	while ((c = fgetc(fp)) != EOF) {
		if (i > BUFSIZE - 2) {
			printf("Error: buffer overflow\n");
			fclose(fp);
			return 1;
		}
		buf[i] = (char)c;
		i++;
	}
	buf[i] = 0;
	fclose(fp);
	return 0;
}

int write_file(char *filename, char *buf)
{
	FILE *fp;
	fp = fopen(filename,"rb+");
	if (fp == NULL) {
		printf("Error: Failed to open file '%s'\n",filename);
		return 1;
	};
	fwrite(buf, 1, strlen(buf), fp);
	fclose(fp);
	return 0;
}

int receive_reply(char *path, char *buf)
{
	int i,err,len;
	char *bufp = NULL;
	char pid[10];

	sprintf(pid,"x%d",getpid());

	for (i=0;i<TIMEOUT;i++) {
		err = read_file(path, buf);
		if(err != 0){
			return 1;
		}
		if (strcmp(buf,NODATA) == 0) {
/*			printf("Waiting.. %d\n",(i+1)); */
		} else {
			break;
		}
		usleep(1000);
	}
	if (i == TIMEOUT) {
		printf("Error. No response from modem\n");
		return 1;
	}
	bufp = NULL;
	len = strlen(buf);
	for (i=0;i<len;i++) {
		if (buf[i] == '\n') {
			buf[i] = 0;
			bufp = buf + i + 1;
			break;
		}
	}
	if (strcmp(buf,pid) == 0) {
/*		printf("Sequence matches.\n"); */
		if(buf[(len-1)] == '\n'){
			printf("%s",bufp);
		} else {
			printf("%s\n",bufp);
		}
	} else {
		printf("Sequence incorrect.\n");
		buf[i] = '\n';
		printf("Expected %s, buffer was:\n%s\n",pid,buf);
	}
	return 0;
}

int check_ready(char *path, char *buf)
{
	int i,err;

	for (i=0;i<TIMEOUT;i++) {
		err = read_file(path, buf);
		if (err != 0) {
			return 1;
		}
		if (strcmp(buf,NODATA) == 0) {
			return 0;
		}
/*		printf("Discarded:\n%s\n",buf); */
		usleep(100);
	}
	printf("Error. Can't clear file\n");
	return 1;
}
