////////////////////////////////////////////////////////////////////////////////
// sudokuki - C++ graphical sudoku game                                       //
// Copyright (C) 2007-2009 Sylvain Vedrenne                                   //
//                                                                            //
// This program is free software; you can redistribute it and/or              //
// modify it under the terms of the GNU General Public License                //
// as published by the Free Software Foundation; either version 2             //
// of the License, or (at your option) any later version.                     //
//                                                                            //
// This program is distributed in the hope that it will be useful,            //
// but WITHOUT ANY WARRANTY; without even the implied warranty of             //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the              //
// GNU General Public License for more details.                               //
//                                                                            //
// You should have received a copy of the GNU General Public License along    //
// with this program; if not, write to the Free Software Foundation, Inc.,    //
// 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.              //
////////////////////////////////////////////////////////////////////////////////
#if DISABLE_NLS
#define _(STR) STR
#else
#include <glibmm/i18n.h>
#endif/*DISABLE_NLS*/

#include "sudokukimm_license.h"
#include "sudokukimm_html.h"
#include "SudokukimmWindow.h"
#include "sudokuki_icon.h"
#include "sudokuki_logo.h"
#include "ui_string.h"
#include <iostream>
#include <vector>
#include <stdio.h>

#include <fstream>

#include <gtkmm.h> //TODO: Replace it by several includes when ready

#include "../generator.h"

/* use this to deactivate local traces for sure (for a release, for instance) */
#define NO_TRACES_SUDOKUKIMMWINDOW_CC 1

/* use this to configure local traces */
#define TRACES_SUDOKUKIMMWINDOW_CC 1

#if (ALL_TRACES/*defined in sudokukimm.h*/||TRACES_SUDOKUKIMMWINDOW_CC)&&!NO_TRACES_SUDOKUKIMMWINDOW_CC
#define DBGM_TRACES 1
#else
#define DBGM_TRACES 0
#endif
#include "../dbgm_traces.h"

std::string SudokukimmWindow::sudokuki_version = std::string(VERSION);
#define GU_JA_UNICODE_ICHI 0x4E00
#define GU_JA_UNICODE_NI 0x4E8C
#define GU_JA_UNICODE_SAN 0x4E09
#define GU_JA_UNICODE_YON 0x56DB
#define GU_JA_UNICODE_GO 0x4E94
#define GU_JA_UNICODE_ROKU 0x516D
#define GU_JA_UNICODE_SHICHI 0x4E03
#define GU_JA_UNICODE_HACHI 0x516B
#define GU_JA_UNICODE_KYU 0x4E5D

Glib::ustring SudokukimmWindow::symbols[20];
std::map<char, int> SudokukimmWindow::values;
Glib::RefPtr<Gdk::Pixbuf> SudokukimmWindow::m_picon = Glib::RefPtr<Gdk::Pixbuf>(0);
Glib::RefPtr<Gdk::Pixbuf> SudokukimmWindow::m_plogo = Glib::RefPtr<Gdk::Pixbuf>(0);

void
SudokukimmWindow::install_menu_hints( const Glib::RefPtr<Gtk::Action>& action, Gtk::Widget* widget )
{
	if(Gtk::MenuItem *item = dynamic_cast<Gtk::MenuItem*>(widget)) {
		Glib::ustring tooltip = action->property_tooltip();
		item->signal_deselect().connect( sigc::bind( sigc::mem_fun(m_pstatusbar, &Gtk::Statusbar::pop), 0) );
		item->signal_select().connect( sigc::hide_return( sigc::bind( sigc::mem_fun(m_pstatusbar, &Gtk::Statusbar::push), tooltip, 0) ) );
	}
}

SudokukimmWindow::SudokukimmWindow( Sudokukimm* psudokukimm ) : m_edit_custom_grid_in_progress(false),m_psudokukimm(psudokukimm)
{
	int x = m_psudokukimm->get_window_x_pos();
	int y = m_psudokukimm->get_window_y_pos();
	if ( -1!=x && -1!=y ) {
		move( x, y );
	}

	m_inhibit_edit = true;
	m_last_cell_with_focus_index = m_focus_idx = -1;
	m_lang = m_psudokukimm->get_locale();
	m_kanji_mode = m_psudokukimm->get_kanji_mode();
	m_level = m_psudokukimm->get_level();
	DBGF("m_level:%d", m_level);
	m_dead_end = m_psudokukimm->get_dead_end_state();
	DBGF("m_dead_end:%d", m_dead_end);

	symbols[0] = Glib::ustring("0");
	symbols[1] = Glib::ustring("1");
	symbols[2] = Glib::ustring("2");
	symbols[3] = Glib::ustring("3");
	symbols[4] = Glib::ustring("4");
	symbols[5] = Glib::ustring("5");
	symbols[6] = Glib::ustring("6");
	symbols[7] = Glib::ustring("7");
	symbols[8] = Glib::ustring("8");
	symbols[9] = Glib::ustring("9");
	symbols[10+1] = Glib::ustring(1, gunichar(GU_JA_UNICODE_ICHI));
	symbols[10+2] = Glib::ustring(1, gunichar(GU_JA_UNICODE_NI));
	symbols[10+3] = Glib::ustring(1, gunichar(GU_JA_UNICODE_SAN));
	symbols[10+4] = Glib::ustring(1, gunichar(GU_JA_UNICODE_YON));
	symbols[10+5] = Glib::ustring(1, gunichar(GU_JA_UNICODE_GO));
	symbols[10+6] = Glib::ustring(1, gunichar(GU_JA_UNICODE_ROKU));
	symbols[10+7] = Glib::ustring(1, gunichar(GU_JA_UNICODE_SHICHI));
	symbols[10+8] = Glib::ustring(1, gunichar(GU_JA_UNICODE_HACHI));
	symbols[10+9] = Glib::ustring(1, gunichar(GU_JA_UNICODE_KYU));

	get_settings()->set_property("gtk-error-bell", false);

	values['1'] = 1; values['2'] = 2; values['3'] = 3;
	values['4'] = 4; values['5'] = 5; values['6'] = 6;
	values['7'] = 7; values['8'] = 8; values['9'] = 9;

	m_peventbox = Gtk::manage( new Gtk::EventBox() );
	m_pmainvbox = Gtk::manage( new Gtk::VBox() );
	m_ptoolbar = Gtk::manage( new Gtk::Toolbar() );
	m_pstatusbar = Gtk::manage( new Gtk::Statusbar() );
	m_paspectframe = Gtk::manage( new Gtk::AspectFrame() );
	m_pvbx = Gtk::manage( new Gtk::VBox() );
	m_pbigtable = Gtk::manage( new Gtk::Table() );

	std::cout << _("Sudokuki, version ") << sudokuki_version << " (" << m_lang << ")\n";

	DBGM("");

	set_icon( get_icon() );
	set_title("Sudokuki");
	set_resizable(false);
	set_border_width(0);
	set_default_size(0,0);

	add_actions();

	m_rpuimgr = Gtk::UIManager::create();
	m_rpuimgr->insert_action_group(m_rpag);
	add_accel_group(m_rpuimgr->get_accel_group());
	m_rpuimgr->add_ui_from_string(ui_info);
	m_custom_grid_merge_id = m_rpuimgr->add_ui_from_string(ui_setup_custom_grid);

	m_rpuimgr->signal_connect_proxy().connect( sigc::mem_fun(this, &SudokukimmWindow::install_menu_hints) );

	m_black_5 = Gdk::Color(); m_black_5.set_grey( 3276 );
	m_grey_90 = Gdk::Color(); m_grey_90.set_grey( 58982 );
	m_peventbox->modify_bg( Gtk::STATE_NORMAL, m_black_5 );

	add(*m_pmainvbox);
	m_pmenubar = static_cast<Gtk::MenuBar*>(m_rpuimgr->get_widget("/MenuBar"));

	m_pmainvbox->pack_start(*m_pmenubar, Gtk::PACK_SHRINK);

	m_ptoolbar = static_cast<Gtk::Toolbar*>(m_rpuimgr->get_widget("/Toolbar"));
	m_ptoolbar->set_toolbar_style(Gtk::TOOLBAR_ICONS);
	m_ptoolbar->set_tooltips(true);

	m_ppopup_menu = dynamic_cast<Gtk::Menu*>(m_rpuimgr->get_widget("/PopupMenu"));

	m_pmainvbox->pack_start(*m_ptoolbar, Gtk::PACK_SHRINK);
	m_pmainvbox->pack_start(*m_peventbox, Gtk::PACK_SHRINK);
	m_pmainvbox->pack_start(*m_pstatusbar, Gtk::PACK_SHRINK);
	m_pstatusbar->set_has_resize_grip(false);

	m_peventbox->add(*m_paspectframe);
	m_paspectframe->set_shadow_type(Gtk::SHADOW_NONE);
	m_paspectframe->unset_label();
	m_paspectframe->add(*m_pvbx);

	m_pvbx->pack_start(*m_pbigtable);

	DBGF("Generated grid string: %s", m_psudokukimm->get_generated_grid().c_str());

	m_psudokukimm->solver_init();

	m_react_when_cell_filled = false;
	for ( int a = 0; a < 3; a++ )
	{
		for ( int b = 0; b < 3; b++ )
		{
			Gtk::Table* pp_table = new Gtk::Table( 3, 3, true );
			m_pbigtable->attach( *pp_table, b, b+1, a, a+1, Gtk::FILL );
			m_pbigtable->set_spacings( 2 );

			for ( int j = 0; j < 3; j++ )
			{
				for ( int i = 0; i < 3; i++ )
				{
					Gtk::Entry* p_ent = Gtk::manage( new Gtk::Entry );
					if ( 0==a && 0==j) { // cell from first line of the grid
						p_ent->signal_focus_out_event().connect( sigc::bind(sigc::mem_fun(this, &SudokukimmWindow::on_cell_focus_out), 3*a+j, 3*b+i));
					}
					p_ent->signal_focus_in_event().connect(sigc::bind(sigc::mem_fun(this, &SudokukimmWindow::on_cell_focus_in), 3*a+j, 3*b+i));
					p_ent->signal_changed().connect(sigc::bind(sigc::mem_fun(this, &SudokukimmWindow::on_cell_filled), 3*a+j, 3*b+i));
					p_ent->signal_button_press_event().connect(sigc::bind(sigc::mem_fun(this, &SudokukimmWindow::on_cell_clicked), 3*a+j, 3*b+i), false);

					DBGF("p_ent: Ox%x", (unsigned int)p_ent);

					m_pentries[9*(3*a+j) + 3*b+i] = p_ent;

					if ( (b==1 || a==1) && b != a )
					{
						p_ent->modify_base( Gtk::STATE_NORMAL, m_grey_90 );
					}
					m_pbold_big_font = new Pango::FontDescription();
					m_pbold_big_font->set_size(16 * PANGO_SCALE);
					m_pbold_big_font->set_weight(Pango::WEIGHT_HEAVY);

					m_pbold_font = new Pango::FontDescription();
					m_pbold_font->set_size(14 * PANGO_SCALE);
					m_pbold_font->set_weight(Pango::WEIGHT_HEAVY);

					m_pother_font = new Pango::FontDescription();
					m_pother_font->set_size(14 * PANGO_SCALE);
#if __SUDOKUKI_WIN32__
					if ( m_kanji_mode ) {
						p_ent->modify_font( *m_pbold_big_font );
					} else {
#endif/*__SUDOKUKI_WIN32__*/
						p_ent->modify_font( *m_pbold_font );
#if __SUDOKUKI_WIN32__
					}
#endif/*__SUDOKUKI_WIN32__*/

					p_ent->set_size_request( 25, 25 );
					p_ent->set_alignment( 0.5 );
					p_ent->set_has_frame( false );
					pp_table->attach( *p_ent, i, i+1, j, j+1, Gtk::FILL );
					pp_table->set_spacings( 1 );

					int value = m_psudokukimm->solver_get_cell_value(3*a+j,3*b+i);
					DBGF("value of cell(%d,%d):%d", 3*a+j, 3*b+i, value);
					if ( 1 <= value && value <= 9 ) {
						DBGF("symbol:%s", symbols[value].c_str());
						string text = symbols[m_kanji_mode*10+value];
						p_ent->modify_text(Gtk::STATE_NORMAL, Gdk::Color("black"));
						p_ent->set_text( text );
						DBGF("char 1 in unicode:%c", g_utf8_get_char("1") );
						p_ent->set_editable( false );
					} else {
						p_ent->modify_font( *m_pother_font );
						p_ent->set_max_length(1);
						p_ent->modify_text(Gtk::STATE_NORMAL, Gdk::Color("blue"));
						int valuePlayer = m_psudokukimm->player_get_cell_value(3*a+j,3*b+i);
						DBGF("valuePlayer:%d", valuePlayer);
						if ( 1 <= valuePlayer && valuePlayer <= 9 ) {
							string textPlayer = symbols[m_kanji_mode*10+valuePlayer];
							p_ent->set_text( textPlayer );
						} else {
							p_ent->set_text("");
						}
						p_ent->set_editable( true );
					}
				}
			}
			pp_table->set_homogeneous( true );
		}
	}

	if ( m_psudokukimm->solver_get_how_many_numbers() < m_psudokukimm->player_get_how_many_numbers() ) {
		m_rpuimgr->get_action( "ui/MenuBar/EditMenu/Undo" )->set_sensitive( true );
		m_rpuimgr->get_action( "ui/MenuBar/EditMenu/Clear" )->set_sensitive( true );
	} else {
		m_rpuimgr->get_action( "ui/MenuBar/EditMenu/Undo" )->set_sensitive( false );
		m_rpuimgr->get_action( "ui/MenuBar/EditMenu/Clear" )->set_sensitive( false );
	}
	m_rpuimgr->get_action( "ui/MenuBar/GameMenu/New" )->set_sensitive( true );
	m_rpuimgr->get_action( "ui/MenuBar/GameMenu/Open" )->set_sensitive( true );
	m_rpuimgr->get_action( "ui/MenuBar/GameMenu/SaveAs" )->set_sensitive( true );
	m_rpuimgr->get_action( "ui/MenuBar/EditMenu/Preferences/LevelMenu" )->set_sensitive( true );
	m_rpuimgr->get_action( "ui/MenuBar/EditMenu/Preferences/PrintSettings" )->set_sensitive( true );

	m_rpuimgr->get_action( "ui/MenuBar/EditMenu/SetupCustomGrid" )->set_sensitive( true );
	m_rpuimgr->get_action( "ui/MenuBar/EditMenu/PlayCustomGrid" )->set_sensitive( false );


	m_react_when_cell_filled = true;

	signal_key_release_event().connect(sigc::mem_fun(*this, &SudokukimmWindow::on_key_pressed), true);
	m_focus_idx = 40;

	m_pentries[m_focus_idx]->grab_focus();

	show_all_children();
	m_inhibit_edit = false;

	signal_reinit_window.connect( sigc::mem_fun( *m_psudokukimm, &Sudokukimm::reinit_window ));

	if ( m_psudokukimm->get_grid_resolved() )  {
		on_game_resolve();
	}
}

SudokukimmWindow::~SudokukimmWindow()
{
	DBGM();
}

void
SudokukimmWindow::on_edit_language( /*int int_locale*/ std::string locale_to_set )
{
	DBGF("\n##### on_edit_language :%s", locale_to_set.c_str() );

	if (m_inhibit_edit) {
		DBGM("\n##### on_edit_* INHIBITED");
		return;
	}

	if ( m_lang != locale_to_set ) {
		DBGF("m_lang:%s => %s hide unrealize reinit_window", m_lang.c_str(), locale_to_set.c_str());
		m_psudokukimm->set_locale( locale_to_set );
		int x, y;
		get_position( x, y );
		DBGF("x:%d, y:%d", x, y);
		m_psudokukimm->set_window_position( x, y );
		hide();
		unrealize();
		signal_reinit_window.emit();
	}
}

void
SudokukimmWindow::on_edit_level( int level )
{
	DBGF("level:%d", level);
	if (m_inhibit_edit) {
		DBGM("\n##### on_edit_* INHIBITED");
		return;
	}

	if ( m_level != level ) {

		m_level = level;
		m_psudokukimm->set_level( m_level );
	}
}

void
SudokukimmWindow::on_edit_kanjimode( int mode )
{
	DBGF("mode:%d", mode);
	if (m_inhibit_edit) {
		DBGM("\n##### on_edit_* INHIBITED");
		return;
	}

	if ( m_kanji_mode != mode ) {
		DBGF("m_kanji_mode:%d => %d hide unrealize reinit_window", m_kanji_mode, mode);
		m_psudokukimm->set_kanji_mode( mode );
		int x, y;
		get_position( x, y );
		DBGF("x:%d, y:%d", x, y);
		m_psudokukimm->set_window_position( x, y );
		hide();
		unrealize();
		signal_reinit_window.emit();
	}
}

void
SudokukimmWindow::add_actions()
{
	m_rpag = Gtk::ActionGroup::create("ActionGroup");

	int i = 0;

	// Game menu
	m_actions.push_back( Gtk::Action::create("GameMenu", _("_Game") ));
	m_rpag->add( m_actions[i++] );

	m_actions.push_back( Gtk::Action::create("Print", Gtk::Stock::PRINT, _("_Print"), _("Print generated grid from HTML browser")) );
	m_rpag->add( m_actions[i++],
			Gtk::AccelKey("<control>P"), sigc::mem_fun(*this, &SudokukimmWindow::on_game_print_generated_grid_in_browser) );

	m_actions.push_back( Gtk::Action::create("New", Gtk::Stock::NEW, _("_New"), _("New game")) );
	m_rpag->add( m_actions[i++],
			Gtk::AccelKey("<control>N"), sigc::mem_fun(*this, &SudokukimmWindow::on_game_new) );

	m_actions.push_back( Gtk::Action::create("Open", Gtk::Stock::OPEN, _("_Open"), _("Open grid...")) );
	m_rpag->add( m_actions[i++],
			Gtk::AccelKey("<control>O"), sigc::mem_fun(*this, &SudokukimmWindow::on_game_open) );

	m_actions.push_back( Gtk::Action::create("SaveAs", Gtk::Stock::SAVE, _("_SaveAs"), _("Save grid as...")) );
	m_rpag->add( m_actions[i++],
			Gtk::AccelKey("<control>S"), sigc::mem_fun(*this, &SudokukimmWindow::on_game_save_as) );

	m_actions.push_back( Gtk::Action::create("Resolve", Gtk::Stock::JUMP_TO, _("_Resolve grid"), _("Resolve current grid")) );

	m_rpag->add( m_actions[i++],
			Gtk::AccelKey("<control>R"),
			sigc::mem_fun(*this, &SudokukimmWindow::on_game_resolve) );

	m_actions.push_back( Gtk::Action::create("ResolveCell", Gtk::Stock::APPLY, _("Resolve cell"), _("Resolve current cell")) );
	m_rpag->add( m_actions[i++],
			Gtk::AccelKey("<control>T"),
			sigc::mem_fun(*this, &SudokukimmWindow::on_cell_resolve) );

	DBGF("m_kanji_mode:%d", m_kanji_mode);

	m_actions.push_back( Gtk::Action::create("Escape", Gtk::Stock::CLOSE, _("Escape") )); m_rpag->add( m_actions[i++] );
	m_actions.push_back( Gtk::Action::create("TryValue_1", symbols[m_kanji_mode*10+1], _("Try value 1 in cell")) );
	m_rpag->add( m_actions[i++], sigc::bind<-1>( sigc::mem_fun(*this, &SudokukimmWindow::on_try_value_in_cell), 1) );
	m_actions.push_back( Gtk::Action::create("TryValue_2", symbols[m_kanji_mode*10+2], _("Try value 2 in cell")) );
	m_rpag->add( m_actions[i++], sigc::bind<-1>( sigc::mem_fun(*this, &SudokukimmWindow::on_try_value_in_cell), 2) );
	m_actions.push_back( Gtk::Action::create("TryValue_3", symbols[m_kanji_mode*10+3], _("Try value 3 in cell")) );
	m_rpag->add( m_actions[i++], sigc::bind<-1>( sigc::mem_fun(*this, &SudokukimmWindow::on_try_value_in_cell), 3) );
	m_actions.push_back( Gtk::Action::create("TryValue_4", symbols[m_kanji_mode*10+4], _("Try value 4 in cell")) );
	m_rpag->add( m_actions[i++], sigc::bind<-1>( sigc::mem_fun(*this, &SudokukimmWindow::on_try_value_in_cell), 4) );
	m_actions.push_back( Gtk::Action::create("TryValue_5", symbols[m_kanji_mode*10+5], _("Try value 5 in cell")) );
	m_rpag->add( m_actions[i++], sigc::bind<-1>( sigc::mem_fun(*this, &SudokukimmWindow::on_try_value_in_cell), 5) );
	m_actions.push_back( Gtk::Action::create("TryValue_6", symbols[m_kanji_mode*10+6], _("Try value 6 in cell")) );
	m_rpag->add( m_actions[i++], sigc::bind<-1>( sigc::mem_fun(*this, &SudokukimmWindow::on_try_value_in_cell), 6) );
	m_actions.push_back( Gtk::Action::create("TryValue_7", symbols[m_kanji_mode*10+7], _("Try value 7 in cell")) );
	m_rpag->add( m_actions[i++], sigc::bind<-1>( sigc::mem_fun(*this, &SudokukimmWindow::on_try_value_in_cell), 7) );
	m_actions.push_back( Gtk::Action::create("TryValue_8", symbols[m_kanji_mode*10+8], _("Try value 8 in cell")) );
	m_rpag->add( m_actions[i++], sigc::bind<-1>( sigc::mem_fun(*this, &SudokukimmWindow::on_try_value_in_cell), 8) );
	m_actions.push_back( Gtk::Action::create("TryValue_9", symbols[m_kanji_mode*10+9], _("Try value 9 in cell")) );
	m_rpag->add( m_actions[i++], sigc::bind<-1>( sigc::mem_fun(*this, &SudokukimmWindow::on_try_value_in_cell), 9) );

	m_actions.push_back( Gtk::Action::create("Unavailable_1", "-")); m_rpag->add( m_actions[i++] );
	m_actions.push_back( Gtk::Action::create("Unavailable_2", "-")); m_rpag->add( m_actions[i++] );
	m_actions.push_back( Gtk::Action::create("Unavailable_3", "-")); m_rpag->add( m_actions[i++] );
	m_actions.push_back( Gtk::Action::create("Unavailable_4", "-")); m_rpag->add( m_actions[i++] );
	m_actions.push_back( Gtk::Action::create("Unavailable_5", "-")); m_rpag->add( m_actions[i++] );
	m_actions.push_back( Gtk::Action::create("Unavailable_6", "-")); m_rpag->add( m_actions[i++] );
	m_actions.push_back( Gtk::Action::create("Unavailable_7", "-")); m_rpag->add( m_actions[i++] );
	m_actions.push_back( Gtk::Action::create("Unavailable_8", "-")); m_rpag->add( m_actions[i++] );
	m_actions.push_back( Gtk::Action::create("Unavailable_9", "-")); m_rpag->add( m_actions[i++] );


	m_actions.push_back( Gtk::Action::create("Quit", Gtk::Stock::QUIT, _("_Quit"), _("Quit game")) );
	m_rpag->add( m_actions[i++],
			Gtk::AccelKey("<control>Q"), sigc::mem_fun(*this, &SudokukimmWindow::on_game_quit) );

	// Edit menu
	m_actions.push_back( Gtk::Action::create("EditMenu", _("_Edit")) );
	m_rpag->add( m_actions[i++] );

	m_actions.push_back( Gtk::Action::create("Undo", Gtk::Stock::UNDO, _("_Undo"), _("Undo move")) );
	m_rpag->add( m_actions[i++],
			Gtk::AccelKey("<control>U"), sigc::mem_fun(*this, &SudokukimmWindow::on_edit_undo) );

	m_actions.push_back( Gtk::Action::create("Clear", Gtk::Stock::CLEAR, _("_Clear all"), _("Clear all moves")) );
	m_rpag->add( m_actions[i++],
			Gtk::AccelKey("<control>X"), sigc::mem_fun(*this, &SudokukimmWindow::on_edit_clear) );

	m_actions.push_back( Gtk::Action::create("SetupCustomGrid", Gtk::Stock::EDIT, _("Custom grid"), _("Set up a custom grid")) );
	m_rpag->add( m_actions[i++],
			Gtk::AccelKey("<control>G"),
			sigc::mem_fun(*this, &SudokukimmWindow::on_edit_setup_custom_grid) );

	m_actions.push_back( Gtk::Action::create("PlayCustomGrid", Gtk::Stock::MEDIA_PLAY, _("Play grid"), _("Play the custom grid")) );
	m_rpag->add( m_actions[i++],
			Gtk::AccelKey("<control>H"),
			sigc::mem_fun(*this, &SudokukimmWindow::on_edit_play_custom_grid) );

	m_actions.push_back( Gtk::Action::create("Preferences", Gtk::Stock::PREFERENCES, _("_Preferences"), _("Edit preferences")) );
	m_rpag->add( m_actions[i++]);

	// KanjiMode menu
	m_actions.push_back( Gtk::Action::create("KanjiNumbersMenu", _("_Numbers")) );
	m_rpag->add( m_actions[i++] );

	Gtk::RadioAction::Group group_kanjimode;
	m_actions.push_back( Gtk::RadioAction::create(group_kanjimode, "Numbers", _("Arabic"), _("Arabic numbers")) );
	if (0 == m_kanji_mode) { ((Glib::RefPtr<Gtk::Action>)m_actions[i])->activate(); }
	m_rpag->add( m_actions[i++],
			sigc::bind<-1>( sigc::mem_fun(*this, &SudokukimmWindow::on_edit_kanjimode), 0 ) );
	m_actions.push_back( Gtk::RadioAction::create(group_kanjimode, "Kanji", _("Chinese"), _("Chinese numbers")) );
	if (1 == m_kanji_mode) { ((Glib::RefPtr<Gtk::Action>)m_actions[i])->activate(); }
	m_rpag->add( m_actions[i++],
			sigc::bind<-1>( sigc::mem_fun(*this, &SudokukimmWindow::on_edit_kanjimode), 1 ) );

	// Edit->Preferences->Level submenu
	m_actions.push_back( Gtk::Action::create("LevelMenu", _("_Level")) );
	m_rpag->add( m_actions[i++] );

	Gtk::RadioAction::Group group_level;
	m_actions.push_back( Gtk::RadioAction::create(group_level, "Level_1", _("Level 1"), _("Sudokuki level 1")) );
	if (1 == m_level) { ((Glib::RefPtr<Gtk::Action>)m_actions[i])->activate(); }
	m_rpag->add( m_actions[i++], sigc::bind<-1>( sigc::mem_fun(*this, &SudokukimmWindow::on_edit_level), 1 ) );
	m_actions.push_back( Gtk::RadioAction::create(group_level, "Level_2", _("Level 2"), _("Sudokuki level 2")) );
	if (2 == m_level) { ((Glib::RefPtr<Gtk::Action>)m_actions[i])->activate(); }
	m_rpag->add( m_actions[i++], sigc::bind<-1>( sigc::mem_fun(*this, &SudokukimmWindow::on_edit_level), 2 ) );
	m_actions.push_back( Gtk::RadioAction::create(group_level, "Level_3", _("Level 3"), _("Sudokuki level 3")) );
	if (3 == m_level) { ((Glib::RefPtr<Gtk::Action>)m_actions[i])->activate(); }
	m_rpag->add( m_actions[i++], sigc::bind<-1>( sigc::mem_fun(*this, &SudokukimmWindow::on_edit_level), 3 ) );
	m_actions.push_back( Gtk::RadioAction::create(group_level, "Level_4", _("Level 4"), _("Sudokuki level 4")) );
	if (4 == m_level) { ((Glib::RefPtr<Gtk::Action>)m_actions[i])->activate(); }
	m_rpag->add( m_actions[i++], sigc::bind<-1>( sigc::mem_fun(*this, &SudokukimmWindow::on_edit_level), 4 ) );
	m_actions.push_back( Gtk::RadioAction::create(group_level, "Level_5", _("Level 5"), _("Sudokuki level 5")) );
	if (5 == m_level) { ((Glib::RefPtr<Gtk::Action>)m_actions[i])->activate(); }
	m_rpag->add( m_actions[i++], sigc::bind<-1>( sigc::mem_fun(*this, &SudokukimmWindow::on_edit_level), 5 ) );

	// Edit->Preferences->Language submenu
	m_actions.push_back( Gtk::Action::create("LanguageMenu", Gtk::Stock::HOME, _("_Language")) );
	m_rpag->add( m_actions[i++] );

	m_actions.push_back( Gtk::Action::create("PrintSettings", Gtk::Stock::PRINT, _("Printing"), _("Print settings")) );
	m_rpag->add( m_actions[i++],
			sigc::mem_fun(*this, &SudokukimmWindow::on_edit_print_settings) );

	Gtk::RadioAction::Group group_language;
	m_actions.push_back( Gtk::RadioAction::create(group_language, "de", _("German"), _("German (de)")) );
	if ("de" == m_lang) { ((Glib::RefPtr<Gtk::Action>)m_actions[i])->activate(); }
	m_rpag->add( m_actions[i++],
			sigc::bind<-1>( sigc::mem_fun(*this, &SudokukimmWindow::on_edit_language), "de" ) );

	m_actions.push_back( Gtk::RadioAction::create(group_language, "el", _("Greek"), _("Greek (el)")) );
	if ("el" == m_lang) { ((Glib::RefPtr<Gtk::Action>)m_actions[i])->activate(); }
	m_rpag->add( m_actions[i++],
			sigc::bind<-1>( sigc::mem_fun(*this, &SudokukimmWindow::on_edit_language), "el" ) );

	m_actions.push_back( Gtk::RadioAction::create(group_language, "en", _("English"), _("English (en)")) );
	if ("en" == m_lang) { ((Glib::RefPtr<Gtk::Action>)m_actions[i])->activate(); }
	m_rpag->add( m_actions[i++],
			sigc::bind<-1>( sigc::mem_fun(*this, &SudokukimmWindow::on_edit_language), "en" ) );

	m_actions.push_back( Gtk::RadioAction::create(group_language, "eo", _("Esperanto"), _("Esperanto (eo)")) );
	if ("eo" == m_lang) { ((Glib::RefPtr<Gtk::Action>)m_actions[i])->activate(); }
	m_rpag->add( m_actions[i++],
			sigc::bind<-1>( sigc::mem_fun(*this, &SudokukimmWindow::on_edit_language), "eo" ) );

	m_actions.push_back( Gtk::RadioAction::create(group_language, "es", _("Spanish"), _("Spanish (es)")) );
	if ("es" == m_lang) { ((Glib::RefPtr<Gtk::Action>)m_actions[i])->activate(); }
	m_rpag->add( m_actions[i++],
			sigc::bind<-1>( sigc::mem_fun(*this, &SudokukimmWindow::on_edit_language), "es" ) );

	m_actions.push_back( Gtk::RadioAction::create(group_language, "fr", _("French"), _("French (fr)")) );
	if ("fr" == m_lang) { ((Glib::RefPtr<Gtk::Action>)m_actions[i])->activate(); }
	m_rpag->add( m_actions[i++],
			sigc::bind<-1>( sigc::mem_fun(*this, &SudokukimmWindow::on_edit_language), "fr" ) );

	m_actions.push_back( Gtk::RadioAction::create(group_language, "ja", _("Japanese"), _("Japanese (ja)")) );
	if ("ja" == m_lang) { ((Glib::RefPtr<Gtk::Action>)m_actions[i])->activate(); }
	m_rpag->add( m_actions[i++],
			sigc::bind<-1>( sigc::mem_fun(*this, &SudokukimmWindow::on_edit_language), "ja") );

	// Help menu
	m_actions.push_back( Gtk::Action::create("HelpMenu", _("_Help")) );
	m_rpag->add( m_actions[i++] );
	m_actions.push_back( Gtk::Action::create("About", Gtk::Stock::ABOUT, _("_About"), _("About Sudokuki...")) );
	m_rpag->add( m_actions[i++],
			Gtk::AccelKey("<control>A"), sigc::mem_fun(*this, &SudokukimmWindow::on_help_about) );
}

bool
SudokukimmWindow::discard_grid_confirm_dialog( bool discard )
{
	m_pconfirm_on_game_new_dialog = new Gtk::MessageDialog(*this,
			_("Discard current grid?"),
			false,
			Gtk::MESSAGE_QUESTION, Gtk::BUTTONS_OK_CANCEL, true );
	m_pconfirm_on_game_new_dialog->set_secondary_text(_("This will discard the current grid and moves."));
	m_pconfirm_on_game_new_dialog->set_icon( get_icon() );
	int button = m_pconfirm_on_game_new_dialog->run();
	switch(button)
	{
	case(Gtk::RESPONSE_OK):
	{
		DBGM("OK clicked.");
		break;
	}
	case(Gtk::RESPONSE_CANCEL):
	{
		DBGM("Cancel clicked.");
		break;
	}
	default:
	{
		DBGM("Unexpected button clicked.");
		break;
	}
	}

	if (discard && 0!=m_pconfirm_on_game_new_dialog) {
		delete m_pconfirm_on_game_new_dialog;
	}

	return (Gtk::RESPONSE_OK == button);
}

void
SudokukimmWindow::on_game_new()
{
	DBGM();

	int confirm = discard_grid_confirm_dialog( false );

	if ( confirm ) {
	} else {
		if (0!=m_pconfirm_on_game_new_dialog) {
			delete m_pconfirm_on_game_new_dialog;
		}
		return;
	}

	m_psudokukimm->reset_generated_grid();
	m_psudokukimm->solver_init(); //using generated grid
	m_psudokukimm->player_reset();
	m_psudokukimm->set_grid_resolved( false );

	m_react_when_cell_filled = false;
	gui_init_grid( false );
	m_react_when_cell_filled = true;

	m_dead_end = false;
	m_psudokukimm->set_dead_end_state( false );
	m_pstatusbar->pop();

	delete m_pconfirm_on_game_new_dialog;

	m_rpuimgr->get_action( "ui/MenuBar/EditMenu/Resolve" )->set_sensitive( true );

	if ( m_edit_custom_grid_in_progress ) {
		m_rpuimgr->remove_ui( m_custom_grid_merge_id );
		m_custom_grid_merge_id = m_rpuimgr->add_ui_from_string(ui_setup_custom_grid);
		m_rpuimgr->get_action( "ui/MenuBar/EditMenu/SetupCustomGrid" )->set_sensitive( true );
		m_rpuimgr->get_action( "ui/MenuBar/EditMenu/PlayCustomGrid" )->set_sensitive( false );
		m_rpuimgr->get_action( "ui/MenuBar/GameMenu/SaveAs" )->set_sensitive( true );
		m_rpuimgr->get_action( "ui/MenuBar/GameMenu/Print" )->set_sensitive( true );
		m_rpuimgr->get_action( "ui/MenuBar/EditMenu/Preferences/LanguageMenu" )->set_sensitive( true );
		m_rpuimgr->get_action( "ui/MenuBar/EditMenu/Preferences/KanjiNumbersMenu" )->set_sensitive( true );
		m_edit_custom_grid_in_progress = false;
	}
}

void
SudokukimmWindow::on_game_open()
{
	DBGM();
	std::string text = "Sudokuki - ";
	text.append( _("Open grid...") );
	Gtk::FileChooserDialog fcd(text, Gtk::FILE_CHOOSER_ACTION_OPEN );

	fcd.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
	fcd.add_button( _("Open"), Gtk::RESPONSE_OK);

	Gtk::FileFilter filter_sudokuki;
	filter_sudokuki.set_name(_("Sudokuki grid files"));
	filter_sudokuki.add_pattern("*.ski");
	fcd.add_filter(filter_sudokuki);

	fcd.set_icon( get_icon() );

	int x, y; fcd.get_position(x,y); DBGF("x:%d y:%d", x, y); // debug

#if __SUDOKUKI_WIN32__
	fcd.set_position(Gtk::WIN_POS_CENTER_ALWAYS);
#else/*__SUDOKUKI_WIN32__*/
	fcd.set_transient_for(*this);
	fcd.set_position(Gtk::WIN_POS_CENTER_ON_PARENT);
#endif/*__SUDOKUKI_WIN32__*/

	int result = fcd.run();

	switch(result)
	{
	case(Gtk::RESPONSE_OK):
	{
		DBGM("Open clicked.");

		std::string fileName = fcd.get_filename();
		DBGF("File selected: %s", fileName.c_str());
		string line;
		ifstream myfile( fileName.c_str() );
		if (myfile.is_open()) {
			if (! myfile.eof() )
			{
				getline (myfile,line);
				DBGF("%s", line.c_str() );
				m_psudokukimm->reset_generated_grid( line );
				m_psudokukimm->solver_init(); //using generated grid
				m_psudokukimm->player_reset( line );

				m_dead_end = false;
				m_psudokukimm->set_dead_end_state( false );
				m_pstatusbar->pop();
				m_rpuimgr->get_action( "ui/MenuBar/EditMenu/Resolve" )->set_sensitive( true );
			} else {
				DBGM("SudokukimmWindow::on_game_open() invalid file couldn't get first line");
			}

			// Retrieve second line containing saved grid
			if (! myfile.eof() )
			{
				getline (myfile, line);
				DBGF("%s", line.c_str());
				m_psudokukimm->player_fill_additional_moves( line );

				m_react_when_cell_filled = false;
				gui_init_grid( false );
				m_react_when_cell_filled = true;
				m_dead_end = false;
				m_psudokukimm->set_dead_end_state( false );
				m_pstatusbar->pop();

				if ( m_psudokukimm->solver_get_how_many_numbers() < m_psudokukimm->player_get_how_many_numbers() ) {
					m_rpuimgr->get_action( "ui/MenuBar/EditMenu/Undo" )->set_sensitive( true );
					m_rpuimgr->get_action( "ui/MenuBar/EditMenu/Clear" )->set_sensitive( true );
				}
			} else {
				DBGM("SudokukimmWindow::on_game_open() invalid file couldn't get second line");
			}

			myfile.close();
		} else {
			DBGM("Unable to open file");
		}
		break;
	}
	case(Gtk::RESPONSE_CANCEL):
	{
		DBGM("Cancel clicked.");
		break;
	}
	default:
	{
		DBGM("Unexpected button clicked.");
		break;
	}
	}
	m_psudokukimm->set_grid_resolved(false);
}

void
SudokukimmWindow::on_game_save_as()
{
	DBGM();
	DBGM("SudokukimmWindow::on_game_save_as()");

	std::string text = "Sudokuki - ";
	text.append( _("Save grid as...") );
	Gtk::FileChooserDialog fcd(text, Gtk::FILE_CHOOSER_ACTION_SAVE );

	fcd.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
	fcd.add_button( _("Save"), Gtk::RESPONSE_OK);

	Gtk::FileFilter filter_sudokuki;
	filter_sudokuki.set_name(_("Sudokuki grid files"));
	filter_sudokuki.add_pattern("*.ski");
	fcd.add_filter(filter_sudokuki);

	fcd.set_icon( get_icon() );

	int x, y; fcd.get_position(x,y); DBGF("x:%d y:%d", x, y); // debug

#if __SUDOKUKI_WIN32__
	fcd.set_position(Gtk::WIN_POS_CENTER_ALWAYS);
#else/*__SUDOKUKI_WIN32__*/
	fcd.set_transient_for(*this);
	fcd.set_position(Gtk::WIN_POS_CENTER_ON_PARENT);
#endif/*__SUDOKUKI_WIN32__*/

	int result = fcd.run();

	switch(result)
	{
	case(Gtk::RESPONSE_OK):
	{
		DBGM("Save clicked.");

		std::string fileName = fcd.get_filename();
		DBGF("File selected: %s", fileName.c_str() );

		int pos = fileName.find(".ski", 0);
		DBGF("fileName '.ski' found? =>%d", pos);
		if ( -1 != pos && (pos+4)==(int)fileName.size() ) {
			DBGF("fileName '.ski' found? =>%d Ok", pos);
		} else {
			DBGF("fileName '.ski' found? =>%d KO", pos);
			fileName.append(".ski");

			/// Just to check // code to remove later
			pos = fileName.find(".ski", 0);
			DBGF("/++/fileName '.ski' found? =>%d", pos);
			if ( -1 != pos && (pos+4)==(int)fileName.size() ) {
				DBGF("/++/fileName '.ski' found? =>%d Ok", pos);
			} else {
				DBGF("/++/fileName '.ski' found? =>%d KO", pos);
			}
			/// Just to check // code to remove later
		}

		fstream file_op(fileName.c_str(), ios::out);
		file_op << m_psudokukimm->get_generated_grid();
		file_op << "\n";
		file_op << m_psudokukimm->get_current_player_values_string();
		file_op << "\n";
		file_op.close();
		break;
	}
	case(Gtk::RESPONSE_CANCEL):
	{
		DBGM("Cancel clicked.");
		break;
	}
	default:
	{
		DBGM("Unexpected button clicked.");
		break;
	}
	}
}

bool
SudokukimmWindow::on_cell_clicked(GdkEventButton* ev, int li, int co)
{
	if (ev->type == GDK_BUTTON_PRESS && ev->button == 3) { // right mouse click
		m_focus_idx = m_last_cell_with_focus_index = (li * 9) + co; // right click on ...

		Gtk::Entry* p_ent = m_pentries[m_focus_idx];
		p_ent->grab_focus();

		resolve_cell_action_update();
		m_rpuimgr->get_action( "ui/PopupMenu/TryValue_1" )->set_visible( m_psudokukimm->player_is_value_possible(li, co, 1) );
		m_rpuimgr->get_action( "ui/PopupMenu/Unavailable_1" )->set_visible( !m_psudokukimm->player_is_value_possible(li, co, 1) );
		m_rpuimgr->get_action( "ui/PopupMenu/TryValue_2" )->set_visible( m_psudokukimm->player_is_value_possible(li, co, 2) );
		m_rpuimgr->get_action( "ui/PopupMenu/Unavailable_2" )->set_visible( !m_psudokukimm->player_is_value_possible(li, co, 2) );
		m_rpuimgr->get_action( "ui/PopupMenu/TryValue_3" )->set_visible( m_psudokukimm->player_is_value_possible(li, co, 3) );
		m_rpuimgr->get_action( "ui/PopupMenu/Unavailable_3" )->set_visible( !m_psudokukimm->player_is_value_possible(li, co, 3) );
		m_rpuimgr->get_action( "ui/PopupMenu/TryValue_4" )->set_visible( m_psudokukimm->player_is_value_possible(li, co, 4) );
		m_rpuimgr->get_action( "ui/PopupMenu/Unavailable_4" )->set_visible( !m_psudokukimm->player_is_value_possible(li, co, 4) );
		m_rpuimgr->get_action( "ui/PopupMenu/TryValue_5" )->set_visible( m_psudokukimm->player_is_value_possible(li, co, 5) );
		m_rpuimgr->get_action( "ui/PopupMenu/Unavailable_5" )->set_visible( !m_psudokukimm->player_is_value_possible(li, co, 5) );
		m_rpuimgr->get_action( "ui/PopupMenu/TryValue_6" )->set_visible( m_psudokukimm->player_is_value_possible(li, co, 6) );
		m_rpuimgr->get_action( "ui/PopupMenu/Unavailable_6" )->set_visible( !m_psudokukimm->player_is_value_possible(li, co, 6) );
		m_rpuimgr->get_action( "ui/PopupMenu/TryValue_7" )->set_visible( m_psudokukimm->player_is_value_possible(li, co, 7) );
		m_rpuimgr->get_action( "ui/PopupMenu/Unavailable_7" )->set_visible( !m_psudokukimm->player_is_value_possible(li, co, 7) );
		m_rpuimgr->get_action( "ui/PopupMenu/TryValue_8" )->set_visible( m_psudokukimm->player_is_value_possible(li, co, 8) );
		m_rpuimgr->get_action( "ui/PopupMenu/Unavailable_8" )->set_visible( !m_psudokukimm->player_is_value_possible(li, co, 8) );
		m_rpuimgr->get_action( "ui/PopupMenu/TryValue_9" )->set_visible( m_psudokukimm->player_is_value_possible(li, co, 9) );
		m_rpuimgr->get_action( "ui/PopupMenu/Unavailable_9" )->set_visible( !m_psudokukimm->player_is_value_possible(li, co, 9) );

		m_ppopup_menu->popup(ev->button, ev->time); // show context menu
		DBGM();
		resolve_cell_action_update();
		m_pop_up_menu_displayed = true;
		return true;
	}
	return false;
}

void
SudokukimmWindow::on_cell_resolve()
{
	if (m_last_cell_with_focus_index < 0) { // focus is not on a cell
		m_pstatusbar->push(_("Select a cell to resolve first"));
		return;
	}

	if ( m_dead_end ) {
		DBGM("calling show_dead_end_dialog");
		show_dead_end_dialog();
		return;
	}

	Gtk::Entry* p_ent = m_pentries[m_last_cell_with_focus_index];
	if ("" != p_ent->get_text()) {
		m_pstatusbar->push(_("The cell is not empty..."));
		return;
	}
	std::map<int, int> solMap = m_psudokukimm->get_solution_values();
	int v = solMap[m_last_cell_with_focus_index]; // at() method not always available? well, let's use []

	int li = m_last_cell_with_focus_index/9;
	int co = m_last_cell_with_focus_index%9;
	DBGF("li:%d, co:%d, v:%d", li, co, v);

	char v_char_roman = symbols[v].c_str()[0];
	DBGF("previous value in cell (should be empty):%d", m_psudokukimm->player_get_cell_value(li, co));

	int cell_filled_result = m_psudokukimm->player_set_cell_value(li, co, v_char_roman);

	/* cell_filled_result: 0=KO, 1=Ok, 100=grid complete, 3=dead end */
	if ( 1 == cell_filled_result/* cell filled Ok */
			|| /* grid completed */100 == cell_filled_result
			|| /* dead end */3 == cell_filled_result ) {
		m_react_when_cell_filled = false;
		p_ent->modify_text(Gtk::STATE_NORMAL, Gdk::Color("dark green"));
		p_ent->set_text( symbols[m_kanji_mode*10+v].c_str() );
		m_react_when_cell_filled = true;
		p_ent->set_editable(false);

		resolve_cell_action_enable( false );
		m_rpuimgr->get_action( "ui/MenuBar/EditMenu/Clear" )->set_sensitive( true );
		m_rpuimgr->get_action( "ui/MenuBar/EditMenu/Undo" )->set_sensitive( true );

		if ( 3 == cell_filled_result ) {
			m_dead_end = true;
			m_psudokukimm->set_dead_end_state( true );

			DBGM("calling show_dead_end_dialog");
			show_dead_end_dialog();
			p_ent->set_editable(false);
		}

	} else {
		DBGM("Cannot resolve cell because one of the previous moves is in the way..."
				"calling show_dead_end_dialog without setting 'dead_end' to true");
		resolve_cell_action_enable( false );
		show_dead_end_dialog();
	}
}

void
SudokukimmWindow::on_game_resolve()
{
	DBGM();
	m_react_when_cell_filled = false;
	m_psudokukimm->player_reset();
	m_psudokukimm->set_grid_resolved(true);
	gui_init_grid( true );

	std::map<int, int> my_map = m_psudokukimm->get_solution_values();

	std::map<int, int>::iterator p;
	for(p = my_map.begin(); p!=my_map.end(); ++p) {
		int cell = p->first;
		int value = p->second;
		DBGF("cell:%d/value:%d", cell, value);

		Gtk::Entry* p_ent = m_pentries[cell];

		string text = " ";
		if ( 1<=value && value<=9) {
			text = symbols[m_kanji_mode*10+value];
		}
		DBGF("text:%s  entry_text:%s", text.c_str(), p_ent->get_text().c_str() );
		if ( text != p_ent->get_text() ) {
			p_ent->modify_text(Gtk::STATE_NORMAL, Gdk::Color("dark green"));
			int li = cell/9;
			int co = cell%9;
			DBGF("text:%s li:%d, co:%d", text.c_str(), li, co);
			char v_char_roman = symbols[value].c_str()[0];
			int result = m_psudokukimm->player_set_cell_value(li, co, v_char_roman);
			assert(100==result || 1==result);
		}
		p_ent->set_text( text.c_str() );
		p_ent->set_editable( false );
	}

	m_rpuimgr->get_action( "ui/MenuBar/EditMenu/Resolve" )->set_sensitive( false );
	resolve_cell_action_enable( false );
	m_rpuimgr->get_action( "ui/MenuBar/EditMenu/Undo" )->set_sensitive( false );
	m_rpuimgr->get_action( "ui/MenuBar/EditMenu/Clear" )->set_sensitive( false );
	m_game_resolved = true;
}
void
SudokukimmWindow::on_game_quit()
{
	DBGM();
	hide();
	Gtk::Main::quit();
	DBGM("SudokukimmWindow::quit() end");
}
void
SudokukimmWindow::on_game_print_generated_grid_in_browser()
{
	std::string gridStr( m_psudokukimm->get_generated_grid() );

	std::string contenu;
	contenu += sudokukimm_html_header;

	for ( int a = 0; a < 3; a++ )
	{
		contenu += "<tr>\n";
		for ( int b = 0; b < 3; b++ )
		{
			// 3x3 table
			contenu += "<td>\n"
				"<table class=\"square\" cellpadding=\"0\" cellspacing=\"0\">\n"
				"<tbody>\n";
			for ( int j = 0; j < 3; j++ )
			{
				contenu += "<tr>\n";
				for ( int i = 0; i < 3; i++ )
				{
					if ( (b==1 || a==1) && b != a )
					{
						//grey_cell
						contenu += "<td><div class=\"cell\">";
					} else {
						// cell
						contenu += "<td><div class=\"cell\">";
					}
					//					int value = m_solver.solver_cell_value_get(3*a+j,3*b+i);
					int value = m_psudokukimm->solver_get_cell_value(3*a+j,3*b+i);
					DBGF("value of cell(%d,%d):%d", 3*a+j, 3*b+i, value);
					if ( 0 <= value && value <= 9 ) {
						string text = symbols[m_kanji_mode*10+value];
						//write the value
						contenu += text;
					} else {
						//empty cell
						contenu += "";
					}
					contenu += "</div></td>\n";
				}
				contenu += "</tr>\n";
			}
			//end of 3x3 table
			contenu += "</tbody>\n";
			contenu += "</table>\n";
			contenu += "</td>\n";
		}
		contenu += "</tr>\n";
	}

	contenu += sudokukimm_html_footer;

	std::string fileName = Glib::build_filename(Glib::get_tmp_dir(), "sudokuki.html");
	std::ofstream myFile(fileName.c_str());
	if ( true == myFile.is_open() ) {
		myFile << contenu << '\n';
		myFile.close();
		DBGM("The grid seems to have been exported with success.");
	}

	std::string browser;
#if __SUDOKUKI_WIN32__
	DBGM("Trying 'firefox.exe'...");
	browser = Glib::find_program_in_path("firefox.exe");
	if ( 0 == browser.length() ) {
		DBGM("Trying 'iexplore.exe'...");
		browser = Glib::find_program_in_path("iexplore.exe");
		if ( 0 == browser.length() ) {
			DBGM("Trying 'C:\\Program Files\\Mozilla Firefox\\firefox.exe'...");
			browser = Glib::find_program_in_path("C:\\Program Files\\Mozilla Firefox\\firefox.exe");
			if ( 0 == browser.length() ) {
				DBGM("Trying 'D:\\Program Files\\Mozilla Firefox\\firefox.exe'...");
				browser = Glib::find_program_in_path("D:\\Program Files\\Mozilla Firefox\\firefox.exe");
				if ( 0 == browser.length() ) {
					DBGM("Trying 'C:\\Program Files\\Internet Explorer\\IEXPLORE.EXE'...");
					browser = Glib::find_program_in_path("C:\\Program Files\\Internet Explorer\\iexplore.exe");
					if ( 0 == browser.length() ) {
						DBGM("Trying 'D:\\Program Files\\Internet Explorer\\IEXPLORE.EXE'...");
						browser = Glib::find_program_in_path("D:\\Program Files\\Internet Explorer\\iexplore.exe");
					}
				}
			}
		}
	}
#endif/*__SUDOKUKI_WIN32__*/
	bool found = false;
	std::string SUDOKUKI_BROWSER = Glib::getenv(std::string("SUDOKUKI_BROWSER"), found);
	DBGM("Environment variable SUDOKUKI_BROWSER was ");
	if (found) { DBGM("found"); } else { DBGM("not found"); }
	DBGF("SUDOKUKI_BROWSER:%s|", SUDOKUKI_BROWSER.c_str());
	if ( found ) browser = SUDOKUKI_BROWSER;
#if !(__SUDOKUKI_WIN32__)
	if ( 0 == browser.length() ) {
		DBGM("SUDOKUKI_BROWSER not set. Looking for a browser to open the HTML grid...");
		DBGM("Trying 'epiphany'...");
		browser = Glib::find_program_in_path("epiphany");
		if ( 0 == browser.length() ) {
			DBGM("Trying 'iceweasel'...");
			browser = Glib::find_program_in_path("iceweasel");
			if ( 0 == browser.length() ) {
				DBGM("Trying 'firefox'...");
				browser = Glib::find_program_in_path("firefox");
				if ( 0 == browser.length() ) {
					DBGM("Trying 'konqueror'...");
					browser = Glib::find_program_in_path("konqueror");
				}
			}
		}
	}
#endif/*!(__SUDOKUKI_WIN32__)*/

	if ( 0 == browser.length() ) {
		DBGM("Please try specifying your browser location in environment variable SUDOKUKI_BROWSER and retry.");
		DBGM("No browser found. Maybe you are using a specific platform? Please report the problem in the hope it can be fixed.");
	} else {
		DBGF("Found: %s", browser.c_str());
#if __SUDOKUKI_WIN32__
		vector<std::string> args;
		args.push_back(browser);
		args.push_back(fileName);
		Glib::spawn_async("", args, Glib::SPAWN_SEARCH_PATH);
#else/*__SUDOKUKI_WIN32__*/
		DBGF("Attempting to open the grid using the following command line:%s %s", browser, fileName );
		Glib::spawn_command_line_async(browser+" "+fileName);
#endif/*__SUDOKUKI_WIN32__*/
	}
	DBGM();
}
void
SudokukimmWindow::on_edit_undo()
{
	DBGM();
	m_react_when_cell_filled = false;

	m_psudokukimm->go_back_one_grid();

	std::map<int, int> my_map = m_psudokukimm->get_current_player_values();

	std::string bidon = m_psudokukimm->get_current_player_values_string();
	DBGF("SudokukimmWindow::on_edit_undo() : %s", bidon.c_str());

	std::map<int, int>::iterator p;
	for(p = my_map.begin(); p!=my_map.end(); ++p) {
		int cell = p->first;
		int value = p->second;
		DBGF("cell:%d/value:%d", cell, value);

		Gtk::Entry* p_ent = m_pentries[cell];

		string text = "";
		if ( 1 <= value && value <= 9 ) {
			text = symbols[m_kanji_mode*10+value];
		} else {
			text = "";
			p_ent->set_editable(true);
			p_ent->modify_text(Gtk::STATE_NORMAL, Gdk::Color("blue"));
		}
		if ( p_ent->get_text() != text )
			p_ent->set_text( text );
		DBGF("text:%s  entry_text:%s", text.c_str(), p_ent->get_text().c_str() );
	}
	m_react_when_cell_filled = true;
	if ( !m_edit_custom_grid_in_progress ) {
		m_rpuimgr->get_action( "ui/MenuBar/EditMenu/Resolve" )->set_sensitive( true );
		resolve_cell_action_update();
	}

	if ( m_psudokukimm->get_current_player_grid_index() == 1 ) {
		m_rpuimgr->get_action( "ui/MenuBar/EditMenu/Undo" )->set_sensitive( false );
		m_rpuimgr->get_action( "ui/MenuBar/EditMenu/Clear" )->set_sensitive( false );
	}

	m_dead_end = false;
	m_psudokukimm->set_dead_end_state( false );
}
void
SudokukimmWindow::on_edit_copy()
{
	DBGM();
}
void
SudokukimmWindow::on_edit_paste()
{
	DBGM();
}
void
SudokukimmWindow::on_edit_clear()
{
	// Clear all moves
	DBGM();
	m_psudokukimm->player_reset();

	m_react_when_cell_filled = false;
	gui_init_grid( false );

	m_react_when_cell_filled = true;

	m_dead_end = false;
	m_psudokukimm->set_dead_end_state( false );
	m_pstatusbar->pop();

	if ( !m_edit_custom_grid_in_progress ) {
		m_rpuimgr->get_action( "ui/MenuBar/EditMenu/Resolve" )->set_sensitive( true );
	}
}
void
SudokukimmWindow::on_edit_setup_custom_grid()
{
	// Allow the user to enter a custom grid
	DBGM();
	if ( discard_grid_confirm_dialog( true ) ) {
		m_edit_custom_grid_in_progress = true;
		m_rpuimgr->get_action( "ui/MenuBar/EditMenu/Preferences/LanguageMenu" )->set_sensitive( false );
		m_rpuimgr->get_action( "ui/MenuBar/EditMenu/Preferences/KanjiNumbersMenu" )->set_sensitive( false );
	} else {
		// do not edit custom grid because user pressed CANCEL
		return;
	}

	m_psudokukimm->reset_generated_grid("000000000000000000000000000000000000000000000000000000000000000000000000000000000");
	m_psudokukimm->solver_init(); //using generated grid
	m_psudokukimm->player_reset();
	m_psudokukimm->set_grid_resolved( false );

	m_react_when_cell_filled = false;
	gui_init_grid( false );
	m_react_when_cell_filled = true;

	m_dead_end = false;
	m_psudokukimm->set_dead_end_state( false );
	m_pstatusbar->pop();

	m_rpuimgr->get_action( "ui/MenuBar/EditMenu/Undo" )->set_sensitive( false );
	m_rpuimgr->get_action( "ui/MenuBar/EditMenu/Clear" )->set_sensitive( false );
	m_rpuimgr->get_action( "ui/MenuBar/GameMenu/New" )->set_sensitive( true );
	m_rpuimgr->get_action( "ui/MenuBar/GameMenu/Open" )->set_sensitive( true );
	m_rpuimgr->get_action( "ui/MenuBar/GameMenu/SaveAs" )->set_sensitive( false );
	m_rpuimgr->get_action( "ui/MenuBar/GameMenu/Print" )->set_sensitive( false );

	m_rpuimgr->remove_ui( m_custom_grid_merge_id );
	m_custom_grid_merge_id = m_rpuimgr->add_ui_from_string(ui_play_custom_grid);

	m_rpuimgr->get_action( "ui/MenuBar/EditMenu/SetupCustomGrid" )->set_sensitive( false );
	m_rpuimgr->get_action( "ui/MenuBar/EditMenu/PlayCustomGrid" )->set_sensitive( true );
	m_rpuimgr->get_action( "ui/MenuBar/EditMenu/Resolve" )->set_sensitive( false );
	resolve_cell_action_enable( false );
	m_rpuimgr->get_action( "ui/MenuBar/EditMenu/Undo" )->set_sensitive( false );
}

void
SudokukimmWindow::on_edit_play_custom_grid()
{
	DBGM("on_edit_play_custom_grid");
	// user stops filling the grid, now let's memorize it!

	DBGM();
	std::string tmp_string;
	for ( int i=0; i<81; i++) {
		DBGF("%s", m_pentries[i]->get_text().c_str() );
		char tmp_char = m_pentries[i]->get_text()[0];
		if ( '1' <= tmp_char && tmp_char <= '9' ) {
			tmp_string.push_back( tmp_char );
			m_pentries[i]->modify_text(Gtk::STATE_NORMAL, Gdk::Color("black"));
			m_pentries[i]->modify_font( *m_pbold_font );
		} else {
			tmp_string.push_back( '-' );
		}
	}
	DBGM();
	DBGM();
	DBGF("String to memorize: %s", tmp_string.c_str() );
	m_psudokukimm->reset_generated_grid( tmp_string );
	m_psudokukimm->solver_init(); //using custom grid

	m_psudokukimm->player_reset();
	m_psudokukimm->set_grid_resolved( false );
	m_dead_end = false;
	m_psudokukimm->set_dead_end_state( false );
	m_pstatusbar->pop();

	m_rpuimgr->get_action( "ui/MenuBar/EditMenu/Preferences/LanguageMenu" )->set_sensitive( true );
	m_rpuimgr->get_action( "ui/MenuBar/EditMenu/Preferences/KanjiNumbersMenu" )->set_sensitive( true );
	m_rpuimgr->get_action( "ui/MenuBar/EditMenu/Undo" )->set_sensitive( false );
	m_rpuimgr->get_action( "ui/MenuBar/EditMenu/Clear" )->set_sensitive( false );
	m_rpuimgr->get_action( "ui/MenuBar/GameMenu/New" )->set_sensitive( true );
	m_rpuimgr->get_action( "ui/MenuBar/GameMenu/Open" )->set_sensitive( true );
	m_rpuimgr->get_action( "ui/MenuBar/GameMenu/SaveAs" )->set_sensitive( true );
	m_rpuimgr->get_action( "ui/MenuBar/GameMenu/Print" )->set_sensitive( true );

	m_rpuimgr->remove_ui( m_custom_grid_merge_id );
	m_custom_grid_merge_id = m_rpuimgr->add_ui_from_string(ui_setup_custom_grid);

	m_rpuimgr->get_action( "ui/MenuBar/EditMenu/SetupCustomGrid" )->set_sensitive( true );
	m_rpuimgr->get_action( "ui/MenuBar/EditMenu/PlayCustomGrid" )->set_sensitive( false );
	m_rpuimgr->get_action( "ui/MenuBar/EditMenu/Resolve" )->set_sensitive( true );
	resolve_cell_action_update();

	m_edit_custom_grid_in_progress = false;
}

void
SudokukimmWindow::resolve_cell_action_update()
{
	//	if ( !m_edit_custom_grid_in_progress && 0 <= m_focus_idx && m_focus_idx <= 80 ) {
	DBGF("m_edit_custom_grid_in_progress:%d m_last_cell_with_focus_index:%d", m_edit_custom_grid_in_progress, m_last_cell_with_focus_index);
	if ( !m_edit_custom_grid_in_progress && 0 <= m_last_cell_with_focus_index && m_last_cell_with_focus_index <= 80 ) {
		Gtk::Entry* p_ent = m_pentries[m_last_cell_with_focus_index];
		DBGF("calling resolve_cell_action_enable(%d)", ("" == p_ent->get_text()) );
		resolve_cell_action_enable( "" == p_ent->get_text() );
	} else {
		DBGM("calling resolve_cell_action_enable(false)");
		resolve_cell_action_enable( false );
	}
}

void
SudokukimmWindow::resolve_cell_action_enable( bool enable )
{
	DBGF("%d", enable);
	m_rpuimgr->get_action( "ui/MenuBar/EditMenu/ResolveCell" )->set_sensitive( enable );
}

void
SudokukimmWindow::on_edit_print_settings()
{
	DBGM();

	bool found = false;
	std::string SUDOKUKI_BROWSER = Glib::getenv(std::string("SUDOKUKI_BROWSER"), found);

	std::string message = _("At the moment, SUDOKUKI_BROWSER is ");
	if (found) {
		message.append( _("set to:") );
		message.append( "\n\n" );
		message.append( SUDOKUKI_BROWSER );
	} else {
		message.append( _("not set.") );
	}
	message.append( "\n\n" );
	message.append( _("This environment variable allows you to select\n"
			"which HTML browser to use for printing grids.\n") );
	message.append( _("It is useful if your browser is not detected\n"
			"automatically.") );
	message.append("\n\n");
	message.append( _("Would you need any help, don't hesitate to ask\n"
			"on the Sudokuki forums:") );
	message.append("\n\n");
	message.append( "http://sudokuki.sourceforge.net/" );
	Gtk::MessageDialog sudokuki_browser_info_dialog(*this,
			_("Print settings"),
			false,
			Gtk::MESSAGE_INFO, Gtk::BUTTONS_CLOSE, true );
	sudokuki_browser_info_dialog.set_secondary_text(message);
	sudokuki_browser_info_dialog.set_icon( get_icon() );
	sudokuki_browser_info_dialog.run();
}
void
SudokukimmWindow::on_help_about()
{
	DBGM();
	Gtk::AboutDialog about_dialog;

	about_dialog.set_transient_for(*this);
	about_dialog.set_position(Gtk::WIN_POS_CENTER_ON_PARENT);

	std::vector<Glib::ustring> authors;
	authors.push_back( _("  -- Author -- ") );
	authors.push_back("Sylvain Vedrenne <sylvain.vedrenne@free.fr>");
	authors.push_back("");
	authors.push_back( _("  -- Contributors -- ") );
	authors.push_back("Claus Poerschke <claus-poerschke@gmx.de>");
	about_dialog.set_authors( authors );

	std::vector<Glib::ustring> documenters;
	documenters.push_back( _("  -- Users mailing list -- ") );
	documenters.push_back("<sudokuki-users@lists.sourceforge.net>");
	documenters.push_back("");
	documenters.push_back( _("  -- Websites -- ") );
	documenters.push_back("http://directory.fsf.org/project/sudokuki/");
	documenters.push_back("http://aur.archlinux.org/packages.php?ID=24354");
	documenters.push_back("http://www.murga-linux.com/puppy/viewtopic.php?p=281347");
	documenters.push_back("http://www.todoprogramas.com/macintosh/descargar/sudokuki");
	documenters.push_back("http://mac.softpedia.com/get/Games/Sudokuki.shtml");
	documenters.push_back("http://linux.softpedia.com");
	documenters.push_back("...");
	about_dialog.set_documenters( documenters );

	std::string trans_creds = "";
	trans_creds.append( _("  -- Translators mailing list -- ") );
	trans_creds.append("\n");
	trans_creds.append("<sudokuki-i18n@lists.sourceforge.net>");
	trans_creds.append("\n");
	trans_creds.append("\n");
	trans_creds.append( _("  -- Translators -- ") );
	trans_creds.append("\n");
	trans_creds.append(_("German"));
	trans_creds.append( " (de): Claus Poerschke" );
	trans_creds.append("\n");
	trans_creds.append(_("Greek"));
	trans_creds.append( " (el): Hélène Damianakos");
	trans_creds.append("\n");
	trans_creds.append(_("Esperanto"));
	trans_creds.append( " (eo): Sylvain Vedrenne");
	trans_creds.append("\n");
	trans_creds.append(_("Spanish"));
	trans_creds.append( " (es): Aritz Beraza");
	trans_creds.append("\n");
	trans_creds.append(_("French"));
	trans_creds.append( " (fr): Sylvain Vedrenne");
	trans_creds.append("\n");
	trans_creds.append(_("Japanese"));
	trans_creds.append( " (ja): Hélène Vedrenne");
	trans_creds.append("\n");
	about_dialog.set_translator_credits( trans_creds);

	std::vector<Glib::ustring> artists;
	artists.push_back("Hélène & Sylvain Vedrenne");
	about_dialog.set_artists( artists );

	about_dialog.set_name( "Sudokuki" );
	about_dialog.set_icon( get_icon() );
	about_dialog.set_version( sudokuki_version );
	about_dialog.set_copyright( "Copyright 2007-2009 Sylvain Vedrenne" );
	about_dialog.set_comments( _("Feel free to request features and report bugs\n"
			"on the Sudokuki Forums hosted by Sourceforge!") );
	about_dialog.set_wrap_license( true );
	about_dialog.set_license( sudokukimm_license );
	about_dialog.set_website("http://sudokuki.sourceforge.net/");
	about_dialog.set_logo( get_logo() );
	about_dialog.run();
}

void
SudokukimmWindow::game_complete_end_game()
{
	DBGM();

	Gtk::MessageDialog game_complete_dialog(*this,
			_("<b><big>Grid complete!</big></b>\n\n"
					"You have succesfully completed the grid.\n\n"
					"<b>- Congratulations -</b>"),
					true,
					Gtk::MESSAGE_INFO );
	game_complete_dialog.set_icon( get_icon() );
	game_complete_dialog.run();

}

bool
SudokukimmWindow::on_key_pressed(GdkEventKey* ev)
{
	assert(-1<=m_focus_idx);
	if( ev->keyval == GDK_Up ) DBGM("     ^");
	if( ev->keyval == GDK_Down ) DBGM("     v");
	if( ev->keyval == GDK_Left ) DBGM("     <");
	if( ev->keyval == GDK_Right ) DBGM("     >");
	DBGF(" m_focus_idx:%d", m_focus_idx);
	DBGF(" m_las_cell_with_focus:%d", m_last_cell_with_focus_index);
	if (-1!=m_focus_idx && ev->keyval==GDK_Left) {
		if (m_focus_idx-1>=0) {
			m_focus_idx--;
			m_pentries[m_focus_idx]->grab_focus();
		}
	} else if (-1!=m_focus_idx && ev->keyval==GDK_Right) {
		if (m_focus_idx+1<=80) {
			m_focus_idx++;
			m_pentries[m_focus_idx]->grab_focus();
		}
	} else if ( (-1==m_focus_idx) && (ev->keyval==GDK_Down) ) {
		m_focus_idx = m_last_cell_with_focus_index;
		m_pentries[m_focus_idx]->grab_focus();
	}

	return true;
}

bool
SudokukimmWindow::on_cell_focus_out(GdkEventFocus* event, int li, int co)
{
	DBGF("cell %d,%d\n", li, co);
	if ( (9*li + co) == m_last_cell_with_focus_index ) {
		m_focus_idx = -1; // focus is now out of the grid
	}
	DBGF(" m_focus_idx:%d", m_focus_idx);
	DBGF(" m_last_cell_with_focus:%d", m_last_cell_with_focus_index);

	if ( !m_pop_up_menu_displayed ) {
		resolve_cell_action_enable( false );
	}

	return false;
}

bool
SudokukimmWindow::on_cell_focus_in(GdkEventFocus* event, int li, int co)
{
	DBGF(" cell: %d,%d", li, co);
	m_pop_up_menu_displayed = false;

	m_focus_idx = 9*li + co;
	if ( m_focus_idx != m_last_cell_with_focus_index ) {
		m_pstatusbar->pop();
	}
	m_last_cell_with_focus_index = 9*li + co;

	DBGF(" m_focus_idx:%d", m_focus_idx);
	DBGF(" m_last_cell_with_focus:%d", m_last_cell_with_focus_index);

	resolve_cell_action_update();
	return false;
}

void
SudokukimmWindow::on_try_value_in_cell(int val)
{
	assert(m_last_cell_with_focus_index >= 0);
	assert(1<=val && val<=9);

	Gtk::Entry* p_ent = m_pentries[m_last_cell_with_focus_index];
	p_ent->set_text(symbols[val]);
}

void
SudokukimmWindow::on_cell_filled(int li, int co)
{
	//TODO: if custom grid: black font !!!!!!
	DBGF("on_cell_filled(%d,%d)", li, co);

	if ( !m_react_when_cell_filled ) {
		DBGM("(DOES NOT REACT WHEN CELL FILLED)");
		return;
	}

	int cell_id = 9*(li) + co;
	DBGF("cell(%d,%d)(cell_id:%d) attempt to fill this cell...", li, co, cell_id);

	Gtk::Entry* p_ent = m_pentries[cell_id];

	if ( m_dead_end ) {
		m_react_when_cell_filled = false;
		p_ent->set_text("");
		m_react_when_cell_filled = true;
		DBGM("calling show_dead_end_dialog");
		show_dead_end_dialog();
		return;
	}

	if (p_ent) {
		const char* ch = p_ent->get_text().c_str();
		if ( '1' <= ch[0] && ch[0] <= '9' ) {
			int cell_filled_result = m_psudokukimm->player_set_cell_value(li, co, ch[0]);

			DBGF("player cell filled?:%d", cell_filled_result);

			int value = ch[0]-'0';
			DBGF("on_cell_filled value:%d", value);
			string text = symbols[m_kanji_mode*10+value];
			DBGF("on_cell_filled text:%s no_kanji_equiv:%s", text.c_str(), symbols[value].c_str() );
			m_react_when_cell_filled = false;
			p_ent->set_text(text);
			m_react_when_cell_filled = true;
			if ( 1 == cell_filled_result || 3 == cell_filled_result ) {
				p_ent->set_editable(false);

				resolve_cell_action_enable( false );
				m_rpuimgr->get_action( "ui/MenuBar/EditMenu/Clear" )->set_sensitive( true );
				m_rpuimgr->get_action( "ui/MenuBar/EditMenu/Undo" )->set_sensitive( true );
			} else if ( 100 == cell_filled_result ) {
				p_ent->set_editable(false);
				m_rpuimgr->get_action( "ui/MenuBar/EditMenu/Resolve" )->set_sensitive( false );
				m_rpuimgr->get_action( "ui/MenuBar/EditMenu/Clear" )->set_sensitive( false );
				m_rpuimgr->get_action( "ui/MenuBar/EditMenu/Undo" )->set_sensitive( false );
				game_complete_end_game();
			}
			if ( 1 != cell_filled_result && 100 != cell_filled_result ) {
				if ( 3 == cell_filled_result ) {

					m_dead_end = true;
					m_psudokukimm->set_dead_end_state( true );
					DBGM("calling show_dead_end_dialog");
					show_dead_end_dialog();
					p_ent->set_editable(false);
				} else {
					m_react_when_cell_filled = false;
					p_ent->set_text("");
					m_react_when_cell_filled = true;
				}
			}
		} else {
			m_react_when_cell_filled = false;
			p_ent->set_text("");
			m_react_when_cell_filled = true;
		}
	} else {
		DBGM("cell not found! This should never happen!");
		assert(false);
	}
}

void
SudokukimmWindow::show_dead_end_dialog()
{
	Gtk::MessageDialog dead_end_dialog(*this,
			_("<b><big>Dead End</big></b>\n\n"
					"No further move is possible.\n\n"
					"Your previous moves seemed valid,\n"
					"but you've gone a wrong way!...\n\n"
					"<b>Go back some moves, and try again.</b>")
					,
					true,
					Gtk::MESSAGE_ERROR );
	dead_end_dialog.set_icon( get_icon() );
	dead_end_dialog.run();
}

void
SudokukimmWindow::gui_init_grid( bool dark_green )
{
	DBGM("");

	if ( dark_green ) {
		//		m_pother_font->set_style(Pango::STYLE_ITALIC);
	} else {
		//		m_pother_font->set_style(Pango::STYLE_NORMAL);
	}

	for ( int a = 0; a < 3; a++ )
	{
		for ( int b = 0; b < 3; b++ )
		{
			for ( int j = 0; j < 3; j++ )
			{
				for ( int i = 0; i < 3; i++ )
				{
					int cell_id = 9*(3*a+j) + 3*b+i;
					DBGF("cell_id: %d", cell_id);

					Gtk::Entry* p_ent = m_pentries[cell_id];

					DBGF("%d", 3);
					int value = m_psudokukimm->solver_get_cell_value(3*a+j,3*b+i);
					int player_value = m_psudokukimm->player_get_cell_value(3*a+j,3*b+i);
					DBGF("previous value of cell(%d,%d):%d", 3*a+j, 3*b+i, value);

					assert ( p_ent != 0 );

					DBGF("value:%d player_value:%d", value, player_value);

					if ( 0 <= value && value <= 9 ) {
						DBGF("%d bis", 3);

#if __SUDOKUKI_WIN32__
						if ( m_kanji_mode ) {
							p_ent->modify_font( *m_pbold_big_font );
						} else {
#endif/*__SUDOKUKI_WIN32__*/
							p_ent->modify_font( *m_pbold_font );
#if __SUDOKUKI_WIN32__
						}
#endif/*__SUDOKUKI_WIN32__*/
						DBGF("%d", 4);
						string text = symbols[m_kanji_mode*10+value];
						p_ent->modify_text(Gtk::STATE_NORMAL, Gdk::Color("black"));
						if ( p_ent->get_text() != text ) {
							p_ent->set_text( text );
						}
						p_ent->set_editable( false );

					} else if ( 0 <= player_value && player_value <= 9 ) {

						p_ent->modify_font( *m_pother_font );
						string text = symbols[m_kanji_mode*10+player_value];
						if ( dark_green ) {
							p_ent->modify_text(Gtk::STATE_NORMAL, Gdk::Color("dark green"));
						} else {
							p_ent->modify_text(Gtk::STATE_NORMAL, Gdk::Color("blue"));
						}
						if ( p_ent->get_text() != text ) {
							p_ent->set_text( text );
						}
						p_ent->set_editable( false );
					} else {

						DBGF("%d 0x%x", 5, (unsigned int)p_ent);

						p_ent->modify_font( *m_pother_font );
						DBGF("%d", 6);
						p_ent->set_max_length(1);
						p_ent->modify_text(Gtk::STATE_NORMAL, Gdk::Color("blue"));
						if ( p_ent->get_text() != "" ) {
							p_ent->set_text("");
						}
						p_ent->set_editable( true );
					}
				}
			}
		}
	}
	m_rpuimgr->get_action( "ui/MenuBar/EditMenu/Undo" )->set_sensitive( false );
	m_rpuimgr->get_action( "ui/MenuBar/EditMenu/Clear" )->set_sensitive( false );
	m_rpuimgr->get_action( "ui/MenuBar/EditMenu/Resolve" )->set_sensitive( true );

}

Glib::RefPtr<Gdk::Pixbuf>
SudokukimmWindow::get_icon()
{
	if ( !m_picon ) {
		m_picon = Gdk::Pixbuf::create_from_inline( -1, sudokuki_icon, false );
	}
	return m_picon;
}

Glib::RefPtr<Gdk::Pixbuf>
SudokukimmWindow::get_logo()
{
	if ( !m_plogo ) {
		m_plogo = Gdk::Pixbuf::create_from_inline( -1, sudokuki_logo, false );
	}
	return m_plogo;
}
