#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <pwd.h>
#include <errno.h>

#include "skk.h"

static char *get_skkdicfullpath(void) {
    char *homedirectory, *skkdicfullpath;
    char skkdicname[] = ".skk-jisyo";

    homedirectory = getenv("HOME");
    if(!homedirectory) {
	struct passwd *pw;
	pw = getpwuid(getuid());
	homedirectory = strdup(pw->pw_dir);
    }

    skkdicfullpath = malloc(strlen(homedirectory) + strlen(skkdicname) + 2);
    if(skkdicfullpath != NULL) {
	snprintf(skkdicfullpath,
		 strlen(homedirectory) + strlen(skkdicname) + 2,
		 "%s/%s", homedirectory, skkdicname);
    }
    return skkdicfullpath;
}

void free_skk_dic(skklist *head) {
    skklist *pos;
    for(pos = head; pos != NULL; pos = pos->next) {
	free(pos->yomi);
	free(pos->kanji);
	free(pos);
    }
}

skklist *skk_dic_list_new(char *yomi, char *kanji, int okuri) {
    skklist *entry;
    entry = malloc(sizeof(skklist));
    if(entry != NULL) {
	entry->kanji = strdup(kanji);
	entry->yomi = strdup(yomi);
	entry->okuri = okuri;
	entry->next = NULL;
    }
    return entry;
}

static skklist *skk_dic_list_last(skklist *list) {
    if(list) {
	while(list->next) {
	    list = list->next;
	}
    }
    return list;
}

skklist *skk_dic_list_append(skklist *list, skklist *entry) {
    skklist *last;
    if(list) {
	last = skk_dic_list_last(list);
	last->next = entry;
    } else {
	list = entry;
    }
    return list;
}

int read_skk_dic(skklist **list, int *num) {
    FILE *fp;
    char *skkdicfullpath;
    char buf[256], kanji[256], yomi[256];
    int okuri = 0, fd;
    skklist *entry, *prev;

    skkdicfullpath = get_skkdicfullpath();
    if(skkdicfullpath == NULL) {
	return -1;
    }

    fp = fopen(skkdicfullpath, "r");
    free(skkdicfullpath);

    if(!fp) {
	return -1;
    }

    prev = NULL; *list = NULL; *num = 0;
    while(fgets(buf, sizeof(buf), fp)) {
	if(buf[0] != '#' && buf[0] != ';') {
	    sscanf(buf, "%s %s", yomi, kanji);

	    entry = skk_dic_list_new(yomi, kanji, okuri);
	    *list = skk_dic_list_append(*list, entry);
	    (*num)++;
	} else {
	    if(strstr(buf, "okuri-ari")) {
		okuri = 0;
	    }
	    if(strstr(buf, "okuri-nasi")) {
		okuri = 1;
	    }
	}
    }
    fclose(fp);
    return 0;
}

int write_skk_dic(skklist *head) {
    char *skkdicfullpath;
    skklist *pos;
    FILE *fp;
    int fd;
    struct flock lock;

    skkdicfullpath = get_skkdicfullpath();
    if(skkdicfullpath == NULL) {
	return -1;
    }

    fp = fopen(skkdicfullpath, "w");
    free(skkdicfullpath);
    if(!fp) {
	return -1;
    }

    fd = fileno(fp);
    lock.l_type = F_WRLCK;
    lock.l_start = 0;
    lock.l_whence = SEEK_SET;
    lock.l_len = 0;

    if(fcntl(fd, F_SETLK, &lock) < 0) {
	fprintf(stderr, "Lock failed: %s\n", strerror(errno));
	fclose(fp);
	return -1;
    }

    fprintf(fp, ";; okuri-ari entries.\n");
    for(pos = head; pos != NULL; pos = pos->next) {
	if(pos->okuri == 0)
	    fprintf(fp, "%s %s\n", pos->yomi, pos->kanji);
    }
    fprintf(fp, ";; okuri-nasi entries.\n");
    for(pos = head; pos != NULL; pos = pos->next) {
	if(pos->okuri == 1)
	    fprintf(fp, "%s %s\n", pos->yomi, pos->kanji);
    }

    lock.l_type = F_UNLCK;
    if(fcntl(fd, F_SETLK, &lock) < 0) {
	fprintf(stderr, "Unlock failed: %s\n",strerror(errno));
	fclose(fp);
	return -1;
    }

    fclose(fp);
    return 0;
}
