/*
 *  anthyaddwordpane.c
 *  Copyright(C) 2003- Masahito Omote <omote@utyuuzin.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "anthy.h"

#include "anthyaddwordpane.h"
#include "charset.h"
#include "intl.h"

static char *touroku_meishi_options[] = {
    N_("Kakujoshi Setsuzoku"),
    N_("Gokan nomi de bunsetsu"),
    N_("Suru setsuzoku"),
    N_("Sa setsuzoku"),
    N_("Na setsuzoku")
};

static char *touroku_fukushi_options[] = {
    N_("Gokan nomi de bunsetsu"),
    N_("Suru setsuzoku"),
    N_("Taru setsuzoku"),
    N_("To setsuzoku")
};

static const gchar *hinshi_items[] = {
    N_("Meishi"),
    N_("Chimei"),
    N_("Zinmei"),
    N_("Doushi"),
    N_("Keiyoushi"),
    N_("Fukushi")
};

typedef enum {
    ANTHY_MEISHI,
    ANTHY_CHIMEI,
    ANTHY_ZINMEI,
    ANTHY_DOUSHI,
    ANTHY_KEIYOUSHI,
    ANTHY_FUKUSHI,
} anthy_hinshi_attr_list;

static anthy_hinshi_attr_list touroku_hinshi_num;

/* callbacks */
static void cb_touroku_button_clicked (GtkButton *, AnthyAddWordPane *);
static void cb_clear_button_clicked   (GtkButton *, AnthyAddWordPane *);

static void enable_checkbox_meishi_options(AnthyAddWordPane *pane) {
    int fukushi_checkbox_num = sizeof(pane->fukushi_options) / sizeof(pane->fukushi_options[0]);
    int meishi_checkbox_num = sizeof(pane->meishi_options) / sizeof(pane->meishi_options[0]);
    int i;
    for(i = 0; i < meishi_checkbox_num; i++) {
	gtk_widget_set_sensitive(GTK_WIDGET(pane->meishi_options[i]), TRUE);
	gtk_widget_show(pane->meishi_options[i]);
    }
    for(i = 0; i < fukushi_checkbox_num; i++) {
	gtk_widget_hide(pane->fukushi_options[i]);
    }
}

static void enable_checkbox_fukushi_options(AnthyAddWordPane *pane) {
    int fukushi_checkbox_num = sizeof(pane->fukushi_options) / sizeof(pane->fukushi_options[0]);
    int meishi_checkbox_num = sizeof(pane->meishi_options) / sizeof(pane->meishi_options[0]);
    int i;

    for(i = 0; i < fukushi_checkbox_num; i++) {
	gtk_widget_set_sensitive(GTK_WIDGET(pane->fukushi_options[i]), TRUE);
	gtk_widget_show(pane->fukushi_options[i]);
    }
    for(i = 0; i < meishi_checkbox_num; i++) {
	gtk_widget_hide(pane->meishi_options[i]);
    }
}

static void disable_checkbox_all_options(AnthyAddWordPane *pane) {
    int i;
    int fukushi_checkbox_num = sizeof(pane->fukushi_options) / sizeof(pane->fukushi_options[0]);
    int meishi_checkbox_num = sizeof(pane->meishi_options) / sizeof(pane->meishi_options[0]);

    for(i = 0; i < meishi_checkbox_num; i++) {
	gtk_widget_set_sensitive(GTK_WIDGET(pane->meishi_options[i]), FALSE);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pane->meishi_options[i]), FALSE);
    }
    for(i = 0; i < fukushi_checkbox_num; i++) {
	gtk_widget_set_sensitive(GTK_WIDGET(pane->fukushi_options[i]), FALSE);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pane->fukushi_options[i]), FALSE);
    }
}

static void cb_touroku_button_clicked(GtkButton *button,
				      AnthyAddWordPane *pane)
{
    GtkWidget *dialog;

    const gchar *utf8_yomi, *utf8_kanji;
    char *euc_yomi, *euc_kanji, *hinshi_code;
    int freq;

    int i, ret = -1;
    int flag = 0;

    utf8_yomi = gtk_entry_get_text(GTK_ENTRY(pane->yomi));
    utf8_kanji = gtk_entry_get_text(GTK_ENTRY(pane->kanji));
    freq = atoi(gtk_entry_get_text(GTK_ENTRY(pane->freq)));

    switch(touroku_hinshi_num) {
    case ANTHY_MEISHI:
	for(i = 4; i >= 0; i--) {
	    flag = flag << 1;
	    if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(pane->meishi_options[i]))) {
		flag = flag | 0x1;
	    }
	}
	hinshi_code = find_meishi_code_from_flag(flag);
	break;
    case ANTHY_FUKUSHI:
	for(i = 3; i >= 0; i--) {
	    flag = flag << 1;
	    if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(pane->fukushi_options[i]))) {
		flag = flag | 0x1;
	    }
	}
	hinshi_code = find_fukushi_code_from_flag(flag);
	break;
    case ANTHY_CHIMEI:
	hinshi_code = strdup("#CN");
	break;
    case ANTHY_ZINMEI:
	hinshi_code = strdup("#JNM");
	break;
    case ANTHY_DOUSHI:
	hinshi_code = NULL;
	break;
    case ANTHY_KEIYOUSHI:
	hinshi_code = NULL;
	break;
    default:
	hinshi_code = NULL;
	break;
    }

    euc_yomi = charset_convert(utf8_yomi, "UTF-8", "EUC-JP");
    euc_kanji = charset_convert(utf8_kanji, "UTF-8", "EUC-JP");
    g_print("freq %d\n", freq);
    ret = add_anthy_priv_dic_with_flags(euc_yomi, euc_kanji,
		    			hinshi_code, freq);

    if(!ret) {
	dialog = gtk_message_dialog_new(NULL,
					GTK_DIALOG_MODAL,
					GTK_MESSAGE_INFO,
					GTK_BUTTONS_CLOSE,
					_("Word registration succeded."));
	/* clear all data */
	cb_clear_button_clicked(NULL, pane);
    } else {
	dialog = gtk_message_dialog_new(NULL,
					GTK_DIALOG_MODAL,
					GTK_MESSAGE_ERROR,
					GTK_BUTTONS_CLOSE,
					_("Word registration failed."));
    }
    gtk_dialog_run(GTK_DIALOG(dialog));
    gtk_widget_destroy(dialog);
    free(hinshi_code);
}

static void cb_clear_button_clicked(GtkButton *button,
				    AnthyAddWordPane *pane) {
    int i;
    int fukushi_checkbox_num = sizeof(pane->fukushi_options) /
				 sizeof(pane->fukushi_options[0]);

    int meishi_checkbox_num = sizeof(pane->meishi_options) /
				sizeof(pane->meishi_options[0]);

    for(i = 0; i < meishi_checkbox_num; i++) {
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pane->meishi_options[i]), FALSE);
    }
    for(i = 0; i < fukushi_checkbox_num; i++) {
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(pane->fukushi_options[i]), FALSE);
    }

    gtk_entry_set_text(GTK_ENTRY(pane->yomi),"");
    gtk_entry_set_text(GTK_ENTRY(pane->kanji), "");
}

static gint cb_toggle_hinshi(GtkWidget *widget,
			     AnthyAddWordPane *pane) {
    touroku_hinshi_num = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widget), "hinshi"));
    switch (touroku_hinshi_num) {
    case ANTHY_MEISHI:
	enable_checkbox_meishi_options(pane);
	break;
    case ANTHY_FUKUSHI:
	enable_checkbox_fukushi_options(pane);
	break;
    case ANTHY_DOUSHI:
    case ANTHY_KEIYOUSHI:
    case ANTHY_CHIMEI:
    case ANTHY_ZINMEI:
	disable_checkbox_all_options(pane);
	break;
    default:
	break;
    }
    return FALSE;
}

int create_addwordpane(AnthyAddWordPane *pane) {
    gint i;
    GtkWidget *label_yomi, *label_kanji, *label_hinshi, *label_freq;
    GtkWidget *table, *table2;
    GtkWidget *vbox1;
    GtkWidget *vbox2;
    GtkWidget *frame;
    GtkWidget *optmenu_alignment;
    GtkWidget *menu, *menu_item;
    gint hinshi_num = sizeof(hinshi_items) / sizeof(gchar*);
    anthy_hinshi_attr_list cattr;

    pane->pane = gtk_hbox_new(FALSE, 15);
    gtk_container_set_border_width(GTK_CONTAINER(pane->pane), 15);

    vbox1 = gtk_vbox_new(FALSE, 0);
    gtk_widget_show(vbox1);
    gtk_box_pack_start(GTK_BOX(pane->pane), vbox1, FALSE, FALSE, 0);

    /* */
    table = gtk_table_new(2, 4, FALSE);
    gtk_widget_show(table);
    gtk_box_pack_start(GTK_BOX(vbox1), table, FALSE, FALSE, 0);

    label_yomi = gtk_label_new(_("Yomi:"));
    gtk_misc_set_alignment(GTK_MISC(label_yomi), 1.0, 0.5);
    gtk_widget_show(label_yomi);
    gtk_table_attach(GTK_TABLE(table), label_yomi,
		     0, 1, 0, 1,
		     GTK_FILL, 0, 5, 5);

    pane->yomi = gtk_entry_new();
    gtk_table_attach(GTK_TABLE(table), pane->yomi, 1, 2, 0, 1, 0, 0, 5, 5);
    gtk_widget_show(pane->yomi);

    label_kanji = gtk_label_new(_("Kanji:"));
    gtk_misc_set_alignment(GTK_MISC(label_kanji), 1.0, 0.5);
    gtk_widget_show(label_kanji);
    gtk_table_attach(GTK_TABLE(table), label_kanji,
		     0, 1, 1, 2,
		     GTK_FILL, 0, 5, 5);

    pane->kanji = gtk_entry_new();
    gtk_table_attach(GTK_TABLE(table), pane->kanji,
		     1, 2, 1, 2,
		     GTK_FILL, 0, 5, 5);
    gtk_widget_show(pane->kanji);

    label_hinshi = gtk_label_new(_("Hinshi:"));
    gtk_misc_set_alignment(GTK_MISC(label_hinshi), 1.0, 0.5);
    gtk_widget_show(label_hinshi);
    gtk_table_attach(GTK_TABLE(table), label_hinshi,
		     0, 1, 2, 3,
		     GTK_FILL, 0, 5, 5);

    pane->freq = gtk_entry_new();
    gtk_entry_set_max_length(GTK_ENTRY(pane->freq), 5); /* 0 - 65536 */
    gtk_entry_set_text(GTK_ENTRY(pane->freq), "1");
    gtk_table_attach(GTK_TABLE(table), pane->freq,
		     1, 2, 3, 4,
		     GTK_FILL, 0, 5, 5);
    gtk_widget_show(pane->freq);

    label_freq = gtk_label_new(_("Freq:"));
    gtk_misc_set_alignment(GTK_MISC(label_freq), 1.0, 0.5);
    gtk_widget_show(label_freq);
    gtk_table_attach(GTK_TABLE(table), label_freq,
		     0, 1, 3, 4,
		     GTK_FILL, 0, 5, 5);


    pane->hinshi = gtk_option_menu_new();
    menu = gtk_menu_new();
    for(i = 0, cattr = ANTHY_MEISHI; i < hinshi_num; i++, cattr++) {
	menu_item = gtk_menu_item_new_with_label(_(hinshi_items[i]));
	g_object_set_data(G_OBJECT(menu_item), "hinshi", GINT_TO_POINTER(cattr));
	g_signal_connect(G_OBJECT(menu_item), "activate",
			 G_CALLBACK(cb_toggle_hinshi), pane);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), menu_item);
	gtk_widget_show(menu_item);
    }
    gtk_option_menu_set_menu(GTK_OPTION_MENU(pane->hinshi), menu);

    optmenu_alignment = gtk_alignment_new(0, 0.5, 0, 0);
    gtk_container_add(GTK_CONTAINER(optmenu_alignment), pane->hinshi);
    gtk_widget_show(optmenu_alignment);
    gtk_table_attach(GTK_TABLE(table), optmenu_alignment,
		     1, 2, 2, 3, GTK_FILL, GTK_FILL, 5, 5);
    gtk_widget_show(pane->hinshi);

    /* Register Options */
    frame = gtk_frame_new(_("Touroku Options"));

    gtk_widget_show(frame);
    gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, TRUE, 10);

    /* register options */
    table2 = gtk_table_new(3, 2, TRUE);
    gtk_container_set_border_width(GTK_CONTAINER(table2), 5);
    gtk_widget_show(table2);
    gtk_container_add(GTK_CONTAINER(frame), table2);

    /* meishi options */
    for(i = 0; i < 5; i++) {
	pane->meishi_options[i] = gtk_check_button_new_with_label(_(touroku_meishi_options[i]));
	gtk_widget_show(pane->meishi_options[i]);
    }
    gtk_table_attach_defaults(GTK_TABLE(table2), pane->meishi_options[0],
			      0, 1, 0, 1);
    gtk_table_attach_defaults(GTK_TABLE(table2), pane->meishi_options[1],
			      1, 2, 0, 1);
    gtk_table_attach_defaults(GTK_TABLE(table2), pane->meishi_options[2],
			      0, 1, 1, 2);
    gtk_table_attach_defaults(GTK_TABLE(table2), pane->meishi_options[3],
			      1, 2, 1, 2);
    gtk_table_attach_defaults(GTK_TABLE(table2), pane->meishi_options[4],
			      0, 1, 2, 3);
    /* fukushi options */
    for(i = 0; i < 4; i++) {
	pane->fukushi_options[i] = gtk_check_button_new_with_label(_(touroku_fukushi_options[i]));
    }
    gtk_table_attach_defaults(GTK_TABLE(table2), pane->fukushi_options[0],
			      0, 1, 0, 1);
    gtk_table_attach_defaults(GTK_TABLE(table2), pane->fukushi_options[1],
			      1, 2, 0, 1);
    gtk_table_attach_defaults(GTK_TABLE(table2), pane->fukushi_options[2],
			      0, 1, 1, 2);
    gtk_table_attach_defaults(GTK_TABLE(table2), pane->fukushi_options[3],
			      1, 2, 1, 2);

    /* */
    vbox2 = gtk_vbutton_box_new();
    gtk_button_box_set_layout(GTK_BUTTON_BOX(vbox2), GTK_BUTTONBOX_START);
    gtk_box_set_spacing(GTK_BOX(vbox2), 10);
    gtk_widget_show(vbox2);
    gtk_box_pack_start(GTK_BOX(pane->pane), vbox2, FALSE, FALSE, 130);

    pane->touroku = gtk_button_new_with_label(_("Touroku"));
    gtk_box_pack_start(GTK_BOX(vbox2), pane->touroku, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(pane->touroku), "clicked",
		     G_CALLBACK(cb_touroku_button_clicked), pane);
    gtk_widget_show(pane->touroku);

    pane->clear = gtk_button_new_with_label(_("Clear"));
    gtk_box_pack_start(GTK_BOX(vbox2), pane->clear, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(pane->clear), "clicked",
		     G_CALLBACK(cb_clear_button_clicked), pane);
    gtk_widget_show(pane->clear);

    return 0;
}
