/*
 *  cannacontainer.c
 *  Copyright(C) 2003- Masahito Omote <omote@utyuuzin.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include <gtk/gtk.h>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "cannacontainer.h"
#include "wordlist_view.h"
#include "charset.h"
#include "intl.h"
#include "canna.h"
#include "cannadic.h"

enum {
    COLUMN_CCLASS_CODE,
    N_COLUMNS_CCLASS_CODE,
};

enum {
    SUBSTANTIVE_KAKUJOSHI,
    SUBSTANTIVE_GOKAN_NOMI,
    SUBSTANTIVE_SURU,
    SUBSTANTIVE_SA,
    SUBSTANTIVE_NA,

    SUBSTANTIVE_VISIBLE,
    SUBSTANTIVE_ACTIVATABLE,
    N_SUBSTANTIVE_COLUMNS,
};

enum {
    ADVERB_GOKAN_NOMI_DE_BUNSETSU,
    ADVERB_SURU,
    ADVERB_TARU,
    ADVERB_TO,

    ADVERB_VISIBLE,
    ADVERB_ACTIVATABLE,
    N_ADVERB_COLUMNS,
};

static char *cclass_code_broad[] = {
    N_("Verb"),
    N_("Substantive"),
    N_("Adjective"),
    N_("Adverb"),
    N_("Etc"),
};

static void cb_canna_viewwordpane_button_load_clicked   (GtkButton *,
						         CannaViewWordPane *);
static void cb_canna_viewwordpane_button_delete_clicked (GtkButton *,
							 CannaViewWordPane *);
static void cb_button_cclass_browse_clicked             (GtkButton *,
							 CannaAddWordPane *);

static void cb_cclass_code_item_toggled (GtkCellRendererToggle *,
					 gchar *,
					 gpointer);

static void cb_add_button_clicked   (GtkButton *, CannaAddWordPane *);
static void cb_clear_button_clicked (GtkButton *, CannaAddWordPane *);

static void cclass_verb_dialog        (CannaAddWordPane *pane);
static void cclass_substantive_dialog (CannaAddWordPane *pane);
static void cclass_adjective_dialog   (CannaAddWordPane *pane);
static void cclass_adverb_dialog      (CannaAddWordPane *pane);
static void cclass_etc_dialog         (CannaAddWordPane *pane);

void create_canna_container(CannaContainer *container) {
    /* I changed coding style a little. */
    GtkWidget *notebook;
    CannaViewWordPane *viewwordpane;
    CannaAddWordPane *addwordpane;
    GtkWidget *label_viewword, *label_addword;

    notebook = gtk_notebook_new();
    viewwordpane = canna_viewword_pane_new();
    addwordpane = canna_addword_pane_new();

    gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook), GTK_POS_TOP);
    gtk_container_set_border_width(GTK_CONTAINER(notebook), 10);

    gtk_container_add(GTK_CONTAINER(notebook), viewwordpane->pane);
    gtk_widget_show(viewwordpane->pane);
    label_viewword = gtk_label_new(_("View word"));
    gtk_widget_show(label_viewword);
    gtk_notebook_set_tab_label(GTK_NOTEBOOK(notebook),
			       gtk_notebook_get_nth_page(GTK_NOTEBOOK(notebook), 0),
			       label_viewword);

    gtk_container_add(GTK_CONTAINER(notebook), addwordpane->pane);
    gtk_widget_show(addwordpane->pane);
    label_addword = gtk_label_new(_("Add word"));
    gtk_widget_show(label_addword);
    gtk_notebook_set_tab_label(GTK_NOTEBOOK(notebook),
			       gtk_notebook_get_nth_page(GTK_NOTEBOOK(notebook), 1),
			       label_addword);

    container->container = notebook;
    container->addwordpane = addwordpane;
}

CannaAddWordPane *canna_addword_pane_new(void) {
    CannaAddWordPane *pane;
    GtkWidget *hbox;
    GtkWidget *vbox1, *vbox2;
    GtkWidget *table1;
    GtkWidget *label_phon, *label_desc;
    GtkWidget *label_cclass_code_narrow, *label_cclass_code_broad;
    GtkWidget *optmenu_cclass_code;
    GtkWidget *alignment_cclass_code;
    GtkWidget *menu_cclass_code, *menuitem_cclass_code;
    GtkWidget *entry_phon, *entry_desc, *entry_cclass_code_narrow;
    GtkWidget *button_cclass_browse;
    GtkWidget *button_add, *button_clear;
    int cclass_code_num;
    int i;

    cclass_code_num = sizeof(cclass_code_broad)/sizeof(cclass_code_broad[0]);

    pane = g_malloc0(sizeof(CannaAddWordPane));

    hbox = gtk_hbox_new(FALSE, 0);
    gtk_container_set_border_width(GTK_CONTAINER(hbox), 15);

    vbox1 = gtk_vbox_new(FALSE, 0);
    gtk_widget_show(vbox1);
    gtk_box_pack_start(GTK_BOX(hbox), vbox1, FALSE, FALSE, 0);

    table1 = gtk_table_new(5, 5, FALSE);
    gtk_widget_show(table1);
    gtk_box_pack_start(GTK_BOX(vbox1), table1, FALSE, FALSE, 0);

    label_phon = gtk_label_new(_("Phon:"));
    gtk_misc_set_alignment(GTK_MISC(label_phon), 1.0, 0.5);
    gtk_widget_show(label_phon);
    gtk_table_attach(GTK_TABLE(table1), label_phon,
		     0, 1, 0, 1, GTK_FILL, 0, 5, 5);

    entry_phon = gtk_entry_new();
    gtk_table_attach(GTK_TABLE(table1), entry_phon,
		     1, 2, 0, 1, 0, 0, 5, 5);
    gtk_widget_show(entry_phon);

    label_desc = gtk_label_new(_("Desc:"));
    gtk_misc_set_alignment(GTK_MISC(label_desc), 1.0, 0.5);
    gtk_widget_show(label_desc);
    gtk_table_attach(GTK_TABLE(table1), label_desc,
		     0, 1, 1, 2, GTK_FILL, 0, 5, 5);

    entry_desc = gtk_entry_new();
    gtk_table_attach(GTK_TABLE(table1), entry_desc,
		     1, 2, 1, 2, 0, 0, 5, 5);
    gtk_widget_show(entry_desc);

    label_cclass_code_broad = gtk_label_new(_("Conjection Class(broad):"));
    gtk_misc_set_alignment(GTK_MISC(label_cclass_code_broad), 1.0, 0.5);
    gtk_widget_show(label_cclass_code_broad);
    gtk_table_attach(GTK_TABLE(table1), label_cclass_code_broad,
		     0, 1, 2, 3, GTK_FILL, 0, 5, 5);

    optmenu_cclass_code = gtk_option_menu_new();
    pane->optmenu_cclass_code = optmenu_cclass_code;
    menu_cclass_code = gtk_menu_new();
    for(i = 0; i < cclass_code_num; i++) {
	menuitem_cclass_code = gtk_menu_item_new_with_label(_(cclass_code_broad[i]));
/*
	g_signal_connect(G_OBJECT(menuitem_cclass_code), "activate",
			 G_CALLBACK(cb_toggle_cclass_code), pane);
*/
	gtk_menu_shell_append(GTK_MENU_SHELL(menu_cclass_code),
			      menuitem_cclass_code);
	gtk_widget_show(menuitem_cclass_code);
    }

    gtk_option_menu_set_menu(GTK_OPTION_MENU(optmenu_cclass_code),
			     menu_cclass_code);
    gtk_widget_show(optmenu_cclass_code);

    alignment_cclass_code = gtk_alignment_new(0, 0.5, 0, 0);
    gtk_container_add(GTK_CONTAINER(alignment_cclass_code),
		      optmenu_cclass_code);
    gtk_widget_show(alignment_cclass_code);
    gtk_table_attach(GTK_TABLE(table1), alignment_cclass_code,
		     1, 2, 2, 3, GTK_FILL, GTK_FILL, 5, 5);

    label_cclass_code_narrow = gtk_label_new(_("Conjection Class(narrow):"));
    gtk_misc_set_alignment(GTK_MISC(label_cclass_code_narrow), 1.0, 0.5);
    gtk_widget_show(label_cclass_code_narrow);
    gtk_table_attach(GTK_TABLE(table1), label_cclass_code_narrow,
		     0, 1, 3, 4, GTK_FILL, 0, 5, 5);

    entry_cclass_code_narrow = gtk_entry_new();
    gtk_widget_show(entry_cclass_code_narrow);
    gtk_table_attach(GTK_TABLE(table1), entry_cclass_code_narrow,
		     1, 2, 3, 4, GTK_FILL, 0, 5, 5);

    button_cclass_browse = gtk_button_new_with_label(_("Browse..."));
    gtk_widget_show(button_cclass_browse);
    gtk_table_attach(GTK_TABLE(table1), button_cclass_browse,
		     2, 3, 3, 4, GTK_FILL, 0, 5, 5);

    g_signal_connect(G_OBJECT(button_cclass_browse), "clicked",
		     G_CALLBACK(cb_button_cclass_browse_clicked), pane);

    vbox2 = gtk_vbutton_box_new();
    gtk_button_box_set_layout(GTK_BUTTON_BOX(vbox2), GTK_BUTTONBOX_START);
    gtk_box_set_spacing(GTK_BOX(vbox2), 10);
    gtk_widget_show(vbox2);
    gtk_box_pack_start(GTK_BOX(hbox), vbox2, FALSE, FALSE, 50);

    button_add = gtk_button_new_with_label(_("Add"));
    gtk_box_pack_start(GTK_BOX(vbox2), button_add, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(button_add), "clicked",
		     G_CALLBACK(cb_add_button_clicked), pane);
    gtk_widget_show(button_add);

    button_clear = gtk_button_new_with_label(_("Clear"));
    gtk_box_pack_start(GTK_BOX(vbox2), button_clear, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(button_clear), "clicked",
		     G_CALLBACK(cb_clear_button_clicked), pane);
    gtk_widget_show(button_clear);

    pane->pane = GTK_WIDGET(hbox);
    pane->phon = GTK_WIDGET(entry_phon);
    pane->desc = GTK_WIDGET(entry_desc);
    pane->cclass_code = GTK_WIDGET(entry_cclass_code_narrow);
    pane->add = GTK_WIDGET(button_add);
    pane->clear = GTK_WIDGET(button_clear);
    return pane;
}

CannaViewWordPane *canna_viewword_pane_new(void) {
    CannaViewWordPane *pane;
    GtkWidget *hbox, *vbox;
    GtkWidget *wordlist;
    GtkWidget *button_load, *button_delete;
    pane = g_malloc0(sizeof(CannaViewWordPane));

    hbox = gtk_hbox_new(FALSE, 10);
    gtk_container_set_border_width(GTK_CONTAINER(hbox), 15);

    wordlist = wordlist_view_new_with_attributes("editable", FALSE,
						 "freq_show", FALSE,
						 "cclass_code_show", TRUE,
						 "selection_mode", GTK_SELECTION_MULTIPLE,
						 NULL);
    gtk_widget_show(wordlist);
    gtk_box_pack_start(GTK_BOX(hbox), wordlist, TRUE, TRUE, 0);

    vbox = gtk_vbutton_box_new();
    gtk_widget_show(vbox);
    gtk_box_pack_start(GTK_BOX(hbox), vbox, FALSE, FALSE, 10);

    button_load = gtk_button_new_with_label(_("Load"));
    gtk_widget_show(button_load);
    gtk_button_box_set_layout(GTK_BUTTON_BOX(vbox), GTK_BUTTONBOX_START);
    gtk_box_set_spacing(GTK_BOX(vbox), 10);
    gtk_box_pack_start(GTK_BOX(vbox), button_load, FALSE, FALSE, 10);

    g_signal_connect(G_OBJECT(button_load), "clicked",
		     G_CALLBACK(cb_canna_viewwordpane_button_load_clicked),
		     pane);

    button_delete = gtk_button_new_with_label(_("Delete"));
    gtk_widget_show(button_delete);
    gtk_box_pack_start(GTK_BOX(vbox), button_delete, FALSE, FALSE, 10);
    gtk_widget_set_sensitive(button_delete, FALSE);

    g_signal_connect(G_OBJECT(button_delete), "clicked",
		     G_CALLBACK(cb_canna_viewwordpane_button_delete_clicked),
		     pane);

    pane->pane = hbox;
    pane->wordlist_canna = wordlist;
    pane->button_load = button_load;
    pane->button_delete = button_delete;
    return pane;
}

void show_canna_container(CannaContainer *container, gboolean flags) {
    if(flags) {
	gtk_widget_show(container->container);
    }
}

/* callbacks */
static void cb_canna_viewwordpane_button_load_clicked(GtkButton *button,
						      CannaViewWordPane *pane)
{
    word *list = NULL, *pos = NULL;

    wordlist_view_clear(WORDLIST_VIEW(pane->wordlist_canna));
    list = canna_get_word_text_priv_dic();
    if(list == NULL)
	return;

    for(pos = list; pos != NULL; pos = pos->next) {
/* for test */
	gchar *eucjp_phon, *eucjp_desc, *eucjp_cclass_code;
	eucjp_phon = pos->phon;
	eucjp_desc = pos->desc;
	eucjp_cclass_code = pos->cclass_code;
	pos->phon = eucjp_to_utf8(eucjp_phon);
	pos->desc = eucjp_to_utf8(eucjp_desc);
	pos->cclass_code = eucjp_to_utf8(eucjp_cclass_code);

	if(pos->phon != NULL && pos->desc != NULL &&
	   pos->cclass_code != NULL) {
	    wordlist_view_set_values(WORDLIST_VIEW(pane->wordlist_canna),
				     pos);
	    free(eucjp_phon);
	    free(eucjp_desc);
	    free(eucjp_cclass_code);
	}
    }
    word_free_list(list);
    gtk_button_set_label(button, _("Reload"));
    gtk_widget_set_sensitive(pane->button_delete, TRUE);
}

static
void cb_canna_viewwordpane_button_delete_clicked (GtkButton *button,
						  CannaViewWordPane *pane)
{
    GList *list, *pos;
    word *data;
    char *utf8_wordinfo, *eucjp_wordinfo;
    int phonlen, desclen, cclasslen, wordinfolen;
    int ret;

    list = wordlist_view_get_selected_data_list(WORDLIST_VIEW(pane->wordlist_canna));

    for(pos = g_list_first(list); pos != NULL; pos = g_list_next(pos)) {
	data = pos->data;
	phonlen = strlen(data->phon);
	desclen = strlen(data->desc);
	cclasslen = strlen(data->cclass_code);
	wordinfolen = phonlen + desclen + cclasslen + 2 + 1;
	utf8_wordinfo = g_malloc(sizeof(char) * wordinfolen);

	snprintf(utf8_wordinfo, wordinfolen, "%s %s %s",
		 data->phon, data->cclass_code, data->desc);
	eucjp_wordinfo = utf8_to_eucjp(utf8_wordinfo);
	ret = canna_delete_dic("user", eucjp_wordinfo);

	g_free(eucjp_wordinfo);
	g_free(utf8_wordinfo);
    }
    /* TODO: must implement free(list) func. */
    wordlist_view_remove_selected_data(WORDLIST_VIEW(pane->wordlist_canna));
}


static void cclass_verb_dialog(CannaAddWordPane *pane) {
    GtkWidget *dialog;
    GtkWidget *scrollwin_cclass_code;
    GtkWidget *treeview_cclass_code;

    GtkTreeViewColumn *column;
    GtkListStore *store;
    GtkTreeSelection *selection;
    GtkCellRenderer *renderer;
    GtkTreeIter iter;
    gint result;
    gchar *cclass_code_utf8 = NULL, *cclass_code_eucjp = NULL;
    int narrow_size, i;

    dialog = gtk_dialog_new_with_buttons(_("Conjection Class"), NULL,
					 GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
					 GTK_STOCK_OK,
					 GTK_RESPONSE_ACCEPT,
					 GTK_STOCK_CANCEL,
					 GTK_RESPONSE_REJECT,
					 NULL);
    gtk_widget_set_size_request(GTK_WIDGET(dialog), 300, 200);

    scrollwin_cclass_code = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollwin_cclass_code),
				   GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox),
		       scrollwin_cclass_code,
		       TRUE, TRUE, 10);
    gtk_widget_show(scrollwin_cclass_code);

    treeview_cclass_code = gtk_tree_view_new();
    gtk_container_add(GTK_CONTAINER(scrollwin_cclass_code),
		      treeview_cclass_code);
    gtk_widget_show(treeview_cclass_code);

    renderer = gtk_cell_renderer_text_new();
    column = gtk_tree_view_column_new_with_attributes("cclass_code",
						      renderer,
						      "text", COLUMN_CCLASS_CODE,
						      NULL);
    gtk_tree_view_append_column(GTK_TREE_VIEW(treeview_cclass_code),column);
    gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(treeview_cclass_code),
				      FALSE);
    selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview_cclass_code));
    gtk_tree_selection_set_mode(selection, GTK_SELECTION_SINGLE);

    store = gtk_list_store_new(N_COLUMNS_CCLASS_CODE, G_TYPE_STRING);
    gtk_tree_view_set_model(GTK_TREE_VIEW(treeview_cclass_code),
			    GTK_TREE_MODEL(store));

    narrow_size = sizeof(verb_code) / sizeof(verb_code[0]);
    for(i = 0; i < narrow_size; i++) {
	gtk_list_store_append(store, &iter);
	gtk_list_store_set(store, &iter,
			   COLUMN_CCLASS_CODE, eucjp_to_utf8(verb_code[i].desc),
			   -1);
    }

    result = gtk_dialog_run(GTK_DIALOG(dialog));

    switch(result) {
    case GTK_RESPONSE_ACCEPT:
	if(gtk_tree_selection_get_selected(selection, NULL, &iter)) {
	    gtk_tree_model_get(GTK_TREE_MODEL(store), &iter,
			       COLUMN_CCLASS_CODE, &cclass_code_utf8,
			       -1);
	    if(cclass_code_utf8 != NULL) {
		cclass_code_eucjp = utf8_to_eucjp(cclass_code_utf8);
		g_free(cclass_code_utf8);
		for(i = 0; i < narrow_size; i++) {
		    if(strcmp(cclass_code_eucjp, verb_code[i].desc) == 0) {
			gtk_entry_set_text(GTK_ENTRY(pane->cclass_code),
					   verb_code[i].code);
			break;
		    }
		}
	    }
	}
	if(cclass_code_eucjp != NULL)
	    g_free(cclass_code_eucjp);
	break;
    default:
	break;
    }

    gtk_widget_destroy(dialog);

}

static void cclass_substantive_dialog(CannaAddWordPane *pane) {
    GtkWidget *dialog;
    GtkWidget *scrollwin_cclass_code;
    GtkWidget *treeview_cclass_code;

    GtkTreeViewColumn *column;
    GtkListStore *store;
    GtkTreeSelection *selection;
    GtkCellRenderer *renderer;
    GtkTreeIter iter;
    gint result;
    gboolean substantive_flag[5];
    int flag;
    int i, code_num;

    dialog = gtk_dialog_new_with_buttons(_("Conjection Class"), NULL,
					 GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
					 GTK_STOCK_OK,
					 GTK_RESPONSE_ACCEPT,
					 GTK_STOCK_CANCEL,
					 GTK_RESPONSE_REJECT,
					 NULL);
    gtk_widget_set_size_request(GTK_WIDGET(dialog), 420, 120);

    scrollwin_cclass_code = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollwin_cclass_code),
				   GTK_POLICY_AUTOMATIC, GTK_POLICY_NEVER);

    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox),
		       scrollwin_cclass_code,
		       TRUE, FALSE, 10);
    gtk_widget_show(scrollwin_cclass_code);

    treeview_cclass_code = gtk_tree_view_new();
    gtk_container_add(GTK_CONTAINER(scrollwin_cclass_code),
		      treeview_cclass_code);
    gtk_widget_show(treeview_cclass_code);

    store = gtk_list_store_new(N_SUBSTANTIVE_COLUMNS,
			       G_TYPE_BOOLEAN,
			       G_TYPE_BOOLEAN,
			       G_TYPE_BOOLEAN,
			       G_TYPE_BOOLEAN,
			       G_TYPE_BOOLEAN,
			       G_TYPE_BOOLEAN,
			       G_TYPE_BOOLEAN);

    gtk_tree_view_set_model(GTK_TREE_VIEW(treeview_cclass_code),
			    GTK_TREE_MODEL(store));

    /* SUBSTANTIVE_KAKUJOSHI */
    renderer = gtk_cell_renderer_toggle_new();
    column = gtk_tree_view_column_new_with_attributes(_("Kakujoshi"),
						      renderer,
						      "active",
						      SUBSTANTIVE_KAKUJOSHI,
						      "visible",
						      SUBSTANTIVE_VISIBLE,
						      "activatable",
						      SUBSTANTIVE_ACTIVATABLE,
						      NULL);
    g_object_set_data (G_OBJECT(renderer), "column",
		       GINT_TO_POINTER(SUBSTANTIVE_KAKUJOSHI));
    g_signal_connect(G_OBJECT(renderer), "toggled",
		     G_CALLBACK(cb_cclass_code_item_toggled), store);
    gtk_tree_view_column_set_sizing(GTK_TREE_VIEW_COLUMN (column),
				    GTK_TREE_VIEW_COLUMN_FIXED);
    gtk_tree_view_column_set_fixed_width(GTK_TREE_VIEW_COLUMN(column), 80);
    gtk_tree_view_append_column(GTK_TREE_VIEW(treeview_cclass_code),column);

    /* SUBSTANTIVE_GOKAN_NOMI */
    renderer = gtk_cell_renderer_toggle_new();
    column = gtk_tree_view_column_new_with_attributes(_("Gokan"),
						      renderer,
						      "active",
						      SUBSTANTIVE_GOKAN_NOMI,
						      "visible",
						      SUBSTANTIVE_VISIBLE,
						      "activatable",
						      SUBSTANTIVE_ACTIVATABLE,
						      NULL);
    g_object_set_data (G_OBJECT(renderer), "column",
		       GINT_TO_POINTER(SUBSTANTIVE_GOKAN_NOMI));
    g_signal_connect(G_OBJECT(renderer), "toggled",
		     G_CALLBACK(cb_cclass_code_item_toggled), store);
    gtk_tree_view_column_set_sizing(GTK_TREE_VIEW_COLUMN (column),
				    GTK_TREE_VIEW_COLUMN_FIXED);
    gtk_tree_view_column_set_fixed_width(GTK_TREE_VIEW_COLUMN(column), 80);
    gtk_tree_view_append_column(GTK_TREE_VIEW(treeview_cclass_code),column);

    /* SUBSTANTIVE_SURU */
    renderer = gtk_cell_renderer_toggle_new();
    column = gtk_tree_view_column_new_with_attributes(_("Suru"),
						      renderer,
						      "active",
						      SUBSTANTIVE_SURU,
						      "visible",
						      SUBSTANTIVE_VISIBLE,
						      "activatable",
						      SUBSTANTIVE_ACTIVATABLE,
						      NULL);
    g_object_set_data (G_OBJECT(renderer), "column",
		       GINT_TO_POINTER(SUBSTANTIVE_SURU));
    g_signal_connect(G_OBJECT(renderer), "toggled",
		     G_CALLBACK(cb_cclass_code_item_toggled), store);
    gtk_tree_view_column_set_sizing(GTK_TREE_VIEW_COLUMN (column),
				    GTK_TREE_VIEW_COLUMN_FIXED);
    gtk_tree_view_column_set_fixed_width(GTK_TREE_VIEW_COLUMN(column), 80);
    gtk_tree_view_append_column(GTK_TREE_VIEW(treeview_cclass_code),column);

    /* SUBSTANTIVE_SA */
    renderer = gtk_cell_renderer_toggle_new();
    column = gtk_tree_view_column_new_with_attributes(_("Sa"),
						      renderer,
						      "active",
						      SUBSTANTIVE_SA,
						      "visible",
						      SUBSTANTIVE_VISIBLE,
						      "activatable",
						      SUBSTANTIVE_ACTIVATABLE,
						      NULL);
    g_object_set_data (G_OBJECT(renderer), "column",
		       GINT_TO_POINTER(SUBSTANTIVE_SA));
    g_signal_connect(G_OBJECT(renderer), "toggled",
		     G_CALLBACK(cb_cclass_code_item_toggled), store);
    gtk_tree_view_column_set_sizing(GTK_TREE_VIEW_COLUMN (column),
				    GTK_TREE_VIEW_COLUMN_FIXED);
    gtk_tree_view_column_set_fixed_width(GTK_TREE_VIEW_COLUMN(column), 80);
    gtk_tree_view_append_column(GTK_TREE_VIEW(treeview_cclass_code), column);

    /* SUBSTANTIVE_NA */
    renderer = gtk_cell_renderer_toggle_new();
    column = gtk_tree_view_column_new_with_attributes(_("Na"),
						      renderer,
						      "active",
						      SUBSTANTIVE_NA,
						      "visible",
						      SUBSTANTIVE_VISIBLE,
						      "activatable",
						      SUBSTANTIVE_ACTIVATABLE,
						      NULL);
    g_object_set_data (G_OBJECT(renderer), "column",
		       GINT_TO_POINTER(SUBSTANTIVE_NA));
    g_signal_connect(G_OBJECT(renderer), "toggled",
		     G_CALLBACK(cb_cclass_code_item_toggled), store);
    gtk_tree_view_column_set_sizing(GTK_TREE_VIEW_COLUMN (column),
				    GTK_TREE_VIEW_COLUMN_FIXED);
    gtk_tree_view_column_set_fixed_width(GTK_TREE_VIEW_COLUMN(column), 80);
    gtk_tree_view_append_column(GTK_TREE_VIEW(treeview_cclass_code), column);

    selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview_cclass_code));
    gtk_tree_selection_set_mode(selection, GTK_SELECTION_SINGLE);

    gtk_list_store_append(store, &iter);
    gtk_list_store_set(store, &iter,
		       SUBSTANTIVE_KAKUJOSHI, FALSE,
		       SUBSTANTIVE_GOKAN_NOMI, FALSE,
		       SUBSTANTIVE_SURU, FALSE,
		       SUBSTANTIVE_SA, FALSE,
		       SUBSTANTIVE_NA, FALSE,
		       SUBSTANTIVE_VISIBLE, TRUE,
		       SUBSTANTIVE_ACTIVATABLE, TRUE,
		       -1);

    result = gtk_dialog_run(GTK_DIALOG(dialog));

    switch(result) {
    case GTK_RESPONSE_ACCEPT:
	flag = 0;
	if(gtk_tree_selection_get_selected(selection, NULL, &iter)) {
	    gtk_tree_model_get(GTK_TREE_MODEL(store), &iter,
			       SUBSTANTIVE_KAKUJOSHI,  &substantive_flag[0],
			       SUBSTANTIVE_GOKAN_NOMI, &substantive_flag[1],
			       SUBSTANTIVE_SURU,       &substantive_flag[2],
			       SUBSTANTIVE_SA,         &substantive_flag[3],
			       SUBSTANTIVE_NA,         &substantive_flag[4],
			       -1);
	    for(i = 4; i >= 0; i--) {
		flag = flag << 1;
		if(substantive_flag[i] == TRUE) {
		    flag = flag | 0x1;
		}
	    }

	    code_num = sizeof(substantive_code) / sizeof(substantive_code[0]);

	    for(i = 0; i < code_num; i++) {
		if(substantive_code[i].flag == flag) {
		    gtk_entry_set_text(GTK_ENTRY(pane->cclass_code),
				       substantive_code[i].code);
		    break;
		}
	    }
	}
	break;
    default:
	break;
    }

    gtk_widget_destroy(dialog);
}

static void cclass_adjective_dialog(CannaAddWordPane *pane) {
    GtkWidget *dialog;
    GtkWidget *scrollwin_cclass_code;
    GtkWidget *treeview_cclass_code;

    GtkTreeViewColumn *column;
    GtkListStore *store;
    GtkTreeSelection *selection;
    GtkCellRenderer *renderer;
    GtkTreeIter iter;
    gint result;
    gchar *cclass_code_utf8 = NULL, *cclass_code_eucjp = NULL;
    int narrow_size, i;

    dialog = gtk_dialog_new_with_buttons(_("Conjection Class"), NULL,
					 GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
					 GTK_STOCK_OK,
					 GTK_RESPONSE_ACCEPT,
					 GTK_STOCK_CANCEL,
					 GTK_RESPONSE_REJECT,
					 NULL);
    gtk_widget_set_size_request(GTK_WIDGET(dialog), 300, 200);

    scrollwin_cclass_code = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollwin_cclass_code),
				   GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox),
		       scrollwin_cclass_code,
		       TRUE, TRUE, 10);
    gtk_widget_show(scrollwin_cclass_code);

    treeview_cclass_code = gtk_tree_view_new();
    gtk_container_add(GTK_CONTAINER(scrollwin_cclass_code),
		      treeview_cclass_code);
    gtk_widget_show(treeview_cclass_code);

    renderer = gtk_cell_renderer_text_new();
    column = gtk_tree_view_column_new_with_attributes("cclass_code",
						      renderer,
						      "text", COLUMN_CCLASS_CODE,
						      NULL);
    gtk_tree_view_append_column(GTK_TREE_VIEW(treeview_cclass_code),column);
    gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(treeview_cclass_code),
				      FALSE);
    selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview_cclass_code));
    gtk_tree_selection_set_mode(selection, GTK_SELECTION_SINGLE);

    store = gtk_list_store_new(N_COLUMNS_CCLASS_CODE, G_TYPE_STRING);
    gtk_tree_view_set_model(GTK_TREE_VIEW(treeview_cclass_code),
			    GTK_TREE_MODEL(store));

    narrow_size = sizeof(adjective_code) / sizeof(adjective_code[0]);
    for(i = 0; i < narrow_size; i++) {
	gtk_list_store_append(store, &iter);
	gtk_list_store_set(store, &iter,
			   COLUMN_CCLASS_CODE,
			   eucjp_to_utf8(adjective_code[i].desc),
			   -1);
    }

    result = gtk_dialog_run(GTK_DIALOG(dialog));

    switch(result) {
    case GTK_RESPONSE_ACCEPT:
	if(gtk_tree_selection_get_selected(selection, NULL, &iter)) {
	    gtk_tree_model_get(GTK_TREE_MODEL(store), &iter,
			       COLUMN_CCLASS_CODE, &cclass_code_utf8,
			       -1);
	    if(cclass_code_utf8 != NULL) {
		cclass_code_eucjp = utf8_to_eucjp(cclass_code_utf8);
		g_free(cclass_code_utf8);
		for(i = 0; i < narrow_size; i++) {
		    if(strcmp(cclass_code_eucjp, adjective_code[i].desc) == 0) {
			gtk_entry_set_text(GTK_ENTRY(pane->cclass_code),
					   adjective_code[i].code);
			break;
		    }
		}
	    }
	}
	if(cclass_code_eucjp != NULL)
	    g_free(cclass_code_eucjp);
	break;
    default:
	break;
    }

    gtk_widget_destroy(dialog);
}

static void cclass_adverb_dialog(CannaAddWordPane *pane) {
    GtkWidget *dialog;
    GtkWidget *scrollwin_cclass_code;
    GtkWidget *treeview_cclass_code;

    GtkTreeViewColumn *column;
    GtkListStore *store;
    GtkTreeSelection *selection;
    GtkCellRenderer *renderer;
    GtkTreeIter iter;
    gint result;
    gboolean adverb_flag[4];
    int flag;
    int i, code_num;

    dialog = gtk_dialog_new_with_buttons(_("Conjection Class"), NULL,
					 GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
					 GTK_STOCK_OK,
					 GTK_RESPONSE_ACCEPT,
					 GTK_STOCK_CANCEL,
					 GTK_RESPONSE_REJECT,
					 NULL);
    gtk_widget_set_size_request(GTK_WIDGET(dialog), 420, 120);

    scrollwin_cclass_code = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollwin_cclass_code),
				   GTK_POLICY_AUTOMATIC, GTK_POLICY_NEVER);

    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox),
		       scrollwin_cclass_code,
		       TRUE, FALSE, 10);
    gtk_widget_show(scrollwin_cclass_code);

    treeview_cclass_code = gtk_tree_view_new();
    gtk_container_add(GTK_CONTAINER(scrollwin_cclass_code),
		      treeview_cclass_code);
    gtk_widget_show(treeview_cclass_code);

    store = gtk_list_store_new(N_ADVERB_COLUMNS,
			       G_TYPE_BOOLEAN,
			       G_TYPE_BOOLEAN,
			       G_TYPE_BOOLEAN,
			       G_TYPE_BOOLEAN,
			       G_TYPE_BOOLEAN,
			       G_TYPE_BOOLEAN);

    gtk_tree_view_set_model(GTK_TREE_VIEW(treeview_cclass_code),
			    GTK_TREE_MODEL(store));

    /* ADVERB_GOKAN_NOMI */
    renderer = gtk_cell_renderer_toggle_new();
    column = gtk_tree_view_column_new_with_attributes(_("Gokan"),
						      renderer,
						      "active",
						      ADVERB_GOKAN_NOMI_DE_BUNSETSU,
						      "visible",
						      ADVERB_VISIBLE,
						      "activatable",
						      ADVERB_ACTIVATABLE,
						      NULL);
    g_object_set_data (G_OBJECT(renderer), "column",
		       GINT_TO_POINTER(ADVERB_GOKAN_NOMI_DE_BUNSETSU));
    g_signal_connect(G_OBJECT(renderer), "toggled",
		     G_CALLBACK(cb_cclass_code_item_toggled), store);
    gtk_tree_view_column_set_sizing(GTK_TREE_VIEW_COLUMN (column),
				    GTK_TREE_VIEW_COLUMN_FIXED);
    gtk_tree_view_column_set_fixed_width(GTK_TREE_VIEW_COLUMN(column), 80);
    gtk_tree_view_append_column(GTK_TREE_VIEW(treeview_cclass_code),column);

    /* ADVERB_SURU */
    renderer = gtk_cell_renderer_toggle_new();
    column = gtk_tree_view_column_new_with_attributes(_("Suru"),
						      renderer,
						      "active",
						      ADVERB_SURU,
						      "visible",
						      ADVERB_VISIBLE,
						      "activatable",
						      ADVERB_ACTIVATABLE,
						      NULL);
    g_object_set_data (G_OBJECT(renderer), "column",
		       GINT_TO_POINTER(ADVERB_SURU));
    g_signal_connect(G_OBJECT(renderer), "toggled",
		     G_CALLBACK(cb_cclass_code_item_toggled), store);
    gtk_tree_view_column_set_sizing(GTK_TREE_VIEW_COLUMN (column),
				    GTK_TREE_VIEW_COLUMN_FIXED);
    gtk_tree_view_column_set_fixed_width(GTK_TREE_VIEW_COLUMN(column), 80);
    gtk_tree_view_append_column(GTK_TREE_VIEW(treeview_cclass_code),column);

    /* ADVERB_TARU */
    renderer = gtk_cell_renderer_toggle_new();
    column = gtk_tree_view_column_new_with_attributes(_("Taru"),
						      renderer,
						      "active",
						      ADVERB_TARU,
						      "visible",
						      ADVERB_VISIBLE,
						      "activatable",
						      ADVERB_ACTIVATABLE,
						      NULL);
    g_object_set_data (G_OBJECT(renderer), "column",
		       GINT_TO_POINTER(ADVERB_TARU));
    g_signal_connect(G_OBJECT(renderer), "toggled",
		     G_CALLBACK(cb_cclass_code_item_toggled), store);
    gtk_tree_view_column_set_sizing(GTK_TREE_VIEW_COLUMN (column),
				    GTK_TREE_VIEW_COLUMN_FIXED);
    gtk_tree_view_column_set_fixed_width(GTK_TREE_VIEW_COLUMN(column), 80);
    gtk_tree_view_append_column(GTK_TREE_VIEW(treeview_cclass_code),column);

    /* ADVERB_TO */
    renderer = gtk_cell_renderer_toggle_new();
    column = gtk_tree_view_column_new_with_attributes(_("To"),
						      renderer,
						      "active",
						      ADVERB_TO,
						      "visible",
						      ADVERB_VISIBLE,
						      "activatable",
						      ADVERB_ACTIVATABLE,
						      NULL);
    g_object_set_data (G_OBJECT(renderer), "column",
		       GINT_TO_POINTER(ADVERB_TO));
    g_signal_connect(G_OBJECT(renderer), "toggled",
		     G_CALLBACK(cb_cclass_code_item_toggled), store);
    gtk_tree_view_column_set_sizing(GTK_TREE_VIEW_COLUMN (column),
				    GTK_TREE_VIEW_COLUMN_FIXED);
    gtk_tree_view_column_set_fixed_width(GTK_TREE_VIEW_COLUMN(column), 80);
    gtk_tree_view_append_column(GTK_TREE_VIEW(treeview_cclass_code), column);

    selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview_cclass_code));
    gtk_tree_selection_set_mode(selection, GTK_SELECTION_SINGLE);

    gtk_list_store_append(store, &iter);
    gtk_list_store_set(store, &iter,
		       ADVERB_GOKAN_NOMI_DE_BUNSETSU, FALSE,
		       ADVERB_SURU, FALSE,
		       ADVERB_TARU, FALSE,
		       ADVERB_TO, FALSE,
		       ADVERB_VISIBLE, TRUE,
		       ADVERB_ACTIVATABLE, TRUE,
		       -1);

    result = gtk_dialog_run(GTK_DIALOG(dialog));

    switch(result) {
    case GTK_RESPONSE_ACCEPT:
	flag = 0;
	if(gtk_tree_selection_get_selected(selection, NULL, &iter)) {
	    gtk_tree_model_get(GTK_TREE_MODEL(store), &iter,
			       ADVERB_GOKAN_NOMI_DE_BUNSETSU, &adverb_flag[0],
			       ADVERB_SURU,       &adverb_flag[1],
			       ADVERB_TARU,       &adverb_flag[2],
			       ADVERB_TO,         &adverb_flag[3],
			       -1);
	    for(i = 3; i >= 0; i--) {
		flag = flag << 1;
		if(adverb_flag[i] == TRUE) {
		    flag = flag | 0x1;
		}
	    }

	    code_num = sizeof(adverb_code) / sizeof(adverb_code[0]);

	    for(i = 0; i < code_num; i++) {
		if(adverb_code[i].flag == flag) {
		    gtk_entry_set_text(GTK_ENTRY(pane->cclass_code),
				       adverb_code[i].code);
		    break;
		}
	    }
	}
	break;
    default:
	break;
    }
    gtk_widget_destroy(dialog);
}

static void cclass_etc_dialog(CannaAddWordPane *pane) {
    GtkWidget *dialog;
    GtkWidget *scrollwin_cclass_code;
    GtkWidget *treeview_cclass_code;

    GtkTreeViewColumn *column;
    GtkListStore *store;
    GtkTreeSelection *selection;
    GtkCellRenderer *renderer;
    GtkTreeIter iter;
    gint result;
    gchar *cclass_code_utf8 = NULL, *cclass_code_eucjp = NULL;
    int narrow_size, i;

    dialog = gtk_dialog_new_with_buttons(_("Conjection Class"), NULL,
					 GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
					 GTK_STOCK_OK,
					 GTK_RESPONSE_ACCEPT,
					 GTK_STOCK_CANCEL,
					 GTK_RESPONSE_REJECT,
					 NULL);
    gtk_widget_set_size_request(GTK_WIDGET(dialog), 300, 200);

    scrollwin_cclass_code = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollwin_cclass_code),
				   GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox),
		       scrollwin_cclass_code,
		       TRUE, TRUE, 10);
    gtk_widget_show(scrollwin_cclass_code);

    treeview_cclass_code = gtk_tree_view_new();
    gtk_container_add(GTK_CONTAINER(scrollwin_cclass_code),
		      treeview_cclass_code);
    gtk_widget_show(treeview_cclass_code);

    renderer = gtk_cell_renderer_text_new();
    column = gtk_tree_view_column_new_with_attributes("cclass_code",
						      renderer,
						      "text", COLUMN_CCLASS_CODE,
						      NULL);
    gtk_tree_view_append_column(GTK_TREE_VIEW(treeview_cclass_code),column);
    gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(treeview_cclass_code),
				      FALSE);
    selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview_cclass_code));
    gtk_tree_selection_set_mode(selection, GTK_SELECTION_SINGLE);

    store = gtk_list_store_new(N_COLUMNS_CCLASS_CODE, G_TYPE_STRING);
    gtk_tree_view_set_model(GTK_TREE_VIEW(treeview_cclass_code),
			    GTK_TREE_MODEL(store));

    narrow_size = sizeof(etc_code) / sizeof(etc_code[0]);
    for(i = 0; i < narrow_size; i++) {
	gtk_list_store_append(store, &iter);
	gtk_list_store_set(store, &iter,
			   COLUMN_CCLASS_CODE,
			   eucjp_to_utf8(etc_code[i].desc),
			   -1);
    }

    result = gtk_dialog_run(GTK_DIALOG(dialog));

    switch(result) {
    case GTK_RESPONSE_ACCEPT:
	if(gtk_tree_selection_get_selected(selection, NULL, &iter)) {
	    gtk_tree_model_get(GTK_TREE_MODEL(store), &iter,
			       COLUMN_CCLASS_CODE, &cclass_code_utf8,
			       -1);
	    if(cclass_code_utf8 != NULL) {
		cclass_code_eucjp = utf8_to_eucjp(cclass_code_utf8);
		g_free(cclass_code_utf8);
		for(i = 0; i < narrow_size; i++) {
		    if(strcmp(cclass_code_eucjp, etc_code[i].desc) == 0) {
			gtk_entry_set_text(GTK_ENTRY(pane->cclass_code),
					   etc_code[i].code);
			break;
		    }
		}
	    }
	}
	if(cclass_code_eucjp != NULL)
	    g_free(cclass_code_eucjp);
	break;
    default:
	break;
    }

    gtk_widget_destroy(dialog);
}

static void cb_button_cclass_browse_clicked(GtkButton *button,
					    CannaAddWordPane *pane)
{
    void (*cclass_dialog[])(CannaAddWordPane*) = {
        cclass_verb_dialog,
	cclass_substantive_dialog,
	cclass_adjective_dialog,
	cclass_adverb_dialog,
	cclass_etc_dialog,
    };
    gint id;
    id = gtk_option_menu_get_history(GTK_OPTION_MENU(pane->optmenu_cclass_code));
    (cclass_dialog[id])(pane);
}

static void cb_cclass_code_item_toggled(GtkCellRendererToggle *cell,
					gchar *path_str,
					gpointer data)
{
    GtkTreeModel *model = GTK_TREE_MODEL(data);
    GtkTreePath *path = gtk_tree_path_new_from_string(path_str);
    GtkTreeIter iter;
    gboolean toggle_item;

    gint *column;
    column = g_object_get_data (G_OBJECT (cell), "column");

    gtk_tree_model_get_iter (model, &iter, path);
    gtk_tree_model_get(model, &iter, column, &toggle_item, -1);

    toggle_item ^= 1;
    gtk_list_store_set (GTK_LIST_STORE(model), &iter, column,
			toggle_item, -1);
    gtk_tree_path_free (path);
}

static void cb_add_button_clicked (GtkButton *button,
				   CannaAddWordPane *pane)
{
    const gchar *utf8_phon, *utf8_desc, *utf8_cclass_code;
    gchar *utf8_word_line;
    gint word_len;
    unsigned char *eucjp_word_line;
    int ret;

    utf8_phon = gtk_entry_get_text(GTK_ENTRY(pane->phon));
    utf8_desc = gtk_entry_get_text(GTK_ENTRY(pane->desc));
    utf8_cclass_code = gtk_entry_get_text(GTK_ENTRY(pane->cclass_code));

    word_len = strlen(utf8_phon) + strlen(utf8_desc) +
	       strlen(utf8_cclass_code) + 2 + 1;
    utf8_word_line = g_malloc(sizeof(gchar) * word_len);
    if(utf8_word_line != NULL) {
	g_snprintf(utf8_word_line, word_len, "%s %s %s",
		   utf8_phon, utf8_cclass_code, utf8_desc);

	eucjp_word_line = utf8_to_eucjp(utf8_word_line);
	if(eucjp_word_line != NULL) {
	    ret = canna_define_dic("user", eucjp_word_line);
	    g_free(eucjp_word_line);
	}
	g_free(utf8_word_line);
    }

    gtk_entry_set_text(GTK_ENTRY(pane->phon), "");
    gtk_entry_set_text(GTK_ENTRY(pane->desc), "");
    gtk_entry_set_text(GTK_ENTRY(pane->cclass_code), "");
    gtk_option_menu_set_history(GTK_OPTION_MENU(pane->optmenu_cclass_code), 0);
}

static void cb_clear_button_clicked (GtkButton *button,
				     CannaAddWordPane *pane)
{
    gtk_entry_set_text(GTK_ENTRY(pane->phon), "");
    gtk_entry_set_text(GTK_ENTRY(pane->desc), "");
    gtk_entry_set_text(GTK_ENTRY(pane->cclass_code), "");
    gtk_option_menu_set_history(GTK_OPTION_MENU(pane->optmenu_cclass_code), 0);
}
