<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_AdmActionForm.php,v 1.1 2009/01/11 05:34:31 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 * フォームクラス
 */
require_once SYL_FRAMEWORK_DIR . '/core/SyL_ActionForm.php';

/**
 * ADM対応アクションフォームクラス
 *
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_AdmActionForm.php,v 1.1 2009/01/11 05:34:31 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_AdmActionForm extends SyL_ActionForm
{
    /**
     * 機能タイプ
     * 
     * @access protected
     * @var string
     */
    var $type = '';
    /**
     * 構成定義設定配列
     * 
     * @access protected
     * @var array
     */
    var $structs_config = array();
    /**
     * SyL_Formクラス外プロパティ配列
     * 
     * @access protected
     * @var array
     */
    var $elements_config_etc = array();
    /**
     * テーブルクラスの配列
     * 
     * @access protected
     * @var array
     */
    var $table_classes = array();
    /**
     * テーブルクラスの関連配列
     * 
     * @access protected
     * @var array
     */
    var $table_relations = array();
    /**
     * メインメンテナンステーブル名
     * 
     * @access protected
     * @var string
     */
    var $maintenance_table = '';
    /**
     * 関連リンクURL
     * 
     * @access protected
     * @var array
     */
    var $related_links = array();
    /**
     * 検索項目名の接頭辞
     * 
     * @access private
     * @var string
     */
    var $search_item_prefix = 'cl.';

    /**
     * 機能タイプを取得する
     *
     * @access public
     * @return string 機能タイプ
     */
    function setType($type)
    {
        switch (strtolower($type)) {
        case 'lst': $this->type = 'lst'; break; // 一覧
        case 'new': $this->type = 'new'; break; // 新規
        case 'upd': $this->type = 'upd'; break; // 更新
        case 'del': $this->type = 'del'; break; // 削除
        case 'vew': $this->type = 'vew'; break; // 詳細表示
        case 'fin': $this->type = 'fin'; break; // 完了表示
        default: SyL_Error::trigger("[Adm error] Invalid type ({$type})");
        }
    }

    /**
     * 機能タイプを取得する
     *
     * @access public
     * @return string 機能タイプ
     */
    function getType()
    {
        return $this->type;
    }

    /**
     * テーブルクラスのリストを取得する
     *
     * @access public
     * @return array テーブルクラスのリスト
     */
    function getTableClasses()
    {
        return $this->table_classes;
    }

    /**
     * テーブルの関連を取得する
     *
     * @access public
     * @return array テーブルの関連
     */
    function getTableRelations()
    {
        return $this->table_relations;
    }

    /**
     * 構成設定値を取得する
     *
     * @access public
     * @return mixed 構成設定値
     */
    function getStructConfig($name)
    {
        return isset($this->structs_config[$name]) ? $this->structs_config[$name] : null;
    }

    /**
     * 関連リンクURLを取得する
     *
     * @access public
     * @return array 関連リンクURL
     */
    function getRelatedLinks()
    {
        return $this->related_links;
    }

    /**
     * メインメンテナンステーブル名を取得する
     *
     * @access public
     * @return string メインメンテナンステーブル名
     */
    function getMaintenanceTable()
    {
        return $this->maintenance_table;
    }

    /**
     * 表示のみ要素に変更する
     *
     * @access public
     * @return string 要素名
     */
    function setReadOnlyColumn($column)
    {
        if (isset($this->elements_config[$column])) {
            switch ($this->type) {
            case 'new': $this->elements_config[$column]['read_only_new'] = true; break;
            case 'upd': $this->elements_config[$column]['read_only_upd'] = true; break;
            }
        }
    }

    /**
     * アクションフォームを構築する
     *
     * @access public
     * @param bool 初期表示フラグ
     */
    function build($init=false)
    {
        $i = 1;
        $elements = array();
        // 要素登録ループ
        foreach ($this->elements_config as $name => $values) {
            if (isset($values['display_detail']) && !$values['display_detail']) {
                // 表示フラグが非表示の場合は取得しない
                continue;
            }
            // 新規／更新フォーム以外は、要素メモを非表示
            switch ($this->type) {
            case 'new':
                if (isset($this->elements_config[$name]['note_new']) && $this->elements_config[$name]['note_new']) {
                    $this->elements_config[$name]['note'] = $this->elements_config[$name]['note_new'];
                }
                break;
            case 'upd':
                if (isset($this->elements_config[$name]['note_upd']) && $this->elements_config[$name]['note_upd']) {
                    $this->elements_config[$name]['note'] = $this->elements_config[$name]['note_upd'];
                }
                break;
            default:
                $this->elements_config[$name]['note'] = '';
            }

            // 要素作成
            $element =& $this->createElement($name, $values);
            // 初期値判定
            if ($init && isset($values['default']) && ($this->type == 'new')) {
                $element->setValue($values['default']);
            }
            if (isset($values['validate']) && (count($values['validate']) > 0)) {
                if ((($this->type == 'new') && isset($values['read_only_new']) && $values['read_only_new']) ||
                    (($this->type == 'upd') && isset($values['read_only_upd']) && $values['read_only_upd'])) {
                    // 読み込み専用なので、バリデーションを追加しない
                    // かつ読み込み専用フラグ追加
                    $element->setReadOnly(true);
                } else {
                    // バリデーション追加
                    $this->setElementValidations($element, $values);
                }
            }

            $this->elements_config_etc[$name]['cols'] = isset($values['cols']) ? $values['cols'] : '1';
            $this->elements_config_etc[$name]['read_only_new'] = isset($values['read_only_new']) ? $values['read_only_new'] : false;
            $this->elements_config_etc[$name]['read_only_upd'] = isset($values['read_only_upd']) ? $values['read_only_upd'] : false;
            $this->elements_config_etc[$name]['default_new']   = isset($values['default_new'])   ? $values['default_new']   : false;
            $this->elements_config_etc[$name]['default_upd']   = isset($values['default_upd'])   ? $values['default_upd']   : false;
            $this->elements_config_etc[$name]['display'] = true; // 検索兼用のダミー

            // ソート順
            $sort = (isset($values['sort_detail']) && $values['sort_detail']) ? $values['sort_detail'] : $i++;
            // 並び順変更のため一時保存
            $elements[$sort] =& $element;
        }

        ksort($elements, SORT_NUMERIC);
        foreach (array_keys($elements) as $name) {
            $this->addElement($elements[$name]);
        }
    }

    /**
     * アクションフォームパラメータセット
     *
     * @access public
     * @param string 要素名
     * @param array アクションフォームパラメータ
     */
    function buildSearch()
    {
        if ($this->getStructConfig('enable_sch')) {
            // 要素登録ループ
            foreach ($this->elements_config as $name => $values) {
                // 要素作成
                $element =& $this->createElement($name, $values);
                $this->addElement($element);

                $this->elements_config_etc[$name]['cols']    = '1'; // 検索は強制的に2列に
                $this->elements_config_etc[$name]['display'] = (isset($values['search']) && $values['search']);
                // 要素メモを非表示
                if (isset($this->elements_config[$name]['note'])) {
                    $this->elements_config[$name]['note'] = '';
                }
            }
        }
    }

    /**
     * 表示カラムのリストを取得する
     *
     * @access public
     * @param string テーブル別名
     * @return array 一覧表示カラムのリスト
     */
    function getListColumns($alias='')
    {
        $i = 1;
        $lists = array();
        // 要素一覧表示ループ
        foreach ($this->elements_config as $name => $values) {
            $sort = 0;
            // 非表示判定
            if ($alias && isset($values['alias']) && ($values['alias'] != $alias)) {
                // 別名が一致しない場合は取得しない
                continue;
            } else {
                switch ($this->type) {
                case 'lst':
                    if (isset($values['display_list']) && !$values['display_list']) {
                        // 一覧表示で、かつ表示フラグが非表示の場合は取得しない
                        continue 2;
                    }
                    $sort = (isset($values['sort_list']) && $values['sort_list']) ? $values['sort_list'] : $i++;
                    break;
                case 'new':
                case 'upd':
                case 'vew':
                    if (isset($values['display_detail']) && !$values['display_detail']) {
                        // 詳細表示で、かつ表示フラグが非表示の場合は取得しない
                        continue 2;
                    }
                    $sort = (isset($values['sort_detail']) && $values['sort_detail']) ? $values['sort_detail'] : $i++;
                    break;
                }
            }
            $lists[$sort] = $name;
        }
        ksort($lists, SORT_NUMERIC);
        return $lists;
    }

    /**
     * 全ての要素の値を取得
     *
     * @access public
     * @param string 検索別名
     * @return array 全ての要素の値
     */
    function getValues($alias='')
    {
        $values = array();
        foreach (parent::getValues() as $name => $value) {
            if ((($this->type == 'new') && $this->elements_config_etc[$name]['read_only_new']) ||
                (($this->type == 'upd') && $this->elements_config_etc[$name]['read_only_upd'])) {
                // 読み込み専用なので、登録／更新項目に含めない
                continue;
            }
            // 別名フィルタリング
            if ($alias && isset($this->elements_config[$name]['alias']) && ($this->elements_config[$name]['alias'] != $alias)) {
                continue;
            }
            $values[$name] = $value;
        }
        return $values;
    }

    /**
     * フォーム全要素を配列で取得する
     *
     * @access public
     * @return array フォーム全要素を配列
     */
    function &getResultArray()
    {
        // 親クラスの結果を取得
        $results =& parent::getResultArray();

        $cols        = '';
        $end_space   = false;
        $dummy_index = 1;
        $elements    = array();
        // 空白用の要素
        $element_space = array(
          'label'     => '&nbsp;',
          'html'      => '&nbsp;',
          'require'   => false,
          'read_only' => true,
          'cols'      => '1'
        );

        // 表示フィルタリング
        foreach (array_keys($results['elements']) as $name) {
            if (!$this->elements_config_etc[$name]['display']) {
                unset($results['elements'][$name]);
            }
        }

        // 表示位置変更
        foreach (array_keys($results['elements']) as $name) {
            $end_space = false;
            switch ($cols) {
            case '':
                // 初期要素
                $elements[$name] = $results['elements'][$name];
                $elements[$name]['cols'] = ($this->elements_config_etc[$name]['cols'] == '1') ? '0' : '2';
                if (count($results['elements']) == 1) {
                    $end_space = true;
                }
                break;
            case '0':
                if ($this->elements_config_etc[$name]['cols'] == '1') {
                    // 2列の右の要素
                    $elements[$name] = $results['elements'][$name];
                    $elements[$name]['cols'] = '1';
                } else if ($this->elements_config_etc[$name]['cols'] == '2') {
                    // 2列の右で空の要素
                    $elements['__dummy_' . $dummy_index++] = $element_space;
                    // 1列の要素
                    $elements[$name] = $results['elements'][$name];
                    $elements[$name]['cols'] = '2';
                }
                break;
            case '1':
                $elements[$name] = $results['elements'][$name];
                if ($this->elements_config_etc[$name]['cols'] == '1') {
                    // 2列の左の要素
                    $elements[$name]['cols'] = '0';
                    $end_space = true;
                } else if ($this->elements_config_etc[$name]['cols'] == '2') {
                    // 1列の要素
                    $elements[$name]['cols'] = '2';
                }
                break;
            case '2':
                $elements[$name] = $results['elements'][$name];
                if ($this->elements_config_etc[$name]['cols'] == '1') {
                    // 2列の左の要素
                    $elements[$name]['cols'] = '0';
                    $end_space = true;
                } else if ($this->elements_config_etc[$name]['cols'] == '2') {
                    // 1列の要素
                    $elements[$name]['cols'] = '2';
                }
                break;
            }
            $cols = $elements[$name]['cols'];
        }

        if ($end_space) {
            // 空の表示項目用
            $elements['__dummy_' . $dummy_index++] = $element_space;
        }

        // 要素再作成
        $results['elements'] = $elements;

        return $results;
    }

    /**
     * データ元を取得する
     *
     * @access public
     * @return array データ元
     */
    function getDataSources()
    {
        $result = array();
        foreach ($this->getListColumns() as $name) {
            if (isset($this->elements_config[$name]['data_source']) && $this->elements_config[$name]['data_source']) {
                $result[$name] = $this->elements_config[$name]['data_source'];
            }
        }
        return $result;
    }

    /**
     * optionsデータを取得する
     *
     * @access public
     * @return array データ元
     */
    function getOption($name)
    {
        if (isset($this->elements_config[$name]['options'])) {
            $option = $this->elements_config[$name]['options'];
            // 未選択は削除
            if (isset($option[''])) {
                unset($option['']);
            }
            return (count($option) > 0) ? $option : null;
        } else {
            return null;
        }
    }

    /**
     * デフォルト値を取得する
     *
     * @access public
     * @return array デフォルト値
     */
    function getDefaults()
    {
        $default_name = '';
        switch ($this->type) {
        case 'new': $default_name = 'default_new'; break;
        case 'upd': $default_name = 'default_upd'; break;
        default:    return array();
        }

        $defaults = array();
        foreach ($this->elements_config_etc as $name => $values) {
            if (isset($values[$default_name]) && ($values[$default_name] !== false)) {
                $defaults[$name] = $values[$default_name];
            }
        }
        return $defaults;
    }
}
