<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_RequestSessionDb.php,v 1.1 2009/01/11 05:34:32 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 * セッション保持クラス
 */
require_once 'SyL_RequestSession.php';
/**
 * DBクラス
 */
require_once SYL_FRAMEWORK_DIR . '/lib/SyL_DB.php';

/**
 * DBセッション保持クラス
 *
 * 例）
 * CREATE TABLE SYL_SESSIONS ( 
 *   session_id varchar(32) NOT NULL, 
 *   session_expires int NOT NULL, 
 *   session_data text,
 *   PRIMARY KEY  (session_id)  
 * ); 
 * のようなテーブルを想定
 *
 * SYL_SESSION_DB_TABLE = 'SYL_SESSIONS:session_id:session_data:session_expires'
 *
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_RequestSessionDb.php,v 1.1 2009/01/11 05:34:32 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_RequestSessionDb extends SyL_RequestSession
{
    /**
     * セッション保持時間
     *
     * @access private
     * @var int
     */
    var $life_time;
    /**
     * セッションテーブル名
     *
     * @access private
     * @var string
     */
    var $table;
    /**
     * セッションIDカラム名
     *
     * @access private
     * @var string
     */
    var $id;
    /**
     * セッションデータカラム名
     *
     * @access private
     * @var string
     */
    var $data;
    /**
     * セッションタイムスタンプカラム名
     *
     * @access private
     * @var string
     */
    var $timestamp;

    /**
     * DBコネクション
     *
     * @access private
     * @var object
     */
    var $conn = null; 

    /**
     * コンストラクタ
     *
     * @access public
     */
    function SyL_RequestSessionDb()
    {
        if (!defined('SYL_SESSION_DB_TABLE')) {
            trigger_error("[SyL error] Undefined constant `SYL_SESSION_DB_TABLE'", E_USER_ERROR);
        }
        // テーブルプロパティ取得
        list($this->table, $this->id, $this->data, $this->timestamp) = explode(':', SYL_SESSION_DB_TABLE);

        ini_set('session.save_handler', 'user');
        session_set_save_handler(
          array(&$this, 'open'),
          array(&$this, 'close'),
          array(&$this, 'read'),
          array(&$this, 'write'),
          array(&$this, 'destroy'),
          array(&$this, 'gc')
        );

        // セッション開始
        parent::SyL_RequestSession();
    }

    /**
     * セッション保存パスを設定
     *
     * @access protected
     */
    function setSavePath()
    {
        // オーバーライドで無効化
    }

    /**
     * セッションオープンイベント
     *
     * @access private
     * @param string セッション保存パス
     * @param string セッション名
     * @return bool true: OK, false: NG
     */
    function open($save_path, $session_name)
    {
        $this->life_time = ini_get('session.gc_maxlifetime');
        if (SYL_SESSION_SAVE_PATH) {
            $this->conn =& SyL_DB::getConnection(SYL_SESSION_SAVE_PATH);
        } else {
            $this->conn =& SyL_DB::getConnection(SYL_DB_DSN);
        }
        return is_object($this->conn);
    }

    /**
     * セッション読み込みイベント
     *
     * @access private
     * @param string セッションID
     * @return string セッションデータ
     */
    function read($session_id)
    {
        $sql  = "";
        $sql .= "SELECT ";
        $sql .=   "{$this->data} ";
        $sql .= "FROM ";
        $sql .=   "{$this->table} ";
        $sql .= "WHERE ";
        $sql .=   "{$this->id} = '$session_id' and ";
        $sql .=   "{$this->timestamp} >= " . time() . " ";

        $this->conn->queryRef($sql, $result, 'one');

        if (!$result) { 
            $sql  = "";
            $sql .= "INSERT INTO {$this->table} (";
            $sql .=   "{$this->id}, ";
            $sql .=   "{$this->timestamp}, ";
            $sql .=   "{$this->data} ";
            $sql .= ") VALUES ( ";
            $sql .=   "'{$session_id}', ";
            $sql .=   " " . (time() + $this->life_time) . ", ";
            $sql .=   "NULL ";
            $sql .= ") ";
            $this->conn->execRef($sql, $result);
            $result = null;
        }
        return $result;
    }

    /**
     * セッション書き込みイベント
     *
     * @access private
     * @param string セッションID
     * @param string セッションデータ
     * @return bool true: OK, false: NG
     */
    function write($session_id, $session_data)
    {
        $sql  = "";
        $sql .= "UPDATE ";
        $sql .=   "{$this->table} ";
        $sql .= "SET ";
        $sql .=   "{$this->timestamp} = " . (time() + $this->life_time) . ", ";
        $sql .=   "{$this->data} = '" . $this->conn->escape($session_data) . "' ";
        $sql .= "WHERE ";
        $sql .=   "{$this->id} = '{$session_id}' ";
        return $this->conn->execRef($sql, $result);
    }

    /**
     * セッションクローズイベント
     *
     * @access private
     * @return bool true
     */
    function close()
    {
        return true;
    }

    /**
     * セッション削除イベント
     *
     * @access private
     * @param string セッションID
     * @return bool true: OK, false: NG
     */
    function destroy($session_id)
    {
        $sql  = "";
        $sql .= "DELETE FROM  ";
        $sql .=   "{$this->table} ";
        $sql .= "WHERE ";
        $sql .=   "{$this->id} = '$session_id'";
        return $this->conn->execRef($sql, $result);
    }

    /**
     * 古いセッション情報の削除イベント
     *
     * @access private
     * @param int セッション保持時間
     * @return bool true: OK, false: NG
     */
    function gc($life_time)
    {
        $sql  = "";
        $sql .= "DELETE FROM  ";
        $sql .=   "{$this->table} ";
        $sql .= "WHERE ";
        $sql .=   "{$this->timestamp} < " . time() . " ";
        return $this->conn->execRef($sql, $result);
    }
}
