<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_Router.php,v 1.1 2009/01/11 05:34:30 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 * 遷移情報クラス
 *
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_Router.php,v 1.1 2009/01/11 05:34:30 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_Router
{
    /**
     * デフォルトアクションファイル名
     * ※先頭大文字
     *
     * @access protected
     * @var string
     */
    var $action_file_default = 'Index.php';
    /**
     * 実行ファイルの拡張子
     *
     * @access protected
     * @var string
     */
    var $exe_ext = '.php';
    /**
     * アクションディレクトリルート
     *
     * @access private
     * @var string
     */
    var $action_dir = '';
    /**
     * テンプレートディレクトリルート
     *
     * @access private
     * @var string
     */
    var $template_dir = '';
    /**
     * アクションファイル
     *
     * @access protected
     * @var string
     */
    var $action_file = '';
    /**
     * アクションファイル（オリジナル）
     *
     * @access protected
     * @var string
     */
    var $action_file_org = '';
    /**
     * テンプレートファイル
     *
     * @access protected
     * @var string
     */
    var $template_file = '';
    /**
     * アクションメソッド名
     *
     * @access protected
     * @var string
     */
    var $action_method = '';
    /**
     * ビュータイプ
     *
     * @access protected
     * @var string
     */
    var $view_type = '';
    /**
     * レイアウト名
     *
     * @access protected
     * @var string
     */
    var $layout_name = '';

    /**
     * コンストラクタ
     *
     * @access public
     * @param string アクション値
     */
    function SyL_Router()
    {
        // アクションディレクトリ定義
        $this->action_dir = SYL_APP_DIR . '/actions/';
        // テンプレートディレクトリ定義
        $this->template_dir = SYL_APP_DIR . '/templates/';
    }

    /**
     * 遷移情報オブジェクトを取得
     *
     * @static
     * @access public
     * @param object データ管理オブジェクト
     * @return object 遷移情報オブジェクト
     */
    function &factory(&$data)
    {
        $classname = 'SyL_Router' . ucfirst(SYL_ROUTER_RULE);
        include_once SYL_FRAMEWORK_DIR . "/core/Router/{$classname}.php";
        $router = new $classname();
        // アクションファイルパスチェック
        if (!$router->validAction()) {
            SyL_Response::forwardFileNotFound();
        }
        $router->createActionInfo($data);
        $router->changeRoutingConfig();
        return $router;
    }

    /**
     * 遷移情報オブジェクトのプロパティを作成
     *
     * $this->action_method
     * $this->action_file
     * $this->action_class
     *
     * @access abstract
     * @param object データ管理オブジェクト
     */
    function createActionInfo(&$data)
    {
    }

    /**
     * アクションディレクトリルートを取得
     *
     * @access public
     * @return string アクションディレクトリルート
     */
    function getActionDir()
    {
        return $this->action_dir;
    }

    /**
     * テンプレートディレクトリルートを取得
     *
     * @access public
     * @return string テンプレートディレクトリルート
     */
    function getTemplateDir()
    {
        return $this->template_dir;
    }

    /**
     * アクションファイルを取得
     *
     * @access public
     * @return string アクションファイル
     */
    function getActionFile()
    {
        return $this->action_file;
    }

    /**
     * テンプレートファイルをセット
     *
     * @access public
     * @param string テンプレートファイル
     */
    function setTemplateFile($template_file)
    {
        if ($template_file[0] == '/') {
            $template_file = substr($template_file, 1);
        }
        $this->template_file = $template_file;
    }

    /**
     * テンプレートファイルを取得
     *
     * @access public
     * @param bool 404エラー起動フラグ
     * @return string テンプレートファイル
     */
    function getTemplateFile($trigger_error_404=true)
    {
        // テンプレートファイル存在チェック
        if ($trigger_error_404 && !is_file($this->template_dir . $this->template_file)) {
            SyL_Loggers::warn("Template file not found dir: {$this->template_dir} file: {$this->template_file}");
            SyL_Response::forwardFileNotFound();
        }
        return $this->template_file;
    }

    /**
     * クラス名を取得
     *
     * @access public
     * @return string クラス名
     */
    function getClassName()
    {
        $classname = str_replace('/', '_', $this->action_file);
        return preg_replace('/(' . preg_quote($this->exe_ext) . ')$/', '', $classname);
    }

    /**
     * アクションメソッド名を取得
     *
     * @access public
     * @return string アクション名
     */
    function getActionName()
    {
        return $this->action_method;
    }

    /**
     * ビュータイプをセット
     *
     * @access public
     * @param string ビュータイプ
     */
    function setViewType($view_type)
    {
        $this->view_type = $view_type;
    }

    /**
     * ビュータイプを取得
     *
     * @access public
     * @return string ビュータイプ
     */
    function getViewType()
    {
        return $this->view_type;
    }

    /**
     * レイアウト名を取得
     *
     * @access public
     * @return string レイアウト名
     */
    function getLayoutName()
    {
        return $this->layout_name;
    }

    /**
     * アクセスパスを階層別に取得
     *
     * @access public
     * @return array 各階層名の配列
     */
    function getUrlNames()
    {
        $url_names = array();
        foreach (explode('/', dirname($this->action_file_org)) as $url_name) {
            switch ($url_name) {
            case '':
            case '.':
            case '\\':
                break;
            default:
                $url_names[] = $url_name;
            }
        }
        return $url_names;
    }

    /**
     * ルーティング設定適用
     *
     * @access public
     */
    function changeRoutingConfig()
    {
        $config =& SyL_Config::factory('routers');
        $config->setRouter($this);
        $config->parseXml();
        $router = $config->getConfig();
        if ($router) {
            if ($router['enableAction'] && $router['forwardAction']) {
                if (preg_match('/^\/(.+)$/', $router['forwardAction'], $matches)) {
                    $this->action_file = $matches[1];
                } else {
                    $this->action_file = $router['forwardAction'];
                }
            }
            if ($router['enableTemplate'] && $router['forwardTemplate']) {
                $this->setTemplateFile($router['forwardTemplate']);
            }
            if ($router['viewType'])   $this->view_type   = $router['viewType'];
            if ($router['layoutName']) $this->layout_name = $router['layoutName'];
        }
    }

    /**
     * アクションファイルのパスチェック
     * ※ディレクトリ／ファイルの先頭「_」はアクセス不可
     *
     * @access public
     * @return bool true: 有効、false: 無効
     */
    function validAction()
    {
        foreach (explode('/', $this->action_file) as $tmp) {
            if ($tmp && ($tmp[0] == '_')) {
                return false;
            }
        }
        return true;
    }
}
