<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_CompressBzip2.php,v 1.1 2009/01/11 05:34:34 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 * 圧縮／解凍クラス（bzip2）
 *
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_CompressBzip2.php,v 1.1 2009/01/11 05:34:34 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_CompressBzip2 extends SyL_Compress
{
    /**
     * ブロックサイズ
     *
     * @access private
     * @var int
     */
     var $blocksize = 9;

    /**
     * コンストラクタ
     *
     * @access public
     */
    function SyL_CompressBzip2()
    {
        parent::SyL_Compress();
        if (!extension_loaded('bz2')) {
            trigger_error("[SyL error] `bzip2' extension not loaded", E_USER_ERROR);
        }
    }

    /**
     * データを圧縮する
     *
     * @static
     * @access public
     * @param string 圧縮前データ
     * @param string 保存ファイル
     * @return string 圧縮後データ
     */
    function compress($data, $out_file='')
    {
        $classname = __CLASS__;
        $obj =& new $classname();
        $data = $obj->processCompress($data);
        if ($out_file !== '') {
            $obj->writeFile($out_file, $data);
        }
        return $data;
    }

    /**
     * ファイルを圧縮する
     *
     * @static
     * @access public
     * @param string 圧縮対象ファイル
     * @param string 保存ファイル
     * @param bool 圧縮対象ファイル削除フラグ
     * @return string 圧縮後データ
     */
    function compressFromFile($in_file, $out_file='', $in_file_remove=false)
    {
        $classname = __CLASS__;
        $obj =& new $classname();
        $data = $obj->readFile($in_file);
        if ($data) {
            $data = call_user_func(array($classname, 'compress'), $data, $out_file);
            if ($in_file_remove && is_file($in_file) && is_file($out_file)) {
                if (filesize($out_file) == strlen($data)) {
                    unlink($in_file);
                }
            }
        }
    }

    /**
     * データを解凍する
     *
     * @static
     * @access public
     * @param string 圧縮データ
     * @param string 保存ファイル
     * @return string 解凍データ
     */
    function extract($data, $out_file='')
    {
        $classname = __CLASS__;
        $obj =& new $classname();
        $data = $obj->processExtract($data);
        if ($out_file !== '') {
            $obj->writeFile($out_file, $data);
        }
        return $data;
    }

    /**
     * ファイルを解凍する
     *
     * @static
     * @access public
     * @param string 解凍対象ファイル
     * @param string 保存ファイル
     * @param bool 解凍対象ファイル削除フラグ
     * @return string 解凍データ
     */
    function extractFromFile($in_file, $out_file='', $in_file_remove=false)
    {
        $classname = __CLASS__;
        $obj =& new $classname();
        $data = $obj->readFile($in_file, 'compress.bzip2');
        if ($out_file !== '') {
            $obj->writeFile($out_file, $data);
        }
        if ($in_file_remove && is_file($in_file) && is_file($out_file)) {
            if (filesize($out_file) == strlen($data)) {
                unlink($in_file);
            }
        }
        return $data;
    }

    /**
     * データを圧縮するロジック
     *
     * @access protected
     * @param string 圧縮前データ
     * @return string 圧縮後データ
     */
    function processCompress($data)
    {
        return bzcompress($data, $this->blocksize);
    }

    /**
     * データを解凍するロジック
     *
     * @access protected
     * @param string 圧縮データ
     * @return string 解凍データ
     */
    function processExtract($data)
    {
        return bzdecompress($data);
    }
}
