<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_DBDriverIbase.php,v 1.1 2009/01/11 05:34:32 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 *  DB操作クラス（Intrebase, Firebird）
 *
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_DBDriverIbase.php,v 1.1 2009/01/11 05:34:32 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_DBDriverIbase extends SyL_DB
{
    /**
     * エンコードフラグ
     *
     * @access private
     * @var bool
     */
    var $client_encode_flag = false;

    /**
     * エンコード変換テーブル
     *
     *  Firebird <=> PHP
     *  (client)     (client)
     * 
     * ※全て小文字で記述
     *
     * @access protected
     * @var string
     */
    var $encode_table = array(
      // PHP, DB
      array('eucjp-win', 'eucj_0208'),
      array('euc-jp',    'eucj_0208'),
      array('euc',       'eucj_0208'),
      array('sjis-win',  'sjis_0208' ),
      array('sjis',      'sjis_0208'),
      array('utf-8',     'unicode_fss')
    );

    /**
     * トランザクションID
     * 
     * @access private
     * @var int トランザクションID
     */
    var $transaction_id = null;
  
    /**
     * DB接続実行
     * 
     * @access public
     * @param string データベース名
     * @param string ユーザー名
     * @param string パスワード
     * @param string ホスト名
     * @param string ポート番号
     * @param bool 持続的接続を行うか
     @ @return boolean 接続OK: true, 接続NG: false
     */
    function openConnection($dbname, $user, $passwd, $host, $port, $persistent)
    {
        $dbname_tmp = $dbname;
        if ($host != '') {
            $dbname_tmp = $host . ':' . $dbname_tmp;
        }

        if ($this->client_encode) {
            $this->connection = $persistent ? ibase_pconnect($dbname_tmp, $user, $passwd, $this->client_encode) : ibase_connect($dbname_tmp, $user, $passwd, $this->client_encode);
            $this->client_encode_flag = true;
        } else {
            $this->connection = $persistent ? ibase_pconnect($dbname_tmp, $user, $passwd) : ibase_connect($dbname_tmp, $user, $passwd);
        }
        return is_resource($this->connection);
    }

    /**
     * DB接続終了
     * 
     * @access public
     */
    function closeConnection()
    {
        if (is_resource($this->connection)) {
            ibase_close($this->connection);
            $this->connection = null;
        }
    }

    /**
     * クライアント側文字エンコーティング設定
     * 
     * @access public
     * @param string DB文字コード
     * @param string PHP側エンコード
     */
    function setClientEncoding($client_encode, $server_encode='')
    {
        $this->client_encode_flag = false;
        parent::setClientEncoding($client_encode, $server_encode);
    }

    /**
     * 最後に起こったエラーメッセージをセット
     *
     * @access public
     * @param string エラーコード
     * @param string エラーメッセージ
     */
    function setErrorMessage($code='', $message='')
    {
        if ($code) {
            parent::setErrorMessage($code, $message);
        } else {
            if (function_exists('ibase_errcode')) {
                if (ibase_errcode()) {
                    $this->last_error_code    = ibase_errcode();
                    $this->last_error_message = ibase_errmsg();
                }
            } else {
                if (ibase_errmsg()) {
                    $this->last_error_code    = '9999';
                    $this->last_error_message = ibase_errmsg();
                }
            }
        }
    }

    /**
     * パラメータサニタイズ（無効化）処理
     * 
     * @access public
     * @param string サニタイズ対象文字列
     * @return string サニタイズ後文字列
     */
    function escape($parameter)
    {
        return str_replace("'", "''", $parameter);
    }

    /**
     * トランザクション開始
     *
     * @access public
     */
    function beginTransaction()
    {
        parent::beginTransaction();
        $this->transaction_id = ibase_trans(IBASE_DEFAULT, $this->connection);
    }

    /**
     * トランザクション破棄
     *
     * @access public
     */
    function rollBack()
    {
        ibase_rollback($this->transaction_id);
        $this->transaction_id = null;
        parent::rollBack();
    }

    /**
     * トランザクション確定
     *
     * @access public
     */
    function commit()
    {
        ibase_commit($this->transaction_id);
        $this->transaction_id = null;
        parent::commit();
    }

    /**
     * SQLを実行し、結果取得
     * 
     * ・SQL文が、select句の場合
     *   実行結果をリソースとして取得
     * ・SQL文が、insert, update, delete句の場合
     *   実行結果影響件数を取得
     * ・SQL文が、上記以外の場合
     *   実行後、true or falseを返却
     *
     * @access public
     * @param string SQL文
     * @param mixed 実行結果件数、またはDBリソース
     * @return boolean 実行OK: true, 実行NG: false
     */
    function execRef($sql, &$result)
    {
        if (!$this->client_encode_flag) {
            $this->convertEncoding($sql);
        }
        parent::execRef($sql, $result);
    }

    /**
     * SQLを実行し、DBリソースを取得
     *
     * @access public
     * @param string SQL文
     * @return mixed 実行OK: DBリソース, 実行NG: false
     */
    function execSelect($sql)
    {
        $result = @ibase_query(($this->transaction_id ? $this->transaction_id : $this->connection), $sql);
        if ($result !== false) {
            return $result;
        } else {
            $this->setErrorMessage();
            $this->triggerError();
            return false;
        }
    }

    /**
     * *** 現状SQL実行成功時は常にtrueを返す ***
     *
     * SQLを実行し、実行結果影響件数を取得
     *
     * @access public
     * @param string SQL文
     * @return mixed 実行OK: 実行結果影響件数, 実行NG: false
     */
    function execUpdate($sql)
    {
        // 実行時エラー判定
        $result = @ibase_query((is_resource($this->transaction_id) ? $this->transaction_id : $this->connection), $sql);
        if ($result !== false) {
            return $result;
        } else {
            $this->setErrorMessage();
            $this->triggerError();
            return false;
        }
    }

    /**
     * SQL実行のみ
     *
     * @access public
     * @param string SQL文
     * @return boolean 実行OK: true, 実行NG: false
     */
    function execNoReturn($sql)
    {
        return $this->execUpdate($sql);
    }

    /**
     * SQL実行し、結果データを取得
     *
     * ・第3パラメータが、'one'の場合
     *    最初の1カラムを文字列として取得
     * ・第3パラメータが、'record'の場合
     *    最初の1レコードを配列として取得
     * ・第3パラメータが、'all'の場合
     *    全レコードを配列として取得
     *
     * @access public
     * @param string SQL文
     * @param mixed  実行結果を取得
     * @param string 結果取得フラグ
     * @return boolean 実行OK: true, 実行NG: false
     */
    function queryRef($sql, &$data, $get='all')
    {
        // SQL実行
        $this->execRef($sql, $query);

        // 実行時エラー判定
        if (!is_resource($query)) {
            $data = $query;
            return ($query !== false);
        }

        switch ($get) {
        // 1フィールド
        case 'one':
            list($data) = ibase_fetch_row($query);
            if ($data === false) {
                $data = '';
            }
            break;
        // 1レコード
        case 'record':
            $data = ibase_fetch_assoc($query);
            if ($data === false) {
                $data = array();
            } else {
                $data = $this->caseFieldName($data);
            }
            break;
        // 全レコード
        case 'all':

            $data = array();
            while ($result = ibase_fetch_assoc($query)) {
              $data[] = $this->caseFieldName($result);
            }
          break;
        }
        ibase_free_result($query);

        if (!$this->client_encode_flag) {
            $this->convertDecoding($data);
        }

        return true;
    }

    /**
     * SQLを実行し、ページ毎にデータを取得
     *
     * @access public
     * @param string SQL文
     * @param array 実行結果格納配列
     * @param object ページオブジェクト
     * @param int 1ページの表示件数
     * @param int 表示対象ページ数 
     * @return bool 実行結果
     */
    function queryPageRef($sql, &$result, &$pager, $limit=20, $page=1)
    {
        // ページングオブジェクト
        $pager = $this->getPager($limit, $page);
        // 総件数取得SQL作成
        $sql_count = "SELECT COUNT(*) FROM (" . $sql . ") AS SyL$";
        // 件数取得SQL実行
        if (!$this->queryRef($sql_count, $record, 'one')) return false;
        $pager->setCountAll($record);

        // ページ単位レコード取得
        $sql_page = trim($sql);
        if (preg_match('/^(select)\s+(.+)$/i', $sql_page, $matches)) {
            $sql_page = $matches[1] . ' first ' . $pager->getCount() . ' skip ' . $pager->getStartOffset() . ' ' . $matches[2];
        } else {
            trigger_error("[SyL error] Ibase SQL invalid ({$sql_page})", E_USER_ERROR);
        }
        // データ取得SQL実行
        return $this->queryRef($sql_page, $result, 'all');
    }

    /**
     * 最後に挿入された行の ID あるいはシーケンスの値を取得
     *
     * @access public
     * @return int 最後に挿入された行のID
     */
    function lastInsertId($seq='')
    {
        $sql = "SELECT GEN_ID({$seq}, 0) FROM rdb$database";
        $this->queryRef($sql, $data, 'one');
        return $data;
    }

    /**
     * 接続しているDBサーバーのバージョンを取得する
     * 
     * @access public
     * @return string DBのバージョン
     */
    function getVersion()
    {
/*
        $sql = "SHOW VERSION";
        $this->queryRef($sql, $version, 'one');
        list($version) = explode('-', $version);
        
        return $version;
*/
        return false;
    }
}
