<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_DBDriverMysqli.php,v 1.1 2009/01/11 05:34:32 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 * DB操作クラス（MySQLi）
 *
 * ※MySQL 4.1以上
 *
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_DBDriverMysqli.php,v 1.1 2009/01/11 05:34:32 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_DBDriverMysqli extends SyL_DB
{
    /**
     * MySQLのバージョン4.1以上
     *
     * @access private
     * @vat bool
     */
     private $ismore41 = false;

    /**
     * エンコード変換テーブル
     *
     *  MySQL    <=> PHP
     *  (client)     (client)
     * 
     * ※全て小文字で記述
     *
     * @access protected
     * @var string
     */
    public $encode_table = array(
      // PHP, DB
      array('eucjp-win', 'ujis'),
      array('euc-jp',    'ujis'),
      array('euc',       'ujis'),
      array('sjis-win',  'sjis' ),
      array('sjis',      'sjis'),
      array('utf-8',     'utf8')
    );

    /**
     * DB接続実行
     * 
     * @access public
     * @param string データベース名
     * @param string ユーザー名
     * @param string パスワード
     * @param string ホスト名
     * @param string ポート番号
     * @param bool 持続的接続を行うか
     * @return boolean 接続OK: true, 接続NG: false
     */
    public function openConnection($dbname, $user, $passwd, $host, $port, $persistent)
    {
        if (!$host) {
            $host = 'localhost';
        }
        if ($port) {
            $this->connection = mysqli_connect($host, $user, $passwd, $dbname, $port);
        } else {
            $this->connection = mysqli_connect($host, $user, $passwd, $dbname);
        }
        if (is_object($this->connection)) {
            $this->ismore41 = (mysqli_get_server_version($this->connection) >= 40100);
            return true;
        } else {
            return false;
        }
    }

    /**
     * クライアント側文字エンコーティング設定
     * 
     * @access public
     * @param string DB文字コード
     * @param string PHP側エンコード
     */
    public function setClientEncoding($client_encode, $server_encode='')
    {
        if ($this->ismore41) {
            $client_encode2 = strtolower($client_encode);
            foreach ($this->encode_table as $encode_table) {
                if ($encode_table[0] == $client_encode2) {
                    $client_encode2 = $encode_table[1];
                    break;
                }
            }
            @mysqli_query($this->connection, "SET NAMES {$client_encode2}");
        }
        parent::setClientEncoding($client_encode, $server_encode);
    }

    /**
     * DB接続終了
     * 
     * @access public
     */
    public function closeConnection()
    {
        if (is_object($this->connection)) {
            mysqli_close($this->connection);
            $this->connection = null;
        }
    }

    /**
     * 最後に起こったエラーメッセージをセット
     *
     * @access public
     * @param string エラーコード
     * @param string エラーメッセージ
     */
    public function setErrorMessage($code='', $message='')
    {
        if ($code) {
            parent::setErrorMessage($code, $message);
        } else {
            $code = mysqli_errno($this->connection);
            if ($code != 0) {
                $this->last_error_code    = $code;
                $this->last_error_message = mysqli_error($this->connection);
            }
        }
    }

    /**
     * パラメータサニタイズ（無効化）処理
     * 
     * @access public
     * @param string サニタイズ対象文字列
     * @return string サニタイズ後文字列
     */
    public function escape($parameter)
    {
        return mysqli_real_escape_string($this->connection, $parameter);
    }

    /**
     * トランザクション開始
     *
     * @access public
     */
    public function beginTransaction()
    {
        parent::beginTransaction();
        mysqli_autocommit($this->connection, false);
    }

    /**
     * トランザクション破棄
     *
     * @access public
     */
    public function rollBack()
    {
        mysqli_rollback($this->connection);
        mysqli_autocommit($this->connection, true);
        parent::rollBack();
    }

    /**
     * トランザクション確定
     *
     * @access public
     */
    public function commit()
    {
        mysqli_commit($this->connection);
        mysqli_autocommit($this->connection, true);
        parent::commit();
    }

    /**
     * SQLを実行し、結果取得
     * 
     * ・SQL文が、select句の場合
     *   実行結果をリソースとして取得
     * ・SQL文が、insert, update, delete句の場合
     *   実行結果影響件数を取得
     * ・SQL文が、上記以外の場合
     *   実行後、true or falseを返却
     *
     * @access public
     * @param string SQL文
     * @param mixed 実行結果件数、またはDBリソース
     * @return boolean 実行OK: true, 実行NG: false
     */
    public function execRef($sql, &$result)
    {
        if (!$this->ismore41) {
            $this->convertEncoding($sql);
        }
        parent::execRef($sql, $result);
    }

    /**
     * SQLを実行し、DBリソースを取得
     *
     * @access public
     * @param string SQL文
     * @return mixed 実行OK: DBリソース, 実行NG: false
     */
    public function execSelect($sql)
    {
        $result = @mysqli_query($this->connection, $sql);
        if ($result !== false) {
            return $result;
        } else {
            $this->setErrorMessage();
            $this->triggerError();
            return false;
        }
    }

    /**
     * SQLを実行し、実行結果影響件数を取得
     *
     * @access public
     * @param string SQL文
     * @return mixed 実行OK: 実行結果影響件数, 実行NG: false
     */
    public function execUpdate($sql)
    {
        $result = @mysqli_query($this->connection, $sql);
        if ($result !== false) {
            return mysqli_affected_rows($this->connection);
        } else {
            $this->setErrorMessage();
            $this->triggerError();
            return false;
        }
    }

    /**
     * SQL実行のみ
     *
     * @access public
     * @param string SQL文
     * @return boolean 実行OK: true, 実行NG: false
     */
    public function execNoReturn($sql)
    {
        $result = @mysqli_query($this->connection, $sql);
        if ($result !== false) {
            return true;
        } else {
            $this->setErrorMessage();
            $this->triggerError();
            return false;
        }
    }

    /**
     * SQL実行し、結果データを取得
     *
     * ・第3パラメータが、'one'の場合
     *    最初の1カラムを文字列として取得
     * ・第3パラメータが、'record'の場合
     *    最初の1レコードを配列として取得
     * ・第3パラメータが、'all'の場合
     *    全レコードを配列として取得
     *
     * @access public
     * @param string SQL文
     * @param mixed  実行結果を取得
     * @param string 結果取得フラグ
     * @return boolean 実行OK: true, 実行NG: false
     */
    public function queryRef($sql, &$data, $get='all')
    {
        // SQL実行
        $this->execRef($sql, $query);

        // 取得判定
        if (!is_object($query)) {
            $data = $query;
            return ($query !== false);
        }

        switch ($get) {
        case 'one':
            if (mysqli_num_rows($query) > 0) {
                list($data) = mysqli_fetch_row($query);
            } else {
                $data = '';
            }
            break;
        case 'record':
            $data = mysqli_fetch_assoc($query);
            if ($data === false) {
                $data = array();
            } else {
                $data = $this->caseFieldName($data);
            }
            break;
        case 'all':
            $data = array();
            while ($result = mysqli_fetch_assoc($query)) {
                $data[] = $this->caseFieldName($result);
            }
            break;
        }
        mysqli_free_result($query);

        if (!$this->ismore41) {
            $this->convertDecoding($data);
        }

        return true;
    }

    /**
     * SQLを実行し、ページ毎にデータを取得
     *
     * @access public
     * @param string SQL文
     * @param array 実行結果格納配列
     * @param object ページオブジェクト
     * @param int 1ページの表示件数
     * @param int 表示対象ページ数 
     * @return bool 実行結果
     */
    public function queryPageRef($sql, &$result, &$pager, $limit=20, $page=1)
    {
        // ページングオブジェクト
        $pager = $this->getPager($limit, $page);
        // 総件数取得SQL作成
        if ($this->ismore41) {
            trigger_error("[SyL error] " . __METHOD__ . " method MySQL ver 4.1 more only", E_USER_ERROR);
        } else {
            if (preg_match('/^select (.+) from (.+)$/i', $sql, $matches)) {
                $sql_count = "SELECT COUNT(*) FROM {$matches[2]}";
            } else {
                trigger_error("[SyL error] Invalid select SQL ({$sql})", E_USER_ERROR);
            }
        }
        // 件数取得SQL実行
        if (!$this->queryRef($sql_count, $record, 'one')) return false;
        $pager->setCountAll($record);

        // ページ単位レコード取得
        $sql_page  = $sql;
        $sql_page .= ' LIMIT ' . $pager->getStartOffset() . ', ' . $pager->getCount();

        // データ取得SQL実行
        return $this->queryRef($sql_page, $result, 'all');
    }

    /**
     * 接続しているDBサーバーのバージョンを取得する
     * 
     * @access public
     * @return string DBのバージョン
     */
    public function getVersion()
    {
        $version = mysqli_get_server_info($this->connection);
        list($version) = explode('-', $version);
        return $version;
    }
}
