<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_DBDriverPgsql.php,v 1.1 2009/01/11 05:34:32 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 *  DB操作クラス（PostgreSQL）
 *
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_DBDriverPgsql.php,v 1.1 2009/01/11 05:34:32 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_DBDriverPgsql extends SyL_DB
{
    /**
     * エンコード変換テーブル
     *
     *  PostgreSQL <=> PHP
     *  (client)       (client)
     * 
     * ※全て小文字で記述
     *
     * @access protected
     * @var string
     */
    var $encode_table = array(
      // PHP, DB
      array('eucjp-win', 'euc-jp'),
      array('euc-jp',    'euc-jp'),
      array('euc',       'euc-jp'),
      array('sjis-win',  'sjis' ),
      array('sjis-win',  'mskanji'),
      array('sjis-win',  'shiftjis'),
      array('sjis-win',  'win932'),
      array('sjis-win',  'windows932'),
      array('sjis',      'sjis'),
      array('utf-8',     'utf8'),
      array('utf-8',     'unicode')
    );

    /**
     * DB接続実行
     * 
     * @access public
     * @param string データベース名
     * @param string ユーザー名
     * @param string パスワード
     * @param string ホスト名
     * @param string ポート番号
     * @param bool 持続的接続を行うか
     * @return boolean 接続OK: true, 接続NG: false
     */
    function openConnection($dbname, $user, $passwd, $host, $port, $persistent)
    {
        $conn_array   = array();
        $conn_array[] = 'dbname=' . $dbname;
        $conn_array[] = 'user='   . $user;
        if ($passwd != '') {
            $conn_array[] = 'password=' . $passwd;
        }
        if ($host != '') {
            $conn_array[] = 'host=' . $host;
        }
        if ($port != '') {
            $conn_array[] = 'port=' . $port;
        }
        $conn_string = implode(' ', $conn_array);

        $this->connection = $persistent ? pg_pconnect($conn_string) : pg_connect($conn_string);
        return is_resource($this->connection);
    }

    /**
     * DB接続終了
     * 
     * @access public
     */
    function closeConnection()
    {
        if (is_resource($this->connection)) {
            pg_close($this->connection);
            $this->connection = null;
        }
    }

    /**
     * クライアント側文字エンコーティング設定
     * 
     * @access public
     * @param string DB文字コード
     * @param string PHP側エンコード
     */
    function setClientEncoding($client_encode, $server_encode='')
    {
        $client_encode2 = strtolower($client_encode);
        foreach ($this->encode_table as $encode_table) {
            if ($encode_table[0] == $client_encode2) {
                $client_encode2 = $encode_table[1];
                break;
            }
        }
        pg_set_client_encoding($this->connection, $client_encode2);
        parent::setClientEncoding($client_encode, $server_encode);
    }

    /**
     * 最後に起こったエラーメッセージをセット
     *
     * @access public
     * @param string エラーコード
     * @param string エラーメッセージ
     */
    function setErrorMessage($code='', $message='')
    {
        if ($code) {
            parent::setErrorMessage($code, $message);
        } else {
            $error_message = $this->connection ? pg_last_error($this->connection) : pg_last_error();
            if ($error_message != '') {
                $this->last_error_code    = '9999';
                $this->last_error_message = $error_message;
            }
        }
    }

    /**
     * パラメータサニタイズ（無効化）処理
     * 
     * @access public
     * @param string サニタイズ対象文字列
     * @return string サニタイズ後文字列
     */
    function escape($parameter)
    {
        return pg_escape_string($parameter);
    }

    /**
     * トランザクション開始
     *
     * @access public
     */
    function beginTransaction()
    {
        parent::beginTransaction();
        @pg_query($this->connection, 'BEGIN');
    }

    /**
     * トランザクション破棄
     *
     * @access public
     */
    function rollBack()
    {
        @pg_query($this->connection, 'ROLLBACK');
        parent::rollBack();
    }

    /**
     * トランザクション確定
     *
     * @access public
     */
    function commit()
    {
        @pg_query($this->connection, 'COMMIT');
        parent::commit();
    }

    /**
     * SQLを実行し、DBリソースを取得
     *
     * @access public
     * @param string SQL文
     * @return mixed 実行OK: DBリソース, 実行NG: false
     */
    function execSelect($sql)
    {
        $result = @pg_query($this->connection, $sql);
        if ($result !== false) {
            return $result;
        } else {
            $this->setErrorMessage();
            $this->triggerError();
            return false;
        }
    }

    /**
     * SQLを実行し、実行結果影響件数を取得
     *
     * @access public
     * @param string SQL文
     * @return mixed 実行OK: 実行結果影響件数, 実行NG: false
     */
    function execUpdate($sql)
    {
        // 実行時エラー判定
        $result = @pg_query($this->connection, $sql);
        if ($result !== false) {
            return pg_affected_rows($result);
        } else {
            $this->setErrorMessage();
            $this->triggerError();
            return false;
        }
    }

    /**
     * SQL実行のみ
     *
     * @access public
     * @param string SQL文
     * @return boolean 実行OK: true, 実行NG: false
     */
    function execNoReturn($sql)
    {
        // 実行時エラー判定
        $result = @pg_query($this->connection, $sql);
        if ($result !== false) {
            return true;
        } else {
            $this->setErrorMessage();
            $this->triggerError();
            return false;
        }
    }
  
    /**
     * SQL実行し、結果データを取得
     *
     * ・第3パラメータが、'one'の場合
     *    最初の1カラムを文字列として取得
     * ・第3パラメータが、'record'の場合
     *    最初の1レコードを配列として取得
     * ・第3パラメータが、'all'の場合
     *    全レコードを配列として取得
     *
     * @access public
     * @param string SQL文
     * @param mixed  実行結果を取得
     * @param string 結果取得フラグ
     * @return boolean 実行OK: true, 実行NG: false
     */
    function queryRef($sql, &$data, $get='all')
    {
        // SQL実行
        $this->execRef($sql, $query);
  
        // 取得判定
        if (!is_resource($query)) {
            $data = $query;
            return ($query !== false);
        }

        switch ($get) {
        // 1フィールド
        case 'one':
            if (pg_num_rows($query) > 0) {
                list($data) = pg_fetch_row($query);
            } else {
                $data = '';
            }
            break;
        // 1レコード
        case 'record':
            $data = pg_fetch_assoc($query);
            if ($data === false) {
                $data = array();
            } else {
                $data = $this->caseFieldName($data);
            }
            break;
        // 全レコード
        case 'all':
            $data = array();
            while ($result = pg_fetch_assoc($query)) {
                $data[] = $this->caseFieldName($result);
            }
            break;
        }
        pg_free_result($query);
        return true;
    }

    /**
     * SQLを実行し、ページ毎にデータを取得
     *
     * @access public
     * @param string SQL文
     * @param array 実行結果格納配列
     * @param object ページオブジェクト
     * @param int 1ページの表示件数
     * @param int 表示対象ページ数 
     * @return bool 実行結果
     */
    function queryPageRef($sql, &$result, &$pager, $limit=20, $page=1)
    {
        // ページングオブジェクト
        $pager = $this->getPager($limit, $page);
        // 総件数取得SQL作成
        $sql_count = "SELECT COUNT(*) FROM (" . $sql . ") AS SyL$";
        // 件数取得SQL実行
        if (!$this->queryRef($sql_count, $record, 'one')) return false;
        $pager->setCountAll($record);

        // ページ単位レコード取得
        $sql_page  = $sql;
        $sql_page .= " LIMIT "  . $pager->getCount();
        $sql_page .= " OFFSET " . $pager->getStartOffset();

        // データ取得SQL実行
        return $this->queryRef($sql_page, $result, 'all');
    }

    /**
     * 最後に挿入された行の ID あるいはシーケンスの値を取得
     *
     * @access public
     * @return int 最後に挿入された行のID
     */
    function lastInsertId($seq='')
    {
        $sql = "SELECT CURRVAL('{$seq}')";
        $this->queryRef($sql, $data, 'one');
        return $data;
    }

    /**
     * 接続しているDBサーバーのバージョンを取得する
     * 
     * @access public
     * @return string DBのバージョン
     */
    function getVersion()
    {
        $sql = "SELECT VERSION()";
        $this->queryRef($sql, $version, 'one');
        if (preg_match('/^PostgreSQL (.+) on .+$/', trim($version), $matches)) {
            $version = $matches[1];
        }
        return $version;
    }
}
