<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_FormElement.php,v 1.1 2009/01/11 05:34:34 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 * 検証クラス
 */
require_once dirname(__FILE__) . '/../SyL_Validation.php';

/**
 * フォーム要素の基底クラス
 * 
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_FormElement.php,v 1.1 2009/01/11 05:34:34 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_FormElement
{
    /**
     * フォーム要素HTMLタグの型
     *
     * @access private
     * @var string
     */
    var $type = '';
    /**
     * フォーム要素HTMLタグ名
     *
     * @access private
     * @var string
     */
    var $name = '';
    /**
     * フォーム要素表示名
     *
     * @access private
     * @var string
     */
    var $display_name = '';
    /**
     * フォーム要素の値
     *
     * @access private
     * @var string
     */
    var $value = '';
    /**
     * フォーム要素HTMLタグの追加属性
     *
     * @access private
     * @var string
     */
    var $attributes = array();
    /**
     * フォーム要素部品配列
     * ※radio, select, checkboxのみ
     *
     * @access private
     * @var array
     */
    var $options = array();
    /**
     * 読み取り専用フラグ
     *
     * @access private
     * @var bool
     */
    var $read_only = false;
    /**
     * エラーメッセージ
     * ※エラーメッセージが''の場合は、エラーが無い
     *
     * @access private
     * @var string
     */
    var $error_message = '';
    /**
     * 検証グループオブジェクト
     *
     * @access private
     * @var object
     */
    var $validators = null;

    /**
     * コンストラクタ
     *
     * @access public
     * @param string フォーム要素HTMLタグの型
     * @param string フォーム要素HTMLタグ名
     * @param string フォーム要素表示名
     * @param array フォーム要素属性配列
     */
    function SyL_FormElement($type, $name, $display_name, $attributes=array())
    {
        // 表示分割子の指定
        if (isset($attributes['separator'])) {
            if (($attributes['separator'] != '') && method_exists($this, 'setSeparator')) {
                $this->setSeparator($attributes['separator']);
            }
            unset($attributes['separator']);
        }

        $this->type         = strtolower($type);
        $this->name         = $name;
        $this->display_name = $display_name;
        $this->attributes   = $attributes;
    }

    /**
     * 個別フォーム要素オブジェクトの取得
     *
     * @access public
     * @param string フォーム要素タイプ
     * @param string フォーム要素名
     * @param string フォーム要素表示名
     * @param mixed リクエスト初期値
     * @param array フォーム要素の部品配列（radio, select, checkboxの場合のみ）
     * @param mixed フォーム要素の値（checkboxの場合は配列、それ以外はstring）
     * @param array フォーム要素の追加属性
     */
    function &create($type, $name, $display_name, $value=null, $options=array(), $attributes=array())
    {
        $class_name = 'SyL_FormElement' . ucfirst($type);
        include_once dirname(__FILE__) . "/FormElement/{$class_name}.php";
        $element =& new $class_name($type, $name, $display_name, $attributes);
        $element->setValue($value);
        $element->setOptions($options);
        return $element;
    }

    /**
     * 追加オプションの設定
     *
     * @access protected
     * @param array フォーム要素の部品配列（radio, select, checkboxの場合のみ）
     */
    function setOptions($options=array())
    {
        if (is_array($options)) {
            $this->options = $options;
        }
    }

    /**
     * 要素名を取得する
     *
     * @access public
     * @return string 要素名
     */
    function getName()
    {
        return $this->name;
    }

    /**
     * 要素名を取得する（複数指定の場合オーバーライド）
     *
     * @access public
     * @return string 要素名
     */
    function getNames()
    {
        return $this->getName();
    }

    /**
     * フォーム要素名を取得する
     *
     * @access public
     * @return string フォーム要素名
     */
    function getType()
    {
        return $this->type;
    }

    /**
     * フォーム要素表示名を取得する
     *
     * @access public
     * @return string 要素名
     */
    function getDisplayName()
    {
        return $this->display_name;
    }

    /**
     * 要素値をセットする
     *
     * @access public
     * @param mixed 要素値
     */
    function setValue($value)
    {
        $this->value = $value;
    }

    /**
     * 要素値を取得する
     *
     * @access public
     * @return string 要素名
     */
    function getValue()
    {
        return $this->value;
    }

    /**
     * エラーメッセージを取得する
     *
     * @access public
     * @return string エラーメッセージ
     */
    function getErrorMessage()
    {
        return $this->error_message;
    }

    /**
     * 要素属性をセット
     *
     * @access public
     * @param string 要素属性値
     * @param string 要素属性名
     */
    function setAttribute($value, $name=null)
    {
        if ($name) {
            $name = strtolower($name);
            $this->attributes[$name] = $value;
        } else {
            $this->attributes[] = $value;
        }
    }

    /**
     * その他属性を取得する
     *
     * @access public
     * @return string その他属性
     */
    function getAttributes()
    {
        $attributes = array();
        foreach ($this->attributes as $k => $v) {
            $attributes[] = is_int($k) ? $this->encode($v) : ($k . '="' . $this->encode($v) . '"');
        }
        return implode(' ', $attributes);
    }

    /**
     * フォーム要素表示判定フラグをセット
     *
     * @access public
     * @param bool フォーム要素表示判定フラグ
     */
    function setReadOnly($read_only)
    {
        $this->read_only = ($read_only === true);
    }

    /**
     * 必須チェック存在判定
     * ※検証グループオブジェクトをセットしてから実行する
     *
     * @access public
     * @return bool true: 必須チェックあり、false: 必須チェック無し
     */
    function isRequire()
    {
        return is_object($this->validators) ? $this->validators->isRequire() : false;
    }

    /**
     * フォーム要素の表示判定を取得
     *
     * @access public
     * @param bool フォーム要素の表示判定を取得
     */
    function isReadOnly()
    {
        return $this->read_only;
    }

    /**
     * フォーム要素HTML出力
     * 読み取り専用になっているかで、子クラスのメソッドを呼び分ける
     *
     * @access public
     * @return string フォーム要素のHTML
     */
    function getHtml()
    {
        if ($this->read_only) {
            return $this->getHtmlView();
        } else {
            return $this->getHtmlTag();
        }
    }

    /**
     * フォーム要素HTML出力（表示）
     *
     * @access public
     * @return string フォーム要素のHTML
     */
    function getHtmlView()
    {
    }

    /**
     * フォーム要素HTML出力（入力項目）
     *
     * @access public
     * @return string フォーム要素のHTML
     */
    function getHtmlTag()
    {
    }

    /**
     * フォーム要素をhiddenタグで出力
     *
     * @access public
     * @param bool true: htmlエンコードを行う、false: htmlエンコードを行わない
     * @return string フォーム要素のHTML
     */
    function getHtmlHidden($name=null, $value=null)
    {
        if (($name === null) || ($value === null)) {
            $name  = $this->name;
            $value = $this->value;
        }
        return '<input type="hidden" name="'  . $name  . '" value="' . $this->encode($value) . '" />';
    }

    /**
     * HTMLエンコードを行う
     *
     * @access public
     * @param string HTMLエンコード前文字列
     * @return string HTMLエンコード後文字列
     */
    function encode($value)
    {
        return htmlspecialchars($value);
    }

    /**
     * 検証グループオブジェクトをセット
     *
     * @access public
     * @param object 検証グループオブジェクト
     */
    function setValidators(&$validators)
    {
        if (version_compare(PHP_VERSION, '5.0.0', '>=')) {
            $this->validators = unserialize(serialize($validators));
        } else {
            $this->validators = $validators;
        }
    }

    /**
     * 検証を実行する
     * 検証グループオブジェクトがない場合は、常にtrue
     *
     * @access public
     * @return bool true: エラー無し、false: エラーあり
     */
    function validate()
    {
        if (!is_object($this->validators)) {
            return true;
        }

        if (!$this->validators->execute($this->getValue(), $this->getDisplayName())) {
            $this->error_message = $this->validators->getErrorMessage();
            return false;
        } else {
            return true;
        }
    }

    /**
     * JavaScript入力チェックタグを取得
     *
     * @access public
     * @param string フォーム名
     * @return string JavaScript入力チェックタグ
     */
    function getJs($formname)
    {
        if (is_object($this->validators)) {
            $element = $this->getName();
            $func    = "check_{$formname}_{$element}(form, errors)";
            $js      = $this->validators->getJs($this->getDisplayName());
            $js = <<< JAVASCRIPT_CODE
function {$func} {
  var validation    = new SyL.Validation.Validation(form);
  var name  = "{$element}";
  var message = "";

{$js}

  if (message) {
    errors.setErrorMessage(name, message);
  }
}
JAVASCRIPT_CODE;
            return array($func, $js);
        } else {
            return array();
        }
    }
}
