<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_Console.php,v 1.1 2009/01/11 05:34:29 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 *  コンソール表示クラス
 *
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_Console.php,v 1.1 2009/01/11 05:34:29 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_Console
{
    /**
     * シェル接頭辞
     *
     * @access private
     * @var string
     */
    var $shell_prefix = '> ';
    /**
     * シェルとして実行するときの引数
     * ※falseの場合はコマンド実行不可
     *
     * @access private
     * @var string
     */
    var $shell_string = '\!';
    /**
     * 一括実行時にコマンドラインを終了する文字列
     *
     * @access private
     * @var array
     */
    var $exit_strings = array(
      'quit',
      'exit',
      '\q'
    );
    /**
     * コマンドラインに表示するメッセージ
     *
     * [message]
     * [default_message] > 
     *
     * array(
     *   [0] => array( [message], [default_message] ),
     *   ...
     * );
     *
     * @access private
     * @var array
     */
    var $command_messages = array();
    /**
     * コマンドラインから取得した入力値を取得するコールバック関数
     *
     * @access private
     * @var array
     */
    var $callback_func = array();
    /**
     * 表示用変換エンコード
     *
     * @access private
     * @var array
     */
    var $output_encode = '';
    /**
     * プログラム側のエンコード
     * 
     * @access protected
     * @var string
     */
    var $internal_encode = '';

    /**
     * コンストラクタ
     *
     * @access public
     * @param string 表示用変換エンコード
     * @param string プログラム側エンコード
     */
    function SyL_Console()
    {
        $this->callback_func = array(&$this, 'doCallback');
    }

    /**
     * シェル接頭辞をセットする
     *
     * @access public
     * @param string シェル接頭辞
     */
    function setShellPrefix($shell_prefix)
    {
        $this->shell_prefix = $shell_prefix;
    }

    /**
     * 表示用変換エンコードをセットする
     *
     * @access public
     * @param string 表示用変換エンコード
     */
    function setOutputEncode($output_encode)
    {
        $this->output_encode = $output_encode;
    }

    /**
     * プログラム側エンコードをセットする
     *
     * @access public
     * @param string プログラム側エンコード
     */
    function setInternalEncode($internal_encode)
    {
        $this->internal_encode = $internal_encode;
    }

    /**
     * 終了文字を追加する
     *
     * @access public
     * @param string 終了文字
     */
    function addExitString($exit_string)
    {
        $this->exit_strings[] = $exit_string;
    }

    /**
     * シェル文字を変更する
     * ※falseの場合はコマンド実行不可
     *
     * @access public
     * @param string シェル文字
     */
    function setShellString($shell_string)
    {
        $this->shell_string = $shell_string;
    }

    /**
     * 一括実行用、表示するメッセージをセット
     *
     * @access public
     * @param string 表示メッセージ
     * @param string コマンド補足メッセージ
     */
    function addMessage($message, $message_default='')
    {
        $this->command_messages[] = array($message, $message_default);
    }

    /**
     * 一括実行用、コマンドラインからの入力値を取得するコールバック関数をセット
     *
     * @access public
     * @param mixed コールバック関数
     */
    function setCallbackFunc($callback_func)
    {
        if (is_callable($callback_func)) {
            $this->callback_func = $callback_func;
        } else {
            trigger_error("[SyL error] invalid callable format", E_USER_ERROR);
        }
    }

    /**
     * デフォルトコールバック関数
     *
     * @access public
     * @param string 入力値
     */
    function doCallback(&$console, $return, $shell)
    {
    }

    /**
     * 一括実行スタート
     *
     * @access public
     * @param bool コマンドループフラグ
     */
    function start($loop=false)
    {
        while (true) {
            $shell = '';
            if ($this->command_messages) {
                $command_message = array_shift($this->command_messages);
                $return = $this->getInput($command_message[0], $command_message[1]);
            } else {
                if (!$loop) {
                    break;
                }
                $return = $this->getInput('');
            }
            if (in_array($return, $this->exit_strings)) {
                break;
            }
            if ($this->shell_string && (substr($return, 0, strlen($this->shell_string)) == $this->shell_string)) {
                $shell = shell_exec(substr($return, strlen($this->shell_string)));
            }
            // コールバック関数実行
            call_user_func_array($this->callback_func, array(&$this, $return, $shell));
        }
    }

    /**
     * 表示を出力する
     *
     * @access public
     * @param string 表示メッセージ
     * @param bool 改行フラグ
     */
    function stdout($message, $newline=true)
    {
        $this->stdmessage($message, 'out', $newline);
    }

    /**
     * エラー表示を出力する
     *
     * @access public
     * @param string エラー表示メッセージ
     * @param bool 改行フラグ
     */
    function stderr($message, $newline=true)
    {
        $this->stdmessage($message, 'err', $newline);
    }

    /**
     * 表示を出力する
     *
     * @access public
     * @param string 表示メッセージ
     * @param string 表示タイプ
     * @param bool 改行フラグ
     */
    function stdmessage($message, $type='out', $newline=true)
    {
        if ($this->output_encode) {
            if ($this->internal_encode) {
                $message = mb_convert_encoding($message, $this->output_encode, $this->internal_encode);
            } else {
                $message = mb_convert_encoding($message, $this->output_encode);
            }
        }
        if ($newline) {
            $message .= "\n";
        }
        if ($type == 'err') {
            fwrite(STDERR, $message);
        } else {
            fwrite(STDOUT, $message);
        }
        ob_flush();
    }

    /**
     * 入力値を取得する
     *
     * @access public
     * @param string 表示メッセージ
     * @param string コマンド補足メッセージ
     */
    function getInput($message, $message_default='')
    {
        if ($message_default != '') {
            $message_default .= ' ';
        }
        $message_default .= $this->shell_prefix;
        if ($message) {
            $this->stdout($message);
        }
        $this->stdout($message_default, false);
        ob_flush();

        return trim(fgets(STDIN,256));
    }
}
