<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_Form.php,v 1.1 2009/01/11 05:34:29 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 * 検証個別クラスの基底クラス
 */
require_once dirname(__FILE__) . '/Form/SyL_FormElement.php';
/**
 * 検証クラス
 */
require_once 'SyL_Validation.php';

/**
 * フォームクラス
 * 
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_Form.php,v 1.1 2009/01/11 05:34:29 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_Form
{
    /**
     * フォーム名
     * 
     * @access protected
     * @var string
     */
    var $name = 'syl_form';
    /**
     * フォーム属性配列
     *
     * @access private
     * @var array
     */
    var $attributes = array();
    /**
     * フォーム要素の基本オブジェクト格納配列
     *
     * @access private
     * @var array
     */
    var $elements = array();
    /**
     * パラメータ取得関数
     *
     * @access private
     * @var mixed
     */
    var $input_callback = null;
    /**
     * 結果表示用配列
     *
     * @access private
     * @var array
     */
    var $results = array();
    /**
     * JavaScriptパラメータ
     *
     * @access private
     * @var array
     */
    var $js_parameters = array(
      'func_name'   => '',
      'all_message' => true,
      'header'      => '',
      'footer'      => '',
      'error_callback' => null
    );

    /**
     * コンストラクタ
     *
     * @access public
     * @param string リクエストアクション
     * @param string リクエストメソッド
     */
    function SyL_Form($action='', $method='POST')
    {
        $this->setAttribute('name',   $this->name);
        $this->setAttribute('id',     $this->name);
        $this->setAttribute('method', strtoupper($method));
        $this->setAttribute('action', $action ? $action : $_SERVER['PHP_SELF']);

        // フォームステータスを初期化
        $this->results['javascript'] = '';
        $this->results['read_only']  = false;
        $this->results['error']      = false;
        // エラーメッセージを初期化
        $this->results['error_message'] = array();
        // フォーム要素初期化
        $this->results['elements'] = array();
        // デフォルトのパラメータ取得関数
        $this->input_callback = array(&$this, 'getParameter');
    }

    /**
     * 個別フォーム要素オブジェクトの取得
     *
     * @access public
     * @param string フォーム要素タイプ
     * @param string フォーム要素名
     * @param string フォーム要素表示名
     * @param array フォーム要素の部品配列（radio, select, checkboxの場合のみ）
     * @param array フォーム要素の追加属性
     */
    function &createElement($type, $element_name, $display_name, $options=array(), $attributes=array())
    {
        $value = call_user_func($this->input_callback, $element_name);
        $element =& SyL_FormElement::create($type, $element_name, $display_name, $value, $options, $attributes);
        if (is_a($element, 'SyL_FormElementGroup')) {
            $element->initGroup($this);
        }
        return $element;
    }

    /**
     * フォーム要素オブジェクトを追加する
     *
     * @access public
     * @param object フォーム要素オブジェクト
     * @param object バリデーションオブジェクト
     */
    function addElement(&$element, $validation=null)
    {
        if ($validation != null) {
            $element->setValidators($validation);
        }
        if ($element->getType() == 'file') {
            $this->setAttribute('enctype', 'multipart/form-data');
        } else if ($element->getType() == 'group') {
            foreach ($element->elements as $element2) {
                if ($element2->getType() == 'file') {
                    $this->setAttribute('enctype', 'multipart/form-data');
                    break;
                }
            }
        }
        $element_name = $element->getName();
        // 要素オブジェクト格納
        $this->elements[$element_name] =& $element;
        // 結果配列初期化
        $this->results['elements'][$element_name] = array(
          'label'         => '',
          'html'          => '',
          'require'       => false,
          'error'         => false,
          'error_message' => '',
          'read_only'     => false
        );
    }

    /**
     * フォーム要素オブジェクトを取得する
     *
     * @access public
     * @param string フォーム要素HTMLタグ名
     * @return object フォーム要素オブジェクト
     */
    function &getElement($element_name)
    {
        return $this->elements[$element_name];
    }

    /**
     * パラメータ取得関数をセット
     *
     * @access public
     * @param mixed パラメータ取得関数
     */
    function registerInputCallback($callback)
    {
        if (is_callable($callback)) {
            $this->input_callback = $callback;
        } else {
            trigger_error("[SyL error] Invalid callback parameter. Not function and method (" . var_export($input_callback) . ")", E_USER_ERROR);
        }
    }

    /**
     * リクエストから初期値を取得する
     *
     * @access private
     * @param string パラメータ名
     * @return mixed パラメータ値
     */
    function getParameter($name)
    {
        switch ($this->getAttribute('method')) {
        case 'POST': return isset($_POST[$name]) ? $_POST[$name] : null;
        case 'GET':  return isset($_GET[$name])  ? $_GET[$name]  : null;
        default:     return null;
        }
    }

    /**
     * フォームタグの属性をセット
     *
     * @access public
     * @param string フォームタグの属性
     * @param string フォームタグの属性値
     */
    function setAttribute($key, $value)
    {
        $key = strtolower($key);
        $this->attributes[$key] = $value;
    }

    /**
     * フォームタグの属性をセット
     *
     * @access public
     * @param string フォームタグの属性
     * @return string フォームタグの属性値
     */
    function getAttribute($key)
    {
        $key = strtolower($key);
        return isset($this->attributes[$key]) ? $this->attributes[$key] : null;
    }

    /**
     * フォームタグの属性を取得
     *
     * @access public
     * @return string フォームタグの属性
     */
    function getAttributes()
    {
        $attributes = array();
        foreach ($this->attributes as $name => $value) {
            $attributes[] = $name . '="' . $value . '"';
        }
        return implode(' ', $attributes);
    }

    /**
     * フォーム要素表示名を取得する
     *
     * @access public
     * @param string フォーム要素名
     * @return string フォーム要素表示名
     */
    function getName($name)
    {
        return $this->elements[$name]->getDisplayName();
    }

    /**
     * フォーム要素のHTMLを取得
     *
     * @access public
     * @param string フォーム要素名
     * @param bool true: htmlエンコードを行う、false: htmlエンコードを行わない
     * @return string フォーム要素のHTML
     */
    function getHtml($name)
    {
        return $this->elements[$name]->getHtml();
    }

    /**
     * 要素の値を取得
     *
     * @access public
     * @param string フォーム要素名
     * @return mixed 要素の値
     */
    function getValue($name)
    {
        return $this->elements[$name]->getValue();
    }

    /**
     * 全ての要素の値を取得
     *
     * @access public
     * @return array 全ての要素の値
     */
    function getValues()
    {
        $values = array();
        foreach (array_keys($this->elements) as $name) {
            $values[$name] = $this->getValue($name);
        }
        return $values;
    }

    /**
     * フォーム要素が必須か判定する
     *
     * @access public
     * @param string フォーム要素名
     * @return bool true: 必須、false: 任意
     */
    function isRequire($name)
    {
        if ($this->results['read_only']) {
            // 表示時は、任意状態とする
            return false;
        } else {
            return $this->elements[$name]->isRequire();
        }
    }

    /**
     * フォーム要素が表示のみか判定する
     *
     * @access public
     * @param string フォーム要素名
     * @return bool true: 必須、false: 任意
     */
    function isReadOnly($name)
    {
        return $this->elements[$name]->isReadOnly();
    }

    /**
     * フォームの全要素を表示のみにするか
     *
     * @access public
     * @param bool フォームの全要素を表示のみフラグ
     */
    function setReadOnly($read_only)
    {
        if (!is_bool($read_only)) {
            return;
        }
        foreach (array_keys($this->elements) as $key) {
            $this->elements[$key]->setReadOnly($read_only);
        }
        $this->results['read_only'] = $read_only;
    }

    /**
     * 全ての要素の検証を実行する
     *
     * @access public
     * @return true: エラーなし、false: エラーあり
     */
    function validate()
    {
        $result = true;
        foreach (array_keys($this->elements) as $key) {
            if (!$this->elements[$key]->validate()) {
                $name          = $this->elements[$key]->getName();
                $error_message = $this->elements[$key]->getErrorMessage();
                $this->results['elements'][$name]['error'] = true;
                $this->results['elements'][$name]['error_message'] = $error_message;
                $this->results['error'] = true;
                $this->results['error_message'][] = $error_message;
                $result = false;
            }
        }
        return $result;
    }

    /**
     * フォーム要素のエラーを判定する
     *
     * @access public
     * @param string フォーム要素名
     * @return bool true: 必須、false: 任意
     */
    function isError($name='')
    {
        if ($name != '') {
            return $this->results['elements'][$name]['error'];
        } else {
            foreach ($this->results['elements'] as $name => $value) {
                if ($value['error']) {
                    return true;
                }
          }
          return false;
        }
    }

    /**
     * エラーメッセージをセットする
     *
     * @access public
     * @param string フォーム要素名
     * @param string エラーメッセージ
     */
    function setErrorMessage($name, $error_message)
    {
        $this->results['elements'][$name]['error'] = true;
        $this->results['elements'][$name]['error_message'] = $error_message;
        $this->results['error'] = true;
        $this->results['error_message'][] = $error_message;
    }

    /**
     * フォーム要素のエラーを取得する
     *
     * @access public
     * @param string フォーム要素名
     * @return string エラーメッセージ
     */
    function getErrorMessage($name)
    {
        return $this->results['elements'][$name]['error_message'];
    }

    /**
     * フォーム要素の全エラーを取得する
     *
     * @access public
     * @param string フォーム要素名
     * @return array エラーメッセージ配列
     */
    function getErrorMessageAll()
    {
        return $this->results['error_message'];
    }

    /**
     * JavaScript使用
     *
     * @access public
     * @param string JavaScriptバリデーション実行関数名
     * @param bool 全エラーメッセージ一括表示フラグ
     */
    function useJs($func_name='sylFormCheck', $all_message=true)
    {
        $this->js_parameters['func_name'] = $func_name;
        $this->js_parameters['all_message'] = $all_message;
    }

    /**
     * JavaScriptエラー時のコールバック関数をセット
     *
     * @access public
     * @param string JavaScriptエラー時のコールバック関数
     */
    function setJsCustomErrorCallback($error_callback)
    {
        $this->js_parameters['error_callback'] = $error_callback;
    }

    /**
     * JavaScriptエラーメッセージヘッダ、フッタの設定
     *
     * @access public
     * @param string JavaScriptエラーメッセージヘッダ
     * @param string JavaScriptエラーメッセージフッタ
     */
    function setJsHeaderFooter($header='', $footer='')
    {
        $this->js_parameters['header'] = $header;
        $this->js_parameters['footer'] = $footer;
    }

    /**
     * JavaScript入力チェックタグを取得
     *
     * @access public
     * @return string JavaScript入力チェックタグ
     */
    function getJs()
    {
        $func_name = $this->js_parameters['func_name'];
        if (!$func_name) {
            return '';
        }

        $all_message = $this->js_parameters['all_message'] ? 'true' : 'false';

        // 各チェックメソッド
        $funcs   = '';
        $checker = '';
        foreach (array_keys($this->elements) as $key) {
            $js = $this->elements[$key]->getJs($this->getAttribute('name'));
            if (count($js) == 2) {
              $funcs   .= '/* ' . $this->elements[$key]->getDisplayName() . ' check */' . "\n" . $js[0] . ';' . "\n";
              $checker .= $js[1] . "\n";
            }
        }

        // 不要な部分（コメントなど）削除
        //$validation = preg_replace('/\/\*[^\/]*\*\//', '', $validation);
        //$validation = preg_replace('/^(\s*)\/\/(.*)$/m', '', $validation);
        //$validation = preg_replace('/^(\s+)(.*)$/m', '$2', $validation);

        $header = str_replace('"', '\\"', $this->js_parameters['header']);
        $footer = str_replace('"', '\\"', $this->js_parameters['footer']);

        return <<< JAVASCRIPT_CODE
<script type="text/javascript">
{$checker}
</script>
<script type="text/javascript">
var {$func_name}_submit = false;
function {$func_name}(form)
{
  if ({$func_name}_submit) {
    return false;
  }

  var errors = new SyL.Validation.Errors();
  errors.setCustomErrorCallback({$this->js_parameters['error_callback']});

{$funcs}

  if (errors.isError()) {
    errors.errorMessageHeader = "{$header}";
    errors.errorMessageHeader = "{$footer}";
    errors.setDisplayAllMessage({$all_message});
    errors.raiseErrorMessage();
    errors.focusElement(form);
    return false;
  } else {
    {$func_name}_submit = true;
    return true;
  }
}
</script>
JAVASCRIPT_CODE;
    }

    /**
     * フォーム全要素を配列で取得する
     *
     * @access public
     * @return array フォーム全要素を配列
     */
    function &getResultArray()
    {
        foreach (array_keys($this->elements) as $name) {
            $this->results['elements'][$name]['label']     = $this->getName($name);
            $this->results['elements'][$name]['html']      = $this->getHtml($name);
            $this->results['elements'][$name]['require']   = $this->isRequire($name);
            $this->results['elements'][$name]['read_only'] = $this->isReadOnly($name);
        }
        // フォームのJSチェック
        $this->results['js_name'] = $this->js_parameters['func_name'];
        $this->results['js_code'] = $this->getJs();

        // フォームタグの属性
        $this->results['attributes'] = $this->getAttributes();
        return $this->results;
    }
}
