<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_ValidationManager.php,v 1.1 2009/01/11 05:34:29 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 * 検証クラス
 */
require_once 'SyL_Validation.php';

/**
 * 検証プロセス管理クラス
 * 
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_ValidationManager.php,v 1.1 2009/01/11 05:34:29 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_ValidationManager
{
    /**
     * 検証クラスをまとめた配列
     *
     * @access private
     * @var array
     */
    var $validators = array();

    /**
     * 検証設定オブジェクトを取得する
     *
     * @access public
     * @param string 設定ファイル名
     * @param string 設定ファイル拡張子
     */
    function &getConfig($filename, $ext='')
    {
        include_once dirname(__FILE__) . '/Validation/SyL_ValidationConfig.php';
        $config =& SyL_ValidationConfig::getConfig($filename, $ext);
        return $config;
    }

    /**
     * 検証オブジェクトを追加する
     *
     * @access public
     * @param object 検証オブジェクト
     * @param string 検証項目値
     * @param string 検証要素名
     * @param string 検証項目名
     */
    function add(&$validator, $value, $element_name='', $name='')
    {
        if ($element_name == '') {
            $element_name = $this->getDefaultElementName();
        }

        $this->validators[$element_name] = array(
          'name'          => $name,
          'value'         => $value,
          'error_massage' => ''
        );

        if (version_compare(PHP_VERSION, '5.0.0', '>=')) {
            $this->validators[$element_name]['validator'] = unserialize(serialize($validator));
        } else {
            $this->validators[$element_name]['validator'] = $validator;
        }
    }

    /**
     * 検証外エラーをセットする
     *
     * @access public
     * @param string エラーメッセージ
     * @param string 検証要素名
     */
    function addError($error_massage, $element_name='')
    {
        if ($element_name == '') {
            $element_name = $this->getDefaultElementName();
        }
        $this->validators[$element_name] = array(
          'name'          => '',
          'value'         => '',
          'validator'     => null,
          'error_massage' => $error_massage
        );
    }

    /**
     * デフォルトの要素名を取得する
     *
     * @access public
     * @return string 要素名
     */
    function getDefaultElementName()
    {
        static $i = 1;
        $element_name = __CLASS__ . $i++;
        return $element_name;
    }

    /**
     * 設定値から検証を実行する
     *
     * @access public
     * @param object 設定値オブジェクト
     * @param array 値取得元配列
     * @return bool true: エラー無し, false: エラーあり
     */
    function executeConfig(&$config, $resource=null)
    {
        if ($resource === null) {
            $resource = ($_SERVER['REQUEST_METHOD'] == 'POST') ? $_POST + $_GET : $_GET + $_POST;
        }
        if (!is_array($resource)) {
            trigger_error("[SyL error] Invalid executeConfig method arg. Array only", E_USER_ERROR);
        }
        foreach ($config->getKeys() as $key) {
            $validators   =& $config->get($key);
            $display_name =  $config->getName($key);
            $value = isset($resource[$key]) ? $resource[$key] : null;
            $this->add($validators, $value, $key, $display_name);
        }
        return $this->execute();
    }

    /**
     * 検証処理を実行する
     *
     * @access public
     * @return bool true: エラー無し, false: エラーあり
     */
    function execute()
    {
        $valid = true;
        foreach (array_keys($this->validators) as $key) {
            if ($this->validators[$key]['validator'] != null) {
                if (!$this->validators[$key]['validator']->execute($this->validators[$key]['value'], $this->validators[$key]['name'])) {
                    $this->validators[$key]['error_massage'] = $this->validators[$key]['validator']->getErrorMessage();
                    $valid = false;
                }
            }
        }
        return $valid;
    }

    /**
     * 全体または項目ごとにエラーがあるか判定する
     *
     * @access public
     * @param string 検証要素名
     * @return bool true: エラーあり、false: エラーなし
     */
    function isError($element_name='')
    {
        if ($element_name != '') {
            return (isset($this->validators[$element_name]) && ($this->validators[$element_name]['error_massage'] != ''));
        } else {
            foreach (array_keys($this->validators) as $key) {
                if ($this->validators[$key]['error_massage'] != '') {
                    return true;
                }
            }
            return false;
        }
    }

    /**
     * 全体または項目ごとにエラーメッセージを取得する
     *
     * @access public
     * @param string 検証要素名
     * @return string エラーメッセージ
     */
    function getErrorMessage($element_name)
    {
        return (isset($this->validators[$element_name]) ? $this->validators[$element_name]['error_massage'] : '');
    }

    /**
     * 全エラーメッセージを取得する
     *
     * @access public
     * @return array エラーメッセージ配列
     */
    function getErrorMessageAll()
    {
        $error_message = array();
        foreach (array_keys($this->validators) as $key) {
            if ($this->validators[$key]['error_massage'] != '') {
                $error_message[] = $this->validators[$key]['error_massage'];
            }
        }
        return $error_message;
    }
}
