<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_UtilPager.php,v 1.1 2009/01/11 05:34:29 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 *  改ページ操作用クラス
 *
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_UtilPager.php,v 1.1 2009/01/11 05:34:29 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_UtilPager
{
    /**
     * 総件数
     * 
     * @access private
     * @var int
     */
    var $sum = 0;
    /**
     * 1ページの表示件数
     * 
     * @access private
     * @var int
     */
    var $num = 0;
    /**
     * 現在ページ数
     * 
     * @access private
     * @var int
     */
    var $current_page = 1;
    /**
     * 改ページ遷移先URL
     * 
     * @access private
     * @var string
     */
    var $url = '';
    /**
     * 改ページ遷移先、パラメータ名（ページ数パラメータ）
     * 
     * @access private
     * @var string
     */
    var $parameter_name = 'page';
    /**
     * 改ページ遷移先、パラメータ名（ページ数以外）
     * 
     * @access private
     * @var array
     */
    var $parameters = array();

    /**
     * コンストラクタ
     *
     * @access public
     */
    function SyL_UtilPager()
    {
        $this->url = $_SERVER['PHP_SELF'];
    }

    /**
     * 1ページの件数をセット
     *
     * @access public
     * @param int 1ページの件数
     */
    function setCount($num)
    {
        $this->num = (!is_numeric($num) || ($num < 0)) ? 0 : (int)$num;
    }

    /**
     * 1ページのレコード数を取得
     *
     * @access public
     * @return int 1ページのレコード数
     */
    function getCount()
    {
        return $this->num;
    }

    /**
     * 総件数をセット
     *
     * @access public
     * @param int 総件数
     */
    function setCountAll($sum)
    {
        $this->sum = (!is_numeric($sum) || ($sum < 0)) ? 0 : (int)$sum;

        // 総件数と現在のページ数の整合性チェック
        // 総レコード数0のときはページ数も0
        if ($this->sum <= 0) {
            $this->current_page = 0;
        // 総ページ数以上のページが指定されたら1ページ目へ
        } else if ($this->current_page > $this->getTotalPage()) {
            $this->current_page = 1;
        }
    }

    /**
     * データ総数を取得
     *
     * @access public
     *
     * @return int データ総数
     */
    function getCountAll()
    {
        return $this->sum;
    }

    /**
     * 現在のページ数をセット
     *
     * @access public
     * @param int 現在のページ数
     */
    function setPage($current_page)
    {
        $this->current_page = (!is_numeric($current_page) || ($current_page < 1)) ? 1 : (int)$current_page;
    }

    /**
     * 現在ページ数取得
     *
     * @access public
     * @return int 現在ページ数
     */
    function getCurrentPage()
    {
        return $this->current_page;
    }

    /**
     * 次ページ数取得
     * カレントが最終ページの場合は、falseを返す
     *
     * @access public
     * @return mixed 次ページ数
     */
    function getNextPage()
    {
        if ($this->getCurrentPage() >= $this->getTotalPage()) {
            return false;
        } else {
            return $this->getCurrentPage() + 1;
        }
    }

    /**
     * 前ページ数取得
     * カレントが最初のページの場合は、falseを返す
     *
     * @access public
     * @return mixed 前ページ数
     */
    function getPrevPage()
    {
        if ($this->isFirstPage()) {
            return false;
        } else {
            return $this->getCurrentPage() - 1;
        }
    }

    /**
     * ページ総数取得
     *
     * @access public
     * @return int ページ総数
     */
    function getTotalPage()
    {
        // 合計レコード数、または1ページ件数が0以下なら0ページ
        return (($this->sum <= 0) || ($this->num <= 0)) ? 0 : ceil($this->sum / $this->num);
    }

    /**
     * 最初のページ判定
     *
     * @access public
     * @param bool true: 最初のページ、false: 最初以外のページ
     */
    function isFirstPage()
    {
        return ($this->getCurrentPage() <= 1);
    }

    /**
     * 最終ページ判定
     *
     * @access public
     * @param bool true: 最終ページ、false: 最終以外のページ
     */
    function isLastPage()
    {
        return ($this->getCurrentPage() == $this->getTotalPage());
    }

    /**
     * データ取得時の最初のレコードを取得。クエリ検索用。
     * ※最初のレコードは0とする。
     *
     * @access public
     * @return int データ取得時の最初のレコード
     */
    function getStartOffset()
    {
        $offset = $this->getCount() * ($this->getCurrentPage() - 1);
        return ($offset < 0) ? 0 : $offset;
    }

    /**
     * データ取得対象の最初のレコードを取得
     *
     * @access public
     * @return int データ取得対象の最初のレコード
     */
    function getStartRecord()
    {
        $offset = $this->getCount() * ($this->getCurrentPage() - 1);
        return ($offset < 0) ? 0 : $offset + 1;
    }

    /**
     * データ取得対象の最後のレコードを取得
     *
     * @access public
     * @return int データ取得対象の最後のレコード
     */
    function getEndRecord()
    {
        if ($this->getCurrentPage() >= $this->getTotalPage()) {
            return $this->getCountAll();
        } else {
            return $this->getCount() * $this->getCurrentPage();
        }
    }

    /**
     * 遷移先URLをセット
     *
     * @access public
     * @param string 遷移先URL
     */
    function setUrl($url)
    {
        $this->url = $url;
    }

    /**
     * 遷移先ページパラメータ名をセット
     *
     * @access public
     * @param string 遷移先ページパラメータ名
     */
    function setParameterName($parameter_name)
    {
        $this->parameter_name = $parameter_name;
    }

    /**
     * 遷移先パラメータをセット
     * ※URLエンコード済みとする
     *
     * @access public
     * @param string 遷移先URL
     */
    function setParameter($name, $value)
    {
        $this->parameters[] = array($name, $value);
    }

    /**
     * クエリパラメータを取得する
     *
     * @access public
     * @param int ページ数
     * @return string クエリパラメータ
     */
    function getBuildParameters($page)
    {
        $parameters = $this->parameter_name . '=' . $page;
        foreach ($this->parameters as $parameter) {
            $parameters .= '&' . $parameter[0] . '=' . urlencode($parameter[1]);
        }
        return $parameters;
    }

    /**
     * 次ページリンク取得
     *
     * @access public
     * @param string リンクメッセージ（リンクあり時）
     * @param string リンクメッセージ（リンク無し時）
     * @return string 次ページリンクタグ
     */
    function createLinkNext($active_message, $noactive_message='')
    {
        // 総レコード数0、次ページがない、または現在MAXページ目状態
        if (($this->sum <= 0) || ($this->getTotalPage() == 1) || ($this->getCurrentPage() == $this->getTotalPage())) {
            return (!$noactive_message ? $active_message : $noactive_message);
        // 総ページ数以上のページが指定されたら1ページ目リンクへ
        } else if ($this->getCurrentPage() > $this->getTotalPage()) {
            return '<a href="' . $this->url . '?' . $this->getBuildParameters(2) . '">' . $active_message . '</a>';
        } else {
            return '<a href="' . $this->url . '?' . $this->getBuildParameters($this->getCurrentPage() + 1) . '">' . $active_message . '</a>';
        }
    }

    /**
     * 前ページリンク取得
     *
     * @access public
     * @param string リンクメッセージ（リンクあり時）
     * @param string リンクメッセージ（リンク無し時）
     * @return string 前ページリンクタグ
     */
    function createLinkPrev($active_message, $noactive_message='')
    {
        // 総レコード数0、次ページがない、または現在1ページ目状態
        if (($this->sum <= 0) || ($this->getTotalPage() == 1) || ($this->getCurrentPage() == 1)) {
            return (!$noactive_message ? $active_message : $noactive_message);
        // 総ページ数以上のページが指定されたら1ページ目リンクへ
        } else if ($this->getCurrentPage() > $this->getTotalPage()) {
            return (!$noactive_message ? $active_message : $noactive_message);
        } else {
            return '<a href="' . $this->url . '?' . $this->getBuildParameters($this->getCurrentPage() - 1) . '">' . $active_message . '</a>';
        }
    }

    /**
     * 指定範囲ページリンク取得
     *
     * @access public
     * @param int ページリンク表示幅（0を指定すると全ページ）
     * @return array ページリンク配列
     */
    function createLinkRange($range=19)
    {
        list($start, $end) = $this->getRange($range);

        $pages = array();
        for ($i=$start; $i<=$end; $i++) {
            if ($i == $this->getCurrentPage()) {
                $pages[] = "<strong>{$i}</strong>";
            } else {
                $pages[] = '<a href="' . $this->url . '?' . $this->getBuildParameters($i) . '">' . $i . '</a>';
            }
        }

        return $pages;
    }

    /**
     * 指定範囲ページSelectタグのオプション取得
     *
     * @access public
     * @param int ページリンク表示幅（0を指定すると全ページ）
     * @return array 指定範囲ページSelectタグのオプション
     */
    function createSelectRange($range=19)
    {
        list($start, $end) = $this->getRange($range);

        $options = array();
        for ($i=$start; $i<=$end; $i++) {
            $active = ($i == $this->getCurrentPage()) ? 'selected' : '';
            $options[] = '<option value="' . $this->url . '?' . $this->getBuildParameters($i) . '" ' . $active . '>&nbsp;' . $i . "\r\n";
        }

        return $options;
    }

    /**
     * 指定範囲ページの最初と最後のページを取得
     *
     * @access public
     * @param int ページリンク表示幅（0を指定すると全ページ）
     * @return array 指定範囲ページの最初と最後のページ
     */
    function getRange($range)
    {
        $start = 1;
        $end   = 1;
        $current_page = $this->getCurrentPage();
        if ($range > 0) {
            $center_page = ceil($range/2);

            if (($current_page - $center_page) > 0) {
                $start = $current_page - $center_page + 1;
            }

            $end = (($current_page + $center_page) > $this->getTotalPage()) ? $this->getTotalPage() : $current_page + $center_page - 1;
        } else {
            $end = $this->getTotalPage();
        }
        return array($start, $end);
    }
}
