<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_XmlXslt.php,v 1.1 2009/01/11 05:34:31 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 * 汎用プロパティクラス
 */
require_once dirname(__FILE__) . '/../Util/SyL_UtilProperty.php';

/** 
 * XSLT適用クラス
 *
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_XmlXslt.php,v 1.1 2009/01/11 05:34:31 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_XmlXslt extends SyL_UtilProperty
{
    /**
     * XML
     *
     * @access private
     * @var string
     */
    var $xml = null;
    /**
     * XMLデータフラグ
     *
     * @access private
     * @var bool
     */
    var $xml_data_flag = false;
    /**
     * XSL
     *
     * @access private
     * @var string
     */
    var $xsl = null;
    /**
     * XSLデータフラグ
     *
     * @access private
     * @var bool
     */
    var $xsl_data_flag = false;
    /**
     * テンプレートのエンコード
     * 
     * @access private
     * @var string
     */
    var $xslt_encode = 'UTF-8';
    /**
     * 内部パラメータエンコード
     * 
     * @access protected
     * @var string
     */
    var $internal_encode = '';
    /**
     * 出力エンコーディング変換フラグ
     * 
     * @access protected
     * @var bool
     */
    var $output_convert_encoding = true;

    /**
     * コンストラクタ
     *
     * @access public
     * @param string XSLファイル
     * @param string XMLファイル
     */
    function SyL_XmlXslt($xsl_file='', $xml_file='')
    {
        if ($xsl_file) $this->setXsl($xsl_file);
        if ($xml_file) $this->setXml($xml_file);
        $this->internal_encode = mb_internal_encoding();
    }

    /**
     * XMLファイル or データをセット
     *
     * @access public
     * @param string XMLファイル or データ
     * @param bool データフラグ
     */
    function setXml($xml, $data_flag=false)
    {
        $this->xml = $xml;
        $this->xml_data_flag = $data_flag;
    }

    /**
     * XSLファイル or データをセット
     *
     * @access public
     * @param string XSLファイル or データ
     * @param bool データフラグ
     */
    function setXsl($xsl, $data_flag=false)
    {
        if (!$data_flag) {
            // 1行目にエンコーディングの定義があったら取得
            $fp = fopen($xsl, 'r');
            if (preg_match ('/<\?xml(.+)encoding=\"([^\"]+)\"(.*)\?>/i', fgets($fp, 4096), $matches)) {
                $this->setXsltEncode($matches[2]);
            }
            fclose($fp);
        }
        $this->xsl = $xsl;
        $this->xsl_data_flag = $data_flag;
    }

    /**
     * XSLTエンコードをセット
     *
     * @access public
     * @param string XSLTエンコード
     */
    function setXsltEncode($xslt_encode)
    {
        $this->xslt_encode = strtoupper($xslt_encode);
    }

    /**
     * 内部パラメータエンコードをセット
     *
     * @access public
     * @param string 内部パラメータエンコード
     */
    function setInternalEncode($internal_encode)
    {
        $this->internal_encode = strtoupper($internal_encode);
    }

    /**
     * 出力エンコーディング変換フラグをセット
     *
     * @access public
     * @param bool 出力エンコーディング変換フラグ
     */
    function setOutputConvertEncoding($output_convert_encoding)
    {
        $this->output_convert_encoding = (bool)$output_convert_encoding;
    }

    /**
     * XSLT適用を行う
     * ※出力ファイルパラメータを指定した場合は戻り値なし
     *
     * @access public
     * @param string 出力ファイル
     * @return mixed XSLT適用後データ
     */
    function transform($out_file=null)
    {
        if ($this->xslt_encode != $this->internal_encode) {
            $this->apply('mb_convert_encoding', $this->xslt_encode, $this->internal_encode);
        }

        $result = false;
        if (version_compare(PHP_VERSION, '5.0.0', '>=')) {
            $xslt = new XSLTProcessor();
            $doc  = new DOMDocument();

            if ($this->xsl_data_flag) {
                $doc->loadXML($this->xsl);
            } else {
                $doc->load($this->xsl);
            }
            $xslt->importStyleSheet($doc);

            if ($this->xml_data_flag) {
                $doc->loadXML($this->xml);
            } else {
                $doc->load($this->xml);
            }

            foreach ($this->gets() as $name => $value) {
                $xslt->setParameter('', $name, $value);
            }

            if ($out_file) {
                $result = $xslt->transformToURI($doc, $out_file);
            } else {
                $result = $xslt->transformToXML($doc);
            }
        } else {
            $xml  = '';
            $xsl  = '';
            $args = array();

            if ($this->xsl_data_flag) {
                $args['/_xsl'] = $this->xsl;
                $xsl = 'arg:/_xsl';
            } else {
                $xsl = $this->xsl;
            }

            if ($this->xml_data_flag) {
                $args['/_xml'] = $this->xml;
                $xml = 'arg:/_xml';
            } else {
                $xml = $this->xml;
            }

            $xslt = xslt_create();
            $result = xslt_process($xslt, $xml, $xsl, $out_file, $args, $this->gets());
            xslt_free($xslt);
        }

        if ($this->output_convert_encoding && is_string($result) && ($this->xslt_encode != $this->internal_encode)) {
            return mb_convert_encoding($result, $this->internal_encode, $this->xslt_encode);
        } else {
            return $result;
        }
    }

}
