/*******************************************************
 *
 * Author: Hirokazu Kato
 *
 *         kato@sys.im.hiroshima-cu.ac.jp
 *
 * Revision: 3.1
 * Date: 01/12/07
 *
*******************************************************/
#define CHECK_CALC 0

#include <stdlib.h>
#define _USE_MATH_DEFINES
#include <math.h>
#include <AR/ar.h>
#include <AR/matrix.h>

static int check_rotation( ARVAL rot[2][3] );
static int check_dir( ARVAL dir[3], ARVAL st[2], ARVAL ed[2], ARVAL cpara[3][4] );

//#define TRUNCATE_MATRIX
#define TRANCATE_VALUE		(~0xfULL)
#define TRANCATE_RADVALUE	(~0xfULL)
#define isTableARCSIN		0

extern const long sinMajorTbl[];	// sin(0)`sin(pi/2)  1/1024[rad] L30
extern const long sinMinorTbl[];	// sin(0)`sin(pi/2/1024)  1/(1024*1024)[rad] L30
extern const long cosMinorTbl[];	// cos(0)`cos(pi/2/1024)  1/(1024*1024)[rad] L30
// rad = 0`pi/2
static ARVAL lookupSinTbl( ARVAL rad )
{
	int a, b;
	long sina, sinb, cosa, cosb;
	long long r;
	ARDBL d = ARVALtoARDBL( rad );
	a = (int)(d >> (32-10));
	b = (((int)d) >> (32-10*2)) & ((1 << 10)-1);
	// @藝 sin(a+b) = sin(a)*cos(b) + sin(b)*cos(a)
	sina = sinMajorTbl[ a ];
	sinb = sinMinorTbl[ b ];
	cosa = sinMajorTbl[ ((int)(M_PI/2*1024)) - a ];
	cosb = cosMinorTbl[ b ];
	r  = sina * (long long)cosb;
	r += sinb * (long long)cosa;
	r >>= 30*2 - 32;
	r &= ~3ULL;	// mCYƂ
	return ARDBLtoARVAL( r );
}

static ARVAL ARVALsin( ARVAL rad )
{
	const ARVAL pi2 = DoubleToARVAL( M_PI*2 );
	const ARVAL pi  = DoubleToARVAL( M_PI );
	const ARVAL pih = DoubleToARVAL( M_PI/2 );

	if ( AbsARVAL( rad ) >= pi2 )
	{
		ARVAL r = DivARVAL( rad, pi2 );
		rad -= MulARVAL( r, pi2 );
	}
	if ( rad < 0 )
		rad += pi2;

	if ( rad <= pih )
		return lookupSinTbl( rad );
	else
	if ( rad <= pi )
		return lookupSinTbl( pi - rad );
	else
	if ( rad <= pi+pih )
		return -lookupSinTbl( rad - pi );
	else
		return -lookupSinTbl( pi2 - rad );
}
static ARVAL ARVALcos( ARVAL rad )
{
	return ARVALsin( DoubleToARVAL( M_PI/2 ) + rad );
}

extern const long acosTbl[];
static long tblacosL28( long xL16 )
{
	long x = xL16 >> 4;
	if ( x >= 1<<12  ||  x <= -1<<12 )
		return 0;
	return ( x >= 0 )? acosTbl[ x ] : (((long)(M_PI*(1<<28))) - acosTbl[ -x ]);
}

static ARVAL ARVALarcsin( ARVAL x )
{
	int i;
	long dx, cx, r;
	ARVAL x2;

	// TailerWJ̌W
	const int factor[ 8 ] = { 
		(long)((1LL<<32)/6),
		(long)((3LL<<32)/40),
		(long)((5LL<<32)/112),
		(long)((35LL<<32)/1152),
		(long)((63LL<<32)/2816),
		(long)((231LL<<32)/13312),
		(long)((143LL<<32)/10240),
		(long)((19305LL<<32)/1671168)
	};

	if ( x == 0 )
		return DoubleToARVAL( M_PI );
	if ( x == DoubleToARVAL( 0.5 ) )
		return DoubleToARVAL( M_PI/6 );
	if ( x >= IntToARVAL( 1 )  ||  x <= -IntToARVAL( 1 ) )
		return 0;
	if ( AbsARVAL( x ) >= DoubleToARVAL( 0.72 ) )
	{
		ARVAL y = SqrtARVAL( IntToARVAL( 1 ) - MulARVAL( x, x ) );
		y = DoubleToARVAL( M_PI/2 ) - ARVALarcsin( y );
		return (x > 0)? y : -y;
	}
	if ( isTableARCSIN )
		return DoubleToARVAL( M_PI/2 ) - ARDBLtoARVAL( (ARDBL)(tblacosL28( (long)(ARVALtoARDBL( x ) >> 16) ) ) << 4 );

	r = (long)(ARVALtoARDBL( x ) >> 1);	// L31
	cx = r;
	x2 = (long)((r * (long long)r) >> 31); // L31
	for ( i=0; i<8; i++ )
	{
		cx = (long)((cx * (long long)x2) >> 31);
		dx = (long)((factor[ i ] * (long long)cx) >> 32);
		r += dx;
	}
	return ARDBLtoARVAL( ((ARDBL)r) << 1 );
}
static ARVAL ARVALarccos( ARVAL x )
{
	return DoubleToARVAL( M_PI/2 ) - ARVALarcsin( x );
}

static long ARVALtoL16( ARVAL x )
{
#if (REALBITS_ARVAL <= 16 )
	return x << (16-REALBITS_ARVAL);
#else
	return (long)((x+((1<<15)+1)) >> (REALBITS_ARVAL-16));
#endif
}
static long ARVALtoL28( ARVAL x )
{
#if (REALBITS_ARVAL <= 28 )
	return x << (28-REALBITS_ARVAL);
#else
	return (long)((x+((1<<27)+1)) >> (REALBITS_ARVAL-28));
#endif
}
static ARVAL L28toARVAL( long x )
{
#if (REALBITS_ARVAL <= 28 )
	return (x+(1<<27)+1) >> (28-REALBITS_ARVAL);
#else
	return ((ARVAL)x) << (REALBITS_ARVAL-28);
#endif
}
static long ARVALtoL8( ARVAL x )
{
#if (REALBITS_ARVAL <= 8 )
	return x << (8-REALBITS_ARVAL);
#else
	return (long)((x+((1<<7)+1)) >> (REALBITS_ARVAL-8));
#endif
}
static ARVAL L15toARVAL( long x )
{
#if (REALBITS_ARVAL <= 15 )
	return (x+(1<<14)+1) >> (15-REALBITS_ARVAL);
#else
	return ((ARVAL)x) << (REALBITS_ARVAL-15);
#endif
}
			
int arGetAngle( ARVAL rot[3][3], ARVAL *wa, ARVAL *wb, ARVAL *wc )
{
    ARVAL      a, b, c;
    ARVAL      sina, cosa, sinb, cosb, sinc, cosc;
		ARVAL      rotdiv;
#if CHECK_CALC
double   w[3];
int      i;
for(i=0;i<3;i++) w[i] = rot[i][0];
for(i=0;i<3;i++) rot[i][0] = rot[i][1];
for(i=0;i<3;i++) rot[i][1] = rot[i][2];
for(i=0;i<3;i++) rot[i][2] = w[i];
#endif

		ARVAL pe = IntToARVAL( 1 );

    if( rot[2][2] > pe ) {
        /* printf("cos(beta) = %f\n", rot[2][2]); */
        rot[2][2] = pe;
    }
    else if( rot[2][2] < -pe ) {
        /* printf("cos(beta) = %f\n", rot[2][2]); */
        rot[2][2] = -pe;
    }
    cosb = rot[2][2];
    b = ARVALarccos( cosb );
    sinb = ARVALsin( b );
#ifdef _DEBUG
{
	double fb = fmod( acos( ARVALtoDouble( cosb ) ), M_PI );
  double fsinb = sin( fb );
	double fb2 = ARVALtoDouble( ARVALarccos( cosb ) );
	double fsinb2 = ARVALtoDouble( ARVALsin( DoubleToARVAL( fb2 ) ) );
	double d1 = fb - fb2;
	double d2 = fsinb - fsinb2;
	int i=1;
}
#endif
		rotdiv = MulARVAL( rot[0][2], rot[0][2] ) + MulARVAL( rot[1][2], rot[1][2] );
    //if ( AbsARVAL( sinb ) >= DoubleToARVAL( 0.000001 ) ) {
    if (  AbsARVAL( sinb ) >= DoubleToARVAL( 0.001 ) // sinbقroťł
			&&  rotdiv >= DoubleToARVAL( 0.001 ) )
		{
			  cosa = DivARVAL( rot[0][2], sinb );
        sina = DivARVAL( rot[1][2], sinb );
        if( cosa > pe ) {
            /* printf("cos(alph) = %f\n", cosa); */
            cosa = pe;
            sina = 0;
        }
        if( cosa < -pe ) {
            /* printf("cos(alph) = %f\n", cosa); */
            cosa = -pe;
            sina =  0;
        }
        if( sina > pe ) {
            /* printf("sin(alph) = %f\n", sina); */
            sina = pe;
            cosa = 0;
        }
        if( sina < -pe ) {
            /* printf("sin(alph) = %f\n", sina); */
            sina = -pe;
            cosa =  0;
        }
        a = ARVALarccos( cosa );
        if( sina < 0 ) a = -a;

        sinc =  DivARVAL( MulARVAL( rot[2][1], rot[0][2] ) - MulARVAL( rot[2][0], rot[1][2] ), rotdiv );
        cosc = -DivARVAL( MulARVAL( rot[0][2], rot[2][0] ) + MulARVAL( rot[1][2], rot[2][1] ), rotdiv );
        if( cosc > pe ) {
            /* printf("cos(r) = %f\n", cosc); */
            cosc = pe;
            sinc = 0;
        }
        if( cosc < -pe ) {
            /* printf("cos(r) = %f\n", cosc); */
            cosc = -pe;
            sinc =  0;
        }
        if( sinc > pe ) {
            /* printf("sin(r) = %f\n", sinc); */
            sinc = pe;
            cosc = 0;
        }
        if( sinc < -pe ) {
            /* printf("sin(r) = %f\n", sinc); */
            sinc = -pe;
            cosc =  0;
        }
        c = ARVALarccos( cosc );
        if( sinc < 0 ) c = -c;
    }
    else {
        a = b = 0;
        cosa = cosb = pe;
        sina = sinb = 0;
        cosc = rot[0][0];
        sinc = rot[1][0];
        if( cosc > pe ) {
            /* printf("cos(r) = %f\n", cosc); */
            cosc = pe;
            sinc = 0;
        }
        if( cosc < -pe ) {
            /* printf("cos(r) = %f\n", cosc); */
            cosc = -pe;
            sinc =  0;
        }
        if( sinc > pe ) {
            /* printf("sin(r) = %f\n", sinc); */
            sinc = pe;
            cosc = 0;
        }
        if( sinc < -pe ) {
            /* printf("sin(r) = %f\n", sinc); */
            sinc = -pe;
            cosc =  0;
        }
        c = ARVALarccos( cosc );
        if( sinc < 0 ) c = -c;
    }

    *wa = a;
    *wb = b;
    *wc = c;

#ifdef TRUNCATE_MATRIX
		*wa &= TRANCATE_RADVALUE;
		*wb &= TRANCATE_RADVALUE;
		*wc &= TRANCATE_RADVALUE;
#endif
    return 0;
}

int arGetRot( ARVAL a, ARVAL b, ARVAL c, ARVAL rot[3][3] )
{
    long   sina, sinb, sinc;
    long   cosa, cosb, cosc;
		ARDBL x;
#if CHECK_CALC
    ARVAL   w[3];
    int      i;
#endif

    sina = (long)(ARVALtoARDBL( ARVALsin( a ) ) >> 4);   cosa = (long)(ARVALtoARDBL( ARVALcos( a ) ) >> 4);
    sinb = (long)(ARVALtoARDBL( ARVALsin( b ) ) >> 4);   cosb = (long)(ARVALtoARDBL( ARVALcos( b ) ) >> 4);
    sinc = (long)(ARVALtoARDBL( ARVALsin( c ) ) >> 4);   cosc = (long)(ARVALtoARDBL( ARVALcos( c ) ) >> 4);
		
		x  = productL28x4toARDBL( cosa, cosa, cosb, cosc );
		x += productL28x3toARDBL( sina, sina, cosc );
		x += productL28x4toARDBL( sina, cosa, cosb, sinc );
		x += -productL28x3toARDBL( sina, cosa, sinc );
		rot[0][0] = ARDBLtoARVAL( x );

		x  = -productL28x4toARDBL( cosa, cosa, cosb, sinc );
		x += -productL28x3toARDBL( sina, sina, sinc );
		x += productL28x4toARDBL( sina, cosa, cosb, cosc );
		x += -productL28x3toARDBL( sina, cosa, cosc );
		rot[0][1] = ARDBLtoARVAL( x );
		rot[0][2] = ARDBLtoARVAL( productL28x2toARDBL( cosa, sinb ) );
		
		x  = productL28x4toARDBL( sina, cosa, cosb, cosc );
		x += -productL28x3toARDBL( sina, cosa, cosc );
		x += productL28x4toARDBL( sina, sina, cosb, sinc );
		x += productL28x3toARDBL( cosa, cosa, sinc );
		rot[1][0] = ARDBLtoARVAL( x );
		
		x  = -productL28x4toARDBL( sina, cosa, cosb, sinc );
		x += productL28x3toARDBL( sina, cosa, sinc );
		x += productL28x4toARDBL( sina, sina, cosb, cosc );
		x += productL28x3toARDBL( cosa, cosa, cosc );
		rot[1][1] = ARDBLtoARVAL( x );
    rot[1][2] = ARDBLtoARVAL( productL28x2toARDBL( sina, sinb ) );

		x  = -productL28x3toARDBL( cosa, sinb, cosc );
		x += -productL28x3toARDBL( sina, sinb, sinc );
		rot[2][0] = ARDBLtoARVAL( x );
		
		x  = productL28x3toARDBL( cosa, sinb, sinc );
		x += -productL28x3toARDBL( sina, sinb, cosc );
		rot[2][1] = ARDBLtoARVAL( x );
		
    rot[2][2] = L28toARVAL( cosb );

#if CHECK_CALC
    for(i=0;i<3;i++) w[i] = rot[i][2];
    for(i=0;i<3;i++) rot[i][2] = rot[i][1];
    for(i=0;i<3;i++) rot[i][1] = rot[i][0];
    for(i=0;i<3;i++) rot[i][0] = w[i];
#endif

#ifdef TRUNCATE_MATRIX
		{
			int i,k;
			for ( i=0; i<3; i++ )
			{
				for ( k=0; k<3; k++ )
					rot[i][k] &= TRANCATE_VALUE;
			}
		}
#endif
    return 0;
}

int arGetNewMatrix( ARVAL a, ARVAL b, ARVAL c,
                    ARVAL trans[3], ARVAL trans2[3][4],
                    ARVAL cpara[3][4], ARVAL ret[3][4] )
{
    ARVAL   cpara2[3][4];
    ARVAL   rot[3][3];
    int      i, j;

    arGetRot( a, b, c, rot );

    if( trans2 != NULL ) {
        for( j = 0; j < 3; j++ ) {
            for( i = 0; i < 4; i++ ) {
                cpara2[j][i] = MulARVAL( cpara[j][0], trans2[0][i] )
                             + MulARVAL( cpara[j][1], trans2[1][i] )
                             + MulARVAL( cpara[j][2], trans2[2][i] );
            }
        }
    }
    else {
        for( j = 0; j < 3; j++ ) {
            for( i = 0; i < 4; i++ ) {
                cpara2[j][i] = cpara[j][i];
            }
        }
    }

    for( j = 0; j < 3; j++ ) {
        for( i = 0; i < 3; i++ ) {
            ret[j][i] = MulARVAL( cpara2[j][0], rot[0][i] )
                      + MulARVAL( cpara2[j][1], rot[1][i] )
                      + MulARVAL( cpara2[j][2], rot[2][i] );
        }
        ret[j][3] = MulARVAL( cpara2[j][0], trans[0] )
                  + MulARVAL( cpara2[j][1], trans[1] )
                  + MulARVAL( cpara2[j][2], trans[2] )
                  + cpara2[j][3];
    }

#ifdef TRUNCATE_MATRIX
		{
			int i,k;
			for ( i=0; i<3; i++ )
			{
				for ( k=0; k<4; k++ )
					ret[i][k] &= TRANCATE_VALUE;
			}
		}
#endif
    return(0);
}

// ԑ傫l擾
static ARVAL* getLargest( ARVAL* v0, ARVAL* v1, ARVAL* v2 )
{
	ARVAL a0 = *v0, a1 = *v1, a2 = *v2;
	if ( a0 > a1 )
	{
		if ( a0 > a2 )
			return v0;
		else
			return v2;
	}
	if ( a1 > a2 )
		return v1;
	return v2;
}

ARVAL vectorNorm( ARVAL x, ARVAL y, ARVAL z )
{
	ARVAL argmax = IntToARVAL( 1LL<<(((sizeof(ARVAL)*8-REALBITS_ARVAL-2) - 2)/2) );
	int shift = 0;
	ARVAL r;
	ARVAL ux = AbsARVAL( x );
	ARVAL uy = AbsARVAL( y );
	ARVAL uz = AbsARVAL( z );
	//return DoubleToARVAL( sqrt( ARVALtoDouble( x )*ARVALtoDouble( x ) + ARVALtoDouble( y )*ARVALtoDouble( y ) + ARVALtoDouble( z )*ARVALtoDouble( z ) ) );

	if ( ux >= argmax  ||  uy >= argmax  ||  uz >= argmax )
	{
		while ( ux >= argmax  ||  uy >= argmax  ||  uz >= argmax )
		{
			ux >>= 1;
			uy >>= 1;
			uz >>= 1;
			shift++;
		}
		x >>= shift;
		y >>= shift;
		z >>= shift;
	}
	else
	{
		argmax >>= 1;
		while ( ((ux < argmax) && ux)  &&  ((uy < argmax) && uy)  &&  ((uz < argmax) && uz) )
		{
			ux <<= 1;
			uy <<= 1;
			uz <<= 1;
			shift--;
		}
		x <<= -shift;
		y <<= -shift;
		z <<= -shift;
	}

	r = SqrtARVAL( MulARVAL( x, x ) + MulARVAL( y, y ) + MulARVAL( z, z ) );
	if ( shift > 0 )
		r <<= shift;
	else
		r >>= -shift;
	return r;
}

ARVAL vector2Norm( ARVAL x, ARVAL y )
{
	ARVAL argmax = IntToARVAL( 1LL<<(((sizeof(ARVAL)*8-REALBITS_ARVAL-2) - 1)/2) );
	int shift = 0;
	ARVAL r;
	ARVAL ux = AbsARVAL( x );
	ARVAL uy = AbsARVAL( y );

	if ( ux >= argmax  ||  uy >= argmax )
	{
		while ( ux >= argmax  ||  uy >= argmax )
		{
			ux >>= 1;
			uy >>= 1;
			shift++;
		}
		x >>= shift;
		y >>= shift;
	}
	else
	{
		argmax >>= 1;
		while ( ((ux < argmax) && ux)  &&  ((uy < argmax) && uy) )
		{
			ux <<= 1;
			uy <<= 1;
			shift--;
		}
		x <<= -shift;
		y <<= -shift;
	}

	r = SqrtARVAL( MulARVAL( x, x ) + MulARVAL( y, y ) );
	if ( shift > 0 )
		r <<= shift;
	else
		r >>= -shift;
	return r;
}

// _ȉ̐xKvȂꍇ x*x + y*y
ARVAL vector2Norm2( ARVAL x, ARVAL y )
{
	//return MulARVAL( x, x ) + MulARVAL( y, y );
	int ix = ARVALtoInt( x );
	int iy = ARVALtoInt( y );
	return IntToARVAL( ix*ix + iy*iy );
}

// sqrt( x*x - ya*yb )
static ARVAL vectorTest1( ARVAL x, int sftx, ARVAL ya, int sftya, ARVAL yb, int sftyb )
{
	ARVAL argmax = IntToARVAL( 1LL<<(((sizeof(ARVAL)*8-REALBITS_ARVAL-2) - 2)/2) );
	int shift = 0, sign = 1;
	ARVAL r;
	ARVAL ux = AbsARVAL( x );
	ARVAL uya = AbsARVAL( ya );
	ARVAL uyb = AbsARVAL( yb );
	if ( ya < 0 )
		sign = -sign;
	if ( yb < 0 )
		sign = -sign;

	if ( sftx > sftya )
		shift = ( sftx > sftyb )? sftx : sftyb;
	else
		shift = ( sftya > sftyb )? sftya : sftyb;
	x >>= shift - sftx;
	ya >>= shift - sftya;
	yb >>= shift - sftyb;
		
	if ( ux >= argmax  ||  uya >= argmax  ||  uyb >= argmax  )
	{
		while ( ux >= argmax  ||  uya >= argmax  ||  uyb >= argmax  )
		{
			ux >>= 1;
			uya >>= 1;
			uyb >>= 1;
			shift++;
		}
		x >>= shift;
		ya >>= shift;
		yb >>= shift;
	}
	else
	{
		argmax >>= 1;
		while ( ((ux < argmax) && ux)  &&  ((uya < argmax) && uya)  &&  ((uyb < argmax) && uyb) )
		{
			ux <<= 1;
			uya <<= 1;
			uyb <<= 1;
			shift--;
		}
		x <<= -shift;
		ya <<= -shift;
		yb <<= -shift;
	}

	r = MulARVAL( x, x ) - (sign > 0)?MulARVAL( ya, yb ):-MulARVAL( ya, yb );
	if ( r >= 0LL )
	{
		r = SqrtARVAL( r );
		if ( shift > 0 )
			r <<= shift;
		else
			r >>= -shift;
	}
	return r;
}

// x*x + y*y + u
static ARVAL vectorTest2( ARVAL x, ARVAL y, int* pshift, int u )
{
	ARVAL argmax = IntToARVAL( 1LL<<(((sizeof(ARVAL)*8-REALBITS_ARVAL-2) - 1)/2) );
	int shift = 0;
	ARVAL ux = AbsARVAL( x );
	ARVAL uy = AbsARVAL( y );

	if ( ux >= argmax  ||  uy >= argmax )
	{
		while ( ux >= argmax  ||  uy >= argmax )
		{
			ux >>= 1;
			uy >>= 1;
			shift++;
		}
		x >>= shift;
		y >>= shift;
	}
	*pshift = shift*2;
	return MulARVAL( x, x ) + MulARVAL( y, y ) + (IntToARVAL( u ) >> (shift*2));
}

// a*b + c*d
static ARVAL vectorTest3( ARVAL a, ARVAL b, ARVAL c, ARVAL d, int* pshift )
{
	ARVAL argmax = IntToARVAL( 1LL<<(((sizeof(ARVAL)*8-REALBITS_ARVAL-2) - 2)/2) );
	int shift = 0;
	ARVAL ua = AbsARVAL( a );
	ARVAL ub = AbsARVAL( b );
	ARVAL uc = AbsARVAL( c );
	ARVAL ud = AbsARVAL( d );

	if ( ua >= argmax  ||  ub >= argmax  ||  uc >= argmax  ||  ud >= argmax )
	{
		while ( ua >= argmax  ||  ub >= argmax  ||  uc >= argmax  ||  ud >= argmax )
		{
			ua >>= 1;
			ub >>= 1;
			uc >>= 1;
			ud >>= 1;
			shift++;
		}
		a >>= shift;
		b >>= shift;
		c >>= shift;
		d >>= shift;
	}
	*pshift = shift*2;
	return MulARVAL( a, b ) + MulARVAL( c, d );
}


int arGetInitRot( ARMarkerInfo *marker_info, ARVAL cpara[3][4], ARVAL rot[3][3] )
{
    ARVAL  wdir[3][3];
    ARVAL  w, w1, w2, w3;
    int     dir;
    int     j;

    dir = marker_info->dir;

    for( j = 0; j < 2; j++ ) {
        w1 = MulARVAL( marker_info->line[(4-dir+j)&3][0], marker_info->line[(6-dir+j)&3][1] )
           - MulARVAL( marker_info->line[(6-dir+j)&3][0], marker_info->line[(4-dir+j)&3][1] );
        w2 = MulARVAL( marker_info->line[(4-dir+j)&3][1], marker_info->line[(6-dir+j)&3][2] )
           - MulARVAL( marker_info->line[(6-dir+j)&3][1], marker_info->line[(4-dir+j)&3][2] );
        w3 = MulARVAL( marker_info->line[(4-dir+j)&3][2], marker_info->line[(6-dir+j)&3][0] )
           - MulARVAL( marker_info->line[(6-dir+j)&3][2], marker_info->line[(4-dir+j)&3][0] );

        wdir[j][0] =  MulARVAL( w1, (MulARVAL( cpara[0][1], cpara[1][2] ) - MulARVAL( cpara[0][2], cpara[1][1] )) )
                   +  MulARVAL( w2, cpara[1][1] )
                   -  MulARVAL( w3, cpara[0][1] );
        wdir[j][1] = -MulARVAL( MulARVAL( w1, cpara[0][0] ), cpara[1][2] )
                   +  MulARVAL( w3, cpara[0][0] );
        wdir[j][2] =  MulARVAL( MulARVAL( w1, cpara[0][0] ), cpara[1][1] );


        w = vectorNorm( wdir[j][0], wdir[j][1], wdir[j][2] );
        wdir[j][0] = DivARVAL( wdir[j][0], w );
        wdir[j][1] = DivARVAL( wdir[j][1], w );
        wdir[j][2] = DivARVAL( wdir[j][2], w );
    }

    if( check_dir(wdir[0], marker_info->vertex[(4-dir)&3],
                  marker_info->vertex[(5-dir)&3], cpara) < 0 ) return -1;
    if( check_dir(wdir[1], marker_info->vertex[(7-dir)&3],
                  marker_info->vertex[(4-dir)&3], cpara) < 0 ) return -1;
    if( check_rotation(wdir) < 0 ) return -1;

    wdir[2][0] = MulARVAL( wdir[0][1], wdir[1][2] ) - MulARVAL( wdir[0][2], wdir[1][1] );
    wdir[2][1] = MulARVAL( wdir[0][2], wdir[1][0] ) - MulARVAL( wdir[0][0], wdir[1][2] );
    wdir[2][2] = MulARVAL( wdir[0][0], wdir[1][1] ) - MulARVAL( wdir[0][1], wdir[1][0] );
    w = vectorNorm( wdir[2][0], wdir[2][1], wdir[2][2] );
		wdir[2][0] = DivARVAL( wdir[2][0], w );
		wdir[2][1] = DivARVAL( wdir[2][1], w );
		wdir[2][2] = DivARVAL( wdir[2][2], w );
/*
    if( wdir[2][2] < 0 ) {
        wdir[2][0] /= -w;
        wdir[2][1] /= -w;
        wdir[2][2] /= -w;
    }
    else {
        wdir[2][0] /= w;
        wdir[2][1] /= w;
        wdir[2][2] /= w;
    }
*/

    rot[0][0] = wdir[0][0];
    rot[1][0] = wdir[0][1];
    rot[2][0] = wdir[0][2];
    rot[0][1] = wdir[1][0];
    rot[1][1] = wdir[1][1];
    rot[2][1] = wdir[1][2];
    rot[0][2] = wdir[2][0];
    rot[1][2] = wdir[2][1];
    rot[2][2] = wdir[2][2];

#ifdef TRUNCATE_MATRIX
		{
			int i,k;
			for ( i=0; i<3; i++ )
			{
				for ( k=0; k<3; k++ )
					rot[i][k] &= TRANCATE_VALUE;
			}
		}
#endif
    return 0;
}

static int check_dir( ARVAL dir[3], ARVAL st[2], ARVAL ed[2],
                      ARVAL cpara[3][4] )
{
    ARMat    mat_a;
    ARVAL    mat_a_ml[ 3*3 ];
    ARVAL    world[2][3];
    ARVAL    camera[2][2];
    ARVAL    v[2][2];
    ARVAL    h;
    int       i, j;

    mat_a.ml = mat_a_ml;
    mat_a.row = 3;
    mat_a.clm = 3;
    for (j=0;j<3;j++) 
    	for (i=0;i<3;i++) 
    		mat_a.ml[j*3+i] = cpara[j][i];
    arMatrixSelfInv( &mat_a );
    world[0][0] = ( MulARVAL( mat_a.ml[0], st[0] ) + MulARVAL( mat_a.ml[1], st[1] ) + mat_a.ml[2] )*10;
    world[0][1] = ( MulARVAL( mat_a.ml[3], st[0] ) + MulARVAL( mat_a.ml[4], st[1] ) + mat_a.ml[5] )*10;
    world[0][2] = ( MulARVAL( mat_a.ml[6], st[0] ) + MulARVAL( mat_a.ml[7], st[1] ) + mat_a.ml[8] )*10;
    world[1][0] = world[0][0] + dir[0];
    world[1][1] = world[0][1] + dir[1];
    world[1][2] = world[0][2] + dir[2];

    for( i = 0; i < 2; i++ ) {
        h = MulARVAL( cpara[2][0], world[i][0] )
          + MulARVAL( cpara[2][1], world[i][1] )
          + MulARVAL( cpara[2][2], world[i][2] );
        if( h == 0 ) return -1;
        camera[i][0] = DivARVAL(
													MulARVAL( cpara[0][0], world[i][0] )
												+ MulARVAL( cpara[0][1], world[i][1] )
												+ MulARVAL( cpara[0][2], world[i][2] ), h );
        camera[i][1] = DivARVAL( 
													MulARVAL( cpara[1][0], world[i][0] )
												+ MulARVAL( cpara[1][1], world[i][1] )
												+ MulARVAL( cpara[1][2], world[i][2] ), h );
    }

    v[0][0] = ed[0] - st[0];
    v[0][1] = ed[1] - st[1];
    v[1][0] = camera[1][0] - camera[0][0];
    v[1][1] = camera[1][1] - camera[0][1];

    if( MulARVAL( v[0][0], v[1][0] ) + MulARVAL( v[0][1], v[1][1] ) < 0 ) {
        dir[0] = -dir[0];
        dir[1] = -dir[1];
        dir[2] = -dir[2];
    }

    return 0;
}

static int check_rotation( ARVAL rot[2][3] )
{
    ARVAL  v1[3], v2[3], v3[3];
    ARVAL  ca, cb, k1, k2, k3, k4;
    ARVAL  a, b, c, d;
    ARVAL  p1, q1, r1;
    ARVAL  p2, q2, r2;
    ARVAL  p3, q3, r3;
    ARVAL  p4, q4, r4;
    ARVAL  w;
    ARVAL  e1, e2, e3, e4;
    int     f, sfta, sftb, sftc;

    v1[0] = rot[0][0];
    v1[1] = rot[0][1];
    v1[2] = rot[0][2];
    v2[0] = rot[1][0];
    v2[1] = rot[1][1];
    v2[2] = rot[1][2];
    v3[0] = MulARVAL( v1[1], v2[2] ) - MulARVAL( v1[2], v2[1] );
    v3[1] = MulARVAL( v1[2], v2[0] ) - MulARVAL( v1[0], v2[2] );
    v3[2] = MulARVAL( v1[0], v2[1] ) - MulARVAL( v1[1], v2[0] );
    w = vectorNorm( v3[0], v3[1], v3[2] );
    if( w == 0 ) return -1;
    v3[0] = DivARVAL( v3[0], w );
    v3[1] = DivARVAL( v3[1], w );
    v3[2] = DivARVAL( v3[2], w );

    cb = MulARVAL( v1[0], v2[0] ) + MulARVAL( v1[1], v2[1] ) + MulARVAL( v1[2], v2[2] );
    cb = AbsARVAL( cb );
    ca = (SqrtARVAL( cb + IntToARVAL( 1 ) ) + SqrtARVAL( IntToARVAL( 1 ) - cb )) >> 1;

    if( MulARVAL( v3[1], v1[0] ) - MulARVAL( v1[1], v3[0] ) != 0 ) {
        f = 0;
    }
    else {
        if( MulARVAL( v3[2], v1[0] ) - MulARVAL( v1[2], v3[0] ) != 0 ) {
            w = v1[1]; v1[1] = v1[2]; v1[2] = w;
            w = v3[1]; v3[1] = v3[2]; v3[2] = w;
            f = 1;
        }
        else {
            w = v1[0]; v1[0] = v1[2]; v1[2] = w;
            w = v3[0]; v3[0] = v3[2]; v3[2] = w;
            f = 2;
        }
    }
    if( MulARVAL( v3[1], v1[0] ) - MulARVAL( v1[1], v3[0] ) == 0 ) return -1;
    k1 = DivARVAL( MulARVAL( v1[1], v3[2] ) - MulARVAL( v3[1], v1[2] ), MulARVAL( v3[1], v1[0] ) - MulARVAL( v1[1], v3[0] ));
    k2 = DivARVAL( MulARVAL( v3[1], ca ), MulARVAL( v3[1], v1[0] ) - MulARVAL( v1[1],v3[0] ));
    k3 = DivARVAL( MulARVAL( v1[0], v3[2] ) - MulARVAL( v3[0], v1[2] ), MulARVAL( v3[0], v1[1] ) - MulARVAL( v1[0], v3[1] ));
    k4 = DivARVAL( MulARVAL( v3[0], ca ), MulARVAL( v3[0], v1[1] ) - MulARVAL( v1[0], v3[1] ));

#if 0
		sfta = sftb = sftc = 0;
		a = MulARVAL( k1, k1 ) + MulARVAL( k3, k3 ) + IntToARVAL( 1 );
    b = MulARVAL( k1, k2 ) + MulARVAL( k3, k4 );
    c = MulARVAL( k2, k2 ) + MulARVAL( k4, k4 ) - IntToARVAL( 1 );
    d = vectorTest1( b, sftb, a, sfta, c, sftc ); //MulARVAL( b, b ) - MulARVAL( a, c );
#else
    a = vectorTest2( k1, k3, &sfta, 1 ); //MulARVAL( k1, k1 ) + MulARVAL( k3, k3 ) + IntToARVAL( 1 );
    b = vectorTest3( k1, k2, k3, k4, &sftb ); //MulARVAL( k1, k2 ) + MulARVAL( k3, k4 );
    c = vectorTest2( k2, k4, &sftc, -1 ); //MulARVAL( k2, k2 ) + MulARVAL( k4, k4 ) - IntToARVAL( 1 );
    d = vectorTest1( b, sftb, a, sfta, c, sftc ); //MulARVAL( b, b ) - MulARVAL( a, c );
#endif
    if( d < 0 ) return -1;
#ifdef _DEBUG
		//if ( sfta || sftb || sftc ) dbgBreak();
		//if ( (a<<sfta)>>sfta != a ) dbgBreak();
		//if ( (b<<sftb)>>sftb != b ) dbgBreak();
		//if ( (c<<sftc)>>sftc != c ) dbgBreak();
		//if ( (a<<sfta) != MulARVAL( k1, k1 ) + MulARVAL( k3, k3 ) + IntToARVAL( 1 ) ) dbgBreak();
		//if ( (b<<sftb) != MulARVAL( k1, k2 ) + MulARVAL( k3, k4 ) ) dbgBreak();
		//if ( (c<<sftc) != MulARVAL( k2, k2 ) + MulARVAL( k4, k4 ) - IntToARVAL( 1 ) ) dbgBreak();
		//if ( d != vectorTest1( b, 0, a, 0, c, 0 ) ) dbgBreak();
#endif

    r1 = DivARVAL(-(b << sftb) + d, (a << sfta) );
    p1 = MulARVAL( k1, r1 ) + k2;
    q1 = MulARVAL( k3, r1 ) + k4;
    r2 = DivARVAL(-(b << sftb) - d, (a << sfta) );
    p2 = MulARVAL( k1, r2 ) + k2;
    q2 = MulARVAL( k3, r2 ) + k4;
    if( f == 1 ) {
        w = q1; q1 = r1; r1 = w;
        w = q2; q2 = r2; r2 = w;
        w = v1[1]; v1[1] = v1[2]; v1[2] = w;
        w = v3[1]; v3[1] = v3[2]; v3[2] = w;
        f = 0;
    }
    if( f == 2 ) {
        w = p1; p1 = r1; r1 = w;
        w = p2; p2 = r2; r2 = w;
        w = v1[0]; v1[0] = v1[2]; v1[2] = w;
        w = v3[0]; v3[0] = v3[2]; v3[2] = w;
        f = 0;
    }

    if( MulARVAL( v3[1], v2[0] ) - MulARVAL( v2[1], v3[0] ) != 0 ) {
        f = 0;
    }
    else {
        if( MulARVAL( v3[2], v2[0] ) - MulARVAL( v2[2], v3[0] ) != 0 ) {
            w = v2[1]; v2[1] = v2[2]; v2[2] = w;
            w = v3[1]; v3[1] = v3[2]; v3[2] = w;
            f = 1;
        }
        else {
            w = v2[0]; v2[0] = v2[2]; v2[2] = w;
            w = v3[0]; v3[0] = v3[2]; v3[2] = w;
            f = 2;
        }
    }
    if( MulARVAL( v3[1], v2[0] ) - MulARVAL( v2[1], v3[0] ) == 0 ) return -1;
    k1 = DivARVAL( MulARVAL( v2[1], v3[2] ) - MulARVAL( v3[1], v2[2] ), MulARVAL( v3[1], v2[0] ) - MulARVAL( v2[1], v3[0] ));
    k2 = DivARVAL( MulARVAL( v3[1], ca), (MulARVAL( v3[1], v2[0] ) - MulARVAL( v2[1], v3[0] )) );
    k3 = DivARVAL( MulARVAL( v2[0], v3[2] ) - MulARVAL( v3[0], v2[2]), MulARVAL( v3[0], v2[1] ) - MulARVAL( v2[0], v3[1] ));
    k4 = DivARVAL( MulARVAL( v3[0], ca ), MulARVAL( v3[0], v2[1] ) - MulARVAL( v2[0], v3[1] ));

#if 0
		sfta = sftb = sftc = 0;
		a = MulARVAL( k1, k1 ) + MulARVAL( k3, k3 ) + IntToARVAL( 1 );
    b = MulARVAL( k1, k2 ) + MulARVAL( k3, k4 );
    c = MulARVAL( k2, k2 ) + MulARVAL( k4, k4 ) - IntToARVAL( 1 );
    d = vectorTest1( b, sftb, a, sfta, c, sftc ); //MulARVAL( b, b ) - MulARVAL( a, c );
#else
		a = vectorTest2( k1, k3, &sfta, 1 ); //MulARVAL( k1, k1 ) + MulARVAL( k3, k3 ) + IntToARVAL( 1 );
    b = vectorTest3( k1, k2, k3, k4, &sftb ); //MulARVAL( k1, k2 ) + MulARVAL( k3, k4 );
    c = vectorTest2( k2, k4, &sftc, -1 ); //MulARVAL( k2, k2 ) + MulARVAL( k4, k4 ) - IntToARVAL( 1 );
    d = vectorTest1( b, sftb, a, sfta, c, sftc ); //MulARVAL( b, b ) - MulARVAL( a, c );
#endif
    if( d < 0 ) return -1;
#ifdef _DEBUG
//if ( sfta || sftb || sftc ) dbgBreak();
//if ( (a<<sfta)>>sfta != a ) dbgBreak();
//if ( (b<<sftb)>>sftb != b ) dbgBreak();
//if ( (c<<sftc)>>sftc != c ) dbgBreak();
//if ( (a<<sfta) != MulARVAL( k1, k1 ) + MulARVAL( k3, k3 ) + IntToARVAL( 1 ) ) dbgBreak();
//if ( (b<<sftb) != MulARVAL( k1, k2 ) + MulARVAL( k3, k4 ) ) dbgBreak();
//if ( (c<<sftc) != MulARVAL( k2, k2 ) + MulARVAL( k4, k4 ) - IntToARVAL( 1 ) ) dbgBreak();
//if ( d != vectorTest1( b, 0, a, 0, c, 0 ) ) dbgBreak();
#endif

    r3 = DivARVAL(-(b << sftb) + d, (a << sfta) );
    p3 = MulARVAL( k1, r3 ) + k2;
    q3 = MulARVAL( k3, r3 ) + k4;
    r4 = DivARVAL(-(b << sftb) - d, (a << sfta) );
    p4 = MulARVAL( k1, r4 ) + k2;
    q4 = MulARVAL( k3, r4 ) + k4;
    if( f == 1 ) {
        w = q3; q3 = r3; r3 = w;
        w = q4; q4 = r4; r4 = w;
        w = v2[1]; v2[1] = v2[2]; v2[2] = w;
        w = v3[1]; v3[1] = v3[2]; v3[2] = w;
        f = 0;
    }
    if( f == 2 ) {
        w = p3; p3 = r3; r3 = w;
        w = p4; p4 = r4; r4 = w;
        w = v2[0]; v2[0] = v2[2]; v2[2] = w;
        w = v3[0]; v3[0] = v3[2]; v3[2] = w;
        f = 0;
    }

    e1 = MulARVAL( p1, p3 ) + MulARVAL( q1, q3 ) + MulARVAL( r1, r3 ); if( e1 < 0 ) e1 = -e1;
    e2 = MulARVAL( p1, p4 ) + MulARVAL( q1, q4 ) + MulARVAL( r1, r4 ); if( e2 < 0 ) e2 = -e2;
    e3 = MulARVAL( p2, p3 ) + MulARVAL( q2, q3 ) + MulARVAL( r2, r3 ); if( e3 < 0 ) e3 = -e3;
    e4 = MulARVAL( p2, p4 ) + MulARVAL( q2, q4 ) + MulARVAL( r2, r4 ); if( e4 < 0 ) e4 = -e4;
    if( e1 < e2 ) {
        if( e1 < e3 ) {
            if( e1 < e4 ) {
                rot[0][0] = p1;
                rot[0][1] = q1;
                rot[0][2] = r1;
                rot[1][0] = p3;
                rot[1][1] = q3;
                rot[1][2] = r3;
            }
            else {
                rot[0][0] = p2;
                rot[0][1] = q2;
                rot[0][2] = r2;
                rot[1][0] = p4;
                rot[1][1] = q4;
                rot[1][2] = r4;
            }
        }
        else {
            if( e3 < e4 ) {
                rot[0][0] = p2;
                rot[0][1] = q2;
                rot[0][2] = r2;
                rot[1][0] = p3;
                rot[1][1] = q3;
                rot[1][2] = r3;
            }
            else {
                rot[0][0] = p2;
                rot[0][1] = q2;
                rot[0][2] = r2;
                rot[1][0] = p4;
                rot[1][1] = q4;
                rot[1][2] = r4;
            }
        }
    }
    else {
        if( e2 < e3 ) {
            if( e2 < e4 ) {
                rot[0][0] = p1;
                rot[0][1] = q1;
                rot[0][2] = r1;
                rot[1][0] = p4;
                rot[1][1] = q4;
                rot[1][2] = r4;
            }
            else {
                rot[0][0] = p2;
                rot[0][1] = q2;
                rot[0][2] = r2;
                rot[1][0] = p4;
                rot[1][1] = q4;
                rot[1][2] = r4;
            }
        }
        else {
            if( e3 < e4 ) {
                rot[0][0] = p2;
                rot[0][1] = q2;
                rot[0][2] = r2;
                rot[1][0] = p3;
                rot[1][1] = q3;
                rot[1][2] = r3;
            }
            else {
                rot[0][0] = p2;
                rot[0][1] = q2;
                rot[0][2] = r2;
                rot[1][0] = p4;
                rot[1][1] = q4;
                rot[1][2] = r4;
            }
        }
    }

#ifdef TRUNCATE_MATRIX
		{
			int i,k;
			for ( i=0; i<2; i++ )
			{
				for ( k=0; k<3; k++ )
					rot[i][k] &= TRANCATE_VALUE;
			}
		}
#endif

    return 0;
}
