﻿// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using System;
using System.Configuration;
using TERASOLUNA.Fw.Common.Logging;

namespace TERASOLUNA.Fw.Client.Configuration.Conversion
{
    /// <summary>
    /// データセット変換設定ファイル内の column 要素のコレクションを格納する構成要素を表します。
    /// </summary>
    /// <remarks>
    /// <see cref="ColumnCollection"/> は、param 要素、または、
    /// result 要素内の column 要素のコレクションを表します。 
    /// </remarks>
    public class ColumnCollection : ConfigurationElementCollection
    {
        /// <summary>
        /// <see cref="ILog"/> 実装クラスのインスタンスです。
        /// </summary>
        /// <remarks>
        /// ログ出力に利用します。
        /// </remarks>
        private static ILog _log = LogFactory.GetLogger(typeof(ColumnCollection));

        /// <summary>
        /// column 要素の要素名です。
        /// </summary>
        /// <remarks>
        /// この定数の値は "column" です。
        /// </remarks>
        private static readonly string ELEMENT_NAME = "column";

        /// <summary>
        /// <see cref="ColumnCollection"/> クラスの新しいインスタンスを初期化します。
        /// </summary>
        /// <remarks>
        /// デフォルトコンストラクタです。
        /// </remarks>
        public ColumnCollection()
        {
        }

        /// <summary>
        /// <see cref="ColumnCollection"/> の型を取得します。
        /// </summary>
        /// <remarks>
        /// このコレクションの <see cref="ConfigurationElementCollectionType"/> は BasicMap です。
        /// </remarks>
        /// <value>
        /// このコレクションの <see cref="ConfigurationElementCollectionType"/> 。
        /// </value>
        public override ConfigurationElementCollectionType CollectionType
        {
            get
            {
                return ConfigurationElementCollectionType.BasicMap;
            }
        }

        /// <summary>
        /// 新しい column 要素を作成します。
        /// </summary>
        /// <remarks>
        /// 新しい <see cref="ColumnElement"/> クラスのインスタンスを作成します。
        /// </remarks>
        /// <returns>新しい param 要素。</returns>
        protected override ConfigurationElement CreateNewElement()
        {
            return new ColumnElement();
        }

        /// <summary>
        /// 指定したインデックス位置にある column 要素を取得します。
        /// </summary>
        /// <param name="index">取得する <see cref="ColumnElement"/> のインデックス位置。</param>
        /// <returns>指定したインデックス位置にある <see cref="ColumnElement"/> 。</returns>
        /// <remarks>
        /// <see cref="ConfigurationElementCollection.BaseGet(int)"/> メソッドを使用して、
        /// <see cref="ColumnCollection"/> から <see cref="ColumnElement"/> を取得します。
        /// </remarks>
        /// <value>
        /// 指定したインデックス位置にある column 要素。
        /// </value>
        public ColumnElement this[int index]
        {
            get
            {
                return (ColumnElement)BaseGet(index);
            }
        }

        /// <summary>
        /// データセット変換設定ファイル内のこの要素のコレクションを識別するために使用する名前を取得します。
        /// </summary>
        /// <remarks>
        /// 要素名は column です。
        /// </remarks>
        /// <value>
        /// この要素のコレクションの要素名。
        /// </value>
        protected override string ElementName
        {
            get
            {
                return ELEMENT_NAME;
            }
        }

        /// <summary>
        /// column 要素の要素キーを取得します。
        /// </summary>
        /// <remarks>
        /// ※column 要素には一意のキーがないので、src 属性の値を返却します。
        /// string 型を返します。
        /// </remarks>
        /// <param name="element">キーを返す <see cref="ColumnElement"/> 。</param>
        /// <returns>指定した <see cref="ColumnElement"/> のキーとなる src 属性の値。</returns>
        /// <exception cref="System.ArgumentNullException">element が null 参照です。</exception>
        /// <exception cref="System.ArgumentException">element が <see cref="ColumnElement"/> ではありません。</exception>
        protected override Object GetElementKey(ConfigurationElement element)
        {
            if (element == null)
            {
                ArgumentNullException exception = new ArgumentNullException("element");
                if (_log.IsErrorEnabled)
                {
                    _log.Error(string.Format(
                        Client.Properties.Resources.E_NULL_ARGUMENT, "element"), exception);
                }
                throw exception;
            }

            ColumnElement columnElement = element as ColumnElement;
            if (columnElement == null)
            {
                string message = string.Format(Client.Properties.Resources.E_INVALID_CAST, element.GetType().FullName, typeof(ColumnElement).FullName);
                ArgumentException exception = new ArgumentException(message);
                if (_log.IsErrorEnabled)
                {
                    _log.Error(message, exception);
                }
                throw exception;
            }

            return columnElement.SrcName;
        }

        /// <summary>
        /// <see cref="ColumnCollection"/> に column 要素を追加します。
        /// </summary>
        /// <param name="element">追加する <see cref="ColumnElement"/> 。</param>
        /// <remarks>
        /// <see cref="ConfigurationElementCollection.BaseAdd(ConfigurationElement)"/> メソッドを使用して、
        /// コレクションに新しい <see cref="ColumnElement"/> を追加します。
        /// </remarks>
        public virtual void Add(ColumnElement element)
        {
            BaseAdd(element);
        }

        /// <summary>
        /// <see cref="ColumnCollection"/> に column 要素を追加します。
        /// </summary>
        /// <param name="element">追加する <see cref="ColumnElement"/> 。</param>
        /// <remarks>
        /// 指定した element が column 要素コレクションに既に存在する場合、例外は発生しません。
        /// </remarks>
        protected override void BaseAdd(ConfigurationElement element)
        {
            BaseAdd(element, false);
        }

        /// <summary>
        /// <see cref="ColumnCollection"/> が読み取り専用かどうかを示す値を取得します。 
        /// </summary>
        /// <returns>
        /// <see cref="ColumnCollection"/> が読み取り専用の場合は true 。それ以外の場合は false 。
        /// </returns>
        /// <remarks>
        /// <see cref="ColumnCollection"/> は読み取り専用ではありません。
        /// </remarks>
        public override bool IsReadOnly()
        {
            return false;
        }
    }
}
