﻿// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Text;
using System.Collections.Generic;
using TERASOLUNA.Fw.Client.Configuration.Conversion;
using System.Configuration;
using TERASOLUNA.Fw.Utlib.Logging;
using TERASOLUNA.Fw.Common.Logging;
using TERASOLUNA.Fw.Common;
namespace TERASOLUNA.Fw.ClientUT.Configuration.Conversion
{
    /// <summary>
    ///TERASOLUNA.Fw.Client.Configuration.Conversion.ColumnCollection のテスト クラスです。すべての
    ///TERASOLUNA.Fw.Client.Configuration.Conversion.ColumnCollection 単体テストをここに含めます
    ///</summary>
    [TestClass()]
    public class ColumnCollectionTest
    {
        private TestContext testContextInstance;

        /// <summary>
        ///現在のテストの実行についての情報および機能を
        ///提供するテスト コンテキストを取得または設定します。
        ///</summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }
        #region 追加のテスト属性
        // 
        //テストを作成するときに、次の追加属性を使用することができます:
        //
        //クラスの最初のテストを実行する前にコードを実行するには、ClassInitialize を使用
        //
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //クラスのすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用
        //
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //各テストを実行する前にコードを実行するには、TestInitialize を使用
        //
        [TestInitialize()]
        public void MyTestInitialize()
        {
            LogUTUtil.Clear();
        }
        //
        //各テストを実行した後にコードを実行するには、TestCleanup を使用
        //
        //[TestCleanup()]
        //public void MyTestCleanup()
        //{
        //}
        //
        #endregion


        ///<summary>
        /// ColumnCollection () のテスト
        ///</summary>
        ///<remarks>
        /// 引数無しのコンストラクタを呼び出した場合、インスタンスが返却されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ConstructorTest01()
        {
            //実行
            ColumnCollection target = new ColumnCollection();

            //検証
            Assert.IsNotNull(target);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ColumnCollection), LogLevel.INFO));
        }

        ///<summary>
        ///Add (ColumnElement) のテスト
        ///</summary>
        ///<remarks>
        /// 要素を追加した場合に例外が発生せず、正しく値が取得できることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void AddTest01()
        {
            //準備
            ColumnCollection target = new ColumnCollection();
            ColumnElement expectElement = new ColumnElement();
            int expectCount = 1;

            //実行
            target.Add(expectElement);

            //検証
            Assert.AreEqual(expectCount, target.Count);
            Assert.AreEqual(expectElement, target[0]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ColumnCollection), LogLevel.INFO));
        }

        ///<summary>
        /// Add (ColumnElement) のテスト
        ///</summary>
        ///<remarks>
        /// 要素を連続で追加した場合に例外が発生せず、FIFOで要素が取得できることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void AddTest02()
        {
            // 準備
            ColumnCollection target = new ColumnCollection();
            ColumnElement expectedElement1 = new ColumnElement("Src1", "dest1");
            ColumnElement expectedElement2 = new ColumnElement("Src2", "dest2");
            int expectedCount = 2;

            // 実行
            target.Add(expectedElement1);
            target.Add(expectedElement2);

            // 検証
            Assert.AreEqual(expectedCount, target.Count);
            Assert.AreEqual(expectedElement1, target[0]);
            Assert.AreEqual(expectedElement2, target[1]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ColumnCollection), LogLevel.INFO));
        }

        ///<summary>
        /// Add (ColumnElement) のテスト
        ///</summary>
        ///<remarks>
        /// 同一IDを持つ要素を複数追加した場合、要素がひとつだけ追加されることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void AddTest03()
        {
            // 準備
            ColumnCollection target = new ColumnCollection();
            ColumnElement expectedElement = new ColumnElement("Src1", "dest1");
            ColumnElement duplicateElement = new ColumnElement("Src1", "dest1");
            int expectedCount = 1;

            // 実行
            target.Add(expectedElement);
            target.Add(duplicateElement);

            // 検証
            Assert.AreEqual(expectedCount, target.Count);
            Assert.AreEqual(expectedElement, target[0]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ColumnCollection), LogLevel.INFO));
        }

        ///<summary>
        /// Add (ColumnElement) のテスト
        ///</summary>
        ///<remarks>
        /// nullを追加した場合、ArgumentNullExceptionが発生することを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void AddTest04()
        {
            // 準備
            ColumnCollection target = new ColumnCollection();

            try
            {
                // 実行
                target.Add(null);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // メッセージ確認
                Assert.AreEqual("element", e.ParamName);
            }
            catch (Exception e)
            {
                Assert.Fail("予期せぬエラーです。" + e.ToString());
            }
            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ColumnCollection), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ColumnCollection), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "element"));
        }

        ///<summary>
        /// CollectionType のテスト
        ///</summary>
        ///<remarks>
        /// 返されたタイプが正しいことを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void CollectionTypeTest01()
        {
            //準備
            ColumnCollection target = new ColumnCollection();

            //実行
            ConfigurationElementCollectionType val = ConfigurationElementCollectionType.BasicMap;

            //検証
            Assert.AreEqual(val, target.CollectionType);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ColumnCollection), LogLevel.INFO));
        }

        ///<summary>
        /// CreateNewElement () のテスト
        ///</summary>
        ///<remarks>
        /// 作られたエレメントがnullでないことを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void CreateNewElementTest01()
        {

            //準備
            ColumnCollectionImpl target = new ColumnCollectionImpl();

            //実行
            ConfigurationElement actual = target.CreateNewElement();

            //検証
            Assert.IsNotNull(actual);
            Assert.IsInstanceOfType(actual, typeof(ConfigurationElement));

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ColumnCollection), LogLevel.INFO));
        }

        ///<summary>
        /// ElementName のテスト
        ///</summary>
        ///<remarks>
        /// エレメント名が正しいことを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ElementNameTest01()
        {
            //準備
            ColumnCollectionImpl target = new ColumnCollectionImpl();

            //実行
            string val = target.ElementName;

            //検証
            Assert.AreEqual("column", val);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ColumnCollection), LogLevel.INFO));
        }

        ///<summary>
        /// GetElementKey (ConfigurationElement) のテスト
        ///</summary>
        ///<remarks>
        /// elementがnullの場合、例外ArgumentNullExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void GetElementKeyTest01()
        {
            //準備
            ColumnCollectionImpl target = new ColumnCollectionImpl();
            ConfigurationElement element = null;

            //実行
            try
            {
                target.GetElementKey(element);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // メッセージ確認
                Assert.AreEqual("element", e.ParamName);
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ColumnCollection), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ColumnCollection), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "element"));
        }

        ///<summary>
        /// GetElementKey (ConfigurationElement) のテスト
        ///</summary>
        ///<remarks>
        /// elementがnullでない、且つelementがColumnElementのインスタンスでない場合、
        /// 例外ArgumentExceptionがスローされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void GetElementKeyTest02()
        {
            //準備
            ColumnCollectionImpl target = new ColumnCollectionImpl();
            ConvertElement notConfigurationElement = new ConvertElement();

            //実行
            try
            {
                target.GetElementKey(notConfigurationElement);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // 確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                Assert.AreEqual(MessageManager.GetMessage("E_INVALID_CAST", notConfigurationElement.GetType().FullName, typeof(ColumnElement).FullName), e.Message);
            }

            // ログの確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ColumnCollection), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ColumnCollection), typeof(ArgumentException), "E_INVALID_CAST", notConfigurationElement.GetType().FullName, typeof(ColumnElement).FullName));
        }

        /// <summary>
        ///GetElementKey (ConfigurationElement) のテスト
        ///</summary>
        ///<remarks>
        /// elementがnullでない、且つelementがColumnElementのインスタンスである場合、
        /// 正常に通信されるされることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void GetElementKeyTest03()
        {
            //準備
            ColumnCollectionImpl target = new ColumnCollectionImpl();
            ConfigurationElement element = new ColumnElement("src", "dest");

            //実行
            String actual = (String)target.GetElementKey(element);

            //検証
            Assert.AreEqual("src", actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ColumnCollection), LogLevel.INFO));
        }

        /// <summary>
        ///IsReadOnly () のテスト
        ///</summary>
        ///<remarks>
        /// falseが返されたことを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void IsReadOnlyTest01()
        {
            //準備
            ColumnCollection target = new ColumnCollection();

            bool expected = false;
            bool actual;

            //実行
            actual = target.IsReadOnly();

            //検証
            Assert.AreEqual(expected, actual);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ColumnCollection), LogLevel.INFO));
        }

        ///<summary>
        /// this[int index] のテスト
        ///</summary>
        ///<remarks>
        /// 追加された要素が正しく返却できることを確認するテスト。
        ///</remarks>
        [TestMethod()]
        public void ItemTest01()
        {
            // 準備
            ColumnCollection target = new ColumnCollection();
            ColumnElement expectedElement = new ColumnElement();
            int expectedCount = 1;

            // 実行
            target.Add(expectedElement);

            // 検証
            Assert.AreEqual(expectedCount, target.Count);
            Assert.AreSame(expectedElement, target[0]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ColumnCollection), LogLevel.INFO));
        }

        ///<summary>
        /// this[int index] のテスト
        ///</summary>
        /// <remarks>
        ///　複数個追加された要素が正しく返却できることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void ItemTest02()
        {
            // 準備
            ColumnCollection target = new ColumnCollection();
            ColumnElement expectedElement1 = new ColumnElement("src1", "dest1");
            ColumnElement expectedElement2 = new ColumnElement("src2", "dest2");
            ColumnElement expectedElement3 = new ColumnElement("src3", "dest3");
            int expectedCount = 3;

            // 実行
            target.Add(expectedElement1);
            target.Add(expectedElement2);
            target.Add(expectedElement3);


            // 検証
            Assert.AreEqual(expectedCount, target.Count);
            Assert.AreSame(expectedElement1, target[0]);
            Assert.AreSame(expectedElement2, target[1]);
            Assert.AreSame(expectedElement3, target[2]);

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ColumnCollection), LogLevel.INFO));
        }

        ///<summary>
        /// this[int index] のテスト
        ///</summary>
        ///<remarks>
        /// インデックスに-1を指定した場合にConfigurationErrorsExceptionが返却されることを確認する。
        ///</remarks>
        [TestMethod()]
        public void ItemTest03()
        {
            // 準備
            ColumnCollection target = new ColumnCollection();
            ColumnElement element1 = new ColumnElement("src1", "dest1");
            target.Add(element1);

            try
            {
                // 実行
                object obj = target[-1];
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ConfigurationErrorsException)
            {
                // 成功
            }
            catch (Exception e)
            {
                Assert.Fail("予期せぬエラーです。" + e.ToString());
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ColumnCollection), LogLevel.INFO));
        }

        ///<summary>
        /// this[int index] のテスト
        ///</summary>
        ///<remarks>
        /// インデックスにサイズより大きい値を指定した場合にConfigurationErrorsExceptionが返却されることを確認する。
        ///</remarks>
        [TestMethod()]
        public void ItemTest04()
        {
            // 準備
            ColumnCollection target = new ColumnCollection();
            ColumnElement element1 = new ColumnElement("src1", "dest1");
            target.Add(element1);

            try
            {
                // 実行
                object obj = target[1];
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ConfigurationErrorsException)
            {
                // 成功
            }
            catch (Exception e)
            {
                Assert.Fail("予期せぬエラーです。" + e.ToString());
            }

            // ログの確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ColumnCollection), LogLevel.INFO));
        }

    }

    ///<summary>
    /// protectedメソッドを呼ぶための実装クラスです。 
    ///</summary>
    public class ColumnCollectionImpl : ColumnCollection
    {

        ///<summary>
        /// 基底クラスの同名のメソッドを呼び出します。
        ///</summary>
        public new ConfigurationElement CreateNewElement()
        {
            return base.CreateNewElement();
        }

        ///<summary>
        /// 基底クラスの同名のメソッドを呼び出します。
        ///</summary>
        public new object GetElementKey(ConfigurationElement element)
        {
            return base.GetElementKey(element);
        }

        ///<summary>
        /// 基底クラスの同名のメソッドを呼び出します。
        ///</summary>
        public new string ElementName
        {
            get
            {
                return base.ElementName;
            }
        }
    }
}
