// Copyright (c) 2008, NTT DATA Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

using System;
using System.Globalization;
using System.Text.RegularExpressions;
using Microsoft.Practices.EnterpriseLibrary.Common.Configuration;
using Microsoft.Practices.EnterpriseLibrary.Validation;
using Microsoft.Practices.EnterpriseLibrary.Validation.Validators;
using TERASOLUNA.Fw.Common.Logging;
using TERASOLUNA.Fw.Common.Validation.Configuration;

namespace TERASOLUNA.Fw.Common.Validation.Validators
{
    /// <summary>
    /// ؃["l`FbN"ɑΉ鐔l`FbNs
    /// <see cref="Microsoft.Practices.EnterpriseLibrary.Validation.Validator"/> pNXłB
    /// </summary>
    /// <remarks>
    /// <para>
    /// ؑΏۂ̕\AƏꂼw肵̐ľ`
    /// ł邱Ƃ`FbN錟؋@\񋟂܂B
    /// ̃`FbNɂ́Aw肵lƓׂv`FbNƁA
    /// w肵ȓ͈̔͂Ɋ܂܂邩ׂ͈̓`FbN𗘗p邱Ƃł܂B
    ///  <seealso cref="IsAccordedInteger"/>  false w肳ꍇǍ`FbN͔͈̓`FbNƂȂ܂B
    ///  <seealso cref="IsAccordedScale"/>  false w肳ꍇǍ`FbN͔͈̓`FbNƂȂ܂B
    /// ͈̓`FbN̏ꍇAw肵l̂͊܂܂܂B
    /// </para>
    /// <para>
    /// {NXpē͒l؂{ɂ́A͒lؐݒt@CɋLq
    ///  validator vf̑ƂāAȉ̐ݒs܂B
    /// ͒lؐݒt@Ĉ̋L@ɂẮA
    ///  Validation Application Block ̃hLgQƂĂB  
    /// </para>
    /// <para>
    /// <list type="bullet">
    /// <item>
    /// <description>negated</description>
    /// <para>؃[𔽓]tOłBtrue ݒ肵ꍇA؃[]܂BftHgĺAfalse łB</para>
    /// </item>
    /// <item>
    /// <description>messageTemplate</description>
    /// <para>؃G[bZ[W̃ev[gƂĎgp镶B
    /// ݒ̏ꍇ́A <see cref="NumberValidator.DefaultNonNegatedMessageTemplate"/> A <see cref="NumberValidator.DefaultNegatedMessageTemplate"/>  
    /// ꂩgp܂B
    /// </para>
    /// </item>
    /// <item>
    /// <description>messageTemplateResourceName</description>
    /// <para>bZ[Wev[g\[Xǂݍލۂ̖OB</para>
    /// </item>
    /// <item>
    /// <description>messageTemplateResourceType</description>
    /// <para>bZ[Wev[g\[Xǂݍލۂ̃\[X̌^B</para>
    /// </item>
    /// <item>
    /// <description>tag</description>
    /// <para>bZ[Wev[g̃v[Xz_{2}ɓn镶B
    /// ؑΏۍڂ̘_w肵܂B</para>
    /// </item>
    /// <item>
    /// <description>type</description>
    /// <para>gp <see cref="Microsoft.Practices.EnterpriseLibrary.Validation.Validator"/> NX̌^SC^Ŏw肵܂B</para>
    /// </item>
    /// <item>
    /// <description>name</description>
    /// <para>gp <see cref="Microsoft.Practices.EnterpriseLibrary.Validation.Validator"/> ̖OłBt[[Nł͗p܂B
    /// ɕ <see cref="Microsoft.Practices.EnterpriseLibrary.Validation.Validator"/> 𗘗p؂sꍇAdOgp
    /// Ƃ͂ł܂B͕̑K{ڂłB</para>
    /// </item>
    /// <item>
    /// <description>integerLength</description>
    /// <para>̌w肵܂BK{łBŏl 1 łB</para>
    /// </item>
    /// <item>
    /// <description>scale</description>
    /// <para>̌w肵܂Bŏl 0 łB</para>
    /// </item>
    /// <item>
    /// <description>isAccordedScale</description>
    /// <para>true łΏ̌v`FbNs܂B
    ///  false A܂͏ȗ͈͓͌̃`FbNłB</para>
    /// </item>
    /// <item>
    /// <description>isAccordedInteger</description>
    /// <para>	true łΐ̌v`FbNs܂B
    ///  falseA܂͏ȗ͈͓͌̃`FbNłB</para>
    /// </item>
    /// </list>
    /// </para>
    /// <example>
    /// <code>
    /// &lt;validator negated="false" integerLength="3" isAccordedInteger="false" 
    /// scale="2" messageTemplate=""  isAccordedScale="false" messageTemplateResourceName=""
    /// messageTemplateResourceType="" tag="ؑΏۍږ" name="number validator"  
    /// type="TERASOLUNA.Fw.Common.Validation.Validators.NumberValidator,TERASOLUNA.Fw.Common"/&gt;
    /// </code>
    /// </example>
    /// </remarks>
    [ConfigurationElementType(typeof(NumberValidatorData))]
    public class NumberValidator : ValueValidator
    {
        /// <summary>
        /// l\邽߂ <see cref="Regex"/> CX^XłB
        /// </summary>
        private static readonly Regex DECIMAL_REG_EXP =
                                    new Regex(@"^[+-]?(\d+)(\.(\d+))?$");

        /// <summary>
        /// <see cref="ILog"/> NX̃CX^XłB
        /// </summary>
        /// <remarks>
        /// Oo͂ɗp܂B
        /// </remarks>
        private static ILog _log = LogFactory.GetLogger(typeof(NumberValidator));

        /// <summary>
        /// ̌łB
        /// </summary>
        /// <remarks>
        /// ftHg̒ĺA 0 łB
        /// </remarks>
        private int _integerLength = 0;

        /// <summary>
        /// ̌łB
        /// </summary>
        /// <remarks>
        /// ftHg̒ĺA 0 łB
        /// </remarks>
        private int _scale = 0;

        /// <summary>
        /// ̌̈v`FbNsǂ̃tOłB
        /// </summary>
        /// <remarks>
        /// ftHg̒ĺA false łB
        /// </remarks>
        private bool _isAccordedInteger = false;

        /// <summary>
        /// ̌̈v`FbNsǂ̃tOłB
        /// </summary>
        /// <remarks>
        /// ftHg̒ĺA false łB
        /// </remarks>
        private bool _isAccordedScale = false;

        /// <summary>
        /// V <see cref="NumberValidator"/> ̃CX^X܂B
        /// </summary>
        /// <param name="messageTemplate">؎s̍ہAbZ[W𐶐邽߂ɗpev[gB
        /// ؂ɗpp[^v[Xz_ɓn܂B
        /// p[^Ƃ̃v[Xz_ԍ͈ȉ̒ʂłB
        /// <list type="table">
        /// <listheader>
        /// <term>v[Xz_</term>
        /// <description>e</description>
        /// </listheader>
        /// <item>
        /// <term>{0}</term>
        /// <description>ؑΏە</description>
        /// </item>
        /// <item>
        /// <term>{1}</term>
        /// <description>key(vpeBE\bh)</description>
        /// </item>
        /// <item>
        /// <term>{2}</term>
        /// <description>Tag</description>
        /// </item>
        /// <item>
        /// <term>{3}</term>
        /// <description>̌</description>
        /// </item>
        /// <item>
        /// <term>{4}</term>
        /// <description>̌</description>
        /// </item>
        /// </list>
        /// </param>
        /// <param name="tag">^OBbZ[Wev[gpăbZ[W𐶐ہA
        /// v[Xz_ 2 Ԗڂɓn܂B</param>
        /// <param name="negated">؏̔]tOB
        /// ]Ō؂{ꍇɂ true w肵܂B</param>
        /// <param name="integerLength">̌B</param>
        /// <param name="isAccordedInteger">̌v`FbNƂ邩ǂ̃tOB
        ///  true łΐ̌v`FbNs܂B
        ///  false w肵ꍇA͈͓̔`FbNƂȂ܂B</param>
        /// <param name="scale">̌B</param>
        /// <param name="isAccordedScale">̌v`FbNƂ邩ǂ̃tOB
        ///  true łΏ̌v`FbNs܂B
        ///  false w肵ꍇA͈͓̔`FbNƂȂ܂B</param>
        /// <exception cref="ArgumentException">
        /// <list type="bullet">
        /// <item>
        /// <paramref name="integerLength"/>  0 ȉłB
        /// </item>
        /// <item>
        /// <paramref name="scale"/> 0łB
        /// </item>
        /// </list>
        /// </exception>  
        public NumberValidator(int integerLength,
                               bool isAccordedInteger,
                               int scale,
                               bool isAccordedScale,
                               string messageTemplate,
                               string tag,
                               bool negated)
            : base(messageTemplate, tag, negated)
        {
            if (integerLength <= 0)
            {
                string message = string.Format(Properties.Resources.E_ZERO_AND_NEGATIVE_NUMBER, "integerLength");
                ArgumentException exception = new ArgumentException(message);
                if (_log.IsErrorEnabled)
                {
                    _log.Error(message, exception);
                }
                throw exception;
            }

            if (scale < 0)
            {
                string message = string.Format(Properties.Resources.E_NEGATIVE_NUMBER, "scale");
                ArgumentException exception = new ArgumentException(message);
                if (_log.IsErrorEnabled)
                {
                    _log.Error(message, exception);
                }
                throw exception;
            }
            _integerLength = integerLength;
            _scale = scale;
            _isAccordedScale = isAccordedScale;
            _isAccordedInteger = isAccordedInteger;
        }

        /// <summary>
        /// ̌v`FbN邩ǂ bool l擾܂B
        /// </summary>
        /// <value>̌v`FbN邩ǂ bool lB</value>
        public bool IsAccordedScale
        {
            get { return _isAccordedScale; }
        }

        /// <summary>
        /// ̌v`FbN邩ǂ bool l擾܂B
        /// </summary>
        /// <value>̌v`FbN邩ǂ bool lB</value>
        public bool IsAccordedInteger
        {
            get { return _isAccordedInteger; }
        }

        /// <summary>
        /// ̌擾܂B
        /// </summary>
        /// <value>̌B</value>
        public int Scale
        {
            get { return _scale; }
        }

        /// <summary>
        /// ̌擾܂B
        /// </summary>
        /// <value>̌B</value>
        public int IntegerLength
        {
            get { return _integerLength; }
        }

        /// <summary>
        /// <seealso cref="ValueValidator.Negated"/>  true łꍇɎgpftHg̃bZ[Wev[g擾܂B
        /// </summary>
        /// <value>؃[]̃ftHgbZ[Wev[gB</value>
        protected override string DefaultNegatedMessageTemplate
        {
            get
            {
                return Properties.Resources.NUMBER_VALIDATOR_NEGATED_MESSAGETEMPLATE;
            }
        }

        /// <summary>
        /// <seealso cref="ValueValidator.Negated"/>  false łꍇɎgpftHg̃bZ[Wev[g擾܂B
        /// </summary>
        /// <value>ftHgbZ[Wev[głB</value>
        protected override string DefaultNonNegatedMessageTemplate
        {
            get
            {
                return Properties.Resources.NUMBER_VALIDATOR_NONNEGATED_MESSAGETEMPLATE;
            }
        }

        /// <summary>
        /// ؃G[bZ[W𐶐郁\bhłB <seealso cref="Validator.MessageTemplate"/> ev[gƂA
        /// ݂̃J`ɏ]ĕtH[}bg܂B
        /// </summary>
        /// <remarks>
        /// <para>tH[}bgɊ܂܂v[Xz_ɓńAȉ̒ʂłB
        /// <list type="table">
        /// <listheader>
        /// <term>v[Xz_</term>
        /// <description>e</description>
        /// </listheader>
        /// <item>
        /// <term>{0}</term>
        /// <description>ؑΏە</description>
        /// </item>
        /// <item>
        /// <term>{1}</term>
        /// <description>key(vpeBE\bh)</description>
        /// </item>
        /// <item>
        /// <term>{2}</term>
        /// <description>Tag</description>
        /// </item>
        /// <item>
        /// <term>{3}</term>
        /// <description>̌</description>
        /// </item>
        /// <item>
        /// <term>{4}</term>
        /// <description>̌</description>
        /// </item>
        /// </list>
        /// </para>
        /// </remarks>
        /// <param name="objectToValidate">ؑΏۂ̃IuWFNgB</param>
        /// <param name="key">IuWFNg擾vpeB܂̓\bhB</param>
        /// <returns>؃G[bZ[WB</returns>
        /// <exception cref="FormatException">
        /// ȉ̃oQƁB
        /// <list type="bullet">
        /// <item>
        /// format ̏ڂłB܂͏ݒ肷 0 菬Aݒ肷w肳ꂽIuWFNg̐ȏłB 
        /// </item>
        /// </list>
        /// </exception>
        protected override string GetMessage(object objectToValidate, string key)
        {
            return string.Format(CultureInfo.CurrentUICulture,
                this.MessageTemplate,
                objectToValidate,
                key,
                this.Tag,
                IntegerLength,
                Scale);
        }

        /// <summary>
        /// <paramref name="objectToValidate"/>lɕϊ\ȕłA
        /// Eꂼw肵ɈvA܂͔͈͓Ɏ܂邩؂郁\bhłB
        /// </summary>
        /// <remarks>
        /// <para>
        /// <paramref name="objectToValidate"/> null ܂͋󕶎̏ꍇA؏͍s܂B
        /// </para>
        /// </remarks>
        /// <param name="objectToValidate">ؑΏۂ̒lłB</param>
        /// <param name="currentTarget">ؑΏۂ̃IuWFNgłB</param>
        /// <param name="key">ؑΏۂ̃vpeB܂̓\bhłB</param>
        /// <param name="validationResults">،ʂi[<see cref="ValidationResults"/>łB</param>
        /// <exception cref="ArgumentNullException">
        /// <paramref name="validationResults"/>  null QƂłB
        /// </exception> 
        /// <exception cref="FormatException">
        /// ȉ̃oQƁB
        /// <list type="bullet">
        /// <item><seealso cref="GetMessage"/></item>
        /// </list>
        /// </exception>
        protected override void DoValidate(object objectToValidate,
                                           object currentTarget,
                                           string key,
                                           ValidationResults validationResults)
        {
            if (validationResults == null)
            {
                ArgumentNullException exception = new ArgumentNullException("validationResults");
                if (_log.IsErrorEnabled)
                {
                    _log.Error(string.Format(
                        Properties.Resources.E_NULL_ARGUMENT, "validationResults"), exception);
                }
                throw exception;
            }

            if (objectToValidate == null)
            {
                return;
            }
            string targetData = objectToValidate.ToString();
            if (targetData.Length == 0)
            {
                return;
            }

            int[] targetArray = GetIntAntScaleLength(targetData);

            bool isError = false;
            if (targetArray.Length == 0)
            {
                isError = true;
            }
            else
            {
                // ̔r
                if ((IsAccordedInteger && targetArray[0] != _integerLength)
                    || (!IsAccordedInteger && targetArray[0] > _integerLength))
                {
                    isError = true;

                }

                if ((IsAccordedScale && targetArray[1] != _scale)
                    || (!IsAccordedScale && targetArray[1] > _scale))
                {
                    isError = true;
                }
            }

            if (!isError == Negated)
            {
                LogValidationResult(validationResults,
                    GetMessage(targetData, key),
                    currentTarget,
                    key);
            }
        }

        /// <summary>
        /// ̐̕Ěi[ int zԂ\bhłB
        /// </summary>
        /// <param name="str">͑Ώۂ̕B</param>
        /// <returns>Ěi[ int zB</returns>
        protected virtual int[] GetIntAntScaleLength(string str)
        {

            Match match = DECIMAL_REG_EXP.Match(str);
            if (!match.Success || match.Groups.Count != 4)
            {
                return new int[0];
            }

            int[] result = new int[2];
            result[0] = match.Groups[1].Length;
            result[1] = match.Groups[3].Length;

            return result;
        }
    }
}
