﻿// <著作権>
// Copyright (c) 2008, NTT DATA Corporation.

using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Text;
using System.Collections.Generic;
using TERASOLUNA.Fw.Common;
using TERASOLUNA.Fw.Utlib.Logging;
using TERASOLUNA.Fw.Common.Logging;
using System.Configuration;
namespace TERASOLUNA.Fw.CommonUT
{
    /// <summary>
    /// TERASOLUNA.Fw.Common.ClassUtil のテスト クラスです。すべての
    /// TERASOLUNA.Fw.Common.ClassUtil 単体テストをここに含めます
    /// </summary>
    [TestClass()]
    public class ClassUtilTest
    {

        private TestContext testContextInstance;

        /// <summary>
        /// 現在のテストの実行についての情報および機能を
        /// 提供するテスト コンテキストを取得または設定します。
        /// </summary>
        public TestContext TestContext
        {
            get
            {
                return testContextInstance;
            }
            set
            {
                testContextInstance = value;
            }
        }
        #region 追加のテスト属性
        // 
        //テストを作成するときに、次の追加属性を使用することができます:
        //
        //クラスの最初のテストを実行する前にコードを実行するには、ClassInitialize を使用
        //
        //[ClassInitialize()]
        //public static void MyClassInitialize(TestContext testContext)
        //{
        //}
        //
        //クラスのすべてのテストを実行した後にコードを実行するには、ClassCleanup を使用
        //
        //[ClassCleanup()]
        //public static void MyClassCleanup()
        //{
        //}
        //
        //各テストを実行する前にコードを実行するには、TestInitialize を使用
        //
        [TestInitialize()]
        public void MyTestInitialize()
        {
            LogUTUtil.Clear();
        }
        //
        //各テストを実行した後にコードを実行するには、TestCleanup を使用
        //
        //[TestCleanup()]
        //public void MyTestCleanup()
        //{
        //}
        //
        #endregion


        /// <summary>
        /// CreateInstanceByType&lt;&gt; (Type) のテスト
        /// </summary>
        /// <remarks>
        /// targetType が null だった場合、
        /// ArgumentNullException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceByTypeTest01()
        {
            Type targetType = null;

            object actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceByType<object>(targetType);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // メッセージ確認
                Assert.AreEqual("targetType", e.ParamName);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "targetType"));
        }

        /// <summary>
        /// CreateInstanceByType&lt;&gt; (Type) のテスト
        /// </summary>
        /// <remarks>
        /// targetType が抽象クラスだった場合、
        /// ArgumentException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceByTypeTest02()
        {
            Type targetType = typeof(ClassUtil_Stub02); 

            object actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceByType<object>(targetType);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // インスタンス生成確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);

                // メッセージ確認
                Assert.AreEqual(MessageManager.GetMessage("E_PARAM_ABSTRACT_TYPE", "targetType", targetType.AssemblyQualifiedName), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(ArgumentException), "E_PARAM_ABSTRACT_TYPE", "targetType", targetType.AssemblyQualifiedName));
        }


        /// <summary>
        /// CreateInstanceByType&lt;&gt; (Type) のテスト
        /// </summary>
        /// <remarks>
        /// targetType が型パラメータが未設定なジェネリック型だった場合、
        /// ArgumentException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceByTypeTest03()
        {
            Type targetType = Type.GetType("TERASOLUNA.Fw.CommonUT.ClassUtil_Stub03`1, TERASOLUNA.Fw.CommonUT", true);

            object actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceByType<object>(targetType);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // インスタンス生成確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);

                // メッセージ確認
                Assert.AreEqual(MessageManager.GetMessage("E_PARAM_GENERIC_TYPE", "targetType", targetType.AssemblyQualifiedName), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(ArgumentException), "E_PARAM_GENERIC_TYPE", "targetType", targetType.AssemblyQualifiedName));
        }

        /// <summary>
        /// CreateInstanceByType&lt;&gt; (Type) のテスト
        /// </summary>
        /// <remarks>
        /// targetType がコンストラクタ呼び出しに失敗する型だった場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceByTypeTest04()
        {
            Type targetType = typeof(ClassUtil_Stub05);

            object actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceByType<object>(targetType);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // メッセージ確認
                Assert.AreEqual(MessageManager.GetMessage("E_MEMBER_ACCESS_EXCEPTION", targetType.AssemblyQualifiedName), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(TerasolunaException), "E_MEMBER_ACCESS_EXCEPTION", targetType.AssemblyQualifiedName));
        }

        /// <summary>
        /// CreateInstanceByType&lt;&gt; (Type) のテスト
        /// </summary>
        /// <remarks>
        /// targetType が呼び出したコンストラクタが例外をスローする型の場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceByTypeTest05()
        {
            Type targetType = typeof(ClassUtil_Stub06);

            object actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceByType<object>(targetType);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // メッセージ確認
                Assert.AreEqual(MessageManager.GetMessage("E_TARGET_INVOCATION_EXCEPTION", targetType.AssemblyQualifiedName), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(TerasolunaException), "E_TARGET_INVOCATION_EXCEPTION", targetType.AssemblyQualifiedName));
        }

        /// <summary>
        /// CreateInstanceByType&lt;&gt; (Type) のテスト
        /// </summary>
        /// <remarks>
        /// targetType が型パラメータ T にキャストができない型の場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceByTypeTest06()
        {
            Type targetType = typeof(ClassUtil_Stub01);

            string actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceByType<string>(targetType);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // メッセージ確認
                Assert.AreEqual(MessageManager.GetMessage("E_INVALID_CAST_TYPE_FROM_GENERICPARAM", targetType.AssemblyQualifiedName, typeof(string).AssemblyQualifiedName), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(TerasolunaException), "E_INVALID_CAST_TYPE_FROM_GENERICPARAM", targetType.AssemblyQualifiedName, typeof(string).AssemblyQualifiedName));
        }

        /// <summary>
        /// CreateInstanceByType&lt;&gt; (Type) のテスト
        /// </summary>
        /// <remarks>
        /// targetType が型パラメータ T にキャストができる型の場合、
        /// 正常にインスタンスが生成されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceByTypeTest07()
        {
            Type targetType = typeof(ClassUtil_Stub01);

            ClassUtil_Stub01 actual;
            actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceByType<ClassUtil_Stub01>(targetType);

            // インスタンス生成確認
            Assert.IsNotNull(actual);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
        }

        /// <summary>
        /// CreateInstanceByTypeName&lt;&gt; (string) のテスト
        /// </summary>
        /// <remarks>
        /// typeName が null の場合、
        /// ArgumentNullException がスローされることを確認するテスト。
        /// </remarks> 
        [TestMethod()]
        public void CreateInstanceByTypeNameTest01()
        {
            string typeName = null;

            object actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceByTypeName<object>(typeName);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // メッセージ確認
                Assert.AreEqual("typeName", e.ParamName);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "typeName"));
        }

        /// <summary>
        /// CreateInstanceByTypeName&lt;&gt; (string) のテスト
        /// </summary>
        /// <remarks>
        /// typeName が空文字の場合、
        /// ArgumentException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceByTypeNameTest02()
        {
            string typeName = string.Empty;

            object actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceByTypeName<object>(typeName);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // インスタンス生成確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);

                // メッセージ確認
                Assert.AreEqual(e.Message, MessageManager.GetMessage("E_EMPTY_STRING", "typeName"));
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(ArgumentException), "E_EMPTY_STRING", "typeName"));
        }

        /// <summary>
        /// CreateInstanceByTypeName&lt;&gt; (string) のテスト
        /// </summary>
        /// <remarks>
        /// typeName が抽象クラス型のアセンブリ修飾名の場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceByTypeNameTest03()
        {
            string typeName = "TERASOLUNA.Fw.CommonUT.ClassUtil_Stub02, TERASOLUNA.Fw.CommonUT";

            object actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceByTypeName<object>(typeName);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // メッセージ確認
                Assert.AreEqual(MessageManager.GetMessage("E_ABSTRACT_TYPE", typeof(ClassUtil_Stub02).AssemblyQualifiedName), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(TerasolunaException), "E_ABSTRACT_TYPE", typeof(ClassUtil_Stub02).AssemblyQualifiedName));
        }

        /// <summary>
        /// CreateInstanceByTypeName&lt;&gt; (string) のテスト
        /// </summary>
        /// <remarks>
        /// typeName が型パラメータ T にキャストができる型のアセンブリ修飾名の場合、
        /// 正常にインスタンスが生成されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceByTypeNameTest04()
        {
            string typeName = "TERASOLUNA.Fw.CommonUT.ClassUtil_Stub01, TERASOLUNA.Fw.CommonUT";

            ClassUtil_Stub01 actual;
            actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceByTypeName<ClassUtil_Stub01>(typeName);

            // インスタンス生成確認
            Assert.IsNotNull(actual);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
        }

        /// <summary>
        /// CreateInstanceByTypeName&lt;&gt; (string) のテスト
        /// </summary>
        /// <remarks>
        /// typeName の型を取得することができないアセンブリ修飾名の場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceByTypeNameTest05()
        {
            string typeName = "TERASOLUNA.Fw.CommonUT.NoExist_ClassUtil_Stub, TERASOLUNA.Fw.CommonUT";

            object actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceByTypeName<object>(typeName);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // メッセージ確認
                Assert.AreEqual(MessageManager.GetMessage("E_TYPE_LOAD_FAILED", typeName), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(TerasolunaException), "E_TYPE_LOAD_FAILED", typeName));
        }

        /// <summary>
        /// CreateInstanceByTypeName&lt;&gt; (string) のテスト
        /// </summary>
        /// <remarks>
        /// typeName がジェネリック型のアセンブリ修飾名の場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceByTypeNameTest06()
        {
            string typeName = "TERASOLUNA.Fw.CommonUT.ClassUtil_Stub03`1, TERASOLUNA.Fw.CommonUT";

            object actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceByTypeName<object>(typeName);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // メッセージ確認
                Assert.AreEqual(MessageManager.GetMessage("E_GENERIC_TYPE", typeof(ClassUtil_Stub03<>).AssemblyQualifiedName), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(TerasolunaException), "E_GENERIC_TYPE", typeof(ClassUtil_Stub03<>).AssemblyQualifiedName));
        }

        /// <summary>
        /// CreateInstanceByTypeName&lt;&gt; (string) のテスト
        /// </summary>
        /// <remarks>
        /// typeName がコンストラクタの呼び出しに失敗するアセンブリ修飾名の場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceByTypeNameTest07()
        {
            string typeName = "TERASOLUNA.Fw.CommonUT.ClassUtil_Stub05, TERASOLUNA.Fw.CommonUT";

            object actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceByTypeName<object>(typeName);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // メッセージ確認
                Assert.AreEqual(MessageManager.GetMessage("E_MEMBER_ACCESS_EXCEPTION", typeof(ClassUtil_Stub05).AssemblyQualifiedName), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(TerasolunaException), "E_MEMBER_ACCESS_EXCEPTION", typeof(ClassUtil_Stub05).AssemblyQualifiedName));
        }

        /// <summary>
        /// CreateInstanceByTypeName&lt;&gt; (string) のテスト
        /// </summary>
        /// <remarks>
        /// typeName のコンストラクタで例外が発生するアセンブリ修飾名の場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceByTypeNameTest08()
        {
            string typeName = "TERASOLUNA.Fw.CommonUT.ClassUtil_Stub06, TERASOLUNA.Fw.CommonUT";

            object actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceByTypeName<object>(typeName);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // メッセージ確認
                Assert.AreEqual(MessageManager.GetMessage("E_TARGET_INVOCATION_EXCEPTION", typeof(ClassUtil_Stub06).AssemblyQualifiedName), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(TerasolunaException), "E_TARGET_INVOCATION_EXCEPTION", typeof(ClassUtil_Stub06).AssemblyQualifiedName));
        }

        /// <summary>
        /// CreateInstanceByTypeName&lt;&gt; (string) のテスト
        /// </summary>
        /// <remarks>
        /// typeName が型パラメータ T にキャストできないアセンブリ修飾名の場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceByTypeNameTest09()
        {
            string typeName = "TERASOLUNA.Fw.CommonUT.ClassUtil_Stub01, TERASOLUNA.Fw.CommonUT";

            object actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceByTypeName<ClassUtil_Stub01AppSettings>(typeName);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // メッセージ確認
                Assert.AreEqual(MessageManager.GetMessage("E_INVALID_CAST_TYPE_FROM_GENERICPARAM", typeof(ClassUtil_Stub01).AssemblyQualifiedName, typeof(ClassUtil_Stub01AppSettings).AssemblyQualifiedName), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(TerasolunaException), "E_INVALID_CAST_TYPE_FROM_GENERICPARAM", typeof(ClassUtil_Stub01).AssemblyQualifiedName, typeof(ClassUtil_Stub01AppSettings).AssemblyQualifiedName));
        }

        /// <summary>
        ///CreateInstanceFromAppSettings&lt;&gt; (string) のテスト
        ///</summary>
        /// <remarks>
        /// appSettingsKey が null の場合、
        /// ArgumentNullException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceFromAppSettingsStringTest01()
        {
            string appSettingsKey = null;

            ClassUtil_Stub01AppSettings actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceFromAppSettings<ClassUtil_Stub01AppSettings>(appSettingsKey);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // メッセージ確認
                Assert.AreEqual("appSettingsKey", e.ParamName);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "appSettingsKey"));
        }

        /// <summary>
        ///CreateInstanceFromAppSettings&lt;&gt; (string) のテスト
        ///</summary>
        /// <remarks>
        /// appSettingsKey が空文字の場合、
        /// ArgumentException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceFromAppSettingsStringTest02()
        {
            string appSettingsKey = string.Empty;

            ClassUtil_Stub01AppSettings actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceFromAppSettings<ClassUtil_Stub01AppSettings>(appSettingsKey);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // インスタンス生成確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);

                // メッセージ確認
                Assert.AreEqual(e.Message, MessageManager.GetMessage("E_EMPTY_STRING", "appSettingsKey"));
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(ArgumentException), "E_EMPTY_STRING", "appSettingsKey"));
        }

        /// <summary>
        ///CreateInstanceFromAppSettings&lt;&gt; (string) のテスト
        ///</summary>
        /// <remarks>
        /// 構成ファイルに設定する引数 appSettingsKey をキーとする add 要素が存在しない場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceFromAppSettingsStringTest03()
        {
            string appSettingsKey = "ClassUtilTest01";

            ClassUtil_Stub01AppSettings actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceFromAppSettings<ClassUtil_Stub01AppSettings>(appSettingsKey);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // メッセージ確認
                Assert.AreEqual(e.Message, MessageManager.GetMessage("E_TARGET_TYPE_NOT_FOUND", appSettingsKey));
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(TerasolunaException), "E_TARGET_TYPE_NOT_FOUND", appSettingsKey));
        }

        /// <summary>
        ///CreateInstanceFromAppSettings&lt;&gt; (string) のテスト
        ///</summary>
        /// <remarks>
        /// 構成ファイルに設定する引数 appSettingsKey をキーとする add 要素に空文字が設定された場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceFromAppSettingsStringTest04()
        {
            string appSettingsKey = "ClassUtilTest02";

            ClassUtil_Stub01AppSettings actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceFromAppSettings<ClassUtil_Stub01AppSettings>(appSettingsKey);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // メッセージ確認
                Assert.AreEqual(MessageManager.GetMessage("E_TARGET_TYPE_NOT_FOUND", appSettingsKey), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(TerasolunaException), "E_TARGET_TYPE_NOT_FOUND", appSettingsKey));
        }

        /// <summary>
        ///CreateInstanceFromAppSettings&lt;&gt; (string) のテスト
        ///</summary>
        /// <remarks>
        /// 構成ファイルに設定する引数 appSettingsKey をキーとする add 要素に、
        /// 型パラメータ T にキャストができる型のアセンブリ修飾名が設定された場合、
        /// 正常にインスタンスが生成されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceFromAppSettingsStringTest05()
        {
            string appSettingsKey = "ClassUtilTest03";

            ClassUtil_Stub01AppSettings actual;

            actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceFromAppSettings<ClassUtil_Stub01AppSettings>(appSettingsKey);

            Assert.IsNotNull(actual);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
        }

        /// <summary>
        ///CreateInstanceFromAppSettings&lt;&gt; (string) のテスト
        ///</summary>
        /// <remarks>
        /// 構成ファイルに設定する引数 appSettingsKey をキーとする add 要素に型を取得することができない文字列が設定された場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceFromAppSettingsStringTest07()
        {
            string appSettingsKey = "ClassUtilTest05";

            object actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceFromAppSettings<object>(appSettingsKey);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // メッセージ確認
                Assert.AreEqual(MessageManager.GetMessage("E_TYPE_LOAD_FAILED", ConfigurationManager.AppSettings[appSettingsKey]), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(TerasolunaException), "E_TYPE_LOAD_FAILED", ConfigurationManager.AppSettings[appSettingsKey]));
        }

        /// <summary>
        ///CreateInstanceFromAppSettings&lt;&gt; (string) のテスト
        ///</summary>
        /// <remarks>
        /// 構成ファイルに設定する引数 appSettingsKey をキーとする add 要素に抽象クラスのアセンブリ修飾名が設定された場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceFromAppSettingsStringTest08()
        {
            string appSettingsKey = "ClassUtilTest04";

            object actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceFromAppSettings<object>(appSettingsKey);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // メッセージ確認
                Assert.AreEqual(MessageManager.GetMessage("E_ABSTRACT_TYPE", typeof(ClassUtil_Stub02).AssemblyQualifiedName), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(TerasolunaException), "E_ABSTRACT_TYPE", typeof(ClassUtil_Stub02).AssemblyQualifiedName));
        }

        /// <summary>
        ///CreateInstanceFromAppSettings&lt;&gt; (string) のテスト
        ///</summary>
        /// <remarks>
        /// 構成ファイルに設定する引数 appSettingsKey をキーとする add 要素にジェネリック型のアセンブリ修飾名が設定された場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceFromAppSettingsStringTest09()
        {
            string appSettingsKey = "ClassUtilTest06";

            ClassUtil_Stub01AppSettings actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceFromAppSettings<ClassUtil_Stub01AppSettings>(appSettingsKey);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // メッセージ確認
                Assert.AreEqual(MessageManager.GetMessage("E_GENERIC_TYPE", typeof(ClassUtil_Stub03<>).AssemblyQualifiedName), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(TerasolunaException), "E_GENERIC_TYPE", typeof(ClassUtil_Stub03<>).AssemblyQualifiedName));
        }

        /// <summary>
        ///CreateInstanceFromAppSettings&lt;&gt; (string) のテスト
        ///</summary>
        /// <remarks>
        /// 構成ファイルに設定する引数 appSettingsKey をキーとする add 要素にコンストラクタ呼び出しに
        /// 失敗する型のアセンブリ修飾名が設定された場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceFromAppSettingsStringTest10()
        {
            string appSettingsKey = "ClassUtilTest07";

            ClassUtil_Stub01AppSettings actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceFromAppSettings<ClassUtil_Stub01AppSettings>(appSettingsKey);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // メッセージ確認
                Assert.AreEqual(MessageManager.GetMessage("E_MEMBER_ACCESS_EXCEPTION", typeof(ClassUtil_Stub05).AssemblyQualifiedName), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(TerasolunaException), "E_MEMBER_ACCESS_EXCEPTION", typeof(ClassUtil_Stub05).AssemblyQualifiedName));
        }

        /// <summary>
        ///CreateInstanceFromAppSettings&lt;&gt; (string) のテスト
        ///</summary>
        /// <remarks>
        /// 構成ファイルに設定する引数 appSettingsKey をキーとする add 要素に呼び出したコンストラクタが
        /// 例外をスローする型のアセンブリ修飾名が設定された場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceFromAppSettingsStringTest11()
        {
            string appSettingsKey = "ClassUtilTest08";

            ClassUtil_Stub01AppSettings actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceFromAppSettings<ClassUtil_Stub01AppSettings>(appSettingsKey);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // メッセージ確認
                Assert.AreEqual(MessageManager.GetMessage("E_TARGET_INVOCATION_EXCEPTION", typeof(ClassUtil_Stub06).AssemblyQualifiedName), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(TerasolunaException), "E_TARGET_INVOCATION_EXCEPTION", typeof(ClassUtil_Stub06).AssemblyQualifiedName));
        }

        /// <summary>
        ///CreateInstanceFromAppSettings&lt;&gt; (string) のテスト
        ///</summary>
        /// <remarks>
        /// 構成ファイルに設定する引数 appSettingsKey をキーとする add 要素に型パラメータ T にキャストできない
        /// アセンブリ修飾名が設定された場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceFromAppSettingsStringTest12()
        {
            string appSettingsKey = "ClassUtilTest03";

            object actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceFromAppSettings<ClassUtil_Stub02>(appSettingsKey);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // メッセージ確認
                Assert.AreEqual(MessageManager.GetMessage("E_INVALID_CAST_TYPE_FROM_GENERICPARAM", typeof(ClassUtil_Stub01AppSettings).AssemblyQualifiedName, typeof(ClassUtil_Stub02).AssemblyQualifiedName), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(TerasolunaException), "E_INVALID_CAST_TYPE_FROM_GENERICPARAM", typeof(ClassUtil_Stub01AppSettings).AssemblyQualifiedName, typeof(ClassUtil_Stub02).AssemblyQualifiedName));
        }

        /// <summary>
        ///CreateInstanceFromAppSettings&lt;&gt; (string, Type) のテスト
        ///</summary>
        /// <remarks>
        /// appSettingsKey にnullが格納された場合、
        /// ArgumentNullException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceFromAppSettingsStringTypeTest01()
        {
            string appSettingsKey = null;
            Type defaultType = typeof(ClassUtil_Stub01DefaultType);

            ClassUtil_Stub01 actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceFromAppSettings<ClassUtil_Stub01>(appSettingsKey, defaultType);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // メッセージ確認
                Assert.AreEqual("appSettingsKey", e.ParamName);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "appSettingsKey"));
        }

        /// <summary>
        ///CreateInstanceFromAppSettings&lt;&gt; (string, Type) のテスト
        ///</summary>
        /// <remarks>
        /// appSettingsKey に空文字が格納された場合、
        /// ArgumentException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceFromAppSettingsStringTypeTest02()
        {
            string appSettingsKey = string.Empty;
            Type defaultType = typeof(ClassUtil_Stub01DefaultType);

            ClassUtil_Stub01 actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceFromAppSettings<ClassUtil_Stub01>(appSettingsKey, defaultType);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // インスタンス生成確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                // メッセージ確認
                Assert.AreEqual(e.Message, MessageManager.GetMessage("E_EMPTY_STRING", "appSettingsKey"));
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(ArgumentException), "E_EMPTY_STRING", "appSettingsKey"));
        }

        /// <summary>
        ///CreateInstanceFromAppSettings&lt;&gt; (string, Type) のテスト
        ///</summary>
        /// <remarks>
        /// defaultType にnullが格納された場合、
        /// ArgumentNullException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceFromAppSettingsStringTypeTest03()
        {
            string appSettingsKey = "ClassUtilTest01";
            Type defaultType = null;

            object actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceFromAppSettings<object>(appSettingsKey, defaultType);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // メッセージ確認
                Assert.AreEqual("defaultType", e.ParamName);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "defaultType"));
        }

        /// <summary>
        ///CreateInstanceFromAppSettings&lt;&gt; (string, Type) のテスト
        ///</summary>
        /// <remarks>
        /// defaultType に抽象クラスのアセンブリ修飾名が格納された場合、
        /// ArgumentException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceFromAppSettingsStringTypeTest04()
        {
            string appSettingsKey = "ClassUtilTest01";
            Type defaultType = typeof(ClassUtil_Stub02);

            ClassUtil_Stub01 actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceFromAppSettings<ClassUtil_Stub01>(appSettingsKey, defaultType);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // インスタンス生成確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                // メッセージ確認
                Assert.AreEqual(e.Message, MessageManager.GetMessage("E_PARAM_ABSTRACT_TYPE", "defaultType", defaultType.AssemblyQualifiedName));
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(ArgumentException), "E_PARAM_ABSTRACT_TYPE", "defaultType", defaultType.AssemblyQualifiedName));
        }

        /// <summary>
        ///CreateInstanceFromAppSettings&lt;&gt; (string, Type) のテスト
        ///</summary>
        /// <remarks>
        /// 構成ファイルに設定する引数 appSettingsKey をキーとする add 要素に
        /// 型パラメータ T にキャストができる型のアセンブリ修飾名、
        /// defaultType に型パラメータ T にキャストができる型が設定された場合、
        /// appSettingsKey の add 要素に設定されたインスタンスが生成されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceFromAppSettingsStringTypeTest05()
        {
            string appSettingsKey = "ClassUtilTest03";
            Type defaultType = typeof(ClassUtil_Stub01DefaultType);

            ClassUtil_Stub01 actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceFromAppSettings<ClassUtil_Stub01>(appSettingsKey, defaultType);

            // appSettingsKeyで指定された型のインスタンスが生成されることの確認
            Assert.IsInstanceOfType(actual, typeof(ClassUtil_Stub01AppSettings));

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
        }

        /// <summary>
        ///CreateInstanceFromAppSettings&lt;&gt; (string, Type) のテスト
        ///</summary>
        /// <remarks>
        /// 構成ファイルに設定する引数 appSettingsKey をキーとする add 要素に空文字が設定され、
        /// defaultType に型パラメータ T にキャストができる型が設定された場合、
        /// defaultType に設定されたインスタンスが生成されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceFromAppSettingsStringTypeTest06()
        {
            string appSettingsKey = "ClassUtilTest02";
            Type defaultType = typeof(ClassUtil_Stub01DefaultType);

            ClassUtil_Stub01 actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceFromAppSettings<ClassUtil_Stub01>(appSettingsKey, defaultType);

            // defaultTypeで指定された型のインスタンスが生成されていることの確認
            Assert.IsInstanceOfType(actual, typeof(ClassUtil_Stub01DefaultType));

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
        }

        /// <summary>
        ///CreateInstanceFromAppSettings&lt;&gt; (string, Type) のテスト
        ///</summary>
        /// <remarks>
        /// 構成ファイルに設定する引数 appSettingsKey をキーとする add 要素が存在せず、
        /// defaultType に型パラメータ T にキャストができる型が設定された場合、
        /// defaultType に設定されたインスタンスが生成されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceFromAppSettingsStringTypeTest07()
        {
            string appSettingsKey = "ClassUtilTest04";
            Type defaultType = typeof(ClassUtil_Stub01DefaultType);

            ClassUtil_Stub01 actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceFromAppSettings<ClassUtil_Stub01>(appSettingsKey, defaultType);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // メッセージ確認
                Assert.AreEqual(MessageManager.GetMessage("E_ABSTRACT_TYPE", typeof(ClassUtil_Stub02).AssemblyQualifiedName), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(TerasolunaException), "E_ABSTRACT_TYPE", typeof(ClassUtil_Stub02).AssemblyQualifiedName));
        }

        /// <summary>
        ///CreateInstanceFromAppSettings&lt;&gt; (string, Type) のテスト
        ///</summary>
        /// <remarks>
        /// 構成ファイルに設定する引数 appSettingsKey キーとする add 要素が存在せず、
        /// defaultType に型パラメータ T にキャストができる型が設定された場合、
        /// defaultType に設定されたインスタンスが生成されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceFromAppSettingsStringTypeTest08()
        {
            string appSettingsKey = "ClassUtilTest01";
            Type defaultType = typeof(ClassUtil_Stub01DefaultType);

            ClassUtil_Stub01 actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceFromAppSettings<ClassUtil_Stub01>(appSettingsKey, defaultType);

            // defaultTypeで指定された型のインスタンスが生成されていることの確認
            Assert.IsInstanceOfType(actual, typeof(ClassUtil_Stub01DefaultType));

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
        }

        /// <summary>
        ///CreateInstanceFromAppSettings&lt;&gt; (string, Type) のテスト
        ///</summary>
        /// <remarks>
        /// 構成ファイルに設定する引数 appSettingsKey をキーとする add 要素が存在せず、
        /// defaultType にジェネリック型が設定された場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceFromAppSettingsStringTypeTest10()
        {
            string appSettingsKey = "ClassUtilTest02";
            Type defaultType = typeof(ClassUtil_Stub03<>);

            object actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceFromAppSettings<object>(appSettingsKey, defaultType);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // メッセージ確認
                Assert.AreEqual(MessageManager.GetMessage("E_PARAM_GENERIC_TYPE", "defaultType", typeof(ClassUtil_Stub03<>).AssemblyQualifiedName), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(ArgumentException), "E_PARAM_GENERIC_TYPE", "defaultType", typeof(ClassUtil_Stub03<>).AssemblyQualifiedName));
        }

        /// <summary>
        ///CreateInstanceFromAppSettings&lt;&gt; (string, Type) のテスト
        ///</summary>
        /// <remarks>
        /// 構成ファイルに設定する引数 appSettingsKey をキーとする add 要素が存在せず、
        /// defaultType にコンストラクタ呼び出しに失敗する型が設定された場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceFromAppSettingsStringTypeTest11()
        {
            string appSettingsKey = "ClassUtilTest02";
            Type defaultType = typeof(ClassUtil_Stub05);

            object actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceFromAppSettings<object>(appSettingsKey, defaultType);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // メッセージ確認
                Assert.AreEqual(MessageManager.GetMessage("E_MEMBER_ACCESS_EXCEPTION", typeof(ClassUtil_Stub05).AssemblyQualifiedName), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(TerasolunaException), "E_MEMBER_ACCESS_EXCEPTION", typeof(ClassUtil_Stub05).AssemblyQualifiedName));
        }

        /// <summary>
        ///CreateInstanceFromAppSettings&lt;&gt; (string, Type) のテスト
        ///</summary>
        /// <remarks>
        /// 構成ファイルに設定する引数 appSettingsKey をキーとする add 要素が存在せず、
        /// defaultType にコンストラクタが例外をスローする型が設定された場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceFromAppSettingsStringTypeTest12()
        {
            string appSettingsKey = "ClassUtilTest02";
            Type defaultType = typeof(ClassUtil_Stub06);

            object actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceFromAppSettings<object>(appSettingsKey, defaultType);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // メッセージ確認
                Assert.AreEqual(MessageManager.GetMessage("E_TARGET_INVOCATION_EXCEPTION", typeof(ClassUtil_Stub06).AssemblyQualifiedName), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(TerasolunaException), "E_TARGET_INVOCATION_EXCEPTION", typeof(ClassUtil_Stub06).AssemblyQualifiedName));
        }

        /// <summary>
        ///CreateInstanceFromAppSettings&lt;&gt; (string, Type) のテスト
        ///</summary>
        /// <remarks>
        /// 構成ファイルに設定する引数 appSettingsKey をキーとする add 要素が存在せず、
        /// defaultType に型パラメータ T にキャストできない型が設定された場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceFromAppSettingsStringTypeTest13()
        {
            string appSettingsKey = "ClassUtilTest02";
            Type defaultType = typeof(ClassUtil_Stub01);

            object actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceFromAppSettings<ClassUtil_Stub02>(appSettingsKey, defaultType);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // メッセージ確認
                Assert.AreEqual(MessageManager.GetMessage("E_INVALID_CAST_TYPE_FROM_GENERICPARAM", typeof(ClassUtil_Stub01).AssemblyQualifiedName, typeof(ClassUtil_Stub02).AssemblyQualifiedName), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(TerasolunaException), "E_INVALID_CAST_TYPE_FROM_GENERICPARAM", typeof(ClassUtil_Stub01).AssemblyQualifiedName, typeof(ClassUtil_Stub02).AssemblyQualifiedName));
        }

        /// <summary>
        ///CreateInstanceFromAppSettings&lt;&gt; (string, Type) のテスト
        ///</summary>
        /// <remarks>
        /// 構成ファイルに設定する引数 appSettingsKey をキーとする add 要素が
        /// 型を取得することができない文字列の場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceFromAppSettingsStringTypeTest14()
        {
            string appSettingsKey = "ClassUtilTest05";
            Type defaultType = typeof(ClassUtil_Stub01);

            object actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceFromAppSettings<object>(appSettingsKey, defaultType);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // メッセージ確認
                Assert.AreEqual(MessageManager.GetMessage("E_TYPE_LOAD_FAILED", ConfigurationManager.AppSettings[appSettingsKey]), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(TerasolunaException), "E_TYPE_LOAD_FAILED", ConfigurationManager.AppSettings[appSettingsKey]));
        }

        /// <summary>
        ///CreateInstanceFromAppSettings&lt;&gt; (string, Type) のテスト
        ///</summary>
        /// <remarks>
        /// 構成ファイルに設定する引数 appSettingsKey をキーとする add 要素が
        /// ジェネリック型のアセンブリ修飾名の場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceFromAppSettingsStringTypeTest15()
        {
            string appSettingsKey = "ClassUtilTest06";
            Type defaultType = typeof(ClassUtil_Stub01);

            object actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceFromAppSettings<object>(appSettingsKey, defaultType);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // メッセージ確認
                Assert.AreEqual(MessageManager.GetMessage("E_GENERIC_TYPE", typeof(ClassUtil_Stub03<>).AssemblyQualifiedName), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(TerasolunaException), "E_GENERIC_TYPE", typeof(ClassUtil_Stub03<>).AssemblyQualifiedName));
        }

        /// <summary>
        ///CreateInstanceFromAppSettings&lt;&gt; (string, Type) のテスト
        ///</summary>
        /// <remarks>
        /// 構成ファイルに設定する引数 appSettingsKey をキーとする add 要素が
        /// コンストラクタ呼び出しに失敗する型のアセンブリ修飾名の場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceFromAppSettingsStringTypeTest16()
        {
            string appSettingsKey = "ClassUtilTest07";
            Type defaultType = typeof(ClassUtil_Stub01);

            object actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceFromAppSettings<object>(appSettingsKey, defaultType);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // メッセージ確認
                Assert.AreEqual(MessageManager.GetMessage("E_MEMBER_ACCESS_EXCEPTION", typeof(ClassUtil_Stub05).AssemblyQualifiedName), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(TerasolunaException), "E_MEMBER_ACCESS_EXCEPTION", typeof(ClassUtil_Stub05).AssemblyQualifiedName));
        }

        /// <summary>
        ///CreateInstanceFromAppSettings&lt;&gt; (string, Type) のテスト
        ///</summary>
        /// <remarks>
        /// 構成ファイルに設定する引数 appSettingsKey をキーとする add 要素が
        /// 呼び出したコンストラクタが例外をスローする型のアセンブリ修飾名の場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceFromAppSettingsStringTypeTest17()
        {
            string appSettingsKey = "ClassUtilTest08";
            Type defaultType = typeof(ClassUtil_Stub01);

            object actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceFromAppSettings<object>(appSettingsKey, defaultType);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // メッセージ確認
                Assert.AreEqual(MessageManager.GetMessage("E_TARGET_INVOCATION_EXCEPTION", typeof(ClassUtil_Stub06).AssemblyQualifiedName), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(TerasolunaException), "E_TARGET_INVOCATION_EXCEPTION", typeof(ClassUtil_Stub06).AssemblyQualifiedName));
        }

        /// <summary>
        ///CreateInstanceFromAppSettings&lt;&gt; (string, Type) のテスト
        ///</summary>
        /// <remarks>
        /// 構成ファイルに設定する引数 appSettingsKey をキーとする add 要素が
        /// 型パラメータ T にキャストできないアセンブリ修飾名の場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceFromAppSettingsStringTypeTest18()
        {
            string appSettingsKey = "ClassUtilTest03";
            Type defaultType = typeof(ClassUtil_Stub01);

            object actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceFromAppSettings<ClassUtil_Stub02>(appSettingsKey, defaultType);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // メッセージ確認
                Assert.AreEqual(MessageManager.GetMessage("E_INVALID_CAST_TYPE_FROM_GENERICPARAM", typeof(ClassUtil_Stub01AppSettings).AssemblyQualifiedName, typeof(ClassUtil_Stub02).AssemblyQualifiedName), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(TerasolunaException), "E_INVALID_CAST_TYPE_FROM_GENERICPARAM", typeof(ClassUtil_Stub01AppSettings).AssemblyQualifiedName, typeof(ClassUtil_Stub02).AssemblyQualifiedName));
        }

        /// <summary>
        ///CreateInstanceWithGenericTypes&lt;&gt; (string, params string[]) のテスト
        ///</summary>
        /// <remarks>
        /// typeName にnullが格納された場合、
        /// ArgumentNullException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceWithGenericTypesTest01()
        {
            string typeName = null;
            string[] genericTypeNames = new string[0];

            object actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceWithGenericTypes<object>(typeName, genericTypeNames);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // メッセージ確認
                Assert.AreEqual("typeName", e.ParamName);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "typeName"));
        }

        /// <summary>
        ///CreateInstanceWithGenericTypes&lt;&gt; (string, params string[]) のテスト
        ///</summary>
        /// <remarks>
        /// typeName に空文字が格納された場合、
        /// ArgumentException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceWithGenericTypesTest02()
        {
            string typeName = string.Empty;
            string[] genericTypeNames = new string[0];

            object actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceWithGenericTypes<object>(typeName, genericTypeNames);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // インスタンス生成確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                // メッセージ確認
                Assert.AreEqual(e.Message, MessageManager.GetMessage("E_EMPTY_STRING", "typeName"));
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(ArgumentException), "E_EMPTY_STRING", "typeName"));
        }

        /// <summary>
        ///CreateInstanceWithGenericTypes&lt;&gt; (string, params string[]) のテスト
        ///</summary>
        /// <remarks>
        /// typeName に抽象クラス型のアセンブリ修飾名が格納された場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceWithGenericTypesTest03()
        {
            string typeName = "TERASOLUNA.Fw.CommonUT.ClassUtil_Stub02, TERASOLUNA.Fw.CommonUT";
            string[] genericTypeNames = new string[0];

            object actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceWithGenericTypes<object>(typeName, genericTypeNames);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // メッセージ確認
                Assert.AreEqual(e.Message, MessageManager.GetMessage("E_PARAM_ABSTRACT_TYPE", "typeName", typeof(ClassUtil_Stub02).AssemblyQualifiedName));
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(TerasolunaException), "E_PARAM_ABSTRACT_TYPE", "typeName", typeof(ClassUtil_Stub02).AssemblyQualifiedName));
        }

        /// <summary>
        ///CreateInstanceWithGenericTypes&lt;&gt; (string, params string[]) のテスト
        ///</summary>
        /// <remarks>
        /// genericTypeNames にnullが格納された場合、
        /// ArgumentNullException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceWithGenericTypesTest04()
        {
            string typeName = "TERASOLUNA.Fw.CommonUT.ClassUtil_Stub01, TERASOLUNA.Fw.CommonUT";
            string[] genericTypeNames = null;

            ClassUtil_Stub01 actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceWithGenericTypes<ClassUtil_Stub01>(typeName, genericTypeNames);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // メッセージ確認
                Assert.AreEqual("genericTypeNames", e.ParamName);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "genericTypeNames"));
        }

        /// <summary>
        ///CreateInstanceWithGenericTypes&lt;&gt; (string, params string[]) のテスト
        ///</summary>
        /// <remarks>
        /// typeName に、型パラメータ T にキャストができる型のアセンブリ修飾名かつ、
        /// ジェネリクスパタメータは持たない型名が設定され、
        /// genericTypeNames の要素数が0個の場合、
        /// 正常にインスタンスが生成されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceWithGenericTypesTest05()
        {
            string typeName = "TERASOLUNA.Fw.CommonUT.ClassUtil_Stub01, TERASOLUNA.Fw.CommonUT";
            string[] genericTypeNames = new string[0];

            ClassUtil_Stub01 actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceWithGenericTypes<ClassUtil_Stub01>(typeName, genericTypeNames);

            Assert.IsNotNull(actual);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
        }

        /// <summary>
        ///CreateInstanceWithGenericTypes&lt;&gt; (string, params string[]) のテスト
        ///</summary>
        /// <remarks>
        /// genericTypeNames の要素にnullが含まれる場合、
        /// ArgumentNullException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceWithGenericTypesTest06()
        {
            string typeName = "TERASOLUNA.Fw.CommonUT.ClassUtil_Stub03`1, TERASOLUNA.Fw.CommonUT";
            string[] genericTypeNames = new string[] { null };

            ClassUtil_Stub03<string> actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceWithGenericTypes<ClassUtil_Stub03<string>>(typeName, genericTypeNames);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // インスタンス生成確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                // メッセージ確認
                Assert.AreEqual(e.Message, MessageManager.GetMessage("E_NULL_ARGUMENT", "genericTypeNames[0]"));
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(ArgumentException), "E_NULL_ARGUMENT", "genericTypeNames[0]"));
        }

        /// <summary>
        ///CreateInstanceWithGenericTypes&lt;&gt; (string, params string[]) のテスト
        ///</summary>
        /// <remarks>
        /// genericTypeNames の要素に空文字が含まれる場合、
        /// ArgumentException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceWithGenericTypesTest07()
        {
            string typeName = "TERASOLUNA.Fw.CommonUT.ClassUtil_Stub03`1, TERASOLUNA.Fw.CommonUT";
            string[] genericTypeNames = new string[] { string.Empty };

            ClassUtil_Stub03<string> actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceWithGenericTypes<ClassUtil_Stub03<string>>(typeName, genericTypeNames);

                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // インスタンス生成確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);
                // メッセージ確認
                Assert.AreEqual(e.Message, MessageManager.GetMessage("E_EMPTY_STRING", "genericTypeNames[0]"));
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(ArgumentException), "E_EMPTY_STRING", "genericTypeNames[0]"));
        }

        /// <summary>
        ///CreateInstanceWithGenericTypes&lt;&gt; (string, params string[]) のテスト
        ///</summary>
        /// <remarks>
        /// typeName に、型パラメータ T にキャストができる型のアセンブリ修飾名かつ
        /// 制約のないジェネリクスパタメータを1つ持つ型名が設定され、
        /// genericTypeNamesの要素数が1個の場合、
        /// 正常にインスタンスが生成されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceWithGenericTypesTest08()
        {
            string typeName = "TERASOLUNA.Fw.CommonUT.ClassUtil_Stub03`1, TERASOLUNA.Fw.CommonUT";
            string[] genericTypeNames = new string[] { "System.String" };

            ClassUtil_Stub03<string>  actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceWithGenericTypes<ClassUtil_Stub03<string>>(typeName, genericTypeNames);

            Assert.IsNotNull(actual);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
        }


        /// <summary>
        ///CreateInstanceWithGenericTypes&lt;&gt; (string, params string[]) のテスト
        ///</summary>
        /// <remarks>
        /// typeName が、型パラメータ T にキャストができる型のアセンブリ修飾名かつ
        /// 制約のあるジェネリクスパタメータを3つ持つ型名で、
        /// genericTypeNames の要素数が3個かつジェネリクスパラメータの制約を満たす場合、
        /// 正常にインスタンスが生成されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceWithGenericTypesTest09()
        {
            string typeName = "TERASOLUNA.Fw.CommonUT.ClassUtil_Stub04`3, TERASOLUNA.Fw.CommonUT";
            string[] genericTypeNames = new string[] { "System.String", "System.Int32", "System.Collections.Generic.IList`1[System.String]" };

            ClassUtil_Stub04<string, Int32, IList<string>> actual = null;
            actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceWithGenericTypes<ClassUtil_Stub04<string, Int32, IList<string>>>(typeName, genericTypeNames);

            Assert.IsNotNull(actual);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
        }

        /// <summary>
        ///CreateInstanceWithGenericTypes&lt;&gt; (string, params string[]) のテスト
        ///</summary>       
        /// <remarks>
        /// typeName が、型パラメータ T にキャストができる型のアセンブリ修飾名かつ
        /// 制約のあるジェネリクスパタメータを1つ持つ型名で、
        /// genericTypeNames の要素が1個かつかつジェネリクスパラメータの制約を満たさない場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceWithGenericTypesTest10()
        {
            string typeName = "TERASOLUNA.Fw.CommonUT.ClassUtil_Stub07`1, TERASOLUNA.Fw.CommonUT";
            string[] genericTypeNames = new string[] { "System.String" };

            ClassUtil_Stub07<List<string>> actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceWithGenericTypes<ClassUtil_Stub07<List<string>>>(typeName, genericTypeNames);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // メッセージ確認
                Assert.AreEqual(e.Message, MessageManager.GetMessage("E_MAKE_GENERICTYPE_FAILED", Type.GetType(typeName, true).AssemblyQualifiedName));
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(TerasolunaException), "E_MAKE_GENERICTYPE_FAILED", Type.GetType(typeName, true).AssemblyQualifiedName));
        }

        /// <summary>
        ///CreateInstanceWithGenericTypes&lt;&gt; (string, params string[]) のテスト
        ///</summary>
        /// <remarks>
        /// typeName がジェネリクスパタメータを1つ持つ型名で、
        /// genericTypeNames の要素が3個の場合、
        /// 正常にインスタンスが生成されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceWithGenericTypesTest11()
        {
            string typeName = "TERASOLUNA.Fw.CommonUT.ClassUtil_Stub03`1, TERASOLUNA.Fw.CommonUT";
            string[] genericTypeNames = new string[] { "System.String", "System.Int32", "System.Collections.Generic.IList`1[System.String]" };

            ClassUtil_Stub03<string> actual = null;
            actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceWithGenericTypes<ClassUtil_Stub03<string>>(typeName, genericTypeNames);

            // インスタンス生成
            Assert.IsNotNull(actual);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
        }

        /// <summary>
        ///CreateInstanceWithGenericTypes&lt;&gt; (string, params string[]) のテスト
        ///</summary>
        /// <remarks>
        /// typeName がジェネリクスパタメータを3つ持つ型名で、
        /// genericTypeNames に要素が1個の場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceWithGenericTypesTest12()
        {
            string typeName = "TERASOLUNA.Fw.CommonUT.ClassUtil_Stub04`3, TERASOLUNA.Fw.CommonUT";
            string[] genericTypeNames = new string[] { "System.String" };

            ClassUtil_Stub04<string, object, List<string>> actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceWithGenericTypes<ClassUtil_Stub04<string, object, List<string>>>(typeName, genericTypeNames);

                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // メッセージ確認
                Assert.AreEqual(e.Message, MessageManager.GetMessage("E_LESS_PARAMS", Type.GetType(typeName, true).AssemblyQualifiedName, 3));
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(TerasolunaException), "E_LESS_PARAMS", Type.GetType(typeName, true).AssemblyQualifiedName, "3"));
        }

        /// <summary>
        ///CreateInstanceWithGenericTypes&lt;&gt; (string, params string[]) のテスト
        ///</summary>
        /// <remarks>
        /// typeName に型を取得できない文字列を指定した場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceWithGenericTypesTest13()
        {
            string typeName = "TERASOLUNA.Fw.CommonUT.NoExist_ClassUtil_Stub, TERASOLUNA.Fw.CommonUT";
            string[] genericTypeNames = new string[0];

            object actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceWithGenericTypes<object>(typeName, genericTypeNames);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // メッセージ確認
                Assert.AreEqual(MessageManager.GetMessage("E_TYPE_LOAD_FAILED", typeName), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(TerasolunaException), "E_TYPE_LOAD_FAILED", typeName));
        }

        /// <summary>
        ///CreateInstanceWithGenericTypes&lt;&gt; (string, params string[]) のテスト
        ///</summary>
        /// <remarks>
        /// typeName にコンストラクタ呼び出しに失敗する型のアセンブリ修飾名を指定した場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceWithGenericTypesTest14()
        {
            string typeName = "TERASOLUNA.Fw.CommonUT.ClassUtil_Stub05, TERASOLUNA.Fw.CommonUT";
            string[] genericTypeNames = new string[0];

            object actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceWithGenericTypes<object>(typeName, genericTypeNames);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // メッセージ確認
                Assert.AreEqual(MessageManager.GetMessage("E_MEMBER_ACCESS_EXCEPTION", typeof(ClassUtil_Stub05).AssemblyQualifiedName), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(TerasolunaException), "E_MEMBER_ACCESS_EXCEPTION", typeof(ClassUtil_Stub05).AssemblyQualifiedName));
        }

        /// <summary>
        ///CreateInstanceWithGenericTypes&lt;&gt; (string, params string[]) のテスト
        ///</summary>
        /// <remarks>
        /// typeName にコンストラクタで例外がスローされる型のアセンブリ修飾名を指定した場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceWithGenericTypesTest15()
        {
            string typeName = "TERASOLUNA.Fw.CommonUT.ClassUtil_Stub06, TERASOLUNA.Fw.CommonUT";
            string[] genericTypeNames = new string[0];

            object actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceWithGenericTypes<object>(typeName, genericTypeNames);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // メッセージ確認
                Assert.AreEqual(MessageManager.GetMessage("E_TARGET_INVOCATION_EXCEPTION", typeof(ClassUtil_Stub06).AssemblyQualifiedName), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(TerasolunaException), "E_TARGET_INVOCATION_EXCEPTION", typeof(ClassUtil_Stub06).AssemblyQualifiedName));
        }

        /// <summary>
        ///CreateInstanceWithGenericTypes&lt;&gt; (string, params string[]) のテスト
        ///</summary>
        /// <remarks>
        /// typeName に型パラメータ T にキャストができない型のアセンブリ修飾名を指定した場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceWithGenericTypesTest16()
        {
            string typeName = "TERASOLUNA.Fw.CommonUT.ClassUtil_Stub01, TERASOLUNA.Fw.CommonUT";
            string[] genericTypeNames = new string[0];

            object actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceWithGenericTypes<ClassUtil_Stub02>(typeName, genericTypeNames);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // メッセージ確認
                Assert.AreEqual(MessageManager.GetMessage("E_INVALID_CAST_TYPE_FROM_GENERICPARAM", typeof(ClassUtil_Stub01).AssemblyQualifiedName, typeof(ClassUtil_Stub02).AssemblyQualifiedName), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(TerasolunaException), "E_INVALID_CAST_TYPE_FROM_GENERICPARAM", typeof(ClassUtil_Stub01).AssemblyQualifiedName, typeof(ClassUtil_Stub02).AssemblyQualifiedName));
        }

        /// <summary>
        ///CreateInstanceWithGenericTypes&lt;&gt; (string, params string[]) のテスト
        ///</summary>
        /// <remarks>
        /// genericTypeNames の要素に型を取得できない文字列が含まれる場合、
        /// TerasolunaException がスローされることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceWithGenericTypesTest17()
        {
            string typeName = "TERASOLUNA.Fw.CommonUT.ClassUtil_Stub03`1, TERASOLUNA.Fw.CommonUT";
            string[] genericTypeNames = new string[] { "TERASOLUNA.Fw.CommonUT.NoExist_ClassUtil_Stub, TERASOLUNA.Fw.CommonUT" };

            ClassUtil_Stub03<string> actual;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceWithGenericTypes<ClassUtil_Stub03<string>>(typeName, genericTypeNames);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // メッセージ確認
                Assert.AreEqual(MessageManager.GetMessage("E_TYPE_LOAD_FAILED", genericTypeNames[0]), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(TerasolunaException), "E_TYPE_LOAD_FAILED", genericTypeNames[0]));
        }

        /// <summary>
        ///CreateInstanceWithGenericTypes&lt;&gt; (string, params string[]) のテスト
        ///</summary>
        /// <remarks>
        /// typeName がジェネリクスパタメータを1つ持つ型名で、
        /// genericTypeNames の要素が3個の場合、
        /// 正常にインスタンスが生成されることを確認するテスト。
        /// </remarks>
        [TestMethod()]
        public void CreateInstanceWithGenericTypesTest18()
        {
            string typeName = "TERASOLUNA.Fw.CommonUT.ClassUtil_Stub03`1, TERASOLUNA.Fw.CommonUT";
            string[] genericTypeNames = new string[] { typeof(IList<>).AssemblyQualifiedName };

            ClassUtil_Stub03<IList<string>> actual = null;
            try
            {
                actual = TERASOLUNA.Fw.Common.ClassUtil.CreateInstanceWithGenericTypes<ClassUtil_Stub03<IList<string>>>(typeName, genericTypeNames);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // メッセージ確認
                Assert.AreEqual(MessageManager.GetMessage("E_GENERIC_TYPE", typeof(IList<>).AssemblyQualifiedName), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(TerasolunaException), "E_GENERIC_TYPE", typeof(IList<>).AssemblyQualifiedName));
        }
        /// <summary>
        ///GetTypeFromTypeName (string) のテスト
        ///</summary>
        ///<remarks>
        ///typeName が null の場合、
        /// ArgumentNullException がスローされることを確認するテスト
        ///</remarks>
        [TestMethod()]
        public void GetTypeFromTypeNameTest01()
        {
            string typeName = null;

            Type actual;

            try
            {
                // 実施
                actual = TERASOLUNA.Fw.Common.ClassUtil.GetType(typeName);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentNullException e)
            {
                // メッセージ確認
                Assert.AreEqual("typeName", e.ParamName);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(ArgumentNullException), "E_NULL_ARGUMENT", "typeName"));
        }

        /// <summary>
        ///GetTypeFromTypeName (string) のテスト
        ///</summary>
        ///<remarks>
        ///typeName が空文字の場合、
        ///ArgumentException がスローされることを確認するテスト
        ///</remarks>
        [TestMethod()]
        public void GetTypeFromTypeNameTest02()
        {
            string typeName = string.Empty;

            Type actual;
            try
            {
                // 実施
                actual = TERASOLUNA.Fw.Common.ClassUtil.GetType(typeName);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (ArgumentException e)
            {
                // インスタンス生成確認
                Assert.AreEqual(typeof(ArgumentException).Name, e.GetType().Name);

                // メッセージ確認
                Assert.AreEqual(MessageManager.GetMessage("E_EMPTY_STRING", "typeName"), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(ArgumentException), "E_EMPTY_STRING", "typeName"));
        }

        /// <summary>
        ///GetTypeFromTypeName (string) のテスト
        ///</summary>
        ///<remarks>
        ///typeName が型を取得することができない文字列の場合、
        ///TerasolunaException がスローされることを確認するテスト
        ///</remarks>
        [TestMethod()]
        public void GetTypeFromTypeNameTest03()
        {
            string typeName = "TERASOLUNA";

            Type actual;
            try
            {
                // 実施
                actual = TERASOLUNA.Fw.Common.ClassUtil.GetType(typeName);
                Assert.Fail("到達してはいけないコードです。");
            }
            catch (TerasolunaException e)
            {
                // メッセージ確認
                Assert.AreEqual(MessageManager.GetMessage("E_TYPE_LOAD_FAILED", typeName), e.Message);
            }
            catch (Exception e)
            {
                Assert.Fail("到達してはいけないコードです。" + e.ToString());
            }

            // ログ確認
            Assert.AreEqual(1, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
            Assert.IsTrue(LogUTUtil.CheckErrorUseMessageId(typeof(ClassUtil), typeof(TerasolunaException), "E_TYPE_LOAD_FAILED", typeName));
        }

        ///<summary>
        ///GetTypeFromTypeName (string) のテスト
        ///</summary>
        ///<remarks>
        ///typeName が型が取得できる文字列の場合、
        ///正常にインスタンスが生成されることを確認するテスト
        ///</remarks>
        [TestMethod()]
        public void GetTypeFromTypeNameTest04()
        {
            string typeName = "TERASOLUNA.Fw.CommonUT.ClassUtil_Stub01, TERASOLUNA.Fw.CommonUT";

            Type actual;

            // 実施
            actual = TERASOLUNA.Fw.Common.ClassUtil.GetType(typeName);

            // インスタンス生成確認
            Assert.IsNotNull(actual);

            // ログ確認
            Assert.AreEqual(0, LogUTUtil.CountLogs(typeof(ClassUtil), LogLevel.INFO));
        }
    }


    /// <summary>
    /// テスト用非ジェネリクスクラス。
    /// </summary>
    public class ClassUtil_Stub01
    {

    }

    /// <summary>
    /// テスト用抽象クラス。
    /// </summary>
    public abstract class ClassUtil_Stub02
    {

    }

    /// <summary>
    /// テスト用ジェネリクスクラス。パラメータ１つ。
    /// </summary>
    public class ClassUtil_Stub03<A>
    {

    }

    /// <summary>
    /// テスト用ジェネリクスクラス。パラメータ３つ。制約あり。
    /// </summary>
    public class ClassUtil_Stub04<A, B, C> where C : IList<string>
    {

    }

    /// <summary>
    /// テスト用コンストラクタ呼び出しに失敗するクラス。
    /// </summary>
    public class ClassUtil_Stub05
    {
        private ClassUtil_Stub05()
        {
        }
    }

    /// <summary>
    /// テスト用コンストラクタで例外が発生するクラス。
    /// </summary>
    public class ClassUtil_Stub06
    {
        public ClassUtil_Stub06()
        {
            throw new InvalidOperationException();
        }
    }


    /// <summary>
    /// テスト用ジェネリクスクラス。パラメータ１つ。制約あり
    /// </summary>
    public class ClassUtil_Stub07<A> where A : List<string>
    {

    }

    /// <summary>
    /// テスト用ジェネリクスクラス。appSettings用。パラメータ１つ。
    /// </summary>
    public class ClassUtil_Stub01AppSettings : ClassUtil_Stub01
    {

    }

    /// <summary>
    /// テスト用ジェネリクスクラス。defaultType用。パラメータ１つ。
    /// </summary>
    public class ClassUtil_Stub01DefaultType : ClassUtil_Stub01
    {

    }


}
