/*
 * Copyright (c) 2007 NTT DATA Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jp.terasoluna.fw.file.dao.standard;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;
import java.io.Writer;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jp.terasoluna.fw.file.annotation.FileFormat;
import jp.terasoluna.fw.file.annotation.OutputFileColumn;
import jp.terasoluna.fw.file.annotation.PaddingType;
import jp.terasoluna.fw.file.annotation.StringConverter;
import jp.terasoluna.fw.file.annotation.TrimType;
import jp.terasoluna.fw.file.dao.FileException;
import jp.terasoluna.fw.file.dao.FileLineException;
import jp.terasoluna.fw.file.dao.FileLineWriter;

import org.apache.commons.lang.StringUtils;

/**
 * t@CANZX(f[^)p̋ʃNXB
 * 
 * <p>
 * t@CANZX(f[^)s3̃NX(CSVAŒ蒷Aϒ) ɋʂ鏈܂Ƃ߂ۃNXB
 * t@C̎ނɑΉTuNXsB<br>
 * gp{@link jp.terasoluna.fw.file.dao.FileLineWriter}QƂ̂ƁB
 * </p>
 * <p>
 * t@C擾͉L̎菇ŌĂяo悤Ɏ邱ƁB
 * <ul>
 * <li>(init()ACX^XKPsȂ)</li>
 * <li>wb_擾(printHeaderLine())</li>
 * <li>f[^擾(printDataLine())</li>
 * <li>gC擾(printTrailerLine())</li>
 * </ul>
 * L̏Ԃł̂ݐmɏo͂łB<br>
 * 
 * </p>
 * 
 * @see jp.terasoluna.fw.file.dao.FileLineWriter
 * @see jp.terasoluna.fw.file.dao.standard.CSVFileLineWriter
 * @see jp.terasoluna.fw.file.dao.standard.FixedFileLineWriter
 * @see jp.terasoluna.fw.file.dao.standard.VariableFileLineWriter
 * @see jp.terasoluna.fw.file.dao.standard.PlainFileLineWriter
 * @param <T> t@CsIuWFNgB
 */
public abstract class AbstractFileLineWriter<T>
        implements FileLineWriter<T> {

    /**
     * ̍sԍB
     */
    private static final int INITIAL_LINE_NO = -1;
    
    /**
     * t@CANZXiójp̕Xg[B
     */
    private Writer writer = null;

    /**
     * t@CANZXst@CB
     */
    private String fileName = null;

    /**
     * t@CGR[fBOB
     */
    private String fileEncoding = System.getProperty("file.encoding");

    /**
     * p[^NX̃NXB
     */
    private Class<T> clazz = null;

    /**
     * s؂蕶B
     */
    private String lineFeedChar = System.getProperty("line.separator");

    /**
     * t@CsIuWFNgFieldiAnnotationji[ϐB
     */
    private Field[] fields = null;

    /**
     * t@CsIuWFNg̏o͐ݒAme[Vi[ϐB
     */
    private OutputFileColumn[] outputFileColumns = null;

    /**
     * eJƂ̃JIndexi[ϐB
     */
    private int[] columnIndexs = null;

    /**
     * eJƂ̃J̃tH[}bgi[ϐB
     */
    private String[] columnFormats = null;

    /**
     * eJƂ̃oCgi[ϐB
     */
    private int[] columnBytes = null;

    /**
     * eJƂ̃pfBOʂi[ϐB
     */
    private PaddingType[] paddingTypes = null;

    /**
     * eJƂ̃pfBOi[ϐB
     */
    private char[] paddingChars = null;

    /**
     * eJƂ̃gʂi[ϐB
     */
    private TrimType[] trimTypes;

    /**
     * eJ̃gi[ϐB
     */
    private char[] trimChars;

    /**
     * t@CsIuWFNg̃XgORo[^i[ϐB
     */
    private StringConverter[] stringConverters = null;

    /**
     * t@CsIuWFNg̃XgORo[^i[}bvB
     */
    private static Map<Class, StringConverter> stringConverterCacheMap =
        new HashMap<Class, StringConverter>();

    /**
     * t@CsIuWFNg̑̃tB[hIuWFNg̔zi[ϐB
     * 
     */
    private Field[] fieldArray = null;

    /**
     * \bhIuWFNg
     */
    private Method[] methods = null;

    /**
     * JtH[}bg(t@Cji[}bvB
     */
    private Map<String, ColumnFormatter> columnFormatterMap = null;

    /**
     * f[^o͊mFptOB
     */
    private boolean writeData = false;

    /**
     * gCo͊mFptOB
     */
    private boolean writeTrailer = false;

    /**
     * ݏς݃f[^̍sB
     */
    private int currentLineCount = 0;

    
    /**
     * stOB
     */
    private boolean calledInit = false;
    
    /**
     * RXgN^B<br>
     * <code>@FileFormat</code>Ame[V̐ݒ`FbNB
     * <code>@FileFormat</code>Ame[Vݒ肳ĂȂꍇ͔񌟍OX[B<br>
     * ؂蕶ƈ͂ݕɓꕶݒ肳Ăꍇ́A񌟍OX[B<br>
     * s؂蕶Rȏ̏ꍇ́A񌟍OX[B<br>
     * 
     * @param fileName t@C
     * @param clazz p[^NX
     * @param columnFormatterMap eLXg擾[
     */
    public AbstractFileLineWriter(String fileName, Class<T> clazz,
            Map<String, ColumnFormatter> columnFormatterMap) {

        if (fileName == null || "".equals(fileName)) {
            throw new FileException("fileName is required.",
                    new IllegalArgumentException(), fileName);
        }

        if (clazz == null) {
            throw new FileException("clazz is required.",
                    new IllegalArgumentException(), fileName);
        }

        if (columnFormatterMap == null || columnFormatterMap.isEmpty()) {
            throw new FileException("columnFormatterMap is required.",
                    new IllegalArgumentException(), fileName);
        }
        
        this.fileName = fileName;
        this.clazz = clazz;
        this.columnFormatterMap = columnFormatterMap;

        
        // FileFormatɊւ`FbNB
        
        // t@CtH[}bg擾B
        FileFormat fileFormat = clazz.getAnnotation(FileFormat.class);

        // @FileFormatꍇAOX[B
        if (fileFormat == null) {
            throw new FileException("FileFormat annotation is not found.",
                    new IllegalStateException(), fileName);
        }
        
        // ؂蕶ƈ͂ݕꍇAOX[B
        if (fileFormat.delimiter() == fileFormat.encloseChar()) {
            throw new FileException("Delimiter is the same as EncloseChar and is no use.",
                    new IllegalStateException(), fileName);
        } 
        
        // R[hB
        if (fileFormat.fileEncoding() != null
                && !"".equals(fileFormat.fileEncoding())) {
            this.fileEncoding = fileFormat.fileEncoding();
        }

        // s؂蕶B
        if (fileFormat.lineFeedChar() != null
                && !"".equals(fileFormat.lineFeedChar())) {
            this.lineFeedChar = fileFormat.lineFeedChar();
        }

        // s؂蕶3ȏ̏ꍇAOX[B
        if (lineFeedChar.length() > 2) {
            throw new FileException(
                    "lineFeedChar length must be 1 or 2. but: "
                    + lineFeedChar.length(),
                    new IllegalStateException(), fileName);
        }
    
    }

    /**
     * sB<br>
     * ōs͈ȉłBB
     * <ul>
     *   <li>t@CsIuWFNg̑(Field)̎擾</li>
     *   <li>ϊʃIuWFNg(stringConverters)̐</li>
     *   <li>t@CsIuWFNg̑ɑ΂Zb^\bh(methods)̎擾</li>
     *   <li>Ώۃt@C̏㏑ݒ̊mF</li>
     *   <li>t@Cւ̃Xg[J</li>
     * </ul>
     * init()AbstracFileLineWriterpNX̃RXgN^
     * Ăԃ\bhłB<br>
     * ʌ݊̂߁A2ȏsłȂ悤ɂĂB
     */
    protected void init() {
        if (!calledInit) {
            // t@CtH[}bg擾B
            FileFormat fileFormat = clazz.getAnnotation(FileFormat.class);
    
            buildFields();
            buildStringConverters();
            buildMethods();
    
            // ㏑tOmF
            if (fileFormat.overWriteFlg()) {
                File file = new File(fileName);
                file.delete();
            }
    
            // t@CI[v
            try {
                writer = new BufferedWriter(new OutputStreamWriter(
                        (new FileOutputStream(fileName, true)), fileEncoding));
            } catch (UnsupportedEncodingException e) {
                throw new FileException("Failed in generation of writer.",
                        e, fileName);
            } catch (FileNotFoundException e) {
                throw new FileException("Failed in generation of writer.",
                        e, fileName);
            }
            calledInit = true;
        }
    }

    /**
     * t@CsIuWFNg̑̃tB[hIuWFNg̔z𐶐B
     */
    private void buildFields() {
        List<Field[]> fieldList = new ArrayList<Field[]>();

        // tB[hIuWFNg𐶐
        Class tempClass = clazz;
        Field[] declaredFieldArray = null;
        int allFieldCount = 0;
        while (tempClass != null) {
            declaredFieldArray = tempClass.getDeclaredFields();
            fieldList.add(declaredFieldArray);
            allFieldCount += declaredFieldArray.length;
            tempClass = tempClass.getSuperclass();
        }

        // JCfbNX̒`̏Ԃɕёւ
        Field[] dataColumnFields = new Field[allFieldCount];

        OutputFileColumn outputFileColumn = null;
        int maxColumnIndex = -1;
        int columnIndex = -1;
        int columnCount = 0;
        
        for (Field[] fields : fieldList) {
            for (Field field : fields) {
                outputFileColumn = field.getAnnotation(OutputFileColumn.class);
                if (outputFileColumn != null) {
                    // }bsO\Ȍ^̃tB[hȂ̂mFB
                    if (columnFormatterMap.get(field.getType().getName())
                            == null) {
                        throw new FileException(
                                "There is a type which isn't supported in a "
                                + "mapping target field in FileLineObject.",
                                new IllegalStateException(), fileName);
                    }
                    
                    columnIndex = outputFileColumn.columnIndex();
                    // JIndex}CiXlȂ̂mFB
                    if (columnIndex < 0) {
                        throw new FileException(
                                "Column Index in FileLineObject is the minus " +
                                "number.",
                                new IllegalStateException(), fileName);
                    }
                    // JIndextB[h𒴂Ă邩邩mFB
                    if (dataColumnFields.length <= columnIndex) {
                        throw new FileException(
                                "Column Index in FileLineObject is bigger than "
                                + "the total number of the field.",
                                new IllegalStateException(), fileName);
                    }
                    // JIndexdĂȂ̂mFB
                    if (dataColumnFields[columnIndex] == null) {
                        dataColumnFields[columnIndex] = field;
                        if (maxColumnIndex < columnIndex) {
                            maxColumnIndex = columnIndex;
                        }
                        columnCount++;
                    } else {
                        throw new FileException("Column Index is duplicate : "
                                + columnIndex, fileName);
                    }
                }
            }
        }
        // columnIndexAԂŒ`Ă邩`FbN
        if (columnCount != (maxColumnIndex + 1)) {
            throw new FileException(
                    "columnIndex in FileLineObject is not sequential order.",
                    new IllegalStateException(), fileName);
        }
        
        this.fieldArray = dataColumnFields;
        
        // tB[hRs[(null̕폜)
        if (dataColumnFields.length == columnCount) {
            this.fields = dataColumnFields;
        } else {        
            this.fields = new Field[columnCount];
            System.arraycopy(dataColumnFields, 0, this.fields, 0, columnCount);
        }

        // OutputFileColumniAme[VjListIuWFNgɊi[
        outputFileColumns = new OutputFileColumn[fields.length];
        columnIndexs = new int[fields.length];
        columnFormats = new String[fields.length];
        columnBytes = new int[fields.length];
        paddingTypes = new PaddingType[fields.length];
        paddingChars = new char[fields.length];
        trimTypes = new TrimType[fields.length];
        trimChars = new char[fields.length];
        for (int i = 0 ; i < fields.length; i++) {
            outputFileColumns[i] = fields[i].getAnnotation(OutputFileColumn.class);
            columnIndexs[i] = outputFileColumns[i].columnIndex();
            columnFormats[i] = outputFileColumns[i].columnFormat();
            columnBytes[i] = outputFileColumns[i].bytes();
            paddingTypes[i] = outputFileColumns[i].paddingType();
            paddingChars[i] = outputFileColumns[i].paddingChar();
            trimTypes[i] = outputFileColumns[i].trimType();
            trimChars[i] = outputFileColumns[i].trimChar();
        }

    }

    /**
     * t@CsIuWFNg̑̕ϊʃIuWFNg̔z𐶐B<br>
     */
    private void buildStringConverters() {

        // ϊʂ̔z𐶐
        StringConverter[] dataColumnStringConverters =
            new StringConverter[fields.length];

        OutputFileColumn outputFileColumn = null;
        Class<? extends StringConverter> converterKind = null;
        
        for (int i = 0; i < fields.length; i++) {
            // JavaBean̓͗p̃Ame[V擾B
//            outputFileColumn = fields[i].getAnnotation(OutputFileColumn.class);
            outputFileColumn = outputFileColumns[i];

            // OutputFileColumn.stringConverter()̓eɂ菈U蕪B
            try {
                // ϊʂ̃Ame[V擾B
                converterKind = outputFileColumn.stringConverter();

                // }bvɎ擾ϊʂƈvL[݂邩肷B
                if (stringConverterCacheMap.containsKey(converterKind)) {
                    // }bvIuWFNg擾Aϊʂ̔zɃZbgB
                    dataColumnStringConverters[i] =
                        stringConverterCacheMap.get(converterKind);
                } else {
                    // CX^X𐶐Aϊʂ̔zɃZbgB
                    dataColumnStringConverters[i] = converterKind.newInstance();
                    stringConverterCacheMap.put(converterKind,
                            dataColumnStringConverters[i]);
                }

            } catch (InstantiationException e) {
                throw new FileLineException(
                        "Failed in an instantiate of a stringConverter.",
                        e, fileName, INITIAL_LINE_NO, 
                        fields[i].getName(), outputFileColumn.columnIndex());

            } catch (IllegalAccessException e) {
                throw new FileLineException(
                        "Failed in an instantiate of a stringConverter.",
                        e, fileName, INITIAL_LINE_NO, 
                        fields[i].getName(), outputFileColumn.columnIndex());
            }
        }
        this.stringConverters = dataColumnStringConverters;
    }

    /**
     * t@CsIuWFNg̑getter\bh̃\bhIuWFNg̔z𐶐B<br>
     * ɑ΂getter\bh͈ȉ̃[ŌB<br>
     * <ul>
     * <li>
     * ̍ŏ̕啶ɂ̐擪Ɂugetv́B
     * </li>
     * <li>
     * is`()Ahas`()Ȃǂgetter\bh͌ΏۊOłB
     * </li>
     * </ul>
     * 
     * getter\bhłȂꍇ͗OB
     * 
     * @throws FileException getter\bhȂꍇB
     */
    private void buildMethods() {
        Method[] dataColumnGetMethods = new Method[fields.length];
        StringBuilder getterName = new StringBuilder();
        String fieldName = null;
        
        for (int i = 0; i < fields.length; i++) {
            // JavaBean珈̑ΏۂƂȂ鑮̑擾B
            fieldName = fields[i].getName();

            // ɁAgetter\bh̖O𐶐B
            getterName.setLength(0);
            getterName.append("get");
            getterName.append(StringUtils.upperCase(fieldName.substring(0,
                    1)));
            getterName.append(fieldName.substring(1, fieldName.length()));

            // getter̃tNVIuWFNg擾B
            try {
                dataColumnGetMethods[i] =
                    clazz.getMethod(getterName.toString());
            } catch (NoSuchMethodException e) {
                throw new FileException(
                        "The getter method of column doesn't exist.",
                        e, fileName);
            }
        }
        this.methods = dataColumnGetMethods;
    }

    /**
     * wb_ւ̏ݏB
     * 
     * @param headerLine wb_֏ޕ̃Xg
     */
    public void printHeaderLine(List<String> headerLine) {
        if (writeData || writeTrailer) {
            throw new FileException("Header part should be called before "
                    + "data part or trailer part.",
                    new IllegalStateException(), fileName);
        }
        printList(headerLine);
    }

    /**
     * f[^ւ̏ݏB
     * 
     * @param t f[^֏ރt@CsIuWFNg
     */
    public void printDataLine(T t) {
        checkWriteTrailer();
        // t@C݂̏
        StringBuilder fileLineBuilder = new StringBuilder();
        
        // Œ蒷t@C̏ꍇ
        // (؂蕶A͂ݕȂꍇ͌Œ蒷t@CƔfB)
        if (getDelimiter() == Character.MIN_VALUE
                && getEncloseChar() == Character.MIN_VALUE) {
            for (int i = 0; i < fields.length; i++) {
                fileLineBuilder.append(getColumn(t, i));
            }
        } else {
            for (int i = 0; i < fields.length; i++) {
                // ͂ݕA؂蕶̒ǉB
                if (getEncloseChar() != Character.MIN_VALUE) {
                    fileLineBuilder.append(getEncloseChar());
                    fileLineBuilder.append(getColumn(t, i));
                    fileLineBuilder.append(getEncloseChar());
                } else {
                    fileLineBuilder.append(getColumn(t, i));
                }
                fileLineBuilder.append(getDelimiter());
            }
            // ԍŌ̋؂蕶폜B
            if (fileLineBuilder.length() > 0) {
                fileLineBuilder.deleteCharAt(fileLineBuilder.length() - 1);
            }
        }

        // s؂蕶ǉB
        fileLineBuilder.append(getLineFeedChar());

        // t@Cւ̏ݏB
        try {
            getWriter().write(fileLineBuilder.toString());
        } catch (IOException e) {
            throw new FileException(
                    "Processing of writer was failed.", e, fileName);
        }
        currentLineCount++;
        setWriteData(true);
    }

    /**
     * gCւ̏ݏB
     * 
     * @param trailerLine gC֏ޕ̃Xg
     */
    public void printTrailerLine(List<String> trailerLine) {
        printList(trailerLine);
        writeTrailer = true;
    }

    /**
     * wb_AgC̏ݗp̋ʃ\bhB
     * 
     * @param stringList ̃Xg
     */
    private void printList(List<String> stringList) {
        for (String stringData : stringList) {
            try {
                writer.write(stringData);
                writer.write(lineFeedChar);
            } catch (IOException e) {
                throw new FileException(
                        "Processing of writer was failed.", e, fileName);
            }
        }
    }

    /**
     * t@CN[YB
     */
    public void closeFile() {
        try {
            writer.flush();
            writer.close();
        } catch (IOException e) {
            throw new FileException(
                    "Closing of writer was failed.", e, fileName);
        }
    }

    /**
     * <p>
     * t@CsIuWFNgJCfbNXƈv鑮̒l擾B
     * </p>
     * 
     * <p>
     * 擾ہAt@CsIuWFNg̃Ame[V̋Lqɂ
     * ȉ̏sB<br>
     * <li>g<br>
     * <li>pfBO<br>
     * <li>ϊ<br>
     * <br>
     * t@CsIuWFNg̃Ame[VŃJ̃oCgw肳ĂꍇA<br>
     * ԋp镶񂪃oCgƈvĂ邩mFB
     * </p>
     * 
     * @param t t@CsIuWFNg
     * @param index J̃CfbNX
     * @return J̕
     */
    protected String getColumn(T t, int index) {
        // t@CɏރJ̕B
        String columnString = null;

        // t@CsIuWFNg(t)JCfbNXƈv鑮̒l擾B
        ColumnFormatter columnFormatter = columnFormatterMap.get(methods[index]
                .getReturnType().getName());
        try {
            columnString = columnFormatter.format(t, methods[index],
                    columnFormats[index]);
        } catch (IllegalArgumentException e) {
            throw new FileLineException("Failed in column data formatting.", e,
                    fileName, currentLineCount + 1, fields[index].getName(),
                    columnIndexs[index]);
        } catch (IllegalAccessException e) {
            throw new FileLineException("Failed in column data formatting.", e,
                    fileName, currentLineCount + 1, fields[index].getName(),
                    columnIndexs[index]);
        } catch (InvocationTargetException e) {
            throw new FileLineException("Failed in column data formatting.", e,
                    fileName, currentLineCount + 1, fields[index].getName(),
                    columnIndexs[index]);
        }

        if (columnString == null) {
            columnString = "";
        }

        // g
        columnString = FileDAOUtility.trim(columnString, fileEncoding,
                trimChars[index], trimTypes[index]);

        // pfBO
        columnString = FileDAOUtility.padding(columnString, fileEncoding,
                columnBytes[index], paddingChars[index], paddingTypes[index]);

        // ϊ
        // OutputFileColumn.stringConverter()̓eɂ菈U蕪B
        columnString = stringConverters[index].convert(columnString);

        // J̃oCg`FbNB
        if (isCheckByte(columnBytes[index])) {
            try {
                // Œ蒷o͎ABytesl̐ݒ肪̗O
                if (columnBytes[index] <= 0) {
                    throw new FileLineException("bytes is not set "
                            + "or a number equal to or less than 0 is set.",
                            new IllegalStateException(), getFileName(),
                            currentLineCount + 1, fields[index].getName(),
                            columnIndexs[index]);
                }
                // ݒ肳ꂽByteslƃf[^̃TCYႤꍇ͗O
                if (columnString.getBytes(fileEncoding).length != columnBytes[index]) {
                    throw new FileLineException(
                            "The data size is different from bytes value of "
                                    + "the set value of the column .",
                            new IllegalStateException(), fileName,
                            currentLineCount + 1, fields[index].getName(),
                            columnIndexs[index]);
                }
            } catch (UnsupportedEncodingException e) {
                throw new FileException(
                        "fileEncoding which isn't supported was set.", e,
                        fileName);
            }
        }
        return columnString;
    }
    
    /**
     * t@C擾B
     * 
     * @return fileName t@C
     */
    public String getFileName() {
        return fileName;
    }

    /**
     * s؂蕶ݒ肷B
     * 
     * @return lineFeedChar s؂蕶
     */
    protected String getLineFeedChar() {
        return lineFeedChar;
    }

    /**
     * JtH[}bg(t@Cji[}bv擾B
     * 
     * @param columnFormatterMap
     *            JtH[}bg(t@Cji[}bv
     */
    public void setColumnFormatterMap(
            Map<String, ColumnFormatter> columnFormatterMap) {
        this.columnFormatterMap = columnFormatterMap;
    }

    /**
     * t@CANZXiójp̕Xg[擾B
     * 
     * @return bufferedWriter t@CANZXiójp̕Xg[
     */
    protected Writer getWriter() {
        return writer;
    }

    /**
     * t@CsIuWFNgFieldiAnnotationji[ϐ擾B
     * 
     * @return fields t@CsIuWFNgFieldiAnnotationji[ϐ
     */
    protected Field[] getFields() {
        return fields;
    }

    /**
     * t@CsIuWFNgFieldɑΉgetter\bhi[ϐ擾B
     * 
     * @return methods t@CsIuWFNgFieldɑΉgetter\bhi[ϐ
     */
    protected Method[] getMethods() {
        return methods;
    }
    
    /**
     * t@CsIuWFNg̃JCfbNX̒`Field(Annotation)i[ϐ擾B
     * 
     * @return fieldArray JCfbNX̒`Ƀ\[gꂽField(Annotation)i[ϐ
     */
    protected Field[] getFieldArray() {
        return fieldArray;
    }

    /**
     * f[^̏o͂JnĂ邩ǂ𔻒肷tOB
     * 
     * @param writeData tO
     */
    protected void setWriteData(boolean writeData) {
        this.writeData = writeData;
    }

    /**
     * gC̏IĂ邩ǂ𔻒肷B<br>
     * ĂꍇAOX[B
     */
    protected void checkWriteTrailer() {
        if (writeTrailer) {
            throw new FileException("Header part or data part should be "
                    + "called before TrailerPart",
                    new IllegalStateException(), fileName);
        }
    }

    /**
     * ؂蕶擾B
     * 
     * @return ؂蕶
     */
    public abstract char getDelimiter();

    /**
     * ͂ݕ擾B
     * 
     * @return ͂ݕ
     */
    public abstract char getEncloseChar();


    /**
     * ΏۃJɑ΂oCg`FbNsԂB
     * <p>
     * Œ蒷t@C`̏ꍇ͏<code>true</code>ԋpăoCg`FbNsB<br>
     * ϒAbru`<code>bytes</code>w肳ĂƂɂ
     * trueԋpoCg`FbNsB
     * </p>
     * @param outputFileColumn ΏۃJOutputFileColumn
     * @return oCgݒ肳Ă(1oCgȏ)ꍇtrueB
     */
    protected boolean isCheckByte(OutputFileColumn outputFileColumn) {

        if (0 < outputFileColumn.bytes()) {
            return true;
        }

        return false;
    }

    /**
     * ΏۃJɑ΂oCg`FbNsԂB
     *
     * @param columnByte ΏۃJ̃oCg
     * @return oCgݒ肳Ă(1oCgȏ)ꍇtrueB
     */
    protected boolean isCheckByte(int columnByte) {
        if (0 < columnByte) {
            return true;
        }
        return false;
    }
}