/*
 * Copyright (c) 2011 NTT DATA Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jp.terasoluna.fw.batch.executor;

import java.io.File;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;

import jp.terasoluna.fw.batch.constants.LogId;
import jp.terasoluna.fw.batch.executor.concurrent.BatchServant;
import jp.terasoluna.fw.batch.executor.vo.BatchJobListResult;
import jp.terasoluna.fw.batch.util.BatchUtil;
import jp.terasoluna.fw.batch.util.JobUtil;
import jp.terasoluna.fw.dao.QueryDAO;
import jp.terasoluna.fw.dao.UpdateDAO;
import jp.terasoluna.fw.logger.TLogger;
import jp.terasoluna.fw.util.PropertyUtil;

import org.apache.commons.logging.Log;
import org.springframework.context.ApplicationContext;
import org.springframework.context.support.AbstractApplicationContext;
import org.springframework.dao.DataAccessException;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.transaction.PlatformTransactionManager;

/**
 * 񓯊ob`GO[L[^B<br>
 * <br>
 * 풓vZXƂċNAWuǗe[uɓo^ꂽWu擾AWu̎sBatchServantNXɈڏB<br>
 * ܂WuǗe[uɃWusʂXVB<br>
 * @see jp.terasoluna.fw.batch.executor.AbstractJobBatchExecutor
 */
public class AsyncBatchExecutor extends AbstractJobBatchExecutor {

    /**
     * K[.
     */
    private static final TLogger LOGGER = TLogger
            .getLogger(AsyncBatchExecutor.class);

    /**
     * ^XNGO[L[^Bean
     */
    private static final String BATCH_TASK_EXECUTOR = "batchTaskExecutor.default";

    /**
     * XbhspBatchServantNXBean
     */
    private static final String BATCH_TASK_SERVANT = "batchTaskExecutor.batchServant";

    /**
     * f[^x[Xُ펞̃gC񐔒`
     */
    private static final String BATCH_DB_ABNORMAL_RETRY_MAX = "batchTaskExecutor.dbAbnormalRetryMax";

    /**
     * f[^x[Xُ펞̃gCԊu`
     */
    private static final String BATCH_DB_ABNORMAL_RETRY_INTERVAL = "batchTaskExecutor.dbAbnormalRetryInterval";

    /**
     * f[^x[Xُ펞̃gC񐔂ZbgO񂩂̔Ԋu`
     */
    private static final String BATCH_DB_ABNORMAL_RETRY_RESET = "batchTaskExecutor.dbAbnormalRetryReset";

    /**
     * f[^x[Xُ펞̃gC񐔂̃ftHgl
     */
    private static final long BATCH_DB_ABNORMAL_RETRY_MAX_DEFAULT = 0;

    /**
     * f[^x[Xُ펞̃gCԊũftHgli~bj
     */
    private static final long BATCH_DB_ABNORMAL_RETRY_INTERVAL_DEFAULT = 20000;

    /**
     * f[^x[Xُ펞̃gC񐔂ZbgO񂩂̔ԊũftHgli~bj
     */
    private static final long BATCH_DB_ABNORMAL_RETRY_RESET_DEFAULT = 600000;

    /**
     * vZXIR[hij
     */
    private static final int PROCESS_END_STATUS_NORMAL = 0;

    /**
     * vZXIR[hiُj
     */
    private static final int PROCESS_END_STATUS_FAILURE = 255;

    /** XbhO[vvtBbNX. */
    public static final String THREAD_GROUP_PREFIX = AsyncBatchExecutor.class
            .getSimpleName()
            + "ThreadGroup";

    /** XbhvtBbNX. */
    public static final String THREAD_NAME_PREFIX = AsyncBatchExecutor.class
            .getSimpleName()
            + "Thread";

    /** XbhO[vZp[^. */
    public static final String THREAD_GROUP_SEPARATOR = "-";

    /** XbhZp[^. */
    public static final String THREAD_NAME_SEPARATOR = "-";

    /** XbhO[vԍ. */
    private static AtomicInteger threadGroupNo = new AtomicInteger(0);

    /**
     * RXgN^
     */
    protected AsyncBatchExecutor() {
        super();
    }

    /**
     * C\bh.
     * @param args
     */
    public static void main(String[] args) {
        long retryCount = 0;
        long retryCountMax = BATCH_DB_ABNORMAL_RETRY_MAX_DEFAULT;
        long retryCountReset = BATCH_DB_ABNORMAL_RETRY_RESET_DEFAULT;
        long retryInterval = BATCH_DB_ABNORMAL_RETRY_INTERVAL_DEFAULT;
        int status = PROCESS_END_STATUS_FAILURE;
        long lastExceptionTime = System.currentTimeMillis();

        // vpeBl擾
        String dbAbnormalRetryMaxStr = PropertyUtil
                .getProperty(BATCH_DB_ABNORMAL_RETRY_MAX);
        String dbAbnormalRetryIntervalStr = PropertyUtil
                .getProperty(BATCH_DB_ABNORMAL_RETRY_INTERVAL);
        String dbAbnormalRetryResetStr = PropertyUtil
                .getProperty(BATCH_DB_ABNORMAL_RETRY_RESET);

        // f[^x[Xُ펞̃gC
        if (dbAbnormalRetryMaxStr != null
                && dbAbnormalRetryMaxStr.length() != 0) {
            try {
                retryCountMax = Long.parseLong(dbAbnormalRetryMaxStr);
            } catch (NumberFormatException e) {
                LOGGER.error(LogId.EAL025046, e, BATCH_DB_ABNORMAL_RETRY_MAX,
                        dbAbnormalRetryMaxStr);
                System.exit(status);
                return;
            }
        }

        // f[^x[Xُ펞̃gCԊui~bj
        if (dbAbnormalRetryIntervalStr != null
                && dbAbnormalRetryIntervalStr.length() != 0) {
            try {
                retryInterval = Long.parseLong(dbAbnormalRetryIntervalStr);
            } catch (NumberFormatException e) {
                LOGGER.error(LogId.EAL025046, e,
                        BATCH_DB_ABNORMAL_RETRY_INTERVAL,
                        dbAbnormalRetryIntervalStr);
                System.exit(status);
                return;
            }
        }

        // f[^x[Xُ펞̃gC񐔂ZbgO񂩂̔Ԋui~bj
        if (dbAbnormalRetryResetStr != null
                && dbAbnormalRetryResetStr.length() != 0) {
            try {
                retryCountReset = Long.parseLong(dbAbnormalRetryResetStr);
            } catch (NumberFormatException e) {
                LOGGER.error(LogId.EAL025046, e, BATCH_DB_ABNORMAL_RETRY_RESET,
                        dbAbnormalRetryResetStr);
                System.exit(status);
                return;
            }
        }

        do {
            try {
                status = executorMain(args);
                break;
            } catch (DataAccessException e) {
                // O񂩂w莞Ԉȏo߂Ă烊gC񐔃Zbg
                if ((System.currentTimeMillis() - lastExceptionTime) > retryCountReset) {
                    retryCount = 0;
                }
                lastExceptionTime = System.currentTimeMillis();

                // gC񐔃`FbN
                if (retryCount >= retryCountMax) {
                    LOGGER.error(LogId.EAL025031, e);
                    break;
                }

                // X[vԑ҂
                try {
                    Thread.sleep(retryInterval);
                } catch (InterruptedException e1) {
                    // ȂɂȂ
                }

                retryCount++;

                LOGGER.info(LogId.IAL025017, retryCount, retryCountMax,
                        retryCountReset, retryInterval);
                if (LOGGER.isTraceEnabled()) {
                    LOGGER.trace(LogId.TAL025010, BatchUtil.getMemoryInfo());
                }
            }
        } while (true);

        System.exit(status);
    }

    /**
     * GO[L[^C\bh.
     * @param args
     */
    public static int executorMain(String[] args) {
        int status = PROCESS_END_STATUS_FAILURE;
        Throwable throwable = null;
        String jobAppCd = null;
        String batchTaskExecutorName = null;
        String batchTaskServantName = null;
        ThreadPoolTaskExecutor taskExecutor = null;

        LOGGER.info(LogId.IAL025005);

        // 1WuƖR[h擾
        if (args.length > 0) {
            jobAppCd = args[0];
        }

        // Ɏw肳ĂȂꍇ͊ϐWuƖR[h擾
        if (jobAppCd == null || jobAppCd.length() == 0) {
            jobAppCd = JobUtil.getenv(ENV_JOB_APP_CD);
            if (jobAppCd != null && jobAppCd.length() == 0) {
                jobAppCd = null;
            }
        }

        if (LOGGER.isInfoEnabled()) {
            LOGGER.info(LogId.IAL025006, jobAppCd == null ? "" : jobAppCd);
        }

        // GO[L[^
        AsyncBatchExecutor executor = new AsyncBatchExecutor();

        // QueryDAO擾
        QueryDAO queryDAO = executor.getSysQueryDAO();
        if (queryDAO == null) {
            LOGGER.info(LogId.IAL025007);
            return status;
        }

        // UpdateDAO擾
        UpdateDAO updateDAO = executor.getSysUpdateDAO();
        if (updateDAO == null) {
            LOGGER.info(LogId.IAL025008);
            return status;
        }

        // PlatformTransactionManager擾
        PlatformTransactionManager transactionManager = executor
                .getSysTransactionManager();
        if (transactionManager == null) {
            LOGGER.info(LogId.IAL025016);
            return status;
        }

        // ^XNGO[L[^BatchServantNXBean擾
        batchTaskExecutorName = PropertyUtil.getProperty(BATCH_TASK_EXECUTOR);
        batchTaskServantName = PropertyUtil.getProperty(BATCH_TASK_SERVANT);

        // ^XNGO[L[^擾
        ApplicationContext ctx = executor.getDefaultApplicationContext();
        if (ctx != null) {
            if (ctx.containsBean(batchTaskExecutorName)) {
                Object batchTaskExecutorObj = null;
                try {
                    batchTaskExecutorObj = ctx.getBean(batchTaskExecutorName,
                            ThreadPoolTaskExecutor.class);
                } catch (Throwable e) {
                    LOGGER.error(LogId.EAL025029, e, batchTaskExecutorName);
                }
                if (batchTaskExecutorObj instanceof ThreadPoolTaskExecutor) {
                    taskExecutor = (ThreadPoolTaskExecutor) batchTaskExecutorObj;
                }
            }
        }
        if (taskExecutor == null) {
            LOGGER.info(LogId.IAL025009);
            return status;
        }

        try {
            do {
                // WuXg1̂ݎ擾isXbhɋ󂫂ꍇ̂ݎ擾j
                List<BatchJobListResult> jobList = null;
                if (checkTaskQueue(taskExecutor)) {
                    if (jobAppCd == null) {
                        jobList = JobUtil.selectJobList(queryDAO, 0, 1);
                    } else {
                        jobList = JobUtil.selectJobList(jobAppCd, queryDAO, 0,
                                1);
                    }
                }

                if (jobList != null && !jobList.isEmpty()) {
                    // XĝPڂ̂ݎ擾
                    BatchJobListResult batchJobListResult = jobList.get(0);

                    if (batchJobListResult != null) {
                        if (LOGGER.isDebugEnabled()) {
                            LOGGER.debug(LogId.DAL025026, batchJobListResult
                                    .getJobSequenceId());
                        }

                        if (LOGGER.isDebugEnabled()) {
                            // Xbhv[^XNGO[L[^̃Xe[^XfobOOɏo
                            logOutputTaskExecutor(LOGGER, taskExecutor);
                        }

                        // sXbhɋ󂫂΃ob`s
                        if (checkTaskQueue(taskExecutor)) {
                            BatchServant job = null;

                            // ReLXgBatchServantCX^X擾
                            if (ctx != null) {
                                try {
                                    job = (BatchServant) ctx.getBean(
                                            batchTaskServantName,
                                            BatchServant.class);
                                } catch (Throwable e) {
                                    LOGGER.error(LogId.EAL025030, e,
                                            batchTaskServantName);
                                    break;
                                }
                            }

                            if (job == null) {
                                LOGGER.error(LogId.EAL025030,
                                        batchTaskServantName);
                                break;
                            } else {
                                // WuXe[^XݒiJnj
                                boolean st = executor
                                        .startBatchStatus(batchJobListResult
                                                .getJobSequenceId(), queryDAO,
                                                updateDAO, transactionManager);
                                if (st) {
                                    // BatchServantɃWuV[PXR[hݒ
                                    job.setJobSequenceId(batchJobListResult
                                            .getJobSequenceId());

                                    if (LOGGER.isDebugEnabled()) {
                                        // XbhO[vvtBbNXݒ
                                        StringBuilder tgn = new StringBuilder();
                                        tgn.append(THREAD_GROUP_PREFIX);
                                        tgn.append(THREAD_GROUP_SEPARATOR);
                                        tgn.append(threadGroupNo
                                                .incrementAndGet());
                                        taskExecutor.setThreadGroupName(tgn
                                                .toString());

                                        // XbhvtBbNXݒ
                                        StringBuilder tn = new StringBuilder();
                                        tn.append(THREAD_NAME_PREFIX);
                                        tn.append(THREAD_NAME_SEPARATOR);
                                        tn.append(threadGroupNo.get());
                                        tn.append(THREAD_NAME_SEPARATOR);
                                        taskExecutor.setThreadNamePrefix(tn
                                                .toString());

                                        LOGGER.debug(LogId.DAL025027, tgn
                                                .toString(), tn.toString());
                                    }

                                    // Wus
                                    taskExecutor.execute(job);
                                } else {
                                    if (LOGGER.isInfoEnabled()) {
                                        LOGGER.info(LogId.IAL025010,
                                                batchJobListResult
                                                        .getJobSequenceId());
                                    }
                                }
                            }
                        }
                    }
                }

                // ItOt@C`FbN
                if (checkEndFile(executor.getExecutorEndMonitoringFile())) {
                    // ItOt@C
                    LOGGER.info(LogId.IAL025011);

                    break;
                }

                // WuXĝƂ or sXbhɋ󂫂ꍇ ͎w莞ԃEFCg
                if (jobList == null || jobList.size() == 0) {
                    // Wu̎sԊui~bj
                    if (executor.getJobIntervalTime() >= 0) {
                        try {
                            Thread.sleep(executor.getJobIntervalTime());
                        } catch (InterruptedException e) {
                            // 荞ݎMŏI
                            if (LOGGER.isInfoEnabled()) {
                                LOGGER.info(LogId.IAL025012, e.getMessage());
                            }

                            break;
                        }
                    }
                }

                if (LOGGER.isTraceEnabled()) {
                    LOGGER.trace(LogId.TAL025010, BatchUtil.getMemoryInfo());
                }
                // 풓pɃ[v
            } while (true);
        } catch (Throwable e) {
            // UۑĂ
            throwable = e;
        } finally {
            LOGGER.debug(LogId.DAL025028);

            // IɃ^XN͂܂ő҂
            taskExecutor.setWaitForTasksToCompleteOnShutdown(true);

            LOGGER.debug(LogId.DAL025029);

            taskExecutor.shutdown();

            LOGGER.debug(LogId.DAL025030);

            while (taskExecutor.getActiveCount() != 0) {
                try {
                    if (LOGGER.isDebugEnabled()) {
                        LOGGER.debug(LogId.DAL025031, taskExecutor
                                .getActiveCount());
                    }

                    Thread.sleep(executor
                            .getExecutorJobTerminateWaitIntervalTime());
                } catch (InterruptedException e) {
                }
            }

            closeRootApplicationContext(ctx);
        }

        if (throwable != null) {
            Throwable cause = throwable.getCause();
            if (cause instanceof DataAccessException) {
                throw (DataAccessException) cause;
            }
            LOGGER.error(LogId.EAL025031, throwable);
        } else {
            LOGGER.info(LogId.IAL025013);

            status = PROCESS_END_STATUS_NORMAL;
        }
        return status;
    }

    /**
     * ^XNGO[L[^ɋ󂫂邩`FbN
     * @param taskExecutor ^XNGO[L[^
     * @return 󂫂̗L
     */
    protected static boolean checkTaskQueue(ThreadPoolTaskExecutor taskExecutor) {
        // Xbh̋󂫃`FbN
        if (taskExecutor.getActiveCount() < taskExecutor.getMaxPoolSize()) {
            // 󂫂
            return true;
        }
        // L[̋󂫃`FbN
        if (taskExecutor.getThreadPoolExecutor().getQueue().remainingCapacity() > 0) {
            // 󂫂
            return true;
        }
        // 󂫂Ȃ
        return false;
    }

    /**
     * ItOt@C`FbN
     * @param endFilePath ItOt@CpX
     * @return ItOt@C`FbN
     */
    protected static boolean checkEndFile(String endFilePath) {
        if (endFilePath != null && endFilePath.length() != 0) {
            File endFile = new File(endFilePath);
            return endFile.exists();
        }
        return false;
    }

    /**
     * Xbhv[^XNGO[L[^̃Xe[^XfobOOɏo
     * @param log Log
     * @param taskExec ThreadPoolTaskExecutor
     */
    protected static void logOutputTaskExecutor(Log log,
            ThreadPoolTaskExecutor taskExec) {
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug(LogId.DAL025032, taskExec.getActiveCount(), taskExec
                    .getCorePoolSize(), taskExec.getMaxPoolSize(), taskExec
                    .getPoolSize(), taskExec.getThreadPoolExecutor()
                    .getActiveCount(), taskExec.getThreadPoolExecutor()
                    .getTaskCount(), taskExec.getThreadPoolExecutor()
                    .getQueue().size(), taskExec.getThreadPoolExecutor()
                    .getQueue().remainingCapacity());
        }
    }

    /**
     * ApplicationContextN[Y.
     * @param context
     */
    protected static void closeRootApplicationContext(ApplicationContext context) {
        if (context instanceof AbstractApplicationContext) {
            AbstractApplicationContext aac = (AbstractApplicationContext) context;
            aac.close();
            aac.destroy();
        }
    }
}
