/*
 * Copyright (c) 2011 NTT DATA Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jp.terasoluna.fw.ex.unit.util;

import java.sql.Timestamp;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.List;

import jp.terasoluna.fw.ex.unit.exception.PopulationFailedException;

import org.apache.commons.lang.StringUtils;
import org.springframework.beans.BeanWrapper;
import org.springframework.beans.BeanWrapperImpl;
import org.springframework.util.Assert;

/**
 * l̔zƃvpeB̔z񂩂CX^X𐶐A{@link jp.terasoluna.fw.ex.unit.util.Populator}
 * ̎NXłB<br>
 * 
 * <pre>
 * ʏ{@link org.springframework.beans.BeanWrapper}gpăCX^Xs܂A
 * {@link java.util.Date}A{@link java.sql.Date}A{@link java.sql.Timestamp}CX^X𐶐̂parseDate\bhgpăCX^Xs܂B
 * ̍{@link java.sql.Timestamp^}Ńp[X݂܂Bp[XoȂꍇ̓ftHg̓ttH[}bgi&quot;yyyy-MM-dd&quot;, &quot;yyyy/MM/dd&quot;, &quot;HH:mm:ss&quot;jŃp[X݂܂B
 * ȊÕtH[}bgŃp[XKvꍇAsetDateFormat\bhŐݒ肵ĂB
 * p[XɎsꍇ{@link java.lang.IllegalArgumentException}X[܂B
 * setIgnoreParseExceptiontrueݒ肷Datẽp[XParseExecptionꍇAOX[Anullݒ肵܂B
 * </pre>
 */
public class DefaultPopulator<T> implements Populator<T> {

    /**
     * ttH[}bg
     */
    private String dateFormat = null;

    /**
     * ftHg̃tH[}bgXg
     */
    private static final List<String> defaultDateFormatList = Arrays.asList(
            "yyyy-MM-dd", "yyyy/MM/dd", "HH:mm:ss");

    /**
     * p[Xs𖳎邩ǂB
     */
    private boolean ignoreParseException = true;

    /**
     * @param ignoreParseException
     *            Zbg ignoreParseException
     */
    public void setIgnoreParseException(boolean ignoreParseException) {
        this.ignoreParseException = ignoreParseException;
    }

    @SuppressWarnings("unchecked")
    public T populate(Class<T> clazz, String[] propertyNames, Object... values) {
        Assert.notEmpty(values);
        Assert.notEmpty(propertyNames);
        Assert.isTrue(values.length == propertyNames.length,
                "The length of propertyNames and values  must be same! "
                        + propertyNames.length + " != " + values.length);

        try {
            BeanWrapper beanWrapper = new BeanWrapperImpl(clazz);

            for (int i = 0; i < values.length; i++) {
                Object value = null;
                String propertyName = propertyNames[i];
                Class<?> propertyType = beanWrapper
                        .getPropertyType(propertyName);
                if (propertyType != null) {
                    // vpeB݂ꍇ̂
                    value = convertPropertyValue(values[i], propertyType);
                    beanWrapper.setPropertyValue(propertyName, value);
                } else if (StringUtils.contains(propertyName, '.')) {
                    // qvfAevpeBnull̏ꍇ (properyNameuxxx.yyyv̗lȎw)
                    String[] names = StringUtils.split(propertyName, '.');
                    StringBuilder prop = new StringBuilder(names[0]);
                    for (int j = 0; j < names.length; j++) {
                        if (j > 0) {
                            prop.append('.');
                            prop.append(names[j]);
                        }
                        String parentName = prop.toString();
                        Class<?> parentType = beanWrapper
                                .getPropertyType(parentName);
                        if (parentType != null) {
                            // etB[h̃CX^X쐬
                            populateParent(beanWrapper, parentType, parentName);
                        } else {
                            // etB[h݂Ȃꍇ͈ȍ~Ȃ
                            break;
                        }
                    }

                    // ^ēx擾
                    propertyType = beanWrapper.getPropertyType(propertyName);
                    if (propertyType != null) {
                        // etB[h̃CX^Xɐꍇ
                        value = convertPropertyValue(values[i], propertyType);
                        beanWrapper.setPropertyValue(propertyName, value);
                    }
                }
            }

            return (T) beanWrapper.getWrappedInstance();
        } catch (Exception e) {
            throw new PopulationFailedException(e);
        }
    }

    private void populateParent(BeanWrapper beanWrapper, Class<?> parentType,
            String parentName) {
        Object parent = beanWrapper.getPropertyValue(parentName);
        if (parent == null) {
            Object o = ReflectionUtils.newInstance(parentType);
            beanWrapper.setPropertyValue(parentName, o);
        }
    }

    @SuppressWarnings("unchecked")
    private Object convertPropertyValue(Object value, Class<?> propertyType) {
        Object result = null;
        if (Date.class.isAssignableFrom(propertyType) && value != null) {
            // 񂩂java.util.Date^ɕϊ
            Date d = parseDate(value.toString());
            // tB[hɑΉ^ɕϊ
            result = DateConvertUtils.convertDateValue(d,
                    (Class<? extends Date>) propertyType);
        } else {
            result = value;
        }
        return result;
    }

    /**
     * Date^̃CX^XsB
     * 
     * @param value
     *            Date^֊i[l
     * @return Date^̃CX^X
     * @throws IllegalArgumentException
     *             valueݒ肵dateFormatAjava.sql.TimestampA
     *             ftHg̃tH[}bĝłp[XłAignoreParseExceptionfalsȅꍇB
     * @see java.util.Date
     */
    private Date parseDate(String value) throws IllegalArgumentException {
        Date date = null;

        if (dateFormat != null) {
            SimpleDateFormat format = new SimpleDateFormat(dateFormat);
            try {
                date = format.parse(value);
                return date;
            } catch (ParseException ignore) {
            }
        }

        try {
            date = Timestamp.valueOf(value);
            return date;
        } catch (IllegalArgumentException ignore) {
        }

        for (String fmt : defaultDateFormatList) {
            SimpleDateFormat format = new SimpleDateFormat(fmt);
            try {
                date = format.parse(value);
            } catch (ParseException ignore) {
            }
        }

        if (date == null && !ignoreParseException) {
            throw new IllegalArgumentException("Cannot parse " + value
                    + ". Please set dateFormat.");
        }
        return date;
    }

    /**
     * ttH[}bg擾B
     * 
     * @return ttH[}bg
     */
    public String getDateFormat() {
        return dateFormat;
    }

    /**
     * ttH[}bgݒ肷B
     * 
     * @param dateFormat
     *            ttH[}bg
     */
    public void setDateFormat(String dateFormat) {
        this.dateFormat = dateFormat;
    }
}
