/*
 * Copyright (c) 2012 NTT DATA Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jp.terasoluna.fw.collector.concurrent;

import java.util.Collection;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantLock;

/**
 * AbstractCollectorpArrayBlockingQueueTuNXB
 * <p>
 * {@link ArrayBlockingQueue#peek()} {@link ArrayBlockingQueue#isEmpty()}ɁA L[ł΁AL[ɗvf邩AL[COItOオ܂ ҂@\ĂB<br>
 * ArrayBlockingQueuéAubNsĂConditiontB[h TuNXɌJĂȂ߁A ̃NXłArrayBlockingQueueƏ璷ȎĂB<br>
 * </p>
 * <p>
 * AbstractCollectorɎgp̂ɍiĂ邽߁A ׂẴ\bhgpł킯ł͂ȂB<br>
 * ̃NXŃI[o[ChĂ郁\bhȊOŁA L[̏ԂύX郁\bhA҂郁\bhsĂ͂ȂȂB
 * </p>
 * <p>
 * L[ɗvflߏÍAL[ɗvfl߂XbhŁAKfinishQueueing\bhs邱ƁB
 * </p>
 * @param <E> RNVɑ݂vf̌^
 */
public class ArrayBlockingQueueEx<E> extends ArrayBlockingQueue<E>
                                                                  implements
                                                                  NotificationBlockingQueue<E> {

    /**
     * serialVersionUID
     */
    private static final long serialVersionUID = 7441765139909417804L;

    /**
     * L[̑𓯊郍bNB
     */
    protected final ReentrantLock queueLock = new ReentrantLock();

    /**
     * L[łȂȂƂɑMVOiB
     */
    protected final Condition notEmpty = queueLock.newCondition();

    /**
     * L[FullłȂȂƂɑMVOiB
     */
    protected final Condition notFull = queueLock.newCondition();

    /**
     * L[TCYB
     */
    protected final int capacity;

    /**
     * L[COItOB
     */
    protected volatile boolean finishQueueingFlag = false;

    /**
     * w肳ꂽ (Œ) eʂюw肳ꂽANZX|V[gpāAArrayBlockingQueue 쐬B
     * @param capacity L[̗e
     * @param fair true ̏ꍇA}܂͍폜ɃubNꂽXbhɑ΂L[ANZX́AFIFO ̏ŏB false ̏ꍇAANZX͎w肳ȂB
     * @see ArrayBlockingQueue#ArrayBlockingQueue(int, boolean)
     */
    public ArrayBlockingQueueEx(int capacity, boolean fair) {
        super(capacity, fair);
        this.capacity = capacity;
    }

    /**
     * w肳ꂽ (Œ) eʂуftHg̃ANZX|V[gpāAArrayBlockingQueue 쐬B
     * @param capacity L[̗e
     * @see ArrayBlockingQueue#ArrayBlockingQueue(int)
     */
    public ArrayBlockingQueueEx(int capacity) {
        super(capacity);
        this.capacity = capacity;
    }

    /**
     * L[CȌIʒmB
     * <p>
     * L[ɗvf̂҂ĂXbhꍇÃubNB L[ɗvfl߂Xbh́AL[COƂŁAK̃\bhs邱ƁB
     * </p>
     */
    public void finishQueueing() {
        queueLock.lock();
        try {
            finishQueueingFlag = true;

            // vf̓҂sĂXbh̃ubN
            notEmpty.signalAll();
        } finally {
            queueLock.unlock();
        }
    }

    /**
     * w肳ꂽvf̃L[̖ɑ}BKvɉAԂp\ɂȂ̂w肳ꂽԂ܂őҋ@B
     * <p>
     * ̃\bh̒`́A{@link ArrayBlockingQueue#offer(Object, long, TimeUnit)}ƓB
     * </p>
     * @param o ǉvf
     * @param timeout 𒆎~܂ł̑ҋ@ԁBPʂ unit
     * @param unit timeout p[^̉ߕ@w肷 TimeUnit
     * @return ꍇ trueAԂp\ɂȂOɎw肳ꂽҋ@Ԃo߂ꍇ false
     * @throws InterruptedException ҋ@Ɋ荞݂ꍇ
     * @throws NullPointerException w肳ꂽvf null łꍇ
     * @see ArrayBlockingQueue#offer(Object, long, TimeUnit)
     */
    @Override
    public boolean offer(E o, long timeout, TimeUnit unit)
                                                          throws InterruptedException {
        if (o == null) {
            throw new NullPointerException();
        }
        queueLock.lockInterruptibly();
        try {
            if (size() == capacity) {

                // L[󂭂̂҂
                if (!notFull.await(timeout, unit)) {

                    // ^CAEg
                    return false;
                }
            }
            boolean success = super.offer(o);
            if (success) {

                // vf̓҂sĂXbh̃ubN
                notEmpty.signal();
            }
            return success;
        } finally {
            queueLock.unlock();
        }
    }

    /**
     * \ł΁ÃL[̖Ɏw肳ꂽvf}B̃L[ςłꍇɂ́AɕԂB
     * <p>
     * ̃\bh̒`́A{@link ArrayBlockingQueue#offer(Object)}ƓB
     * </p>
     * @param o ǉvf
     * @return vf̃L[ɒǉ\ȏꍇ trueAłȂꍇ false
     * @throws NullPointerException w肳ꂽvf null łꍇ
     * @see ArrayBlockingQueue#offer(Object)
     */
    @Override
    public boolean offer(E o) {
        queueLock.lock();
        try {
            if (size() == capacity) {
                return false;
            }
            boolean success = super.offer(o);
            if (success) {

                // vf̓҂sĂXbh̃ubN
                notEmpty.signal();
            }
            return success;
        } finally {
            queueLock.unlock();
        }
    }

    /**
     * w肳ꂽvf̃L[̖ɒǉBKvɉAԂp\ɂȂ܂őҋ@B
     * @param o ǉvf
     * @throws InterruptedException ҋ@Ɋ荞݂ꍇ
     * @throws NullPointerException w肳ꂽvf null łꍇ
     */
    @Override
    public void put(E o) throws InterruptedException {
        if (o == null) {
            throw new NullPointerException();
        }
        queueLock.lock();
        try {
            if (size() == capacity) {

                // L[󂭂̂҂
                notFull.await();
            }
            super.put(o);

            // vf̓҂sĂXbh̃ubN
            notEmpty.signal();
        } finally {
            queueLock.unlock();
        }
    }

    /**
     * L[̐擪擾邪A폜ȂB
     * <p>
     * gdlF<b> L[̏ꍇ́AL[ɗvf邩AL[CȌIʒm܂ő҂B<br>
     * L[CȌIʒmꂽAL[̏ꍇ null ԂB
     * </p>
     * <p>
     * L[ɗvfꍇAL[CȌIʒmꂽ̎dĺA {@link ArrayBlockingQueue#peek()}ƓB
     * </p>
     * @return L[̐擪BL[COIɃL[̏ꍇ null
     */
    @Override
    public E peek() {
        queueLock.lock();
        try {
            while (!finishQueueingFlag && size() == 0) {
                try {

                    // L[ɗvf̂̂҂
                    notEmpty.await();
                } catch (InterruptedException e) {
                    return null;
                }
            }
            return super.peek();
        } finally {
            queueLock.unlock();
        }
    }

    /**
     * ̃L[̐擪擾э폜B̃L[ɗvf݂Ȃꍇ́AKvɉĎw肳ꂽԂҋ@B
     * <p>
     * gdlF<b> L[CȌIʒmꂽAL[̏ꍇ́A^CAEg҂ null ԂB
     * </p>
     * <p>
     * L[CȌIʒmO̎dĺA {@link ArrayBlockingQueue#poll(long, TimeUnit)}ƓB
     * </p>
     * @param timeout 𒆎~܂ł̑ҋ@ԁBPʂ unit
     * @param unit timeout p[^̉ߕ@w肷 TimeUnit
     * @return ̃L[̐擪Bw肳ꂽҋ@Ԃo߁A邢̓L[CȌIʒmꂽvf݂Ȃꍇ null
     * @throws InterruptedException ҋ@Ɋ荞݂ꍇ
     */
    @Override
    public E poll(long timeout, TimeUnit unit) throws InterruptedException {
        queueLock.lock();
        try {
            if (!finishQueueingFlag && size() == 0) {

                // L[ɗvf̂̂҂
                if (!notEmpty.await(timeout, unit)) {

                    // ^CAEg
                    return null;
                }
            }
            if (finishQueueingFlag && size() == 0) {
                // L[CȌIʒmꂽAAL[
                return null;
            }
            E elm = super.poll(timeout, unit);
            if (elm != null) {

                // L[̋󂫑҂sĂXbh̃ubN
                notFull.signal();
            }
            return elm;
        } finally {
            queueLock.unlock();
        }
    }

    /**
     * L[ɗvfȂꍇ true ԂB
     * <p>
     * gdlF<b> L[̏ꍇ́AL[ɗvf邩AL[CȌIʒm܂ő҂B<br>
     * L[CȌIʒmꂽAL[̏ꍇ true ԂB
     * </p>
     */
    @Override
    public boolean isEmpty() {
        queueLock.lock();
        try {
            while (!finishQueueingFlag && size() == 0) {
                try {

                    // L[ɗvf̂̂҂
                    notEmpty.await();
                } catch (InterruptedException e) {
                    return true;
                }
            }
            return super.isEmpty();
        } finally {
            queueLock.unlock();
        }
    }

}
