/*
   Copyright (c) 2019 by The ThreadDB Project
   All Rights Reserved.

   ThreadDB undergoes the BSD License 2.0. You should have received a copy along with this program; if not, write to the ThreadDB Project.
   To obtain a full unlimited version contact thethreaddbproject(at)gmail.com.

   threaddbTypes.h - Basic type definitions
*/

#pragma once

/**
   \file
   \brief         Basic data types used by the interface.
   \details       This module provides the underlying data structures required to interact with the
                  database.
*/

#if defined(_WIN32)
#if defined(THREADDB_)
#define DLLEXPORT_ __declspec(dllexport)
#else
#define DLLEXPORT_ __declspec(dllimport)
#endif
#else
#define DLLEXPORT_
#endif

#include <stdint.h>
#include <stddef.h>

#if defined(__cplusplus)
extern "C"
{
#endif

    /**
    *  @brief Data item entry index.
    *
    *  @details
    *   This structure reflects the information required to uniquely identify a data item in the database.
    *   It provides information on the temporary database file, the file position of the data record and the position within the data record.
    */

    typedef struct threadDB_LinkInfo
    {
        volatile uint64_t m_FilePos; ///< Position of the related record within the temporary database file.
        volatile uint32_t m_FileID; ///< Unique database file identification.
        volatile uint32_t m_Filling; ///< Position of the data item within the data record.
    } threadDB_LinkInfo;

    /**
    *  @brief Random access managment information.
    *
    *  @details
    *   This structure reflects the information required for random access of indivdual database items.
    *   It provides information on the position of the item and the size of the stored data record.
    *   The datastructure has to be preallocated and - if required - provided to the @ref ThreadDB_Store routine.
    *   It can then be used in the @ref ThreadDB_OpenItem and ThreadDB_RecoverRandom function to read individual data items.
    */

    typedef struct threadDB_ItemInfo
    {
        threadDB_LinkInfo m_LinkInfo; ///< Unique index of the data item.

        volatile uint32_t m_PackageSize; ///< Size of the data record in bytes.
    } threadDB_ItemInfo;

    /**
    *  @brief Stream reading control information.
    *
    *  @details
    *   This structure reflects the information required for continious stream reading of data items.
    *   It provides information on the position of the item, the size of the stored data record,
    *   a temporary record buffer and the allocated buffers size.
    *   The read information is created by the functions @ref ThreadDB_Open and @ ref ThreadDB_OpenItem and
    *   required for the call to @ref ThreadDB_RecoverContinous function.
    */

    typedef struct threadDB_ReadInfo
    {
        threadDB_LinkInfo m_LinkInfo; ///< Uniquely index of the data item.

        size_t m_BlockOffset; ///< Position of the current data item in the record.
        char* m_pBuffer; ///< Temporary read buffer.
        uint32_t m_BufferSize; ///< Size of the temporary read buffer.
    } threadDB_ReadInfo;

    /**
    *  @brief Type of action to be performed when relocating a temporary database file
    *
    *  @details
    *   The parameter defines if a temporary database file should be copied or moved when transferred to
    *   another path or partition.
    */

    typedef enum {
        eCopyFileTo = 0, ///< Indicates copy operation has to be executed
        eMoveFileTo = 1  ///< Indicates move operation has to be exectued
    } threadDB_RelocationType;

    typedef int32_t threadDB_ReturnCode;

#if defined(__cplusplus)
}
#endif
