/*
    TiMidity++ -- MIDI to WAVE converter and player
    Copyright (C) 1999-2004 Masanao Izumo <iz@onicos.co.jp>
    Copyright (C) 1995 Tuukka Toivonen <tt@cgs.fi>

    This program is free software; you can redistribute it and/or modify
    it under the terms timip_of the GNU General Public License as published by
    the Free Software Foundation; either version 2 timip_of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty timip_of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy timip_of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

    rtsyn_common.c
        Copyright (c) 2003  Keishi Suenaga <s_keishi@mutt.freemail.ne.jp>

    I referenced following sources.
        alsaseq_c.c - ALSA sequencer server interface
            Copyright (c) 2000  Takashi Iwai <tiwai@suse.de>
        readmidi.c
*/

#ifdef HAVE_CONFIG_H
#include "timip_config.h"
#endif /* HAVE_CONFIG_H */
#include "timip_interface.h"

#include <stdio.h>

#include <stdarg.h>
#ifdef	TIMIP_HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <sys/types.h>
#ifdef	TIMIP_TIME_WITH_SYS_TIME
#include <sys/time.h>
#endif
#ifndef NO_STRING_H
#include <string.h>
#else
#include <strings.h>
#endif
#include <math.h>
#include <signal.h>

#include "timip_server_defs.h"

#ifdef TIMIP___W32__
#include <windows.h>
#endif

#include "timip_timidity.h"
#include "timip_common.h"
#include "timip_controls.h"
#include "timip_instrum.h"
#include "timip_playmidi.h"
#include "timip_readmidi.h"
#include "timip_recache.h"
#include "timip_output.h"
#include "timip_aq.h"
#include "timip_timer.h"

#include "timip_rtsyn.h"


//int timip_seq_quit;

int timip_rtsyn_system_mode=DEFAULT_SYSTEM_MODE;

static int32 event_time_offset;
static double starttime;
double timip_rtsyn_reachtime;
static int time_advance;
static int set_time_first=2;

//acitive sensing
static int active_sensing_flag=0;
static double active_sensing_time=0;

//timer interrupt
#ifdef USE_WINSYN_TIMER_I
rtsyn_mutex_t timip_timerMUTEX;

#ifdef TIMIP___W32__
MMRESULT timip_timerID;
#else 
pthread_t timer_thread;
int thread_on_f=0;
#endif

#endif

#define EX_RESET_NO 7
static char sysex_resets[EX_RESET_NO][11]={
		'\xf0','\x7e','\x7f','\x09','\x00','\xf7','\x00','\x00','\x00','\x00','\x00',
		'\xf0','\x7e','\x7f','\x09','\x01','\xf7','\x00','\x00','\x00','\x00','\x00',
		'\xf0','\x7e','\x7f','\x09','\x03','\xf7','\x00','\x00','\x00','\x00','\x00',
		'\xf0','\x41','\x10','\x42','\x12','\x40','\x00','\x7f','\x00','\x41','\xf7',
		'\xf0','\x41','\x10','\x42','\x12','\x00','\x00','\x7f','\x00','\x01','\xf7',
		'\xf0','\x41','\x10','\x42','\x12','\x00','\x00','\x7f','\x01','\x00','\xf7',
		'\xf0','\x43','\x10','\x4c','\x00','\x00','\x7E','\x00','\xf7','\x00','\x00' };
/*
#define EX_RESET_NO 9
static char sysex_resets[EX_RESET_NO][11]={
	'\xf0','\x7e','\x7f','\x09','\x00','\xf7','\x00','\x00','\x00','\x00','\x00', //gm off
	'\xf0','\x7e','\x7f','\x09','\x01','\xf7','\x00','\x00','\x00','\x00','\x00', //gm1
	'\xf0','\x7e','\x7f','\x09','\x02','\xf7','\x00','\x00','\x00','\x00','\x00', //gm off
	'\xf0','\x7e','\x7f','\x09','\x03','\xf7','\x00','\x00','\x00','\x00','\x00', //gm2
	'\xf0','\x41','\x10','\x42','\x12','\x40','\x00','\x7f','\x00','\x41','\xf7', //GS
	'\xf0','\x41','\x10','\x42','\x12','\x40','\x00','\x7f','\x7f','\x41','\xf7', //GS off
	'\xf0','\x41','\x10','\x42','\x12','\x00','\x00','\x7f','\x00','\x01','\xf7', //88
	'\xf0','\x41','\x10','\x42','\x12','\x00','\x00','\x7f','\x01','\x00','\xf7', //88
	'\xf0','\x43','\x10','\x4c','\x00','\x00','\x7E','\x00','\xf7','\x00','\x00'  //XG on
	};
*/

static void seq_set_time(MidiEvent *timip_ev);



void timip_rtsyn_gm_reset(){
	MidiEvent timip_ev;

	timip_rtsyn_server_reset();
	timip_ev.type=ME_RESET;
	timip_ev.a=GM_SYSTEM_MODE;
	timip_rtsyn_play_event(&timip_ev);

}


void timip_rtsyn_gs_reset(){
	MidiEvent timip_ev;

	timip_rtsyn_server_reset();
	timip_ev.type=ME_RESET;
	timip_ev.a=GS_SYSTEM_MODE;
	timip_rtsyn_play_event(&timip_ev);
}


void timip_rtsyn_xg_reset(){
	MidiEvent timip_ev;

	timip_rtsyn_server_reset();
	timip_ev.type=ME_RESET;
	timip_ev.a=XG_SYSTEM_MODE;
	timip_ev.time=0;
	timip_rtsyn_play_event(&timip_ev);
}


void timip_rtsyn_normal_reset(){
	MidiEvent timip_ev;

	timip_rtsyn_server_reset();
	timip_ev.type=ME_RESET;
	timip_ev.a=timip_rtsyn_system_mode;
	timip_rtsyn_play_event(&timip_ev);
}
void timip_rtsyn_gm_modeset(){
	MidiEvent timip_ev;

	timip_rtsyn_server_reset();
	timip_rtsyn_system_mode=GM_SYSTEM_MODE;
	timip_ev.type=ME_RESET;
	timip_ev.a=GM_SYSTEM_MODE;
	timip_rtsyn_play_event(&timip_ev);
	timip_change_system_mode(timip_rtsyn_system_mode);
}


void timip_rtsyn_gs_modeset(){
	MidiEvent timip_ev;

	timip_rtsyn_server_reset();
	timip_rtsyn_system_mode=GS_SYSTEM_MODE;
	timip_ev.type=ME_RESET;
	timip_ev.a=GS_SYSTEM_MODE;
	timip_rtsyn_play_event(&timip_ev);
	timip_change_system_mode(timip_rtsyn_system_mode);
}


void timip_rtsyn_xg_modeset(){
	MidiEvent timip_ev;

	timip_rtsyn_server_reset();
	timip_rtsyn_system_mode=XG_SYSTEM_MODE;
	timip_ev.type=ME_RESET;
	timip_ev.a=XG_SYSTEM_MODE;
	timip_rtsyn_play_event(&timip_ev);
	timip_change_system_mode(timip_rtsyn_system_mode);
}


void timip_rtsyn_normal_modeset(){
	MidiEvent timip_ev;

	timip_rtsyn_server_reset();
	timip_rtsyn_system_mode=DEFAULT_SYSTEM_MODE;
	timip_ev.type=ME_RESET;
	timip_ev.a=GS_SYSTEM_MODE;
	timip_rtsyn_play_event(&timip_ev);
	timip_change_system_mode(timip_rtsyn_system_mode);
}



#ifdef USE_WINSYN_TIMER_I
#ifdef TIMIP___W32__
VOID CALLBACK timip_timercalc(UINT uTimerID, UINT uMsg, DWORD dwUser, DWORD dummy1, DWORD dummy2){
	MidiEvent timip_ev;
	double time_div, currenttime;
	
	rtsyn_mutex_lock(timip_timerMUTEX);
	currenttime=timip_get_current_calender_time();
	time_div=currenttime-starttime;
	if( time_div > 1.0/TICKTIME_HZ*2.0){
		time_div= 1.0/TICKTIME_HZ-3/(timip_play_mode->rate);
	}
	timip_ev.time= ((double)timip_current_sample
			+ (timip_play_mode->rate)*time_div+0.5);
	starttime=currenttime;
	timip_ev.type = ME_NONE;
	timip_play_event(&timip_ev);
//	compute_data(tmdy_struct,(tmdy_struct->output->timip_play_mode->rate)*time_div);
	timip_aq_fill_nonblocking();
	timip_rtsyn_reachtime=currenttime +  (double)(1.0f/TICKTIME_HZ);
	rtsyn_mutex_unlock(timip_timerMUTEX);
	return;
}
#else
void *timip_timercalc(void *arg){
 	MidiEvent timip_ev;
	unsigned int slt;
	double reachtime,delay;
	delay=(double)(1.0/TICKTIME_HZ);
	while(thread_on_f==1){		
		rtsyn_mutex_lock(timip_timerMUTEX);
		reachtime=timip_get_current_calender_time()+delay;
		timip_ev.type = ME_NONE;
		seq_set_time(&timip_ev);
		timip_play_event(&timip_ev);
		timip_aq_fill_nonblocking();
		rtsyn_mutex_unlock(timip_timerMUTEX);
		do{
			sleep(0);
		}while(timip_get_current_calender_time()<reachtime);
	}
	return NULL;
}
#endif
#endif
void timip_rtsyn_init(void){
	int i,j;
		/* set constants */
	timip_opt_realtime_playing = 1; /* Enable loading patch while playing */
	timip_allocate_cache_size = 0; /* Don't use pre-calclated samples */
	timip_auto_reduce_polyphony = 0;
	timip_current_keysig = (timip_opt_init_keysig == 8) ? 0 : timip_opt_init_keysig;
	timip_note_key_offset = timip_key_adjust;
	time_advance=timip_play_mode->rate/TICKTIME_HZ*2;
	if (!(timip_play_mode->encoding & PE_MONO))
		time_advance >>= 1;
	if (timip_play_mode->encoding & PE_24BIT)
		time_advance /= 3;
	else if (timip_play_mode->encoding & PE_16BIT)
		time_advance >>= 1;

	i = timip_current_keysig + ((timip_current_keysig < 8) ? 7 : -9), j = 0;
	while (i != 7)
		i += (i < 7) ? 5 : -7, j++;
	j += timip_note_key_offset, j -= floor(j / 12.0) * 12;
	timip_current_freq_table = j;
	
#ifdef USE_WINSYN_TIMER_I

	rtsyn_mutex_init(timip_timerMUTEX);
#ifdef TIMIP___W32__
	timeBeginPeriod(1);
	{
		DWORD data = 0;
		UINT delay;
		delay=(1000/TICKTIME_HZ);
		 
		
		timip_timerID = timeSetEvent( delay, 0, timip_timercalc, data,
			TIME_PERIODIC | TIME_CALLBACK_FUNCTION );
        if( !timip_timerID ){
        	timip_ctl->cmsg(  CMSG_ERROR, VERB_NORMAL,"Fail to setup Timer Interrupt (winsyn) \n");
        }
	}
#else
	thread_on_f=1;
	if(0!=pthread_create(&timer_thread,NULL,timip_timercalc,NULL)){
        	timip_ctl->cmsg(  CMSG_ERROR, VERB_NORMAL,"Fail to setup Timer Interrupt (winsyn) \n");
	}
#endif

#endif
	timip_rtsyn_server_reset();
}

void timip_rtsyn_close(void){
#ifdef USE_WINSYN_TIMER_I
#ifdef TIMIP___W32__
	timeKillEvent( timip_timerID );
	timeEndPeriod(1);
#else
	thread_on_f=0;
	pthread_join(timer_thread, NULL);
#endif
	rtsyn_mutex_destroy(timip_timerMUTEX);
#endif 
}

void timip_rtsyn_play_event(MidiEvent *timip_ev)
{
  int gch;
  int32 cet;
#ifdef USE_WINSYN_TIMER_I
	rtsyn_mutex_lock(timip_timerMUTEX);
#endif 

	gch = GLOBAL_CHANNEL_EVENT_TYPE(timip_ev->type);
	if(gch || !IS_SET_CHANNELMASK(timip_quietchannels, timip_ev->channel) ){
//    if ( !timip_seq_quit ) {
			timip_ev->time=0;
			timip_play_event(timip_ev);
//		}
	}
#ifdef USE_WINSYN_TIMER_I
	rtsyn_mutex_unlock(timip_timerMUTEX);
#endif 

}

void timip_rtsyn_reset(void){
		timip_rtsyn_stop_playing();
#ifdef USE_WINSYN_TIMER_I
	        rtsyn_mutex_lock(timip_timerMUTEX);
#endif

		timip_free_instruments(0);        //also in timip_rtsyn_server_reset
		timip_free_global_mblock();
#ifdef USE_WINSYN_TIMER_I
	        rtsyn_mutex_unlock(timip_timerMUTEX);
#endif

		timip_rtsyn_server_reset();
//		printf("system reseted\n");
}

void timip_rtsyn_server_reset(void){
#ifdef USE_WINSYN_TIMER_I
	rtsyn_mutex_lock(timip_timerMUTEX);
#endif 
	timip_play_mode->close_output();	// PM_REQ_PLAY_START wlll called in timip_playmidi_stream_init()
	timip_play_mode->open_output();	// but w32_a.c does not have it.
	timip_readmidi_read_init();
	timip_playmidi_stream_init();
	starttime=timip_get_current_calender_time();
	timip_reduce_voice_threshold = 0; // * Disable auto reduction timip_voice *
	timip_auto_reduce_polyphony = 0;
	event_time_offset = 0;
#ifdef USE_WINSYN_TIMER_I
	rtsyn_mutex_unlock(timip_timerMUTEX);
#endif 
}

void timip_rtsyn_stop_playing(void)
{
	if(timip_upper_voices) {
		MidiEvent timip_ev;
		timip_ev.type = ME_EOT;
		timip_ev.a = 0;
		timip_ev.b = 0;
		timip_rtsyn_play_event(&timip_ev);
#ifdef USE_WINSYN_TIMER_I
	        rtsyn_mutex_lock(timip_timerMUTEX);
#endif
		timip_aq_flush(1);
#ifdef USE_WINSYN_TIMER_I
		        rtsyn_mutex_unlock(timip_timerMUTEX);
#endif
	}
}
extern int32 timip_current_sample;
extern FLOAT_T timip_midi_time_ratio;
extern int volatile timip_stream_max_compute;


static void seq_set_time(MidiEvent *timip_ev)
{
	double currenttime, time_div;
	
	currenttime=timip_get_current_calender_time();
	time_div=currenttime-starttime;
	timip_ev->time=((double) timip_current_sample
			+ (timip_play_mode->rate)*time_div+0.5);
	starttime=currenttime;
	
	timip_rtsyn_reachtime=currenttime +  (double)(1.0f/TICKTIME_HZ);
}

#if 0
static void seq_set_time(MidiEvent *timip_ev)
{
	double past_time,btime;
	static int shift=0;

	if(set_time_first==2){
		starttime=timip_get_current_calender_time()-(double)timip_current_sample/(double)timip_play_mode->rate;
	}
	past_time = (int32)((timip_get_current_calender_time() - starttime)*timip_play_mode->rate);	
//	printf("%f,%f\n",(double)past_time,(  (double)timip_current_sample-(double)past_time )  );
	if (set_time_first==1){
		shift=(double)past_time-(double)timip_current_sample;
///		printf("%d\n",shift);
	}
	if (set_time_first>0) set_time_first--;
	event_time_offset=timip_play_mode->rate/TICKTIME_HZ;
	timip_ev->time = past_time;
	if(set_time_first==0 && (past_time-timip_current_sample>timip_stream_max_compute*timip_play_mode->rate/1000)){ 
		starttime=timip_get_current_calender_time()-(double)(timip_current_sample+shift)/(double)timip_play_mode->rate;
		timip_ev->time=timip_current_sample+shift;
	}
	timip_ev->time += (int32)event_time_offset;

	
	timip_rtsyn_reachtime=timip_get_current_calender_time()+  (double)(1.0f/TICKTIME_HZ);
	
#if 0
	btime = (double)((timip_ev->time-timip_current_sample/timip_midi_time_ratio)/timip_play_mode->rate);
	btime *= 1.01; /* to be sure */
	timip_aq_set_soft_queue(btime, 0.0);
#endif
}
#endif

void timip_rtsyn_play_calculate(){
	MidiEvent timip_ev;

#ifndef USE_WINSYN_TIMER_I	
	timip_ev.type = ME_NONE;
	seq_set_time(&timip_ev);
	timip_play_event(&timip_ev);
	timip_aq_fill_nonblocking();
#endif
	
	if(active_sensing_flag==~0 && (timip_get_current_calender_time() > active_sensing_time+0.5)){
//normaly acitive sensing expiering time is 330ms(>300ms) but this loop is heavy
		timip_play_mode->close_output();
		timip_play_mode->open_output();
		timip_ctl->cmsg(  CMSG_ERROR, VERB_NORMAL,"Active Sensing Expired\n");
		active_sensing_flag=0;
	}
}
	
int timip_rtsyn_play_one_data (int port, int32 dwParam1){
	MidiEvent timip_ev;

	timip_ev.type = ME_NONE;
	timip_ev.channel = dwParam1 & 0x0000000f;
	timip_ev.channel = timip_ev.channel+port*16;
	timip_ev.a = (dwParam1 >> 8) & 0xff;
	timip_ev.b = (dwParam1 >> 16) & 0xff;
	switch ((int) (dwParam1 & 0x000000f0)) {
	case 0x80:
		timip_ev.type = ME_NOTEOFF;
//		timip_rtsyn_play_event(&timip_ev);
		break;
	case 0x90:
		timip_ev.type = (timip_ev.b) ? ME_NOTEON : ME_NOTEOFF;
//		timip_rtsyn_play_event(&timip_ev);
		break;
	case 0xa0:
		timip_ev.type = ME_KEYPRESSURE;
//		timip_rtsyn_play_event(&timip_ev);
		break;
	case 0xb0:
		if (! timip_convert_midi_control_change(timip_ev.channel, timip_ev.a, timip_ev.b, &timip_ev))
		timip_ev.type = ME_NONE;
		break;
	case 0xc0:
		timip_ev.type = ME_PROGRAM;
//		timip_rtsyn_play_event(&timip_ev);
		break;
	case 0xd0:
		timip_ev.type = ME_CHANNEL_PRESSURE;
//		timip_rtsyn_play_event(&timip_ev);
		break;
	case 0xe0:
		timip_ev.type = ME_PITCHWHEEL;
//		timip_rtsyn_play_event(&timip_ev);
		break;
	case 0xf0:
#ifdef IA_PORTMIDISYN
		if ( (dwParam1 & 0x000000ff) == 0xf0) {
			//SysEx
			return 1;
		}
#endif
		if ((dwParam1 & 0x000000ff) == 0xf2) {
			timip_ev.type = ME_PROGRAM;
//			timip_rtsyn_play_event(&timip_ev);
		}
#if 0
		if ((dwParam1 & 0x000000ff) == 0xf1)
			//MIDI Time Code Qtr. Frame (not need)
			printf("MIDI Time Code Qtr\n");
		if ((dwParam1 & 0x000000ff) == 0xf3)
			//Song Select(Song #) (not need)
		if ((dwParam1 & 0x000000ff) == 0xf6)
			//Tune request (not need)
			printf("Tune request\n");
		if ((dwParam1 & 0x000000ff) == 0xf8)
			//Timing Clock (not need)
			printf("Timing Clock\n");
		if ((dwParam1&0x000000ff)==0xfa)
			//Start
		if ((dwParam1 & 0x000000ff) == 0xfb)
			//Continue
		if ((dwParam1 & 0x000000ff) == 0xfc) {
			//Stop
			printf("Stop\n");
		}
#endif
		if ((dwParam1 & 0x000000ff) == 0xfe) {
			//Active Sensing
//			printf("Active Sensing\n");
			active_sensing_flag = ~0;
			active_sensing_time = timip_get_current_calender_time();
		}
		if ((dwParam1 & 0x000000ff) == 0xff) {
			//System Reset
			printf("System Reset\n");
		}
		break;
	default:
//		printf("Unsup/ed event %d\n", aevp->type);
		break;
	}
	if (timip_ev.type != ME_NONE) {
		timip_rtsyn_play_event(&timip_ev);
	}
	return 0;
}


void timip_rtsyn_play_one_sysex (char *sysexbuffer, int exlen ){
	int i,j,chk,ne;
	MidiEvent timip_ev;
	MidiEvent evm[260];

	if(sysexbuffer[exlen-1] == '\xf7'){            // I don't konw why this need
		for(i=0;i<EX_RESET_NO;i++){
			chk=0;
			for(j=0;(j<exlen)&&(j<11);j++){
				if(chk==0 && sysex_resets[i][j]!=sysexbuffer[j]){
					chk=~0;
				}
			}
			if(chk==0){
				 timip_rtsyn_server_reset();
			}
		}
/*
		printf("SyeEx length=%x bytes \n", exlen);
		for(i=0;i<exlen;i++){
			printf("%x ",sysexbuffer[i]);
		}
		printf("\n");
*/
		if(timip_parse_sysex_event(sysexbuffer+1,exlen-1,&timip_ev)){
			if(timip_ev.type==ME_RESET && timip_rtsyn_system_mode!=DEFAULT_SYSTEM_MODE){
				timip_ev.a=timip_rtsyn_system_mode;
				timip_change_system_mode(timip_rtsyn_system_mode);
				timip_rtsyn_play_event(&timip_ev);
			}else{
				timip_rtsyn_play_event(&timip_ev);
			}
		}
		if(ne=timip_parse_sysex_event_multi(sysexbuffer+1,exlen-1, evm)){
			for (i = 0; i < ne; i++){
					timip_rtsyn_play_event(&evm[i]);
			}
		}
	}
}
