/*      MikMod sound library
   (c) 1998, 1999 Miodrag Vallat and others - see file AUTHORS for
   complete list.

   This library is free software; you can redistribute it and/or modify
   it under the terms timip_of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 timip_of
   the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty timip_of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy timip_of the GNU Library General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA.
 */

/*==============================================================================

  $Id: timip_load_ult.c,v 1.27 1999/10/25 16:31:41 miod Exp $

  Ultratracker (ULT) module loader

==============================================================================*/

#ifdef HAVE_CONFIG_H
#include "timip_config.h"
#endif

#include <string.h>

#include "timip_unimod_priv.h"

/*========== Module structure */

/* header */
typedef struct ULTHEADER
  {
    CHAR id[16];
    CHAR songtitle[32];
    UBYTE reserved;
  }
ULTHEADER;

/* sample information */
typedef struct ULTSAMPLE
  {
    CHAR samplename[32];
    CHAR dosname[12];
    SLONG loopstart;
    SLONG loopend;
    SLONG sizestart;
    SLONG sizeend;
    UBYTE volume;
    UBYTE flags;
    UWORD speed;
    SWORD timip_finetune;
  }
ULTSAMPLE;

typedef struct ULTEVENT
  {
    UBYTE note, sample, eff, dat1, dat2;
  }
ULTEVENT;

/*========== Loader variables */

#define ULTS_16BITS     4
#define ULTS_LOOP       8
#define ULTS_REVERSE    16

#define ULT_VERSION_LEN 18
static CHAR ULT_Version[ULT_VERSION_LEN] = "Ultra Tracker v1.x";

static ULTEVENT timip_ev;

/*========== Loader code */

BOOL 
timip_ULT_Test (void)
{
  CHAR id[16];

  if (!_mm_read_string (id, 15, timip_modreader))
    return 0;
  if (strncmp (id, "MAS_UTrack_V00", 14))
    return 0;
  if ((id[14] < '1') || (id[14] > '4'))
    return 0;
  return 1;
}

BOOL 
timip_ULT_Init (void)
{
  return 1;
}

void 
timip_ULT_Cleanup (void)
{
}

static UBYTE 
ReadUltEvent (ULTEVENT * event)
{
  UBYTE flag, rep = 1;

  flag = _mm_read_UBYTE (timip_modreader);
  if (flag == 0xfc)
    {
      rep = _mm_read_UBYTE (timip_modreader);
      event->note = _mm_read_UBYTE (timip_modreader);
    }
  else
    event->note = flag;

  event->sample = _mm_read_UBYTE (timip_modreader);
  event->eff = _mm_read_UBYTE (timip_modreader);
  event->dat1 = _mm_read_UBYTE (timip_modreader);
  event->dat2 = _mm_read_UBYTE (timip_modreader);

  return rep;
}

BOOL 
timip_ULT_Load (BOOL curious)
{
  int t, u, tracks = 0;
  SAMPLE *q;
  ULTSAMPLE s;
  ULTHEADER mh;
  UBYTE nos, noc, nop;

  /* try to read module header */
  _mm_read_string (mh.id, 15, timip_modreader);
  _mm_read_string (mh.songtitle, 32, timip_modreader);
  mh.reserved = _mm_read_UBYTE (timip_modreader);

  if (_mm_eof (timip_modreader))
    {
      _mm_errno = MMERR_LOADING_HEADER;
      return 0;
    }

  ULT_Version[ULT_VERSION_LEN - 1] = '3' + (mh.id[14] - '1');
  timip_of.modtype = timip_DupStr (ULT_Version, ULT_VERSION_LEN, 1);
  timip_of.initspeed = 6;
  timip_of.inittempo = 125;
  timip_of.reppos = 0;

  /* read songtext */
  if ((mh.id[14] > '1') && (mh.reserved))
    if (!timip_ReadLinedComment (mh.reserved, 32))
      return 0;

  nos = _mm_read_UBYTE (timip_modreader);
  if (_mm_eof (timip_modreader))
    {
      _mm_errno = MMERR_LOADING_HEADER;
      return 0;
    }

  timip_of.songname = timip_DupStr (mh.songtitle, 32, 1);
  timip_of.numins = timip_of.numsmp = nos;

  if (!timip_AllocSamples ())
    return 0;
  q = timip_of.samples;
  for (t = 0; t < nos; t++)
    {
      /* try to read sample info */
      _mm_read_string (s.samplename, 32, timip_modreader);
      _mm_read_string (s.dosname, 12, timip_modreader);
      s.loopstart = _mm_read_I_ULONG (timip_modreader);
      s.loopend = _mm_read_I_ULONG (timip_modreader);
      s.sizestart = _mm_read_I_ULONG (timip_modreader);
      s.sizeend = _mm_read_I_ULONG (timip_modreader);
      s.volume = _mm_read_UBYTE (timip_modreader);
      s.flags = _mm_read_UBYTE (timip_modreader);
      s.speed = (mh.id[14] >= '4') ? _mm_read_I_UWORD (timip_modreader) : 8363;
      s.timip_finetune = _mm_read_I_SWORD (timip_modreader);

      if (_mm_eof (timip_modreader))
	{
	  _mm_errno = MMERR_LOADING_SAMPLEINFO;
	  return 0;
	}

      q->samplename = timip_DupStr (s.samplename, 32, 1);
      /* The correct formula for the coefficient would be
         pow(2,(double)s.finetume/OCTAVE/32768), but to avoid floating point
         here, we'll use a first order approximation here.
         1/567290 == Ln(2)/OCTAVE/32768 */
      q->speed = s.speed + s.speed * (((SLONG) s.speed * (SLONG) s.timip_finetune) / 567290);
      q->length = s.sizeend - s.sizestart;
      q->volume = s.volume >> 2;
      q->loopstart = s.loopstart;
      q->loopend = s.loopend;
      q->flags = SF_SIGNED;
      if (s.flags & ULTS_LOOP)
	q->flags |= SF_LOOP;
      if (s.flags & ULTS_16BITS)
	{
	  s.sizeend += (s.sizeend - s.sizestart);
	  s.sizestart <<= 1;
	  q->flags |= SF_16BITS;
	  q->loopstart >>= 1;
	  q->loopend >>= 1;
	}
      q++;
    }

  if (!timip_AllocPositions (256))
    return 0;
  for (t = 0; t < 256; t++)
    timip_of.positions[t] = _mm_read_UBYTE (timip_modreader);
  for (t = 0; t < 256; t++)
    if (timip_of.positions[t] == 255)
      break;
  timip_of.numpos = t;

  noc = _mm_read_UBYTE (timip_modreader);
  nop = _mm_read_UBYTE (timip_modreader);

  timip_of.numchn = ++noc;
  timip_of.numpat = ++nop;
  timip_of.numtrk = timip_of.numchn * timip_of.numpat;
  if (!timip_AllocTracks ())
    return 0;
  if (!timip_AllocPatterns ())
    return 0;
  for (u = 0; u < timip_of.numchn; u++)
    for (t = 0; t < timip_of.numpat; t++)
      timip_of.patterns[(t * timip_of.numchn) + u] = tracks++;

  /* read pan position table for v1.5 and higher */
  if (mh.id[14] >= '3')
    for (t = 0; t < timip_of.numchn; t++)
      timip_of.panning[t] = _mm_read_UBYTE (timip_modreader) << 4;

  for (t = 0; t < timip_of.numtrk; t++)
    {
      int rep, row = 0;

      timip_UniReset ();
      while (row < 64)
	{
	  rep = ReadUltEvent (&timip_ev);

	  if (_mm_eof (timip_modreader))
	    {
	      _mm_errno = MMERR_LOADING_TRACK;
	      return 0;
	    }

	  while (rep--)
	    {
	      UBYTE eff;
	      int offset;

	      if (timip_ev.sample)
		UniInstrument (timip_ev.sample - 1);
	      if (timip_ev.note)
		UniNote (timip_ev.note + 2 * OCTAVE - 1);

	      /* first effect - various fixes by Alexander Kerkhove and
	         Thomas Neumann */
	      eff = timip_ev.eff >> 4;
	      switch (eff)
		{
		case 0x3:	/* tone portamento */
		  timip_UniEffect (UNI_ITEFFECTG, timip_ev.dat2);
		  break;
		case 0x5:
		  break;
		case 0x9:	/* sample offset */
		  offset = (timip_ev.dat2 << 8) | ((timip_ev.eff & 0xf) == 9 ? timip_ev.dat1 : 0);
		  timip_UniEffect (UNI_ULTEFFECT9, offset);
		  break;
		case 0xb:	/* panning */
		  timip_UniPTEffect (8, timip_ev.dat2 * 0xf);
		  break;
		case 0xc:	/* volume */
		  timip_UniPTEffect (eff, timip_ev.dat2 >> 2);
		  break;
		default:
		  timip_UniPTEffect (eff, timip_ev.dat2);
		  break;
		}

	      /* second effect */
	      eff = timip_ev.eff & 0xf;
	      switch (eff)
		{
		case 0x3:	/* tone portamento */
		  timip_UniEffect (UNI_ITEFFECTG, timip_ev.dat1);
		  break;
		case 0x5:
		  break;
		case 0x9:	/* sample offset */
		  if ((timip_ev.eff >> 4) != 9)
		    timip_UniEffect (UNI_ULTEFFECT9, ((UWORD) timip_ev.dat1) << 8);
		  break;
		case 0xb:	/* panning */
		  timip_UniPTEffect (8, timip_ev.dat1 * 0xf);
		  break;
		case 0xc:	/* volume */
		  timip_UniPTEffect (eff, timip_ev.dat1 >> 2);
		  break;
		default:
		  timip_UniPTEffect (eff, timip_ev.dat1);
		  break;
		}

	      timip_UniNewline ();
	      row++;
	    }
	}
      if (!(timip_of.tracks[t] = timip_UniDup ()))
	return 0;
    }
  return 1;
}

CHAR *
timip_ULT_LoadTitle (void)
{
  CHAR s[32];

  _mm_fseek (timip_modreader, 15, SEEK_SET);
  if (!_mm_read_UBYTES (s, 32, timip_modreader))
    return NULL;

  return (timip_DupStr (s, 32, 1));
}

/*========== Loader information */

MLOADER timip_load_ult =
{
  NULL,
  "ULT",
  "ULT (UltraTracker)",
  timip_ULT_Init,
  timip_ULT_Test,
  timip_ULT_Load,
  timip_ULT_Cleanup,
  timip_ULT_LoadTitle
};


/* ex:set ts=4: */
