/*      MikMod sound library
   (c) 1998, 1999 Miodrag Vallat and others - see file AUTHORS for
   complete list.

   This library is free software; you can redistribute it and/or modify
   it under the terms timip_of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 timip_of
   the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty timip_of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy timip_of the GNU Library General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA.
 */

/*==============================================================================

  $Id: timip_load_xm.c,v 1.32 1999/10/25 16:31:41 miod Exp $

  Fasttracker (XM) module loader

==============================================================================*/

#ifdef HAVE_CONFIG_H
#include "timip_config.h"
#endif

#include <string.h>
extern char *timip_safe_strdup(const char *s);

#include "timip_unimod_priv.h"

/*========== Module structure */

typedef struct XMHEADER
  {
    CHAR id[17];		/* ID text: 'Extended module: ' */
    CHAR songname[21];		/* Module name */
    CHAR trackername[20];	/* Tracker name */
    UWORD version;		/* Version number */
    ULONG headersize;		/* Header size */
    UWORD songlength;		/* Song length (in patten order table) */
    UWORD restart;		/* Restart position */
    UWORD numchn;		/* Number timip_of channels (2,4,6,8,10,...,32) */
    UWORD numpat;		/* Number timip_of patterns (max 256) */
    UWORD numins;		/* Number timip_of instruments (max 128) */
    UWORD flags;
    UWORD tempo;		/* Default tempo */
    UWORD bpm;			/* Default BPM */
    UBYTE orders[256];		/* Pattern order table  */
  }
XMHEADER;

typedef struct XMINSTHEADER
  {
    ULONG size;			/* Instrument size */
    CHAR name[22];		/* Instrument name */
    UBYTE type;			/* Instrument type (always 0) */
    UWORD numsmp;		/* Number timip_of samples in instrument */
    ULONG ssize;
  }
XMINSTHEADER;

#define XMENVCNT (12*2)
#define XMNOTECNT (8*OCTAVE)
typedef struct XMPATCHHEADER
  {
    UBYTE what[XMNOTECNT];	/*  Sample number for all notes */
    UWORD volenv[XMENVCNT];	/*  Points for volume envelope */
    UWORD panenv[XMENVCNT];	/*  Points for panning envelope */
    UBYTE volpts;		/*  Number timip_of volume points */
    UBYTE panpts;		/*  Number timip_of panning points */
    UBYTE volsus;		/*  Volume sustain point */
    UBYTE volbeg;		/*  Volume loop start point */
    UBYTE volend;		/*  Volume loop end point */
    UBYTE pansus;		/*  Panning sustain point */
    UBYTE panbeg;		/*  Panning loop start point */
    UBYTE panend;		/*  Panning loop end point */
    UBYTE volflg;		/*  Volume type: bit 0: On; 1: Sustain; 2: Loop */
    UBYTE panflg;		/*  Panning type: bit 0: On; 1: Sustain; 2: Loop */
    UBYTE vibflg;		/*  Vibrato type */
    UBYTE vibsweep;		/*  Vibrato sweep */
    UBYTE vibdepth;		/*  Vibrato depth */
    UBYTE vibrate;		/*  Vibrato rate */
    UWORD volfade;		/*  Volume fadeout */
  }
XMPATCHHEADER;

typedef struct XMWAVHEADER
  {
    ULONG length;		/* Sample length */
    ULONG loopstart;		/* Sample loop start */
    ULONG looplength;		/* Sample loop length */
    UBYTE volume;		/* Volume  */
    SBYTE timip_finetune;		/* Finetune (signed byte -128..+127) */
    UBYTE type;			/* Loop type */
    UBYTE panning;		/* Panning (0-255) */
    SBYTE relnote;		/* Relative note number (signed byte) */
    UBYTE reserved;
    CHAR samplename[22];	/* Sample name */
    UBYTE vibtype;		/* Vibrato type */
    UBYTE vibsweep;		/* Vibrato sweep */
    UBYTE vibdepth;		/* Vibrato depth */
    UBYTE vibrate;		/* Vibrato rate */
  }
XMWAVHEADER;

typedef struct XMPATHEADER
  {
    ULONG size;			/* Pattern header length  */
    UBYTE packing;		/* Packing type (always 0) */
    UWORD numrows;		/* Number timip_of rows in pattern (1..256) */
    SWORD packsize;		/* Packed patterndata size */
  }
XMPATHEADER;

typedef struct XMNOTE
  {
    UBYTE note, ins, vol, eff, dat;
  }
XMNOTE;

/*========== Loader variables */

static XMNOTE *xmpat = NULL;
static XMHEADER *mh = NULL;

/* increment unit for sample array reallocation */
#define XM_SMPINCR 64
static ULONG *nextwav = NULL;
static XMWAVHEADER *wh = NULL, *s = NULL;

/*========== Loader code */

BOOL 
timip_XM_Test (void)
{
  UBYTE id[38];

  if (!_mm_read_UBYTES (id, 38, timip_modreader))
    return 0;
  if (memcmp (id, "Extended Module: ", 17))
    return 0;
  if (id[37] == 0x1a)
    return 1;
  return 0;
}

BOOL 
timip_XM_Init (void)
{
  if (!(mh = (XMHEADER *) _mm_malloc (sizeof (XMHEADER))))
    return 0;
  return 1;
}

void 
timip_XM_Cleanup (void)
{
  _mm_free (mh);
}

static int 
XM_ReadNote (XMNOTE * n)
{
  UBYTE cmp, result = 1;

  memset (n, 0, sizeof (XMNOTE));
  cmp = _mm_read_UBYTE (timip_modreader);

  if (cmp & 0x80)
    {
      if (cmp & 1)
	{
	  result++;
	  n->note = _mm_read_UBYTE (timip_modreader);
	}
      if (cmp & 2)
	{
	  result++;
	  n->ins = _mm_read_UBYTE (timip_modreader);
	}
      if (cmp & 4)
	{
	  result++;
	  n->vol = _mm_read_UBYTE (timip_modreader);
	}
      if (cmp & 8)
	{
	  result++;
	  n->eff = _mm_read_UBYTE (timip_modreader);
	}
      if (cmp & 16)
	{
	  result++;
	  n->dat = _mm_read_UBYTE (timip_modreader);
	}
    }
  else
    {
      n->note = cmp;
      n->ins = _mm_read_UBYTE (timip_modreader);
      n->vol = _mm_read_UBYTE (timip_modreader);
      n->eff = _mm_read_UBYTE (timip_modreader);
      n->dat = _mm_read_UBYTE (timip_modreader);
      result += 4;
    }
  return result;
}

static UBYTE *
XM_Convert (XMNOTE * xmtrack, UWORD rows)
{
  int t;
  UBYTE note, ins, vol, eff, dat;

  timip_UniReset ();
  for (t = 0; t < rows; t++)
    {
      note = xmtrack->note;
      ins = xmtrack->ins;
      vol = xmtrack->vol;
      eff = xmtrack->eff;
      dat = xmtrack->dat;

      if (note)
	{
	  if (note > XMNOTECNT)
	    timip_UniEffect (UNI_KEYFADE, 0);
	  else
	    UniNote (note - 1);
	}
      if (ins)
	UniInstrument (ins - 1);

      switch (vol >> 4)
	{
	case 0x6:		/* volslide down */
	  if (vol & 0xf)
	    timip_UniEffect (UNI_XMEFFECTA, vol & 0xf);
	  break;
	case 0x7:		/* volslide up */
	  if (vol & 0xf)
	    timip_UniEffect (UNI_XMEFFECTA, vol << 4);
	  break;

	  /* volume-row fine volume slide is compatible with protracker
	     EBx and EAx effects i.e. a zero nibble means DO NOT SLIDE, as
	     opposed to 'take the last sliding value'. */
	case 0x8:		/* finevol down */
	  timip_UniPTEffect (0xe, 0xb0 | (vol & 0xf));
	  break;
	case 0x9:		/* finevol up */
	  timip_UniPTEffect (0xe, 0xa0 | (vol & 0xf));
	  break;
	case 0xa:		/* set vibrato speed */
	  timip_UniPTEffect (0x4, vol << 4);
	  break;
	case 0xb:		/* vibrato */
	  timip_UniPTEffect (0x4, vol & 0xf);
	  break;
	case 0xc:		/* set panning */
	  timip_UniPTEffect (0x8, vol << 4);
	  break;
	case 0xd:		/* panning slide left (only slide when data not zero) */
	  if (vol & 0xf)
	    timip_UniEffect (UNI_XMEFFECTP, vol & 0xf);
	  break;
	case 0xe:		/* panning slide right (only slide when data not zero) */
	  if (vol & 0xf)
	    timip_UniEffect (UNI_XMEFFECTP, vol << 4);
	  break;
	case 0xf:		/* tone porta */
	  timip_UniPTEffect (0x3, vol << 4);
	  break;
	default:
	  if ((vol >= 0x10) && (vol <= 0x50))
	    timip_UniPTEffect (0xc, vol - 0x10);
	}

      switch (eff)
	{
	case 0x4:
	  timip_UniEffect (UNI_XMEFFECT4, dat);
	  break;
	case 0xa:
	  timip_UniEffect (UNI_XMEFFECTA, dat);
	  break;
	case 0xe:		/* Extended effects */
	  switch (dat >> 4)
	    {
	    case 0x1:		/* XM fine porta up */
	      timip_UniEffect (UNI_XMEFFECTE1, dat & 0xf);
	      break;
	    case 0x2:		/* XM fine porta down */
	      timip_UniEffect (UNI_XMEFFECTE2, dat & 0xf);
	      break;
	    case 0xa:		/* XM fine volume up */
	      timip_UniEffect (UNI_XMEFFECTEA, dat & 0xf);
	      break;
	    case 0xb:		/* XM fine volume down */
	      timip_UniEffect (UNI_XMEFFECTEB, dat & 0xf);
	      break;
	    default:
	      timip_UniPTEffect (eff, dat);
	    }
	  break;
	case 'G' - 55:		/* G - set global volume */
	  timip_UniEffect (UNI_XMEFFECTG, dat > 64 ? 64 : dat);
	  break;
	case 'H' - 55:		/* H - global volume slide */
	  timip_UniEffect (UNI_XMEFFECTH, dat);
	  break;
	case 'K' - 55:		/* K - keyOff and KeyFade */
	  timip_UniEffect (UNI_KEYFADE, dat);
	  break;
	case 'L' - 55:		/* L - set envelope position */
	  timip_UniEffect (UNI_XMEFFECTL, dat);
	  break;
	case 'P' - 55:		/* P - panning slide */
	  timip_UniEffect (UNI_XMEFFECTP, dat);
	  break;
	case 'R' - 55:		/* R - multi retrig note */
	  timip_UniEffect (UNI_S3MEFFECTQ, dat);
	  break;
	case 'T' - 55:		/* T - Tremor */
	  timip_UniEffect (UNI_S3MEFFECTI, dat);
	  break;
	case 'X' - 55:
	  switch (dat >> 4)
	    {
	    case 1:		/* X1 - Extra Fine Porta up */
	      timip_UniEffect (UNI_XMEFFECTX1, dat & 0xf);
	      break;
	    case 2:		/* X2 - Extra Fine Porta down */
	      timip_UniEffect (UNI_XMEFFECTX2, dat & 0xf);
	      break;
	    }
	  break;
	default:
	  if (eff <= 0xf)
	    {
	      /* the pattern jump destination is written in decimal,
	         but it seems some poor tracker software writes them
	         in hexadecimal... (sigh) */
	      if (eff == 0xd)
		/* don't change anything if we're sure it's in hexa */
		if ((((dat & 0xf0) >> 4) <= 9) && ((dat & 0xf) <= 9))
		  /* otherwise, convert from dec to hex */
		  dat = (((dat & 0xf0) >> 4) * 10) + (dat & 0xf);
	      timip_UniPTEffect (eff, dat);
	    }
	  break;
	}
      timip_UniNewline ();
      xmtrack++;
    }
  return timip_UniDup ();
}

static BOOL 
LoadPatterns (BOOL dummypat)
{
  int t, u, v, numtrk;

  if (!timip_AllocTracks ())
    return 0;
  if (!timip_AllocPatterns ())
    return 0;

  numtrk = 0;
  for (t = 0; t < mh->numpat; t++)
    {
      XMPATHEADER ph;

      ph.size = _mm_read_I_ULONG (timip_modreader);
      if (ph.size < (mh->version == 0x0102 ? 8 : 9))
	{
	  _mm_errno = MMERR_LOADING_PATTERN;
	  return 0;
	}
      ph.packing = _mm_read_UBYTE (timip_modreader);
      if (ph.packing)
	{
	  _mm_errno = MMERR_LOADING_PATTERN;
	  return 0;
	}
      if (mh->version == 0x0102)
	ph.numrows = _mm_read_UBYTE (timip_modreader) + 1;
      else
	ph.numrows = _mm_read_I_UWORD (timip_modreader);
      ph.packsize = _mm_read_I_UWORD (timip_modreader);

      ph.size -= (mh->version == 0x0102 ? 8 : 9);
      if (ph.size)
	_mm_fseek (timip_modreader, ph.size, SEEK_CUR);

      timip_of.pattrows[t] = ph.numrows;

      if (ph.numrows)
	{
	  if (!(xmpat = (XMNOTE *) _mm_calloc (ph.numrows * timip_of.numchn, sizeof (XMNOTE))))
	    return 0;

	  /* when packsize is 0, don't try to load a pattern.. it's empty. */
	  if (ph.packsize)
	    for (u = 0; u < ph.numrows; u++)
	      for (v = 0; v < timip_of.numchn; v++)
		{
		  if (!ph.packsize)
		    break;

		  ph.packsize -= XM_ReadNote (&xmpat[(v * ph.numrows) + u]);
		  if (ph.packsize < 0)
		    {
		      free (xmpat);
		      xmpat = NULL;
		      _mm_errno = MMERR_LOADING_PATTERN;
		      return 0;
		    }
		}

	  if (ph.packsize)
	    {
	      _mm_fseek (timip_modreader, ph.packsize, SEEK_CUR);
	    }

	  if (_mm_eof (timip_modreader))
	    {
	      free (xmpat);
	      xmpat = NULL;
	      _mm_errno = MMERR_LOADING_PATTERN;
	      return 0;
	    }

	  for (v = 0; v < timip_of.numchn; v++)
	    timip_of.tracks[numtrk++] = XM_Convert (&xmpat[v * ph.numrows], ph.numrows);

	  free (xmpat);
	  xmpat = NULL;
	}
      else
	{
	  for (v = 0; v < timip_of.numchn; v++)
	    timip_of.tracks[numtrk++] = XM_Convert (NULL, ph.numrows);
	}
    }

  if (dummypat)
    {
      timip_of.pattrows[t] = 64;
      if (!(xmpat = (XMNOTE *) _mm_calloc (64 * timip_of.numchn, sizeof (XMNOTE))))
	return 0;
      for (v = 0; v < timip_of.numchn; v++)
	timip_of.tracks[numtrk++] = XM_Convert (&xmpat[v * 64], 64);
      free (xmpat);
      xmpat = NULL;
    }

  return 1;
}

static BOOL 
LoadInstruments (void)
{
  int t, u;
  INSTRUMENT *d;
  long next = 0;
  UWORD wavcnt = 0;

  if (!timip_AllocInstruments ())
    return 0;
  d = timip_of.instruments;
  for (t = 0; t < timip_of.numins; t++, d++)
    {
      XMINSTHEADER ih;
      long headend;

      memset (d->samplenumber, 0xff, INSTNOTES * sizeof (UWORD));

      /* read instrument header */
      headend = _mm_ftell (timip_modreader);
      ih.size = _mm_read_I_ULONG (timip_modreader);
      headend += ih.size;
      _mm_read_string (ih.name, 22, timip_modreader);
      ih.type = _mm_read_UBYTE (timip_modreader);
      ih.numsmp = _mm_read_I_UWORD (timip_modreader);

      d->insname = timip_DupStr (ih.name, 22, 1);

      if ((SWORD) ih.size > 29)
	{
	  ih.ssize = _mm_read_I_ULONG (timip_modreader);
	  if (((SWORD) ih.numsmp > 0) && (ih.numsmp <= XMNOTECNT))
	    {
	      XMPATCHHEADER pth;
	      int p;

	      _mm_read_UBYTES (pth.what, XMNOTECNT, timip_modreader);
	      _mm_read_I_UWORDS (pth.volenv, XMENVCNT, timip_modreader);
	      _mm_read_I_UWORDS (pth.panenv, XMENVCNT, timip_modreader);
	      pth.volpts = _mm_read_UBYTE (timip_modreader);
	      pth.panpts = _mm_read_UBYTE (timip_modreader);
	      pth.volsus = _mm_read_UBYTE (timip_modreader);
	      pth.volbeg = _mm_read_UBYTE (timip_modreader);
	      pth.volend = _mm_read_UBYTE (timip_modreader);
	      pth.pansus = _mm_read_UBYTE (timip_modreader);
	      pth.panbeg = _mm_read_UBYTE (timip_modreader);
	      pth.panend = _mm_read_UBYTE (timip_modreader);
	      pth.volflg = _mm_read_UBYTE (timip_modreader);
	      pth.panflg = _mm_read_UBYTE (timip_modreader);
	      pth.vibflg = _mm_read_UBYTE (timip_modreader);
	      pth.vibsweep = _mm_read_UBYTE (timip_modreader);
	      pth.vibdepth = _mm_read_UBYTE (timip_modreader);
	      pth.vibrate = _mm_read_UBYTE (timip_modreader);
	      pth.volfade = _mm_read_I_UWORD (timip_modreader);

	      /* read the remainder timip_of the header
	         (2 bytes for 1.03, 22 for 1.04) */
	      for (u = headend - _mm_ftell (timip_modreader); u; u--)
		_mm_read_UBYTE (timip_modreader);

	      /* #@!$&% fix for K_OSPACE.XM and possibly others */
	      if(pth.volpts > XMENVCNT/2)
	        pth.volpts = XMENVCNT/2;
	      if(pth.panpts > XMENVCNT/2)
	        pth.panpts = XMENVCNT/2;

	      if ((_mm_eof (timip_modreader)) || (pth.volpts > XMENVCNT / 2) || (pth.panpts > XMENVCNT / 2))
		{
		  if (nextwav)
		    {
		      free (nextwav);
		      nextwav = NULL;
		    }
		  if (wh)
		    {
		      free (wh);
		      wh = NULL;
		    }
		  _mm_errno = MMERR_LOADING_SAMPLEINFO;
		  return 0;
		}

	      for (u = 0; u < XMNOTECNT; u++)
		d->samplenumber[u] = pth.what[u] + timip_of.numsmp;
	      d->volfade = pth.volfade;

#define XM_ProcessEnvelope(name) 											\
 				for (u = 0; u < (XMENVCNT / 2); u++) {						\
 					d->name##env[u].pos = pth.name##env[2*u];		\
 					d->name##env[u].val = pth.name##env[2*u + 1];	\
 				}															\
				memcpy(d->name##env,pth.name##env,XMENVCNT);			\
				if (pth.name##flg&1) d->name##flg|=EF_ON;				\
				if (pth.name##flg&2) d->name##flg|=EF_SUSTAIN;			\
				if (pth.name##flg&4) d->name##flg|=EF_LOOP;				\
				d->name##susbeg=d->name##susend=pth.name##sus;		\
				d->name##beg=pth.name##beg;								\
				d->name##end=pth.name##end;								\
				d->name##pts=pth.name##pts;								\
																			\
				/* scale envelope */										\
				for (p=0;p<XMENVCNT/2;p++)									\
					d->name##env[p].val<<=2;								\
																			\
				if ((d->name##flg&EF_ON)&&(d->name##pts<2))				\
					d->name##flg&=~EF_ON;

	      XM_ProcessEnvelope (vol);
	      XM_ProcessEnvelope (pan);
#undef XM_ProcessEnvelope

	      /* Samples are stored outside the instrument struct now, so we
	         have to load them all into a temp area, count the timip_of.numsmp
	         along the way and then do an timip_AllocSamples() and move
	         everything over */
	      if (mh->version > 0x0103)
		next = 0;
	      for (u = 0; u < ih.numsmp; u++, s++)
		{
		  /* Allocate more room for sample information if necessary */
		  if (timip_of.numsmp + u == wavcnt)
		    {
		      wavcnt += XM_SMPINCR;
		      if (!(nextwav = realloc (nextwav, wavcnt * sizeof (ULONG))))
			{
			  if (wh)
			    {
			      free (wh);
			      wh = NULL;
			    }
			  _mm_errno = MMERR_OUT_OF_MEMORY;
			  return 0;
			}
		      if (!(wh = realloc (wh, wavcnt * sizeof (XMWAVHEADER))))
			{
			  free (nextwav);
			  nextwav = NULL;
			  _mm_errno = MMERR_OUT_OF_MEMORY;
			  return 0;
			}
		      s = wh + (wavcnt - XM_SMPINCR);
		    }

		  s->length = _mm_read_I_ULONG (timip_modreader);
		  s->loopstart = _mm_read_I_ULONG (timip_modreader);
		  s->looplength = _mm_read_I_ULONG (timip_modreader);
		  s->volume = _mm_read_UBYTE (timip_modreader);
		  s->timip_finetune = _mm_read_SBYTE (timip_modreader);
		  s->type = _mm_read_UBYTE (timip_modreader);
		  s->panning = _mm_read_UBYTE (timip_modreader);
		  s->relnote = _mm_read_SBYTE (timip_modreader);
		  s->vibtype = pth.vibflg;
		  s->vibsweep = pth.vibsweep;
		  s->vibdepth = pth.vibdepth * 4;
		  s->vibrate = pth.vibrate;
		  s->reserved = _mm_read_UBYTE (timip_modreader);
		  _mm_read_string (s->samplename, 22, timip_modreader);

		  nextwav[timip_of.numsmp + u] = next;
		  next += s->length;

		  if (_mm_eof (timip_modreader))
		    {
		      free (nextwav);
		      free (wh);
		      nextwav = NULL;
		      wh = NULL;
		      _mm_errno = MMERR_LOADING_SAMPLEINFO;
		      return 0;
		    }
		}

	      if (mh->version > 0x0103)
		{
		  for (u = 0; u < ih.numsmp; u++)
		    nextwav[timip_of.numsmp++] += _mm_ftell (timip_modreader);
		  _mm_fseek (timip_modreader, next, SEEK_CUR);
		}
	      else
		timip_of.numsmp += ih.numsmp;
	    }
	  else
	    {
	      /* read the remainder timip_of the header */
	      for (u = headend - _mm_ftell (timip_modreader); u; u--)
		_mm_read_UBYTE (timip_modreader);

	      if (_mm_eof (timip_modreader))
		{
		  free (nextwav);
		  free (wh);
		  nextwav = NULL;
		  wh = NULL;
		  _mm_errno = MMERR_LOADING_SAMPLEINFO;
		  return 0;
		}
	    }
	}
    }

  /* sanity check */
  if (!timip_of.numsmp)
    {
      if (nextwav)
	{
	  free (nextwav);
	  nextwav = NULL;
	}
      if (wh)
	{
	  free (wh);
	  wh = NULL;
	}
      _mm_errno = MMERR_LOADING_SAMPLEINFO;
      return 0;
    }

  return 1;
}

BOOL 
timip_XM_Load (BOOL curious)
{
  INSTRUMENT *d;
  SAMPLE *q;
  int t, u;
  BOOL dummypat = 0;
  char tracker[21], modtype[60];

  /* try to read module header */
  _mm_read_string (mh->id, 17, timip_modreader);
  _mm_read_string (mh->songname, 21, timip_modreader);
  _mm_read_string (mh->trackername, 20, timip_modreader);
  mh->version = _mm_read_I_UWORD (timip_modreader);
  if ((mh->version < 0x102) || (mh->version > 0x104))
    {
      _mm_errno = MMERR_NOT_A_MODULE;
      return 0;
    }
  mh->headersize = _mm_read_I_ULONG (timip_modreader);
  mh->songlength = _mm_read_I_UWORD (timip_modreader);
  mh->restart = _mm_read_I_UWORD (timip_modreader);
  mh->numchn = _mm_read_I_UWORD (timip_modreader);
  mh->numpat = _mm_read_I_UWORD (timip_modreader);
  mh->numins = _mm_read_I_UWORD (timip_modreader);
  mh->flags = _mm_read_I_UWORD (timip_modreader);
  mh->tempo = _mm_read_I_UWORD (timip_modreader);
  mh->bpm = _mm_read_I_UWORD (timip_modreader);
  if (!mh->bpm)
    {
      _mm_errno = MMERR_NOT_A_MODULE;
      return 0;
    }
  _mm_read_UBYTES (mh->orders, 256, timip_modreader);

  if (_mm_eof (timip_modreader))
    {
      _mm_errno = MMERR_LOADING_HEADER;
      return 0;
    }

  /* set module variables */
  timip_of.initspeed = mh->tempo;
  timip_of.inittempo = mh->bpm;
  strncpy (tracker, mh->trackername, 20);
  tracker[20] = 0;
  for (t = 20; (tracker[t] <= ' ') && (t >= 0); t--)
    tracker[t] = 0;

  /* some modules have the tracker name empty */
  if (!tracker[0])
    strcpy (tracker, "Unknown tracker");

#ifdef	TIMIP_HAVE_SNPRINTF
  snprintf (modtype, 60, "%s (XM format %d.%02d)",
	    tracker, mh->version >> 8, mh->version & 0xff);
#else
  sprintf (modtype, "%s (XM format %d.%02d)",
	   tracker, mh->version >> 8, mh->version & 0xff);
#endif
  timip_of.modtype = timip_safe_strdup (modtype);
  timip_of.numchn = mh->numchn;
  timip_of.numpat = mh->numpat;
  timip_of.numtrk = (UWORD) timip_of.numpat * timip_of.numchn;	/* get number timip_of channels */
  timip_of.songname = timip_DupStr (mh->songname, 20, 1);
  timip_of.numpos = mh->songlength;	/* copy the songlength */
  timip_of.reppos = mh->restart < mh->songlength ? mh->restart : 0;
  timip_of.numins = mh->numins;
  timip_of.flags |= UF_XMPERIODS | UF_INST | UF_BGSLIDES | UF_NOWRAP | UF_FT2QUIRKS;
  if (mh->flags & 1)
    timip_of.flags |= UF_LINEAR;

  memset (timip_of.chanvol, 64, timip_of.numchn);	/* store channel volumes */

  if (!timip_AllocPositions (timip_of.numpos + 1))
    return 0;
  for (t = 0; t < timip_of.numpos; t++)
    timip_of.positions[t] = mh->orders[t];

  /* We have to check for any pattern numbers in the order list greater than
     the number timip_of patterns total. If one or more is found, we set it equal to
     the pattern total and make a dummy pattern to workaround the problem */
  for (t = 0; t < timip_of.numpos; t++)
    {
      if (timip_of.positions[t] >= timip_of.numpat)
	{
	  timip_of.positions[t] = timip_of.numpat;
	  dummypat = 1;
	}
    }
  if (dummypat)
    {
      timip_of.numpat++;
      timip_of.numtrk += timip_of.numchn;
    }

  if (mh->version < 0x0104)
    {
      if (!LoadInstruments ())
	return 0;
      if (!LoadPatterns (dummypat))
	return 0;
      for (t = 0; t < timip_of.numsmp; t++)
	nextwav[t] += _mm_ftell (timip_modreader);
    }
  else
    {
      if (!LoadPatterns (dummypat))
	return 0;
      if (!LoadInstruments ())
	return 0;
    }

  if (!timip_AllocSamples ())
    {
      free (nextwav);
      free (wh);
      nextwav = NULL;
      wh = NULL;
      return 0;
    }
  q = timip_of.samples;
  s = wh;
  for (u = 0; u < timip_of.numsmp; u++, q++, s++)
    {
      q->samplename = timip_DupStr (s->samplename, 22, 1);
      q->length = s->length;
      q->loopstart = s->loopstart;
      q->loopend = s->loopstart + s->looplength;
      q->volume = s->volume;
      q->speed = s->timip_finetune + 128;
      q->panning = s->panning;
      q->seekpos = nextwav[u];
      q->vibtype = s->vibtype;
      q->vibsweep = s->vibsweep;
      q->vibdepth = s->vibdepth;
      q->vibrate = s->vibrate;

      if (s->type & 0x10)
	{
	  q->length >>= 1;
	  q->loopstart >>= 1;
	  q->loopend >>= 1;
	}

      q->flags |= SF_OWNPAN;
      if (s->type & 0x3)
	q->flags |= SF_LOOP;
      if (s->type & 0x2)
	q->flags |= SF_BIDI;
      if (s->type & 0x10)
	q->flags |= SF_16BITS;
      q->flags |= SF_DELTA | SF_SIGNED;
    }

  d = timip_of.instruments;
  s = wh;
  for (u = 0; u < timip_of.numins; u++, d++)
    for (t = 0; t < XMNOTECNT; t++)
      {
	if (d->samplenumber[t] >= timip_of.numsmp)
	  d->samplenote[t] = 255;
	else
	  {
	    int note = t + s[d->samplenumber[t]].relnote;
	    d->samplenote[t] = (note < 0) ? 0 : note;
	  }
      }

  free (wh);
  free (nextwav);
  wh = NULL;
  nextwav = NULL;
  return 1;
}

CHAR *
timip_XM_LoadTitle (void)
{
  CHAR s[21];

  _mm_fseek (timip_modreader, 17, SEEK_SET);
  if (!_mm_read_UBYTES (s, 21, timip_modreader))
    return NULL;

  return (timip_DupStr (s, 21, 1));
}

/*========== Loader information */

MLOADER timip_load_xm =
{
  NULL,
  "XM",
  "XM (FastTracker 2)",
  timip_XM_Init,
  timip_XM_Test,
  timip_XM_Load,
  timip_XM_Cleanup,
  timip_XM_LoadTitle
};

/* ex:set ts=4: */
