/*
 * Copyright (c) 2007, 2008 University of Tsukuba
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the University of Tsukuba nor the names of its
 *    contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
/*
 * Copyright (c) 2010-2012 Yuichi Watanabe
 */

#include <core/types.h>
#include <core/initfunc.h>
#include "asm.h"
#include "constants.h"
#include "cpu_interpreter.h"
#include "cpu_mmu.h"
#include "current.h"
#include "io_io.h"
#include "panic.h"

/* #define IOIO_DEBUG */
#ifdef IOIO_DEBUG
#include <core/printf.h>
#define IOIO_DBG(...)						\
	do {							\
		printf(__VA_ARGS__);				\
	} while (0)
#else
#define IOIO_DBG(...)
#endif

struct call_iopass_data {
	ioport_t port;
	bool pass;
};

int
do_io_nothing (iotype_t type, ioport_t port, void *data)
{
	switch (type) {
	case IOTYPE_INB:
		*(u8 *)data = 0xFF;
		break;
	case IOTYPE_INW:
		*(u16 *)data = 0xFFFF;
		break;
	case IOTYPE_INL:
		*(u32 *)data = 0xFFFFFFFF;
		break;
	case IOTYPE_OUTB:
	case IOTYPE_OUTW:
	case IOTYPE_OUTL:
		break;
	default:
		panic ("do_io_nothing: Bad type");
	}
	return 1; /* no pass */
}

int
do_io_pass (iotype_t type, ioport_t port, void *data)
{
	switch (type) {
	case IOTYPE_INB:
		asm_inb (port, (u8 *)data);
#ifdef IOIO_DEBUG
		if (port > 0x1000) {
			IOIO_DBG("ioio: inb port 0x%x data 0x%x\n", port, *(u8 *)data);
		}
#endif
		break;
	case IOTYPE_INW:
		asm_inw (port, (u16 *)data);
#ifdef IOIO_DEBUG
		if (port > 0x1000) {
			IOIO_DBG("ioio: inw port 0x%x data 0x%x\n", port, *(u16 *)data);
		}
#endif
		break;
	case IOTYPE_INL:
		asm_inl (port, (u32 *)data);
#ifdef IOIO_DEBUG
		if (port > 0x1000) {
			IOIO_DBG("ioio: inl port 0x%x data 0x%x\n", port, *(u32 *)data);
		}
#endif
		break;
	case IOTYPE_OUTB:
#ifdef IOIO_DEBUG
		if (port > 0x1000) {
			IOIO_DBG("ioio: outb port 0x%x data 0x%x\n", port, *(u8 *)data);
		}
#endif
		asm_outb (port, *(u8 *)data);
		break;
	case IOTYPE_OUTW:
#ifdef IOIO_DEBUG
		if (port > 0x1000) {
			IOIO_DBG("ioio: outw port 0x%x data 0x%x\n", port, *(u16 *)data);
		}
#endif
		asm_outw (port, *(u16 *)data);
		break;
	case IOTYPE_OUTL:
#ifdef IOIO_DEBUG
		if (port > 0x1000) {
			IOIO_DBG("ioio: outl port 0x%x data 0x%x\n", port, *(u32 *)data);
		}
#endif
		asm_outl (port, *(u32 *)data);
		break;
	default:
		panic ("do_iopass_default: Bad type");
	}
	return 1; /* no pass */
}

static bool
call_iopass (struct vcpu *p, void *q)
{
	struct call_iopass_data *d;

	d = q;
	p->vmctl.extern_iopass (p, d->port, d->pass);
	return false;
}

iofunc_t
set_iofunc (ioport_t port, iofunc_t func)
{
	iofunc_t old, *p;
	struct call_iopass_data d;

	p = &current->vm->io.iofunc[port & 0xFFFF];
	/* old = *p; *p = func; */
	old = (iofunc_t)asm_lock_swap_ulong ((ulong *)p, (ulong)func);
	d.port = port;
	if (func == do_io_pass)
		d.pass = true;
	else
		d.pass = false;
	vcpu_list_foreach (current->vm, call_iopass, &d);
	return old;
}

void
iotype_to_size_wr(iotype_t type, ioport_t *size, bool *wr)
{
	switch (type) {
	case IOTYPE_INB:
		*wr = false;
		*size = 1;
		break;
	case IOTYPE_INW:
		*wr = false;
		*size = 2;
		break;
	case IOTYPE_INL:
		*wr = false;
		*size = 4;
		break;
	case IOTYPE_OUTB:
		*wr = true;
		*size = 1;
		break;
	case IOTYPE_OUTW:
		*wr = true;
		*size = 2;
		break;
	case IOTYPE_OUTL:
		*wr = true;
		*size = 4;
		break;
	default:
		panic("iotype_to_size_wr: Invalid type. type %d", type);
	}
}

void
call_io (iotype_t type, ioport_t port, void *data)
{
	int no_pass;
	iofunc_t func;
	iotype_t smalltype;

	/*
	 * Divide a unaligned access to two, three aligned accesses.
	 */
	if (type == IOTYPE_INW || type == IOTYPE_OUTW) {
		if (port & 0x1) {
			smalltype = IOTYPE_INW ? IOTYPE_INB : IOTYPE_OUTB;
			call_io(smalltype, port, data);
			call_io(smalltype, port + 1, (void *)((ulong)data + 1));
			return;
		}
	} else if (type == IOTYPE_INL || type == IOTYPE_OUTL) {
		switch (port & 0x3) {
		case 0:
			break;
		case 1:
		case 3:
			smalltype = IOTYPE_INL ? IOTYPE_INB : IOTYPE_OUTB;
			call_io(smalltype, port, data);
			smalltype = IOTYPE_INL ? IOTYPE_INW : IOTYPE_OUTW;
			call_io(smalltype, port + 1, (void *)((ulong)data + 1));
			smalltype = IOTYPE_INL ? IOTYPE_INB : IOTYPE_OUTB;
			call_io(smalltype, port + 3, (void *)((ulong)data + 3));
			return;
		case 2:
			smalltype = IOTYPE_INL ? IOTYPE_INW : IOTYPE_OUTW;
			call_io(smalltype, port, data);
			call_io(smalltype, port + 2, (void *)((ulong)data + 2));
			return;
		}
	}

	/*
	 * Return all F on reading by default.
	 */
	do_io_nothing(type, port, data);

	func = current->vm->io.iofunc[port & 0xFFFF];
	if (func == NULL) {
		return;
	}
	no_pass = func(type, port & 0xFFFF, data);
	if (!no_pass) {
		do_io_pass(type, port & 0xFFFF, data);
	}
}
