/*
 * Copyright (c) 2007, 2008 University of Tsukuba
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the University of Tsukuba nor the names of its
 *    contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
/*
 * Copyright (c) 2010-2014 Yuichi Watanabe
 */

#include <core/assert.h>
#include <core/vm.h>
#include <core/initfunc.h>
#include <core/panic.h>
#include <core/printf.h>
#include <core/string.h>
#include "mm.h"
#include "gbiosloader.h"

#define GBIOS_GPHYS_END	0xfffff
#define GBIOS_MAX_SIZE	0x20000

static void *saved_gbios = 0;
static size_t gbios_size;

void
gbios_save (u32 start, u32 size)
{
	void *tmp;

	if (size > GBIOS_MAX_SIZE) {
		panic("Guest BIOS is too large.");
	}
	saved_gbios = alloc (size);
	if (saved_gbios == NULL) {
		panic("Failed to allocate guest BIOS save area.");
	}
	tmp = mapmem (MAPMEM_HPHYS, start, size);
	if (tmp == NULL) {
		panic("Failed to map guest BIOS loaded by GRUB or another loader.");
	}
	memcpy (saved_gbios, tmp, size);
	gbios_size = size;
	unmapmem (tmp, size);
	return;
}

vmmerr_t
gbios_load (void)
{
	phys_t gphys_start;
	void *start;

	if (vm_get_id() == 0) {
		/*
		 * VM0 boots with host BIOS.
		 */
		return VMMERR_SUCCESS;
	}

	if (saved_gbios == NULL) {
		panic("Guest BIOS is not loaded by GRUB or another loader");
	}

	gphys_start = GBIOS_GPHYS_END + 1 - gbios_size;
	start = mapmem(MAPMEM_GPHYS, gphys_start, gbios_size);
	if (start == NULL) {
		printf("Failed to load guest BIOS.\n");
		return VMMERR_NOMEM;
	}
	memcpy(start, saved_gbios, gbios_size);
	unmapmem(start, gbios_size);
	return VMMERR_SUCCESS;
}
