/*
 * Copyright (c) 2010-2014 Yuichi Watanabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of Yuichi Watanabe nor the names of its
 *    contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <common/calc.h>
#include <core/panic.h>
#include <core/printf.h>
#include <core/rm.h>
#include <core/string.h>
#include <core/vm_config.h>
#include "current.h"
#include "mm.h"
#include "vm.h"
#include "uefi.h"
#include "calluefi.h"

/* #define GMM_DEBUG */
#ifdef GMM_DEBUG
#define GMM_DBG(...)						\
	do {							\
		printf("GMMASSIGN: " __VA_ARGS__);		\
	} while (0)
#else
#define GMM_DBG(...)
#endif

static void
gmm_assign_reserve_mem(phys_t start, phys_t end)
{
	u64 npages;
	int ret;

	if (!uefi_booted) {
		return;
	}
	npages = (end - start + 1) >> PAGESIZE_SHIFT;
	ret = call_uefi_allocate_pages(2 /* AllocateAddress */,
				       8 /* EfiUnusableMemory */,
				       npages, &start);
	if (ret) {
		panic("Failed to reserve memory 0x%llx-0x%llx 0x%x",
		      start, end, ret);
	}
}

static int
gmm_alloc_avail(struct vm *vm, int count, struct resource *parent,
		phys_t start, phys_t end, phys_t *mem_size)
{
	struct resource *new;
	phys_t size = *mem_size;
	phys_t low_mem_size = ROUND_DOWN(mm_top_of_low_avail_mem() + 1,
					 PAGESIZE);

	while(count < MAXNUM_OF_SYSMEMMAP) {
		new = vm->resource + count;
		if (rm_alloc_avail_resource(parent, new,
					    start, end, PAGESIZE,
					    RESOURCE_TYPE_MEM,
					    MEM_TYPE_AVAILABLE,
					    vm->name) != VMMERR_SUCCESS) {
			break;
		}
		if (size <= low_mem_size) {
			size += (new->end - new->start + 1);
			if (size > low_mem_size) {
				GMM_DBG("size > low_mem_size\n");
				new->end -= (size - low_mem_size);
			}
		}
		gmm_assign_reserve_mem(new->start, new->end);
		GMM_DBG("%s %016llx-%016llx\n",
			vm->name, new->start, new->end);
		count++;
	}
	*mem_size = size;
	return count;
}

static void
gmm_assign_specified_memory(struct vm *vm, struct vm_config *vm_config)
{
	struct resource *parent = NULL;
	struct mem_config *mem_config;
	phys_t mem_size = 0;
	int count = 0;

	LIST2_FOREACH(vm_config->mem_list, mem_list, mem_config) {
		while ((parent = mm_next_phys_mem_map(parent)) != NULL) {
			if (parent->data != MEM_TYPE_AVAILABLE) {
				continue;
			}
			count = gmm_alloc_avail(vm, count, parent,
						mem_config->start,
						mem_config->end,
						&mem_size);
			if (count >= MAXNUM_OF_SYSMEMMAP) {
				printf("Too many resource for 0x%s\n",
				       vm->name);
				goto out;
			}
		}
	}
 out:
	vm->resource_count = count;
}

static void
gmm_assign_available_memory(struct vm *vm)
{
	int count = 0;
	phys_t avail_start, avail_end;
	struct resource *parent = NULL, *new, *next, *prev;

	while ((parent = mm_next_phys_mem_map(parent)) != NULL) {
		if (parent->data != MEM_TYPE_AVAILABLE) {
			new = vm->resource + count++;
			rm_init_resource(new, parent->start,
					 parent->end, parent->type,
					 parent->data, vm->name);
			rm_insert_resource(parent, new);
			if (count >= MAXNUM_OF_SYSMEMMAP) {
				goto out;
			}
			continue;
		}
		next = NULL;
		prev = NULL;
		for (;;) {
			next = RM_RESOURCE_NEXT(parent, next);
			avail_start = prev ? prev->end + 1 : parent->start;
			avail_end = next ? next->start - 1 : parent->end;
			if (avail_start <= avail_end) {
				new = vm->resource + count++;
				rm_init_resource(new, avail_start,
						 avail_end, parent->type,
						 parent->data, vm->name);
				rm_insert_resource(parent, new);
			}
			if (next == NULL) {
				break;
			}
			new = vm->resource + count++;
			rm_init_resource(new, next->start,
					 next->end, parent->type,
					 MEM_TYPE_RESERVED, vm->name);
			if (count >= MAXNUM_OF_SYSMEMMAP) {
				printf("Too many resource for 0x%s\n",
				       vm->name);
				goto out;
			}
			prev = next;
		}
	}
 out:
	vm->resource_count = count;
}

void
gmm_assign_all_memory(void)
{
	struct vm_config *vm_config;
	struct vm *vm;

	LIST2_FOREACH(vm_config_list.head, vm_config_list, vm_config) {
		if (strcmp(vm_config->name, VM0_NAME) == 0) {
				continue;
		}
		vm = vm_find_by_name(vm_config->name);
		if (vm == NULL) {
			continue;
		}
		gmm_assign_specified_memory(vm, vm_config);
	}

	vm = vm_find_by_name(VM0_NAME);
	if (vm) {
		gmm_assign_available_memory(vm);
	}
}

phys_t
gmm_gp2hp(phys_t gphys, bool *fakerom)
{
	return current->gmm.gp2hp(gphys, fakerom);
}

int
gmm_get_mem_map(int index, phys_t *base, phys_t *len, u32 *type,
		bool *restrict_access)
{
	return current->gmm.get_mem_map(index, base, len, type,
					    restrict_access);
}

phys32_t
gmm_top_of_low_avail_mem(void)
{
	phys_t top_of_low = 0;
	phys_t gphys, len, end;
	u32 type;
	int index = 0;

	while (gmm_get_mem_map(index++, &gphys, &len, &type, NULL)) {
		if (type == MEM_TYPE_AVAILABLE) {
			end = gphys + len -1;
			if (end <= 0xffffffff &&
			    end > top_of_low) {
				top_of_low = end;
			}
		}

	}
	return top_of_low;
}

phys_t
gmm_top_of_high_avail_mem(void)
{
	phys_t top_of_high = 0;
	phys_t gphys, len, end;
	u32 type;
	int index = 0;

	while (gmm_get_mem_map(index++, &gphys, &len, &type, NULL)) {
		if (type == MEM_TYPE_AVAILABLE) {
			end = gphys + len -1;
			if (end >= 0x100000000LL &&
			    end > top_of_high) {
				top_of_high = end;
			}
		}

	}
	return top_of_high;
}
