/*
 * Copyright (c) 2007, 2008 University of Tsukuba
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the University of Tsukuba nor the names of its
 *    contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
/*
 * Copyright (c) 2014 Yuichi Watanabe
 */

#include <core/vm_config.h>
#include <core/cpu.h>
#include "ap.h"
#include "apic.h"
#include "asm.h"
#include "assert.h"
#include "constants.h"
#include "entry.h"
#include "int.h"
#include "linkage.h"
#include "localapic.h"
#include "mm.h"
#include "panic.h"
#include "pcpu.h"
#include "printf.h"
#include "seg.h"
#include "sleep.h"
#include "spinlock.h"
#include "string.h"
#include "thread.h"
#include "uefi.h"

#define APINIT_SIZE		(cpuinit_end - cpuinit_start)
#define APINIT_POINTER(n)	((void *)(apinit + ((u8 *)&n - cpuinit_start)))

static void ap_start (void);

volatile int num_of_processors; /* number of application processors */
bool delayed_ap_init;
static void (*initproc_bsp) (void), (*initproc_ap) (void);
static spinlock_t ap_lock;
static void *newstack_tmp;
static spinlock_t sync_lock;
static u32 sync_id;
static volatile u32 sync_count;
static spinlock_t *apinitlock;
static u32 apinit_addr;
static bool ap_started;

/* this function is called after starting AP and switching a stack */
/* unlock the spinlock because the stack is switched */
static asmlinkage void
apinitproc1 (void)
{
	void (*proc) (void);
	int n;
	void *newstack;

	newstack = newstack_tmp;
	spinlock_unlock (apinitlock);
	spinlock_lock (&ap_lock);
	num_of_processors++;
	n = num_of_processors;
	proc = initproc_ap;
	printf ("Processor %d (AP)\n", num_of_processors);
	spinlock_unlock (&ap_lock);
	segment_init_ap (n);
	currentcpu->stackaddr = newstack;
	int_init_ap ();
	proc ();
}

static asmlinkage void
bspinitproc1 (void)
{
	printf ("Processor 0 (BSP)\n");
	spinlock_init (&sync_lock);
	sync_id = 0;
	sync_count = 0;
	num_of_processors = 0;
	spinlock_init (&ap_lock);

	if (!uefi_booted || shell_apic_id == get_apic_id()) {
		delayed_ap_init = false;
		apinit_addr = 0xF000;
		ap_start ();
	} else {
		delayed_ap_init = true;
		apinit_addr = alloc_realmodemem (APINIT_SIZE + 15);
		ASSERT (apinit_addr >= 0x1000);
		while ((apinit_addr & 0xF) != (APINIT_OFFSET & 0xF))
			apinit_addr++;
		ASSERT (apinit_addr > APINIT_OFFSET);
		localapic_delayed_ap_start (ap_start);
	}
	initproc_bsp ();
	panic ("bspinitproc1");
}

/* this function is called on AP by entry.s */
/* other APs are waiting for spinlock */
/* the stack is cpuinit_tmpstack defined in entry.s */
/* this function allocates a stack and calls apinitproc1 with the new stack */
asmlinkage void
apinitproc0 (void)
{
	newstack_tmp = alloc (VMM_STACKSIZE);
	asm_wrrsp_and_jmp ((ulong)newstack_tmp + VMM_STACKSIZE, apinitproc1);
}

void
ap_start_addr (u8 addr, bool (*loopcond) (void *data), void *data)
{
	if (!apic_available ())
		return;
	apic_start_ap(addr);
}

static bool
ap_start_loopcond (void *data)
{
	int *p;

	p = data;
	return (*p)++ < 3;
}

static void
ap_start (void)
{
	volatile u32 *num;
	u8 *apinit;
	u32 tmp;
	int i;
	u8 buf[5];
	u8 *p;
	u32 apinit_segment;

	apinit_segment = (apinit_addr - APINIT_OFFSET) >> 4;
	/* Put a "ljmpw" instruction to the physical address 0 */
	p = mapmem_hphys (0, 5, MAPMEM_WRITE);
	memcpy (buf, p, 5);
	p[0] = 0xEA;		/* ljmpw */
	p[1] = APINIT_OFFSET & 0xFF;
	p[2] = APINIT_OFFSET >> 8;
	p[3] = apinit_segment & 0xFF;
	p[4] = apinit_segment >> 8;
	apinit = mapmem (MAPMEM_HPHYS | MAPMEM_WRITE, apinit_addr,
			 APINIT_SIZE);
	ASSERT (apinit);
	memcpy (apinit, cpuinit_start, APINIT_SIZE);
	num = (volatile u32 *)APINIT_POINTER (apinit_procs);
	apinitlock = (spinlock_t *)APINIT_POINTER (apinit_lock);
	*num = 0;
	spinlock_init (apinitlock);
	i = 0;
	ap_start_addr (0, ap_start_loopcond, &i);
	for (;;) {
		spinlock_lock (&ap_lock);
		tmp = num_of_processors;
		spinlock_unlock (&ap_lock);
		if (*num == tmp)
			break;
		usleep (1000000);
	}
	unmapmem ((void *)apinit, APINIT_SIZE);
	memcpy (p, buf, 5);
	unmapmem (p, 5);
	ap_started = true;
}

static void
bsp_continue (asmlinkage void (*initproc_arg) (void))
{
	void *newstack;

	newstack = alloc (VMM_STACKSIZE);
	currentcpu->stackaddr = newstack;
	asm_wrrsp_and_jmp ((ulong)newstack + VMM_STACKSIZE, initproc_arg);
}

void
panic_wakeup_all (void)
{
	if (!apic_available ())
		return;
	if (!ap_started)
		return;
	apic_broadcast_nmi();
}

void
sync_all_processors (void)
{
	u32 id = 0;
	bool ret = false;

	spinlock_lock (&sync_lock);
	asm_lock_cmpxchgl (&sync_id, &id, id);
	sync_count++;
	if (sync_count == num_of_processors + 1) {
		sync_count = 0;
		asm_lock_incl (&sync_id);
		ret = true;
	}
	spinlock_unlock (&sync_lock);
	while (!ret) {
		ret = asm_lock_cmpxchgl (&sync_id, &id, id);
		asm_pause ();
	}
}

void
start_all_processors (void (*bsp_initproc) (void), void (*ap_initproc) (void))
{
	initproc_bsp = bsp_initproc;
	initproc_ap = ap_initproc;
	bsp_continue (bspinitproc1);
}
