/*
 * Copyright (c) 2010-2014 Yuichi Watanabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its
 *    contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * Serial controller (UART 16550) emulator.
 * I implement only writing for serial console of guest Linux and
 * guest BIOS.
 */
#include <core.h>
#include <core/io.h>
#include <core/panic.h>
#include <core/printf.h>
#include <core/serial.h>
#include <core/serial_regs.h>
#include <core/spinlock.h>
#include <core/types.h>
#include <core/vm.h>
#include "ioapic_emu.h"

#define SERIAL_GSI 4
#define SERIAL_PORT SERIAL_PORT1

/* #define SERIAL_DEBUG */
#ifdef SERIAL_DEBUG
#define SERIAL_DBG(...)						\
	do {							\
		printf("SERIAL: " __VA_ARGS__);			\
	} while (0)
#else
#define SERIAL_DBG(...)
#endif

struct serial_emu {
	spinlock_t lock;
	u8 ratelsb;
	u8 ratemsb;
	u8 intrctl;
	u8 fifoctl;
	u8 linectl;
	u8 modectl;
	u8 modestat;
	u8 user;
	bool tbr_empty;
};

static drvdata_hdl_t serial_handle;

static void
serial_inb(ioport_t port_off, u8 *data)
{
	struct serial_emu *serial;

	serial = vm_get_driver_data(serial_handle);
	spinlock_lock(&serial->lock);

	if (serial->linectl & LINECTL_DLAB_BIT) {
		switch (port_off) {
		case PORT_RATELSB: /* 0 */
			*data = serial->ratelsb;
			spinlock_unlock(&serial->lock);
			return;
		case PORT_RATEMSB: /* 1 */
			*data = serial->ratemsb;
			spinlock_unlock(&serial->lock);
			return;
		}
	}
	switch (port_off) {
	case PORT_DATA: /* 0 */
		*data = 0;
		break;
	case PORT_INTRCTL: /* 1 */
		*data = serial->intrctl;
		break;
	case PORT_INTRSTAT: /* 2 */
		*data = 0;
		if (serial->intrctl & INTRCTL_ETBREI_BIT &&
		    serial->tbr_empty) {
			*data |= INTRSTAT_TBR_EMPTY_INTR;
			serial->tbr_empty = false; /* Reading ISR clears
						      TBR empty interrupt. */
		} else {
			*data |= INTRSTAT_INT_NO_PENDING; /* 0 if interrupt
							    pending */
		}
		if (serial->fifoctl & FIFOCTL_ENABLE) {
			*data |= INTRSTAT_FIFO_ENABLED;
		}
		break;
	case PORT_LINECTL: /* 3 */
		*data = serial->linectl;
		break;
	case PORT_MODECTL: /* 4 */
		*data = serial->modectl;
		break;
	case PORT_LINESTAT: /* 5 */
		*data = LINESTAT_TXSTART_BIT | LINESTAT_TX_BIT;
		break;
	case PORT_MODESTAT: /* 6 */
		*data = serial->modestat | MODESTAT_CTS_BIT;
		break;
	case PORT_USER: /* 7 */
		*data = serial->user;
		break;
	default:
		panic("serial_inb: Invalid port_off. port_off 0x%x", port_off);
	}
	spinlock_unlock(&serial->lock);
}

static void
serial_outb(ioport_t port_off, u8 data)
{
	struct serial_emu *serial;
	u8 oldval;

	serial = vm_get_driver_data(serial_handle);
	spinlock_lock(&serial->lock);

	if (serial->linectl & LINECTL_DLAB_BIT) {
		switch (port_off) {
		case PORT_RATELSB: /* 0 */
			serial->ratelsb = data;
			spinlock_unlock(&serial->lock);
			return;
		case PORT_RATEMSB: /* 1 */
			serial->ratemsb = data;
			spinlock_unlock(&serial->lock);
			return;
		}
	}
	switch (port_off) {
	case PORT_DATA: /* 0 */
		serial_out(data);
		if (serial->intrctl & INTRCTL_ETBREI_BIT) {
			serial->tbr_empty = true;
			ioapic_trigger_int(SERIAL_GSI);
		}

		break;
	case PORT_INTRCTL: /* 1 */
		oldval = serial->intrctl;
		serial->intrctl = data;
		SERIAL_DBG("Write INTRCTL 0x%x\n", data);
		if ((oldval & INTRCTL_ETBREI_BIT) == 0 &&
		    serial->intrctl & INTRCTL_ETBREI_BIT) {
			serial->tbr_empty = true;
			ioapic_trigger_int(SERIAL_GSI);
		}
		break;
	case PORT_FIFOCTL: /* 2 */
		serial->fifoctl = data;
		SERIAL_DBG("Write FIFOCTL 0x%x\n", data);
		break;
	case PORT_LINECTL: /* 3 */
		serial->linectl = data;
		break;
	case PORT_MODECTL: /* 4 */
		serial->modectl = data;
		break;
	case PORT_LINESTAT: /* 5 */
		break;
	case PORT_MODESTAT: /* 6 */
		serial->modestat = data;
		break;
	case PORT_USER: /* 7 */
		serial->user = data;
		break;
	default:
		panic("serial_outb: Invalid port_off. port_off 0x%x", port_off);
	}
	spinlock_unlock(&serial->lock);
}

static ioact_t
serial_io_hook(iotype_t type, u32 port, void *data)
{
	ioport_t port_off;

	port_off = port - SERIAL_PORT;
	switch (type) {
	case IOTYPE_INB:
		serial_inb(port_off, data);
		break;
	case IOTYPE_OUTB:
		serial_outb(port_off, *(u8 *)data);
		break;
	default:
		do_io_nothing(type, port, data);
		break;
	}
	return IOACT_CONT; /* emulated */
}

static void
serial_vminit(void)
{
	struct serial_emu *serial;
	u32 i;

	serial = vm_get_driver_data(serial_handle);
	spinlock_init(&serial->lock);

	for (i = SERIAL_PORT; i < SERIAL_PORT + SERIAL_NUM_OF_IOPORT; i++)
		set_iofunc (i, serial_io_hook);
}

static void
serial_preinit(void)
{
	serial_handle = vm_alloc_driver_data(sizeof(struct serial_emu));
}

static void
serial_reset(void)
{
	struct serial_emu *serial = vm_get_driver_data(serial_handle);

	spinlock_lock(&serial->lock);
	serial->ratelsb = 0;
	serial->ratemsb = 0;
	serial->intrctl = 0;
	serial->fifoctl = 0;
	serial->linectl = 0;
	serial->modectl = 0;
	serial->modestat = 0;
	serial->user = 0;
	serial->tbr_empty = false;
	spinlock_unlock(&serial->lock);
}

DRIVER_VMINIT(serial_vminit);
DRIVER_PREINIT(serial_preinit);
DRIVER_RESET(serial_reset);
