/* glinfo.c */

#include	"tazel.h"

/************************************************************************************************/
// *glinfo[]* is a Ground Location Table. ( Name & Location ) 
// You can edit data in *glinfo[]* for your location. ( case insensitive )
/************************************************************************************************/
// (0) Line description:
//    First line is used as default data. ( Required )
//    Last line ( all NULL ) is used as End-Of-Line marker. ( Required )
// (1) Column description:
//  * First column is Name, Second column is its Alias or NULL.
//  * Third column is Longitude in Degree.
//    You need to attach 'E'(East)  or 'W'(West)  at the End-Of-String. ( Required )
//  * Fourth column is Latitude in Degree.
//    You need to attach 'N'(North) or 'S'(South) at the End-Of-String. ( Required )
//  * Last column is Altitude in Meter.  If you are not sure, you can fill it with 0.000.
/************************************************************************************************/

struct glinfo{							/* Struct for Ground Location							*/
	char	*name;						/* >(String) Name										*/
	char	*alias;						/* >(String) Alias Name or NULL							*/
	char	*lon;						/* >(String) Longitude [DegE]or[DegW]					*/
	char	*lat;						/* >(String) Latitude  [DegN]or[DegS]					*/
	char	*alt;						/* >(String) Altitude  [M   ]							*/
}glinfo[GL_NUMBER]={					/* Ground Location Table								*/

/***  Name        ,Alias      ,Lon[deg]    ,Lat[deg]   ,Alt[m]  ***/
	{"Akihabara"  , "Akiba"   , "+139.774E", "+35.699N", "+4.000"},		/* ( Default  )	*/
	{"Nipponbashi", "Ponbashi", "+135.507E", "+34.667N", "+4.000"},
	{"AllZero"    , "Test"    , "+000.000E", "+00.000N", "+0.000"},
	{ NULL        ,  NULL     ,  NULL      ,  NULL     ,  NULL   },		/* ( Required )	*/

};

/************************************************************************************************/
// This function looks up GLINFO[] Table by "p_name", then set result into p_save.
// If "p_name"==NULL then it select default ( i.e. glinfo[0] ) data.
/************************************************************************************************/
extern void set_obspos(char *p_name /* => */ ,obs_t *p_save){

dbl		lon,lat,alt;					/* Lon/Lat/Alt											*/
char	*ptr;							/* Pointer to "EeWwNnSs" char							*/
int		i;

/* Loop Up */
	for( i=0 ; i<GL_NUMBER && glinfo[i].name!=NULL ; i++ ){
		if( p_name==NULL || strcasecmp(glinfo[i].name,p_name)==0 || strcasecmp(glinfo[i].alias,p_name)==0 ){

		/* >Lon/Lat/Alt */
			lon=atof(glinfo[i].lon);				/* [DegE]			*/
			lat=atof(glinfo[i].lat);				/* [DegN]			*/
			alt=atof(glinfo[i].alt);				/* [M   ]			*/

		/* >DegE/DegW Adjustment */
			ptr=strpbrk(glinfo[i].lon,"EeWw");
			if( ptr==NULL || lon<-360.0 || +360.0<lon ){
				fprintf(stderr,"Error: Illegal Longitude [Name=\"%s\"]\n",p_name);
				exit(1);
			}
			if( *ptr=='E' || *ptr=='e' )			/* [+-]###.###E		*/
				;
			else									/* [+-]###.###W		*/
				lon=-lon;

		/* >DegN/DegS Adjustment */
			ptr=strpbrk(glinfo[i].lat,"NnSs");
			if( ptr==NULL || lat<-90.0 || +90.0<lat ){
				fprintf(stderr,"Error: Illegal Latitude [Name=\"%s\"]\n",p_name);
				exit(1);
			}
			if( *ptr=='N' || *ptr=='n' )			/* [+-]##.###N		*/
				;
			else									/* [+-]##.###S		*/
				lat=-lat;

		/* >( Debug ) */
#ifdef DEBUG
	printf("### ges_setpos3d() ###\n");
	printf("p_name=\"%s\"\n",p_name);
	printf("lon=%f [DegE]\n",lon);
	printf("lat=%f [DegN]\n",lat);
	printf("alt=%f [m   ]\n",alt);
#endif

		/* >Convert Unit & Save Result */
			p_save->lon=rad(lon);					/* [DegE] -> [RadE]	*/
			p_save->lat=rad(lat);					/* [DegN] -> [RadN]	*/
			p_save->alt=alt/1000.0;					/* [M   ] -> [KM  ]	*/

		/* >Return */
			return;

		}
	}

/* Error */
	fprintf(stderr,"Error: No such Ground Location [Name=\"%s\"]\n",p_name);
	exit(1);

}

/************************************************************************************************/
// This function lists all the GLINFO[] Table Information.
/************************************************************************************************/
extern void list_obsdata(){

dbl		lon,lat,alt;					/* OBS Lon/Lat/Alt										*/
char	*ptr;							/* Pointer to "E[ast]/W[est]/N[orth]/S[outh]" char		*/
int		i;

/* Loop Up - for all OBS items */
	for( i=0 ; i<GL_NUMBER && glinfo[i].name!=NULL ; i++ ){

	/* >Set Lon/Lat/Alt */
		lon=atof(glinfo[i].lon);					/* [ DegE | DegW ]	*/
		lat=atof(glinfo[i].lat);					/* [ DegN | DegS ]	*/
		alt=atof(glinfo[i].alt);					/* [ Meter       ]	*/

	/* >DegE/DegW Adjustment */
		ptr=strpbrk(glinfo[i].lon,"EeWw");
		if( ptr==NULL || lon<-360.0 || +360.0<lon ){
			fprintf(stderr,"Error: Illegal Longitude [Name=\"%s\"]\n",glinfo[i].name);
			exit(1);
		}
		if( *ptr=='E' || *ptr=='e' )				/* [+-]###.###E		*/
			;
		else										/* [+-]###.###W		*/
			lon=-lon;

	/* >DegN/DegS Adjustment */
		ptr=strpbrk(glinfo[i].lat,"NnSs");
		if( ptr==NULL || lat<-90.0 || +90.0<lat ){
			fprintf(stderr,"Error: Illegal Latitude [Name=\"%s\"]\n",glinfo[i].name);
			exit(1);
		}
		if( *ptr=='N' || *ptr=='n' )				/* [+-]##.###N		*/
			;
		else										/* [+-]##.###S		*/
			lat=-lat;

	/* >List Information */
		printf("Name(Alias) = %-14s(%-14s)\n", glinfo[i].name, glinfo[i].alias);
		printf("> Lon/Lat/Alt = %+8.3f[DegE](%+8.3f[DegW])/%+8.3f[DegN](%+8.3f[DegS])/%+8.3f[m]\n", lon, -lon, lat, -lat, alt);

	}

}
