/* tazel.h - TAZEL System Header File */

/************************************************************************************************/
// Include Files
/************************************************************************************************/
#define		_XOPEN_SOURCE
#define		_C99_SOURCE
#define		_GNU_SOURCE

#include	<stdio.h>
#include	<stdlib.h>
#include	<string.h>
#include	<strings.h>
#include	<ctype.h>
#include	<limits.h>
#include	<math.h>
#include	<time.h>
#include	<stdint.h>
#include	<inttypes.h>

/************************************************************************************************/
// Define Values
/************************************************************************************************/
#define		VER_TAZEL		"31"		/* TAZEL Version Number  ( String )						*/
#define		LEN_SATCNAME	10			/* Length for Satellite-ConciseName						*/
#define		GL_NUMBER		4096		/* Number of Ground Locations ( MAX )					*/

#define		TRUE	1
#define		FALSE	0

/************************************************************************************************/
// Define Types
/************************************************************************************************/
typedef double dbl;

typedef struct{					/* Struct for TLE-Data ( TLE = Two Line Element )				*/
	char	name[BUFSIZ];				/* > TLE Satellite Name (Line0)							*/
	char	line1[BUFSIZ];				/* > TLE Line1 Raw Data									*/
	char	line2[BUFSIZ];				/* > TLE Line2 Raw Data									*/
	int		catno;						/* > TLE NORAD Catalog Number (!)						*/
	char	idesg[9];					/* > TLE International Designator						*/
	dbl		epoch,drag1,drag2,bstar;
	dbl		incl,raan,eccn,argp,mano,mmot;
	int		setno,revno;
	struct tle_t *prev,*next;			/* > Link Pointer for B-Tree (Prev/Next)				*/
}tle_t;									/* (!) CATNO is Negative if TLE-data is Invalid			*/

typedef struct{					/* Struct for SAT-Data ( SAT = Satellite )						*/
	dbl		lon,lat;					/* > SAT Longitude [DegE] / Latitude   [DegN]			*/
	dbl		alt,vel;					/* > SAT Altitude  [Km]   / Velocity   [m/s]			*/
	dbl		az,el;						/* > SAT Azimth    [Deg]  / Elevation  [Deg]			*/
	dbl		range,rrate;				/* > SAT Range     [Km]   / Range Rate [m/s]			*/
	dbl		x,y,z;						/* > SAT ECI-Position (Px,Py,Pz) [Km]					*/
	dbl		xdot,ydot,zdot;				/* > SAT ECI-Velocity (Vx,Vy,Vz) [m/s]					*/
}sat_t;

typedef struct{					/* Struct for OBS-Data ( OBS = Observer )						*/
	dbl		lon,lat;					/* > OBS Longitude [DegE] / Latitude [DegN]				*/
	dbl		alt;						/* > OBS Altitude  [km]									*/
	dbl		theta;						/* > OBS Theta Angle { ATAN2(ECI-Y,ECI-X) } [Rad]		*/
}obs_t;

typedef struct{					/* Struct for 3D Vector											*/
	dbl		x,y,z,r;					/* > (X,Y,Z) [Km] + R { SQRT(X**2+Y**2+Z**2) } [Km]		*/
}vec_t;

/************************************************************************************************/
// Declare Functions
/************************************************************************************************/
/* Display Related Functions */
extern void	disp_usage();
extern void	disp_version();
extern void	disp_format(FILE *fp,int flag_tledump);
extern void	disp_result(time_t t_abs,int t_off,dbl t_msec,tle_t tle_main,sat_t sat_main,char *buf_datetime,char *buf_satcname,char *buf_tleepoch,int flag_tledump);

/* Argument Related Functions */
extern void set_optparam(char *argv[] /* => */ ,int *tledump,int *exemode,int *gapline,char **p_gname,char **p_fname,char **p_date,char **p_time,char **p_looptick);
extern void set_looptick(char *p_looptick /* => */ ,int *loop,int *delt,int *tick);
extern void set_datetime(char *p_date,char *p_time /* => */ ,struct tm *tm_init,dbl *msec);

/* Observer Related Functions */
extern void	list_obsdata();
extern void	set_obspos(char *p_name /* => */ ,obs_t *p_save);

/* Epoch Related Functions */
extern void	tleepoch2str(dbl epoch,char *buf);
extern struct tm tleepoch2tm(tle_t tle);
extern dbl	unix2tleepoch(time_t u_time,dbl u_msec);
extern dbl	diff_epoch2utc(dbl epoch,time_t t_abs,time_t t_off);

/* Mathmatics Related Functions */
extern dbl	fmod2p(dbl x);
extern dbl	rad(dbl x);
extern dbl	deg(dbl x);
extern void	kvec(dbl k,vec_t *p_vec);
extern void	vmag(vec_t *p_vec);

/* String Related Functions */
extern char	*substr(char *ptr,int ps,int pe);
extern tle_t *tledup(tle_t tle);

extern dbl	jd_year(int yyyy);
extern dbl	jd_epoch(dbl epoch);

extern int	rd_tledata(FILE *fp,tle_t *p_tle,int flag_preserve);
extern int	tle_cksum(char *line1,char *line2);
extern void	do_predict(time_t tt_utc,dbl t_msec,tle_t *p_tle,sat_t *p_sat,obs_t *p_ges);
