/*
    misc
    copyright (c) 1998-2004 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "misc.h"
#include <gdk/gdkkeysyms.h>


/******************************************************************************
*                                                                             *
* ja:メッセージボックス関数群                                                 *
*                                                                             *
******************************************************************************/
/* ja:ボタンが押された */
static void
message_box_clicked (GtkWidget *widget,
                     GtkWidget *dialog)
{
  *(gint *)g_object_get_data (G_OBJECT (dialog), "user_data")
                    = (gint)g_object_get_data (G_OBJECT (widget), "user_data");
  gtk_widget_destroy (dialog);
}


/* ja:ESCが押された */
static gboolean
message_box_dialog_key_press (GtkWidget   *widget,
                              GdkEventKey *event,
                              gpointer     user_data)
{
  if (event->keyval == GDK_Escape)
    gtk_widget_destroy (widget);
  return FALSE;
}


/*  ja:メッセージボックスを表示する
    title,タイトル
     text,本文
      def,デフォルトのボタン
      ...,ボタンのラベル
      RET,押されたボタン(-1:キャンセル)                                     */
gint
misc_message_box (const gchar *title,
                  const gchar *text,
                  const gint def, ...)
{
  va_list ap;
  gchar *name;
  gint i = 0, result = -1;
  GtkWidget *dialog, *button, *label, *hbox0, *hbox1, *vbox, *focus = NULL;

  va_start (ap, def);
  /* ja:メインウインドウ */
  dialog = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title (GTK_WINDOW (dialog), title);
  g_signal_connect_after (G_OBJECT (dialog), "key-press-event",
                            G_CALLBACK (message_box_dialog_key_press), NULL);
  g_signal_connect (G_OBJECT (dialog), "destroy", gtk_main_quit, NULL);
  g_object_set_data (G_OBJECT (dialog), "user_data", &result);
  /* ja:ラベル */
  label = gtk_label_new(text);
  /* ja:ボックス */
  hbox0 = gtk_hbox_new (FALSE, SPACING);
  /* ja:ボタン */
  while ((name = va_arg (ap, gchar *)))
    {
      button = gtk_button_new_with_mnemonic (name);
      g_signal_connect (G_OBJECT (button), "clicked",
                                    G_CALLBACK (message_box_clicked), dialog);
      g_object_set_data (G_OBJECT (button), "user_data", (gpointer)i);
      GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
      if (i == def)
        focus = button;
      gtk_box_pack_start (GTK_BOX (hbox0), button, FALSE, FALSE, 0);
      i++;
    }
  va_end (ap);

  /* ja:ボックス */
  vbox = gtk_vbox_new (FALSE, SPACING);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), SPACING);
  gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);
  hbox1 = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_end (GTK_BOX (hbox1), hbox0, TRUE, FALSE, 0);
  gtk_box_pack_end (GTK_BOX (vbox), hbox1, FALSE, FALSE, 0);
  gtk_container_add (GTK_CONTAINER (dialog), vbox);

  /* ja:表示 */
  if (focus)
    gtk_widget_grab_focus (focus);

  gtk_grab_add (dialog);
  gtk_widget_show_all (dialog);
  gtk_main ();

  return result;
}


/******************************************************************************
*                                                                             *
* ja:低レベル関数群                                                           *
*                                                                             *
******************************************************************************/
/*  ja:スクロールバーを設定する
       scroll,スクロールウイジット
         func,シグナル関数
    func_data,データ
          min,最小値
          max,最大値
         page,ページ
          pos,位置                                                          */
void
misc_set_scroll_bar (GtkWidget     *scroll,
                     GtkSignalFunc  func,
                     gpointer       func_data,
                     const gint     min,
                     const gint     max,
                     const gint     page,
                     const gint     pos)
{
  GtkAdjustment *adjust;

  adjust = GTK_ADJUSTMENT (gtk_adjustment_new (pos, min, max, 1, page, page));
  gtk_range_set_adjustment (GTK_RANGE (scroll), adjust);
  if (func)
    g_signal_connect (G_OBJECT (adjust), "value-changed", func, func_data);
}


/*  ja:Windowの内容をスクロールする
    widget,ウイジット
        dx,X軸方向の移動
        dy,Y軸方向の移動
         x,スクロールする範囲の右上X座標
         y,スクロールする範囲の右上Y座標
     width,スクロールする範囲の幅
    height,スクロールする範囲の高さ                                         */
void
misc_scroll_window (GtkWidget  *widget,
                    const gint  dx,
                    const gint  dy,
                    const gint  x,
                    const gint  y,
                    const gint  width,
                    const gint  height)
{
  GdkGC *gc;
  GdkRectangle rc;

  if (ABS (dx) > width || ABS (dy) > height)
    {
      rc.x = x;
      rc.y = y;
      rc.width = width;
      rc.height = height;
      gtk_widget_draw (widget, &rc);
    }
  else if (dx != 0 || dy != 0)
    {
      gc = gdk_gc_new (widget->window);
      gdk_gc_set_exposures (gc, TRUE);
      gdk_window_copy_area (widget->window,
                            gc,
                            dx > 0 ? x + dx : x,
                            dy > 0 ? y + dy : y,
                            widget->window,
                            dx > 0 ? x : x - dx,
                            dy > 0 ? y : y - dy,
                            width - ABS (dx),
                            height - ABS (dy));
      gdk_gc_destroy (gc);
      if (dx != 0)
        {
          rc.x = dx > 0 ? x : x + width + dx;
          rc.y = y;
          rc.width = ABS (dx);
          rc.height = height;
          gtk_widget_draw (widget, &rc);
        }
      if (dy != 0)
        {
          rc.x = x;
          rc.y = dy > 0 ? y : y + height + dy;
          rc.width = width;
          rc.height = ABS (dy);
          gtk_widget_draw (widget, &rc);
        }
    }
}


/******************************************************************************
*                                                                             *
* ja:数値文字列関数群                                                         *
*                                                                             *
******************************************************************************/
const static gchar hex[16]={'0', '1', '2', '3', '4', '5', '6', '7',
                            '8', '9', 'A', 'B', 'C', 'D', 'E', 'F'};


/*  ja:数値→文字列
    value,数値
    radix,基数
     wide,桁数
     flag,TRUE:符号あり,FALSE:符号なし
      RET,文字列                                                            */
gchar *
misc_str_from_val (const gint     value,
                   const gint     radix,
                   const gint     wide,
                   const gboolean flag)
{
  gchar c, *str = NULL;
  gint i = 0, j, t;

  t = value;
  if (value == 0)
    {
      str = g_malloc (sizeof (gchar));
      str[i++] = hex[0];
    }
  else
    {
      if (flag && value < 0)
        {
          while (t != 0)
            {
              str = g_realloc (str, (i + 1) * sizeof (gchar));
              str[i++] = hex[ABS (t % radix)];
              t /= radix;
            }
        }
      else
        {
          while (t != 0)
            {
              str = g_realloc (str, (i + 1) * sizeof (gchar));
              str[i++] = hex[(guint)t % radix];
              t = (guint)t / radix;
            }
        }
    }
  str = g_realloc (str, (i + 1) * sizeof (gchar));
  str[i] = '\0';
  for (j = 0; j < i / 2; j++)
    {
      c = str[j];
      str[j] = str[i - j - 1];
      str[i - j - 1] = c;
    }
  if (flag && value < 0)
    {
      for (j = i; j >= 0; j--)
        str[j + 1] = str[j];
      str[0] = '-';
      i++;
    }
  if (i < ABS (wide))
    {
      str = g_realloc (str, (ABS (wide) + 1) * sizeof (gchar));
      for (j = i; j >= 0; j--)
        str[j + ABS (wide) - i] = str[j];
      if (wide > 0)
        {
          for (j = 0; j < wide - i; j++)
            str[j] = ' ';
        }
      else
        {
          for (j = 0; j < -wide - i; j++)
            str[j] = hex[0];
          if (str[j] == '-')
            {
              str[0] = '-';
              str[j] = hex[0];
            }
        }
    }
  return str;
}


/*  ja:文字列→数値
    value,数値
      str,文字列
    radix,基数
     flag,TRUE:符号あり,FALSE:符号なし
      RET,TRUE:正常終了,FALSE:エラー                                        */
gboolean
misc_str_to_val (gint           *value,
                 const gchar    *str,
                 const gint      radix,
                 const gboolean  flag)
{
  gchar c;
  gint i, j, t;

  *value = 0;
  for (i = 0; str[i] != '\0' && str[i] == ' '; i++);
  if (str[i] == '\0')
    return FALSE;
  if (flag && str[i] == '-')
    {
      i++;
      while (str[i] != '\0')
        {
          t = *value;
          *value *= radix;
          c = g_ascii_toupper (str[i]);
          for (j = 0; j < radix; j++)
            if (hex[j] == c)
              break;
          *value += j;
          if (j == radix || *value < t)
            {
              *value = t;
              return FALSE;
            }
          i++;
        }
      if (*value < 0)
        {
          *value=0;
          return FALSE;
        }
      *value =- *value;
    }
  else
    {
      while (str[i] != '\0')
        {
          t = *value;
          *value *= radix;
          c = g_ascii_toupper (str[i]);
          for (j = 0; j < radix; j++)
            if (hex[j] == c)
              break;
          *value += j;
          if (j == radix || *value < t)
            {
              *value = t;
              return FALSE;
            }
          i++;
        }
    }
  return TRUE;
}


/*  ja:数値→文字列
     value,数値
    divide,除算値
       RET,文字列                                                           */
gchar *
misc_str_from_float (const gint value,
                     const gint divide)
{
    gchar *text = NULL;
    gint i = 0, n, t;

#ifdef G_HAVE_GINT64
  n = t = (gint64)value * 100000 / divide;
#else /* not G_HAVE_GINT64 */
  n = t = value * 100000 / divide;
#endif /* not G_HAVE_GINT64 */
  if (t == 0)
    {
      text = g_malloc (sizeof (gchar) * 4);
      text[i++] = hex[0];
    }
  else
    {
      t = ABS (t);
      while (t > 0)
        {
          text = g_realloc (text, (i + 4) * sizeof (gchar));
          text[i++] = hex[t % 10];
          t /= 10;
        }
    }
  text[i] = '\0';
  g_strreverse (text);
  if (g_strlen (text) < 6)
    {
      g_memmove (text + 2, text, (i + 1) * sizeof (gchar));
      text[0] = '0';
      text[1] = '.';
      i++;
    }
  else
    {
      g_memmove (text + i - 4, text + i - 5, sizeof (gchar) * 6);
      text[i - 5] = '.';
    }
  while (text[i] == '0')
    text[i--] = '\0';
  if (text[i] == '.')
    text[i--] = '\0';
  if (n >= 0)
    {
      text = g_realloc (text, (i + 2) * sizeof (gchar));
    }
  else
    {
      text = g_realloc (text, (i + 3) * sizeof (gchar));
      g_memmove (text + 1, text, (i + 2) * sizeof (gchar));
      text[0] = '-';
    }
  return text;
}


/*  ja:文字列→数値
     value,数値
    divide,除算値
      text,文字列
       RET,TRUE:正常終了,FALSE:エラー                                       */
gboolean
misc_str_to_float (gint        *value,
                   gint        *divide,
                   const gchar *text)
{
  gchar *p;
  gint i, j, t;

  *value = 0;
  *divide = 1;
  p = g_strdup (text);
  for (i = j = 0; p[i] != '\0'; i++)
    if (p[i] != ' ')
      p[j++] = p[i];
  if (j <= 0)
    return FALSE;
  p[j] = '\0';
  for (i = 0; p[i] != '\0'; i++)
    if (p[i] == '.')
      break;
  if (p[i] != '\0')
    {
      for (j = g_strlen (p) - 1; j > 0; j--)
        if (p[j] == '0')
          p[j] = '\0';
        else
          break;
      while (p[++i] != '\0')
        {
          p[i-1] = p[i];
          *divide *= 10;
        }
      p[i - 1] = '\0';
    }
  if (p[0] == '-' && p[1] == '\0')
    {
      g_free (p);
      return FALSE;
    }
  i = p[0] == '-' ? 1 : 0;
  while (p[i] != '\0')
    {
      t = *value;
      *value *= 10;
      for (j = 0; j < 10; j++)
        if (hex[j] == p[i])
          break;
      *value += j;
      if (j == 10 || *value < t)
        {
          *value = t;
          g_free (p);
          return FALSE;
        }
      i++;
  }
  while (*value % 2 == 0 && *divide % 2 == 0)
    *value /= 2, *divide /= 2;
  while (*value % 5 == 0 && *divide % 5 == 0)
    *value /= 5, *divide /= 5;
  *value *= (p[0] == '-' ? -1 : 1);
  g_free (p);
  return TRUE;
}


/*  ja:数列→文字列
    array,数列
     size,要素数
     bits,ビット数
    radix,基数
     wide,桁数
     flag,TRUE:符号あり,FALSE:符号なし
      RET,文字列                                                            */
gchar *
misc_str_from_array (const gpointer *array,
                     const gsize     size,
                     const gint      bits,
                     const gint      radix,
                     const gint      wide,
                     const gboolean  flag)
{
  gchar *p, *text = NULL;
  gint i, leng, length = 0;

  if (!array || size <= 0
                    || (bits != 0 && bits != 8 && bits != 16 && bits != 32))
    return NULL;
  for (i = 0; i < size; i++)
    {
      switch (bits)
        {
          case 8:
            p = flag
                ? misc_str_from_val (((gint8 *)array)[i],  radix, wide, flag)
                : misc_str_from_val (((guint8 *)array)[i], radix, wide, flag);
            break;
          case 16:
            p = flag
                ? misc_str_from_val (((gint16 *)array)[i],  radix, wide, flag)
                : misc_str_from_val (((guint16 *)array)[i], radix, wide, flag);
            break;
          case 32:
            p = flag
                ? misc_str_from_val (((gint32 *)array)[i],  radix, wide, flag)
                : misc_str_from_val (((guint32 *)array)[i], radix, wide, flag);
          default:
            p = flag
                ? misc_str_from_val (((gint *)array)[i],  radix, wide, flag)
                : misc_str_from_val (((guint *)array)[i], radix, wide, flag);
            break;
        }
      leng = g_strlen (p);
      text = g_realloc (text, (length + leng + 1) * sizeof (gchar));
      g_memmove (text + length, p, g_strlen (p));
      g_free (p);
      length += leng;
      text[length++] = ' ';
    }
  text[length - 1] = '\0';
  return text;
}


/*  ja:文字列→数列
     size,要素数
     bits,ビット数
     text,文字列
    radix,基数
     flag,TRUE:符号あり,FALSE:符号なし
      RET,数列                                                              */
gpointer
misc_str_to_array (gsize          *size,
                   const gint      bits,
                   const gchar    *text,
                   const gint      radix,
                   const gboolean  flag)
{
  gchar *p;
  gpointer array = NULL;
  gint i, j, value;

  if (!size)
    return NULL;
  *size = 0;
  if (!text || (bits != 0 && bits != 8 && bits != 16 && bits != 32))
    return NULL;
  p = g_malloc ((g_strlen (text) + 2) * sizeof (gchar));
  g_strcpy (p, text);
  for (i = 0; p[i] != '\0'; i++)
    {
      for (j = 0; j < radix; j++)
        if (p[i] == hex[j])
          break;
      if (j >= radix)
        p[i] = '\0';
    }
  p[i + 1] = '\0';
  i = 0;
  while (p[i] != '\0')
    {
      if (!misc_str_to_val (&value, p + i, radix, flag))
        {
          g_free (p);
          g_free (array);
          *size = 0;
          return NULL;
        }
      switch (bits)
        {
          case 8:
            array = g_realloc (array, (*size + 1) * sizeof (guint8));
            ((guint8 *)array)[(*size)++] = value;
            break;
          case 16:
            array = g_realloc (array, (*size + 1) * sizeof (guint16));
            ((guint16 *)array)[(*size)++] = value;
            break;
          case 32:
            array = g_realloc (array, (*size + 1) * sizeof (guint32));
            ((guint32 *)array)[(*size)++] = value;
          default:
            array = g_realloc (array, (*size + 1) * sizeof (guint));
            ((guint *)array)[(*size)++] = value;
        }
      i += g_strlen (p + i) + 1;
    }
  g_free (p);
  return array;
}


/******************************************************************************
*                                                                             *
* ISO-10646                                                                   *
*                                                                             *
******************************************************************************/
/*  ja:マルチバイト文字列→ワイド文字列
       src,マルチバイト文字列
    srclen,文字数(-1:NULL終了)
       dst,ワイド文字列
    dstlen,文字数
       RET,文字数                                                           */
gssize
misc_multibyte_to_widechar (const gchar  *src,
                            const gssize  srclen,
                            guint16      *dst,
                            const gssize  dstlen)
{
  gssize i, j, leng;

  if (!src)
    return 0;
  leng = srclen < 0 ? g_strlen (src) + 1 : srclen;
  i = j = 0;
  if (!dst || dstlen <= 0)  /* ja:必要な文字数を数える */
    while (i < leng)
      {
        if (i + 1 < leng && 0xc0 <= (guint8)src[i] && (guint8)src[i] <= 0xdf)
          i += 2;
        else if (i + 2 < leng && 0xe0 <= (guint8)src[i])
          i += 3;
        else
          i++;
        j++;
      }
  else                      /* ja:変換する */
    while (i < leng && j < dstlen)
      {
        if (i + 1 < leng && 0xc0 <= (guint8)src[i] && (guint8)src[i] <= 0xdf)
          {
            dst[j] = (src[i] & 0x1f) << 6 | (src[i + 1] & 0x3f);
            i += 2;
          }
        else if (i + 2 < leng && 0xe0 <= (guint8)src[i])
          {
            dst[j] = (src[i] & 0xf) << 12 | (src[i + 1] & 0x3f) << 6
                                                        | (src[i + 2] & 0x3f);
            i += 3;
          }
        else
          {
            dst[j] = src[i];
            i++;
          }
        j++;
      }
  return j;
}


/*  ja:ワイド文字列→マルチバイト文字列
       src,ワイド文字列
    srclen,文字数(-1:NULL終了)
       dst,マルチバイト文字列
    dstlen,文字数
       RET,文字数                                                           */
gssize
misc_widechar_to_multibyte (const guint16 *src,
                            const gssize   srclen,
                            gchar         *dst,
                            const gssize   dstlen)
{
  gssize i, j, leng;

  if (!src)
    return 0;
  if (srclen < 0)
    {
      for (leng = 0; src[leng] > 0; leng++);
        leng++;
    }
  else
    {
      leng = srclen;
    }
  j = 0;
  if (!dst || dstlen <= 0)  /* ja:必要な文字数を数える */
    for (i = 0; i < leng; i++)
      if (src[i] <= 0x007f)
        j++;
      else if (src[i] <= 0x07ff)
        j += 2;
      else
        j += 3;
  else                      /* ja:変換する */
    for (i = 0; i < leng && j < dstlen; i++)
      if (j + 1 == dstlen || src[i] <= 0x007f)
        {
          dst[j++] = src[i];
        }
      else if (j + 2 == dstlen || src[i] <= 0x07ff)
        {
          dst[j++] = 0xc0 | src[i] >> 6;
          dst[j++] = 0x80 | (src[i] & 0x3f);
        }
      else
        {
          dst[j++] = 0xe0 | src[i] >> 12;
          dst[j++] = 0x80 | (src[i] >> 6 & 0x3f);
          dst[j++] = 0x80 | (src[i] & 0x3f);
        }
  return j;
}


/*  ja:マルチバイト文字列→ワイド文字列
    src,マルチバイト文字列
    RET,ワイド文字数,NULL:エラー                                            */
guint16 *
misc_multistring_to_widestring (const gchar *src)
{
  gssize leng;
  guint16 *dst;

  leng = misc_multibyte_to_widechar (src, -1, NULL, -1);
  dst = g_malloc (leng * sizeof (guint16));
  misc_multibyte_to_widechar (src, -1, dst, leng);
  return dst;
}


/*  ja:ワイド文字列→マルチバイト文字列
    src,ワイド文字列
    RET,マルチバイト文字列,NULL:エラー                                      */
gchar *
misc_widestring_to_multistring (const guint16 *src)
{
  gssize leng;
  gchar *dst;

  leng = misc_widechar_to_multibyte (src, -1, NULL, -1);
  dst = g_malloc (leng * sizeof (gchar));
  misc_widechar_to_multibyte (src, -1, dst, leng);
  return dst;
}
