/*
    winmenu
    copyright (c) 1998-2005 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "misc_winmenu.h"


static void misc_winmenu_class_init (MiscWinmenuClass *klass);
static void misc_winmenu_init       (MiscWinmenu      *winmenu);
static void misc_winmenu_dispose    (GObject          *object);


static GObjectClass *parent_class = NULL;


/******************************************************************************
*                                                                             *
******************************************************************************/
GType
misc_winmenu_get_type (void)
{
  static GType type = 0;

  if (!type)
    {
      static const GTypeInfo info =
      {
        sizeof (MiscWinmenuClass),
        NULL,               /* base_init */
        NULL,               /* base_finalize */
        (GClassInitFunc)misc_winmenu_class_init,
        NULL,               /* class_finalize */
        NULL,               /* class_data */
        sizeof (MiscWinmenu),
        0,              /* n_preallocs */
        (GInstanceInitFunc)misc_winmenu_init,
      };

      type = g_type_register_static (G_TYPE_OBJECT, "MiscWinmenu", &info, 0);
    }

  return type;
}


static void
misc_winmenu_class_init (MiscWinmenuClass *klass)
{
  GObjectClass *object_class;

  parent_class = g_type_class_peek_parent (klass);
  object_class = (GObjectClass *) klass;

  object_class->dispose = misc_winmenu_dispose;
}


/*  ja:新規作成
    RET,オブジェクト                                                        */
static void
misc_winmenu_init (MiscWinmenu *winmenu)
{
  winmenu->position             = 0;
  winmenu->handler_add          = 0;
  winmenu->handler_change       = 0;
  winmenu->handler_remove       = 0;
  winmenu->handler_destroy_mdi  = 0;
  winmenu->handler_destroy_menu = 0;
  winmenu->mdi                  = NULL;
  winmenu->menu_shell           = NULL;
  winmenu->menu_item_head       = NULL;
  winmenu->menu_item_foot       = NULL;
  winmenu->glist                = NULL;
}


static void
misc_winmenu_dispose (GObject *object)
{
  MiscWinmenu *winmenu = MISC_WINMENU (object);

  if (winmenu->mdi)
    {
      if (winmenu->handler_add)
        g_signal_handler_disconnect (G_OBJECT (winmenu->mdi),
                                                winmenu->handler_add);
      if (winmenu->handler_remove)
        g_signal_handler_disconnect (G_OBJECT (winmenu->mdi),
                                                winmenu->handler_remove);
      if (winmenu->handler_destroy_mdi)
        g_signal_handler_disconnect (G_OBJECT (winmenu->mdi),
                                                winmenu->handler_destroy_mdi);
      winmenu->mdi = NULL;
    }
  if (winmenu->menu_shell)
    {
      if (winmenu->handler_destroy_menu)
        g_signal_handler_disconnect (G_OBJECT (winmenu->menu_shell),
                                                winmenu->handler_destroy_menu);
      winmenu->menu_shell = NULL;
    }
  if (G_OBJECT_CLASS (parent_class)->dispose)
    G_OBJECT_CLASS (parent_class)->dispose (object);
}


static void
misc_winmenu_activate (GtkWidget   *widget,
                       GtkNotebook *notebook)
{
  GtkWidget *child;

  child = g_object_get_data (G_OBJECT (widget), "user_data");
  gtk_notebook_set_current_page (notebook,
                                    gtk_notebook_page_num (notebook, child));
}


static void
misc_winmenu_add (MiscMdi     *mdi,
                  guint        page_num,
                  gpointer    *page_data,
                  MiscWinmenu *winmenu)
{
  if (winmenu && winmenu->mdi == mdi && winmenu->menu_shell)
    {
      gchar *label;
      gint length;
      GList *glist;
      GtkWidget *menu_item;

      glist = gtk_container_get_children (GTK_CONTAINER (winmenu->menu_shell));
      length = g_list_length (glist);
      label = g_filename_to_utf8 (misc_mdi_get_file (mdi, page_num), -1,
                                                            NULL, NULL, NULL);
      menu_item = gtk_menu_item_new_with_label (label);
      g_free (label);
      g_signal_connect (G_OBJECT (menu_item), "activate",
                            G_CALLBACK (misc_winmenu_activate), winmenu->mdi);
      g_object_set_data (G_OBJECT (menu_item), "user_data",
                    gtk_notebook_get_nth_page (GTK_NOTEBOOK (mdi), page_num));
      gtk_widget_show (menu_item);
      if (winmenu->glist)
        {
          gint i;

          for (i = length - 2; i >= 0; i--)
            if (g_list_nth_data (glist, i)
                                    == (g_list_last (winmenu->glist))->data)
            gtk_menu_shell_insert (GTK_MENU_SHELL (winmenu->menu_shell),
                                                            menu_item, i + 1);
          if (i < 0)
            gtk_menu_shell_append (GTK_MENU_SHELL (winmenu->menu_shell),
                                                                    menu_item);
        }
      else
        {
          gint position;

          position = 0 <= winmenu->position
                        && winmenu->position < length ? winmenu->position : -1;
          if (length > 0)
            {
              if (position != 0)
                {
                  winmenu->menu_item_head = gtk_separator_menu_item_new ();
                  gtk_menu_shell_insert (GTK_MENU_SHELL (winmenu->menu_shell),
                                            winmenu->menu_item_head, position);
                  if (position > 0)
                    position++;
                }
              if (position >= 0)
                {
                  winmenu->menu_item_foot = gtk_separator_menu_item_new ();
                  gtk_menu_shell_insert (GTK_MENU_SHELL (winmenu->menu_shell),
                                            winmenu->menu_item_foot, position);
                }
            }
          if (position < 0)
            gtk_menu_shell_append (GTK_MENU_SHELL (winmenu->menu_shell),
                                                                    menu_item);
          else if (position == 0)
            gtk_menu_shell_prepend (GTK_MENU_SHELL (winmenu->menu_shell),
                                                                    menu_item);
          else
            gtk_menu_shell_insert (GTK_MENU_SHELL (winmenu->menu_shell),
                                                        menu_item, position);
          if (GTK_IS_MENU (winmenu->menu_shell))
            {
              GtkWidget *menu_item_attach;

              menu_item_attach = gtk_menu_get_attach_widget
                                            (GTK_MENU (winmenu->menu_shell));
              if (menu_item_attach)
                gtk_widget_set_sensitive (menu_item_attach, TRUE);
            }
        }
      winmenu->glist = g_list_append (winmenu->glist, menu_item);
      g_list_free (glist);
    }
}


static void
misc_winmenu_change (MiscMdi     *mdi,
                     guint        page_num,
                     gpointer    *page_data,
                     MiscWinmenu *winmenu)
{
  if (winmenu && winmenu->mdi == mdi && winmenu->menu_shell && winmenu->glist)
    {
      gchar *label;
      gint i;
      GtkWidget *child;

      child = gtk_notebook_get_nth_page (GTK_NOTEBOOK (mdi), page_num);
      label = g_filename_to_utf8 (misc_mdi_get_file (mdi, page_num), -1,
                                                            NULL, NULL, NULL);
      for (i = g_list_length (winmenu->glist) - 1; i >= 0; i--)
        {
          GtkWidget *menu_item;

          menu_item = g_list_nth_data (winmenu->glist, i);
          if (g_object_get_data (G_OBJECT (menu_item), "user_data") == child)
            gtk_label_set_text (GTK_LABEL (gtk_bin_get_child
                                                (GTK_BIN (menu_item))), label);
        }
      g_free (label);
    }
}


static void
misc_winmenu_remove (MiscMdi     *mdi,
                     guint        page_num,
                     gpointer    *page_data,
                     MiscWinmenu *winmenu)
{
  if (winmenu && winmenu->mdi == mdi && winmenu->menu_shell && winmenu->glist)
    {
      gint i;
      GtkWidget *child, *menu_item;

      child = gtk_notebook_get_nth_page (GTK_NOTEBOOK (mdi), page_num);
      for (i = g_list_length (winmenu->glist) - 1; i >= 0; i--)
        {
          menu_item = g_list_nth_data (winmenu->glist, i);
          if (g_object_get_data (G_OBJECT (menu_item), "user_data") == child)
            {
              gtk_container_remove (GTK_CONTAINER (winmenu->menu_shell),
                                                                    menu_item);
              winmenu->glist = g_list_remove (winmenu->glist, menu_item);
              break;
            }
        }
      if (!winmenu->glist)
        {
          if (winmenu->menu_item_head)
            {
              gtk_container_remove (GTK_CONTAINER (winmenu->menu_shell),
                                                    winmenu->menu_item_head);
              winmenu->menu_item_head = NULL;
            }
          if (winmenu->menu_item_foot)
            {
              gtk_container_remove (GTK_CONTAINER (winmenu->menu_shell),
                                                    winmenu->menu_item_foot);
              winmenu->menu_item_foot = NULL;
            }
          if (GTK_IS_MENU (winmenu->menu_shell))
            {
              menu_item = gtk_menu_get_attach_widget
                                            (GTK_MENU (winmenu->menu_shell));
              if (menu_item)
                gtk_widget_set_sensitive (menu_item, FALSE);
            }
        }
    }
}


static void
misc_winmenu_destroy_mdi (GtkWidget   *widget,
                          MiscWinmenu *winmenu)
{
  if (winmenu && winmenu->mdi == MISC_MDI (widget))
    {
      gint i;

      for (i = misc_mdi_get_n_pages (winmenu->mdi) - 1; i >= 0; i--)
        misc_winmenu_remove (winmenu->mdi,
                             i,
                             misc_mdi_get_data (winmenu->mdi, i),
                             winmenu);
      winmenu->mdi = NULL;
    }
}


static void
misc_winmenu_destroy_menu_shell (GtkWidget   *widget,
                                 MiscWinmenu *winmenu)
{
  if (winmenu && winmenu->menu_shell == widget)
    {
      g_list_free (winmenu->glist);
      winmenu->glist = NULL;
      winmenu->menu_item_head = NULL;
      winmenu->menu_item_foot = NULL;
      winmenu->menu_shell = NULL;
    }
}


/******************************************************************************
*                                                                             *
* ja:ウインドウメニュー関数群                                                 *
*                                                                             *
******************************************************************************/
/*  ja:新規作成
    RET,オブジェクト                                                        */
GObject*
misc_winmenu_new (void)
{
  return g_object_new (MISC_TYPE_WINMENU, NULL);
}


static void
misc_winmenu_create_menu (MiscWinmenu *winmenu)
{
  if (winmenu && winmenu->mdi)
    {
      gint i, n_pages;

      n_pages = misc_mdi_get_n_pages (winmenu->mdi);
      for (i =  0; i < n_pages; i++)
        misc_winmenu_add (winmenu->mdi,
                          i,
                          misc_mdi_get_data (winmenu->mdi, i),
                          winmenu);
    }
}


static void
misc_winmenu_destroy_menu (MiscWinmenu *winmenu)
{
  if (winmenu && winmenu->mdi)
    {
      gint i;

      for (i =  misc_mdi_get_n_pages (winmenu->mdi) - 1; i >= 0; i--)
        misc_winmenu_remove (winmenu->mdi,
                             i,
                             misc_mdi_get_data (winmenu->mdi, i),
                             winmenu);
    }
}


/*  ja:MDIを設定する
    winmenu,オブジェクト
       mdi,MDI                                                              */
void
misc_winmenu_set_mdi (MiscWinmenu *winmenu,
                      MiscMdi     *mdi)
{
  if (winmenu)
    {
      if (winmenu->mdi)
        {
          if (winmenu->handler_add)
            g_signal_handler_disconnect (G_OBJECT (winmenu->mdi),
                                                winmenu->handler_add);
          if (winmenu->handler_change)
            g_signal_handler_disconnect (G_OBJECT (winmenu->mdi),
                                                winmenu->handler_change);
          if (winmenu->handler_remove)
            g_signal_handler_disconnect (G_OBJECT (winmenu->mdi),
                                                winmenu->handler_remove);
          if (winmenu->handler_destroy_mdi)
            g_signal_handler_disconnect (G_OBJECT (winmenu->mdi),
                                                winmenu->handler_destroy_mdi);
        }
      misc_winmenu_destroy_menu (winmenu);
      winmenu->mdi = mdi;
      if (winmenu->mdi)
        {
          winmenu->handler_add = g_signal_connect (G_OBJECT (mdi),
                "add-window", G_CALLBACK (misc_winmenu_add), winmenu);
          winmenu->handler_change = g_signal_connect (G_OBJECT (mdi),
                "change-window", G_CALLBACK (misc_winmenu_change), winmenu);
          winmenu->handler_remove = g_signal_connect (G_OBJECT (mdi),
                "remove-window", G_CALLBACK (misc_winmenu_remove), winmenu);
          winmenu->handler_destroy_mdi = g_signal_connect (G_OBJECT (mdi),
                "destroy", G_CALLBACK (misc_winmenu_destroy_mdi), winmenu);
          misc_winmenu_create_menu (winmenu);
        }
    }
}


/*  ja:メニューを設定する
       winmenu,オブジェクト
    menu_shell,メニュー
      position,位置(0...,-1:末尾)                                           */
void
misc_winmenu_set_menu (MiscWinmenu *winmenu,
                       GtkWidget   *menu_shell,
                       gint         position)
{
  if (winmenu)
    {
      if (winmenu->menu_shell)
        {
          if (winmenu->handler_destroy_menu)
            g_signal_handler_disconnect (G_OBJECT (winmenu->menu_shell),
                                                winmenu->handler_destroy_menu);
          misc_winmenu_destroy_menu (winmenu);
        }
      winmenu->menu_shell = menu_shell;
      winmenu->position = position;
      if (winmenu->menu_shell)
        {
          winmenu->handler_destroy_menu
                        = g_signal_connect (G_OBJECT (menu_shell), "destroy",
                        G_CALLBACK (misc_winmenu_destroy_menu_shell), winmenu);
          misc_winmenu_create_menu (winmenu);
          if (!winmenu->glist && GTK_IS_MENU (winmenu->menu_shell))
            {
              GList *glist;

              glist = gtk_container_get_children (GTK_CONTAINER (menu_shell));
              if (g_list_length (glist) <= 0)
                {
                  GtkWidget *menu_item;

                  menu_item = gtk_menu_get_attach_widget
                                                    (GTK_MENU (menu_shell));
                  if (menu_item)
                    gtk_widget_set_sensitive (menu_item, FALSE);
                }
              g_list_free(glist);
            }
        }
    }
}
