/*
    Text maid
    copyright (c) 1998-2006 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "ftype.h"
#include "general.h"
#include "prop.h"
#include "orz/orzmdi.h"


/******************************************************************************
*                                                                             *
* ja:ファイルタイプ関数群                                                     *
*                                                                             *
******************************************************************************/
/*  ja:ファイルタイプをコピーする
    ft_dst,コピー先
    ft_src,コピー元                                                         */
static void
ftype_copy (FileType *ft_dst,
            FileType *ft_src)
{
  ft_dst->ft_id = ft_src->ft_id;
  ft_dst->crlf = ft_src->crlf;
  ft_dst->margin = ft_src->margin;
  ft_dst->tab = ft_src->tab;
  ft_dst->writemode = ft_src->writemode;
  ft_dst->autoindent = ft_src->autoindent;
  ft_dst->code = ft_src->code;
  ft_dst->eof = ft_src->eof;
  ft_dst->hexcase = ft_src->hexcase;
  ft_dst->html = ft_src->html;
  ft_dst->esc = ft_src->esc;
  ft_dst->limit = ft_src->limit;
  ft_dst->overwrite = ft_src->overwrite;
  ft_dst->space = ft_src->space;
  ft_dst->syscol = ft_src->syscol;
  ft_dst->tabconv = ft_src->tabconv;
  ft_dst->gline = ft_src->gline;
  ft_dst->mline = ft_src->mline;
  ft_dst->nline = ft_src->nline;
  ft_dst->uline = ft_src->uline;
  ft_dst->vline = ft_src->vline;
  g_free (ft_dst->charset);
  ft_dst->charset = ft_src->charset ? g_strdup (ft_src->charset) : NULL;
  g_free (ft_dst->font_name);
  ft_dst->font_name = ft_src->font_name ? g_strdup (ft_src->font_name) : NULL;
  g_free (ft_dst->text);
  ft_dst->text = ft_src->text ? g_strdup (ft_src->text) : NULL;
  g_free (ft_dst->ext);
  ft_dst->ext = ft_src->ext ? g_strdup (ft_src->ext) : NULL;
  g_memmove (ft_dst->color, ft_src->color, sizeof (GdkColor) * 12);
}


/******************************************************************************
*                                                                             *
* ja:ファイルタイプダイアログ関数群                                           *
*                                                                             *
******************************************************************************/
typedef struct _FileTypeDialog
{
  gboolean edited;
  gint row;
  GtkWidget *dialog;
  GtkWidget *tview;
  GtkWidget *button_add;
  GtkWidget *button_remove;
  GtkWidget *button_up;
  GtkWidget *button_down;
  GtkWidget *button_edit;
  GtkWidget *check;
} FileTypeDialog;


enum
{
  TARGET_GTK_TREE_MODEL_ROW
};


/* ja:リストボックスの選択 */
static void
ftype_dialog_changed (GtkTreeSelection *select,
                      FileTypeDialog   *ftypedlg)
{
  gint i;
  GtkTreeIter iter;
  GtkTreeModel *model;

  model = gtk_tree_view_get_model (GTK_TREE_VIEW (ftypedlg->tview));
  ftypedlg->row = -1;
  for (i = 0; gtk_tree_model_iter_nth_child (model, &iter, NULL, i); i++)
    if (gtk_tree_selection_iter_is_selected (select, &iter))
      {
        ftypedlg->row = i;
        break;
      }
  if (ftypedlg->row >= 0)
    {
      gtk_widget_set_sensitive (ftypedlg->button_add, TRUE);
      gtk_widget_set_sensitive (ftypedlg->button_remove, i > 1);
      gtk_widget_set_sensitive (ftypedlg->button_up, ftypedlg->row > 0);
      gtk_widget_set_sensitive (ftypedlg->button_down, ftypedlg->row < i - 1);
      gtk_widget_set_sensitive (ftypedlg->button_edit, TRUE);
      gtk_dialog_set_response_sensitive (GTK_DIALOG (ftypedlg->dialog),
                        GTK_RESPONSE_OK, ftypedlg->check || ftypedlg->edited);
    }
  else
    {
      gtk_widget_set_sensitive (ftypedlg->button_add, FALSE);
      gtk_widget_set_sensitive (ftypedlg->button_remove, FALSE);
      gtk_widget_set_sensitive (ftypedlg->button_up, FALSE);
      gtk_widget_set_sensitive (ftypedlg->button_down, FALSE);
      gtk_widget_set_sensitive (ftypedlg->button_edit, FALSE);
      gtk_dialog_set_response_sensitive (GTK_DIALOG (ftypedlg->dialog),
                                            GTK_RESPONSE_OK, ftypedlg->edited
      || (ftypedlg->check
      && gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (ftypedlg->check))));
    }
}


/* ja:追加 */
static void
ftype_dialog_clicked_add (GtkWidget      *widget,
                          FileTypeDialog *ftypedlg)
{
  GtkListStore *store;
  GtkTreeIter iter;

  store = GTK_LIST_STORE (gtk_tree_view_get_model
                                            (GTK_TREE_VIEW (ftypedlg->tview)));
  if (gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store),
                                                &iter, NULL, ftypedlg->row))
    {
      FileType *ft_dst, *ft_src;

      gtk_tree_model_get (GTK_TREE_MODEL (store), &iter, 1, &ft_src, -1);
      ft_dst = g_malloc0 (sizeof (FileType));
      ftype_copy (ft_dst, ft_src);
      ft_dst->ft_id = tcount++;
      gtk_list_store_insert (store, &iter, ftypedlg->row + 1);
      gtk_list_store_set (store, &iter,
                                    0, ft_dst->text, 1, ft_dst, 2, FALSE, -1);
      gtk_dialog_set_response_sensitive (GTK_DIALOG (ftypedlg->dialog),
                                                        GTK_RESPONSE_OK, TRUE);
      ftypedlg->edited = TRUE;
    }
}


/* ja:削除 */
static void
ftype_dialog_clicked_remove (GtkWidget      *widget,
                             FileTypeDialog *ftypedlg)
{
  GtkListStore *store;
  GtkTreeIter iter;

  store = GTK_LIST_STORE (gtk_tree_view_get_model
                                            (GTK_TREE_VIEW (ftypedlg->tview)));
  if (gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store),
                                                &iter, NULL, ftypedlg->row))
    {
      FileType *ft;

      gtk_tree_model_get (GTK_TREE_MODEL (store), &iter, 1, &ft, -1);
      gtk_list_store_remove (store, &iter);
      g_free (ft->text);
      g_free (ft->ext);
      g_free (ft->font_name);
      g_free (ft);
      gtk_dialog_set_response_sensitive (GTK_DIALOG (ftypedlg->dialog),
                                                        GTK_RESPONSE_OK, TRUE);
      ftypedlg->edited = TRUE;
    }
}


/* ja:上へ */
static void
ftype_dialog_clicked_up (GtkWidget      *widget,
                         FileTypeDialog *ftypedlg)
{
  GtkListStore *store;
  GtkTreeIter iter0, iter1;
  GtkTreeSelection *select;

  store = GTK_LIST_STORE (gtk_tree_view_get_model
                                            (GTK_TREE_VIEW (ftypedlg->tview)));
  select = gtk_tree_view_get_selection (GTK_TREE_VIEW (ftypedlg->tview));
  if (gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store),
                                            &iter0, NULL, ftypedlg->row)
        && gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store),
                                            &iter1, NULL, ftypedlg->row - 1))
    {
#if GTK_CHECK_VERSION(2,2,0)
      gtk_list_store_swap (store, &iter0, &iter1);
#else /* not GTK_CHECK_VERSION(2,2,0) */
      gboolean edited0, edited1;
      FileType *ft0, *ft1;

      gtk_tree_model_get (GTK_TREE_MODEL (store), &iter0,
                                                    1, &ft0, 2, &edited0, -1);
      gtk_tree_model_get (GTK_TREE_MODEL (store), &iter1,
                                                    1, &ft1, 2, &edited1, -1);
      gtk_list_store_set (store, &iter0, 0, ft1->text, 1, ft1, 2, edited1, -1);
      gtk_list_store_set (store, &iter1, 0, ft0->text, 1, ft0, 2, edited0, -1);
      gtk_tree_selection_select_iter (select, &iter1);
      gtk_dialog_set_response_sensitive (GTK_DIALOG (ftypedlg->dialog),
                                                        GTK_RESPONSE_OK, TRUE);
#endif /* not GTK_CHECK_VERSION(2,2,0) */
      ftypedlg->edited = TRUE;
#if GTK_CHECK_VERSION(2,2,0)
      ftype_dialog_changed (select, ftypedlg);
#endif /* GTK_CHECK_VERSION(2,2,0) */
    }
}


/* ja:下へ */
static void
ftype_dialog_clicked_down (GtkWidget      *widget,
                           FileTypeDialog *ftypedlg)
{
  GtkListStore *store;
  GtkTreeIter iter0, iter1;
  GtkTreeSelection *select;

  store = GTK_LIST_STORE (gtk_tree_view_get_model
                                            (GTK_TREE_VIEW (ftypedlg->tview)));
  select = gtk_tree_view_get_selection (GTK_TREE_VIEW (ftypedlg->tview));
  if (gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store),
                                            &iter0, NULL, ftypedlg->row)
        && gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store),
                                            &iter1, NULL, ftypedlg->row + 1))
    {
#if GTK_CHECK_VERSION(2,2,0)
      gtk_list_store_swap (store, &iter0, &iter1);
#else /* not GTK_CHECK_VERSION(2,2,0) */
      gboolean edited0, edited1;
      FileType *ft0, *ft1;

      gtk_tree_model_get (GTK_TREE_MODEL (store), &iter0,
                                                    1, &ft0, 2, &edited0, -1);
      gtk_tree_model_get (GTK_TREE_MODEL (store), &iter1,
                                                    1, &ft1, 2, &edited1, -1);
      gtk_list_store_set (store, &iter0, 0, ft1->text, 1, ft1, 2, edited1, -1);
      gtk_list_store_set (store, &iter1, 0, ft0->text, 1, ft0, 2, edited0, -1);
      gtk_tree_selection_select_iter (select, &iter1);
      gtk_dialog_set_response_sensitive (GTK_DIALOG (ftypedlg->dialog),
                                                        GTK_RESPONSE_OK, TRUE);
#endif /* not GTK_CHECK_VERSION(2,2,0) */
      ftypedlg->edited = TRUE;
#if GTK_CHECK_VERSION(2,2,0)
      ftype_dialog_changed (select, ftypedlg);
#endif /* GTK_CHECK_VERSION(2,2,0) */
    }
}


/* ja:編集 */
static void
ftype_dialog_clicked_edit (GtkWidget      *widget,
                           FileTypeDialog *ftypedlg)
{
  GtkListStore *store;
  GtkTreeIter iter;

  store = GTK_LIST_STORE (gtk_tree_view_get_model
                                            (GTK_TREE_VIEW (ftypedlg->tview)));
  if (gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store),
                                                &iter, NULL, ftypedlg->row))
    {
      FileType *ft;

      gtk_tree_model_get (GTK_TREE_MODEL (store), &iter, 1, &ft, -1);
      if (property_dialog (GTK_WINDOW (ftypedlg->dialog), ft))
        {
          gtk_list_store_set (store, &iter, 0, ft->text, 1, ft, 2, TRUE, -1);
          gtk_dialog_set_response_sensitive (GTK_DIALOG (ftypedlg->dialog),
                                                        GTK_RESPONSE_OK, TRUE);
          ftypedlg->edited = TRUE;
        }
    }
}


/* ja:チェックボックスが変化したとき */
void
ftype_dialog_toggled (GtkToggleButton *togglebutton,
                      FileTypeDialog  *ftypedlg)
{
  if (gtk_toggle_button_get_active (togglebutton))
    {
      gtk_tree_selection_unselect_all (gtk_tree_view_get_selection
                                            (GTK_TREE_VIEW (ftypedlg->tview)));
      gtk_widget_set_sensitive (ftypedlg->tview, FALSE);
      ftypedlg->row = -1;
    }
  else
    {
      gtk_widget_set_sensitive (ftypedlg->tview, TRUE);
    }
  gtk_dialog_set_response_sensitive (GTK_DIALOG (ftypedlg->dialog),
                                                        GTK_RESPONSE_OK, TRUE);
  ftypedlg->edited = TRUE;
}


/* ja:ダイアログが破棄されるとき */
static void
ftype_dialog_destroy (GtkObject      *object,
                      FileTypeDialog *ftypedlg)
{
  gint i;
  GtkTreeIter iter;
  GtkTreeModel *model;

  model = gtk_tree_view_get_model (GTK_TREE_VIEW (ftypedlg->tview));
  for (i = 0; gtk_tree_model_iter_nth_child (model, &iter, NULL, i); i++)
    {
      FileType *ft;

      gtk_tree_model_get (model, &iter, 1, &ft, -1);
      if (ft)
        {
          g_free (ft->text);
          g_free (ft->ext);
          g_free (ft->font_name);
          g_free (ft);
        }
    }
}


/* ja:ドロップされたとき */
static void
ftype_dialog_drag_end (GtkWidget      *widget,
                       GdkDragContext *drag_context,
                       FileTypeDialog *ftypedlg)
{
  ftypedlg->edited = TRUE;
  ftype_dialog_changed (gtk_tree_view_get_selection
                                (GTK_TREE_VIEW (ftypedlg->tview)), ftypedlg);
}


/* ja:ダブルクリックされたとき */
static gboolean
ftype_dialog_button_press (GtkWidget      *widget,
                           GdkEventButton *event,
                           FileTypeDialog *ftypedlg)
{
  gboolean result = FALSE;

  if (event->type == GDK_2BUTTON_PRESS && event->button == 1)
    {
      GtkTreePath *path;

      if (gtk_tree_view_get_path_at_pos (GTK_TREE_VIEW (ftypedlg->tview),
                    (gint)event->x, (gint)event->y, &path, NULL, NULL, NULL))
        {
          GtkTreeIter iter;
          GtkTreeModel *model;

          model = gtk_tree_view_get_model (GTK_TREE_VIEW (ftypedlg->tview));
          if (gtk_tree_model_get_iter (model, &iter, path))
            {
              ftype_dialog_clicked_edit (widget, ftypedlg);
              result = TRUE;
            }
          gtk_tree_path_free (path);
        }
    }
  return result;
}


/*  ja:ファイルタイプダイアログを表示する
    parent,親ウインドウ
       ind,TRUE:独立ファイルタイプあり,FALSE:なし
       RET,選択されたファイルタイプ(ftnum:独立,-1:キャンセル)               */
gint
ftype_dialog (GtkWindow *parent,
              gboolean   ind)
{
  gint i;
  GtkCellRenderer *renderer;
  GtkListStore *store;
  GtkTreeSelection *select;
  GtkTreeViewColumn *column;
  GtkWidget *scroll, *hbox, *vbox;
  GtkTargetEntry dnd_entries[1] = {{"GTK_TREE_MODEL_ROW",
    GTK_TARGET_SAME_APP | GTK_TARGET_SAME_WIDGET, TARGET_GTK_TREE_MODEL_ROW}};
  FileTypeDialog ftypedlg;

  ftypedlg.edited = FALSE;
  ftypedlg.row = -1;
  /* ja:メインウインドウ */
  ftypedlg.dialog = gtk_dialog_new_with_buttons
                            (ind ? _("Associating File Type") : _("File Type"),
                            parent,
                            GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                            GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                            GTK_STOCK_OK, GTK_RESPONSE_OK,
                            NULL);
  gtk_window_set_default_size (GTK_WINDOW (ftypedlg.dialog), 320, 240);
  g_signal_connect (G_OBJECT (ftypedlg.dialog), "key-press-event",
                                G_CALLBACK (misc_dialog_key_press), NULL);
  g_signal_connect (G_OBJECT (ftypedlg.dialog), "destroy",
                                G_CALLBACK (ftype_dialog_destroy), &ftypedlg);
  /* ja:ボタン */
  ftypedlg.button_add = gtk_button_new_from_stock (GTK_STOCK_ADD);
  ftypedlg.button_remove = gtk_button_new_from_stock (GTK_STOCK_REMOVE);
  ftypedlg.button_up = gtk_button_new_from_stock (GTK_STOCK_GO_UP);
  ftypedlg.button_down = gtk_button_new_from_stock (GTK_STOCK_GO_DOWN);
#if GTK_CHECK_VERSION(2,6,0)
  ftypedlg.button_edit = gtk_button_new_from_stock (GTK_STOCK_EDIT);
#else /* not GTK_CHECK_VERSION(2,6,0) */
  ftypedlg.button_edit = gtk_button_new_with_mnemonic (_("_Edit"));
#endif /* not GTK_CHECK_VERSION(2,6,0) */
  g_signal_connect (G_OBJECT (ftypedlg.button_add), "clicked",
                        G_CALLBACK (ftype_dialog_clicked_add), &ftypedlg);
  g_signal_connect (G_OBJECT (ftypedlg.button_remove), "clicked",
                        G_CALLBACK (ftype_dialog_clicked_remove), &ftypedlg);
  g_signal_connect (G_OBJECT (ftypedlg.button_up), "clicked",
                        G_CALLBACK (ftype_dialog_clicked_up), &ftypedlg);
  g_signal_connect (G_OBJECT (ftypedlg.button_down), "clicked",
                        G_CALLBACK (ftype_dialog_clicked_down), &ftypedlg);
  g_signal_connect (G_OBJECT (ftypedlg.button_edit), "clicked",
                        G_CALLBACK (ftype_dialog_clicked_edit), &ftypedlg);
  /* ja:チェックボタン */
  if (ind)
    {
      ftypedlg.check = gtk_check_button_new_with_mnemonic
                                                (_("Independent File Type"));
      g_signal_connect (G_OBJECT (ftypedlg.check), "toggled",
                                G_CALLBACK (ftype_dialog_toggled), &ftypedlg);
    }
  else
    {
      ftypedlg.check = NULL;
    }
  /* ja:リストボックス */
  store = gtk_list_store_new (3,
                                G_TYPE_STRING, G_TYPE_POINTER, G_TYPE_BOOLEAN);
  ftypedlg.tview = gtk_tree_view_new_with_model (GTK_TREE_MODEL (store));
  gtk_tree_view_set_enable_search (GTK_TREE_VIEW (ftypedlg.tview), FALSE);
  renderer = gtk_cell_renderer_text_new ();
  column = gtk_tree_view_column_new_with_attributes
                        (_("Registered File Type"), renderer, "text", 0, NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (ftypedlg.tview), column);
  select = gtk_tree_view_get_selection (GTK_TREE_VIEW (ftypedlg.tview));
  g_signal_connect (G_OBJECT (select), "changed",
                                G_CALLBACK (ftype_dialog_changed), &ftypedlg);
  for (i = 0; i < ftnum; i++)
    {
      FileType *ft;
      GtkTreeIter iter;

      ft = g_malloc0 (sizeof (FileType));
      ftype_copy (ft, ftype + i);
      gtk_list_store_append (store, &iter);
      gtk_list_store_set (store, &iter, 0, ft->text, 1, ft, 2, FALSE, -1);
    }
  g_signal_connect (G_OBJECT (ftypedlg.tview), "button-press-event",
                            G_CALLBACK (ftype_dialog_button_press), &ftypedlg);
  gtk_widget_add_events (ftypedlg.tview, GDK_BUTTON_PRESS_MASK);
  /* en:Drag & Drop */
  gtk_tree_view_enable_model_drag_dest (GTK_TREE_VIEW (ftypedlg.tview),
                                            dnd_entries, 1, GDK_ACTION_MOVE);
  gtk_tree_view_enable_model_drag_source (GTK_TREE_VIEW (ftypedlg.tview),
                            GDK_BUTTON1_MASK, dnd_entries, 1, GDK_ACTION_MOVE);
  g_signal_connect (G_OBJECT (ftypedlg.tview), "drag-end",
                            G_CALLBACK (ftype_dialog_drag_end), &ftypedlg);
  /* ja:スクロールウインドウ */
  scroll = gtk_scrolled_window_new (NULL, NULL);
  gtk_container_add (GTK_CONTAINER (scroll), ftypedlg.tview);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll),
                                    GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
  /* ja:ボックス */
  vbox = gtk_vbox_new (FALSE, SPACING);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), SPACING);
  gtk_box_pack_start (GTK_BOX (vbox), scroll, TRUE, TRUE, 0);
  if (ftypedlg.check)
    gtk_box_pack_start (GTK_BOX (vbox), ftypedlg.check, FALSE, FALSE, 0);
  hbox = gtk_hbox_new (FALSE, SPACING);
  gtk_box_pack_start (GTK_BOX (hbox), ftypedlg.button_add, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), ftypedlg.button_remove, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), ftypedlg.button_up, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), ftypedlg.button_down, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), ftypedlg.button_edit, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (ftypedlg.dialog)->vbox), vbox);

  /* ja:表示 */
  gtk_widget_set_sensitive (ftypedlg.button_add, FALSE);
  gtk_widget_set_sensitive (ftypedlg.button_remove, FALSE);
  gtk_widget_set_sensitive (ftypedlg.button_up, FALSE);
  gtk_widget_set_sensitive (ftypedlg.button_down, FALSE);
  gtk_widget_set_sensitive (ftypedlg.button_edit, FALSE);
  gtk_dialog_set_response_sensitive (GTK_DIALOG (ftypedlg.dialog),
                                                    GTK_RESPONSE_OK, FALSE);
  gtk_widget_show_all (vbox);

  if (gtk_dialog_run (GTK_DIALOG (ftypedlg.dialog)) == GTK_RESPONSE_OK)
    {
      GtkTreeIter iter;
      FileType *ft;

      for (i = 0; i < ftnum; i++)
        {
          g_free (ftype[i].text);
          g_free (ftype[i].ext);
          g_free (ftype[i].font_name);
        }
      g_free (ftype);
      ftype = g_malloc0 (gtk_tree_model_iter_n_children
                        (GTK_TREE_MODEL (store), NULL) * sizeof (FileType));
      ftnum = 0;
      for (i = 0; gtk_tree_model_iter_nth_child
                                (GTK_TREE_MODEL (store), &iter, NULL, i); i++)
        {
          gtk_tree_model_get (GTK_TREE_MODEL (store), &iter, 1, &ft, -1);
          ftype_copy (ftype + ftnum++, ft);
        }
      for (i = 0; gtk_tree_model_iter_nth_child
                                (GTK_TREE_MODEL (store), &iter, NULL, i); i++)
        {
          gboolean edited;

          gtk_tree_model_get (GTK_TREE_MODEL (store), &iter,
                                                    1, &ft, 2, &edited, -1);
          if (edited)
            {
              gint j;

              /* ja:同一ファイルタイプをすべて変更 */
              for (j = orz_mdi_get_n_pages (ORZ_MDI (mdi)) - 1; j >= 0; j--)
                {
                  TmaidWindow *tmaid;

                  tmaid = orz_mdi_get_data (ORZ_MDI (mdi), j);
                  if (tmaid->ft.ft_id == ft->ft_id)
                    property_change (tmaid, ft);
                }
              charlist_renewal_all (ft->charset, -1);
            }
        }
      if (ftypedlg.check
        && gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (ftypedlg.check)))
        ftypedlg.row = ftnum;
    }
  else
    {
      ftypedlg.row = -1;
    }
  gtk_widget_destroy (ftypedlg.dialog);

  return ftypedlg.row;
}
