/*
    Text maid
    copyright (c) 1998-2007 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "tmaid.h"
#include <locale.h>
#ifdef G_OS_WIN32
# include <windows.h>
#endif /* G_OS_WIN32 */
#include "command.h"
#include "file.h"
#include "find.h"
#include "general.h"
#include "icons.h"
#include "jump.h"
#include "print.h"
#include "replace.h"
#include "sigfile.h"
#include "sigmain.h"
#include "misc/argument.h"
#include "misc/charuty.h"
#include "misc/fileio.h"
#include "misc/gnugpl.h"
#include "misc/license.h"
#include "misc/profile.h"
#include "orz/orzaccel.h"
#include "orz/orzcomm.h"
#include "orz/orzhistory.h"
#include "orz/orzmdi.h"
#include "orz/orzwinmenu.h"


gboolean caret = FALSE;             /* ja:キャレットの形 */
gboolean ins = TRUE;                /* ja:TRUE:挿入,FALSE:上書き */
gboolean newfile = FALSE;           /* ja:TRUE:新規ファイル,FALSE:なし */
gboolean second = FALSE;            /* ja:TRUE:新規,FALSE:既存 */
gchar *open_path = NULL;            /* ja:開くのパス */
gchar *save_path = NULL;            /* ja:名前を付けて保存のパス */
gchar *clipboard_text = NULL;       /* ja:クリップボードのテキスト */
gint def_width, def_height;         /* ja:デフォルトのウインドウのサイズ */
gint ftnum = 0;                     /* ja:ファイルタイプの数 */
#if GTK_CHECK_VERSION(2,4,0) || defined (G_OS_WIN32)
gint findex = 0;                    /* ja:ファイルタイプのフィルタ */
#endif /* GTK_CHECK_VERSION(2,4,0) || defined (G_OS_WIN32) */
gint tcount = 0;                    /* ja:ファイルタイプのカウンタ */
guint timer_id = 0;                 /* ja:タイマID */
FileType *ftype = NULL;             /* ja:ファイルタイプ */
GObject *history;                   /* ja:ファイルの履歴 */
GdkAtom atom_targets;               /* ja:ターゲット */
GdkAtom atom_text;                  /* ja:テキスト */
GdkAtom atom_compound;              /* ja:コンパウンド */
GdkAtom atom_utf8;                  /* ja:UTF-8 */
GtkPositionType n_pos = GTK_POS_TOP;/* ja:ノートブックの位置 */
GtkWidget *window;                  /* ja:ウインドウ */
GtkWidget *mdi;                     /* en:MDI */
GtkWidget *status;                  /* ja:ステータスウインドウ */
GtkWidget *label_cursor;            /* ja:カーソル座標 */
PangoFontDescription *system_font = NULL;
GdkColor system_color[12] = {
    {0, 0x0000, 0x0000, 0x0000}, {0, 0xffff, 0xffff, 0xffff},
    {0, 0x8000, 0x8000, 0x8000}, {0, 0x8000, 0x8000, 0x8000},
    {0, 0x8000, 0x8000, 0x8000}, {0, 0x8000, 0x8000, 0x8000},
    {0, 0x8000, 0x8000, 0x8000}, {0, 0xa000, 0xa000, 0xa000},
    {0, 0xffff, 0xffff, 0xffff}, {0, 0x0000, 0x0000, 0x8000},
    {0, 0x0000, 0x0000, 0x0000}, {0, 0xffff, 0xffff, 0xffff}};
GtkTargetEntry targets[4] = {       /* ja:Text maidが提供可能なセレクション */
    {"STRING",        0, TARGET_STRING},
    {"TEXT",          0, TARGET_TEXT},
    {"COMPOUND_TEXT", 0, TARGET_COMPOUND_TEXT},
    {"UTF8_STRING",   0, TARGET_UTF8}};
MiscCreateMenuEntry menu_entries[] = {
{"/",                   NULL,                          NULL,          NULL,             MISC_CREATE_MENU_BAR,                               NULL,             NULL,                    NULL},
{"/file",               N_("_File"),                   NULL,          NULL,             MISC_CREATE_MENU_ITEM,                              NULL,             NULL,                    NULL},
{"/file/",              NULL,                          NULL,          NULL,             MISC_CREATE_MENU_SHELL,                             NULL,             NULL,                    NULL},
{"/file/new",           GTK_STOCK_NEW,                 NULL,          NULL,             MISC_CREATE_MENU_STOCK,                             command_new,      NULL,                    NULL},
{"/file/open",          GTK_STOCK_OPEN,                NULL,          NULL,             MISC_CREATE_MENU_STOCK | MISC_CREATE_MENU_ELLIPSIS, command_open,     NULL,                    NULL},
{"/file/close",         GTK_STOCK_CLOSE,               NULL,          NULL,             MISC_CREATE_MENU_STOCK,                             command_close,    NULL,                    NULL},
{"/file/save",          GTK_STOCK_SAVE,                NULL,          NULL,             MISC_CREATE_MENU_STOCK,                             command_save,     NULL,                    NULL},
{"/file/saveas",        GTK_STOCK_SAVE_AS,             NULL,          NULL,             MISC_CREATE_MENU_STOCK | MISC_CREATE_MENU_ELLIPSIS, command_saveas,   NULL,                    NULL},
{"/file/reload",        GTK_STOCK_REVERT_TO_SAVED,     "<control>R",  NULL,             MISC_CREATE_MENU_STOCK | MISC_CREATE_MENU_ELLIPSIS, command_reload,   NULL,                    NULL},
{"/file/sep1",          NULL,                          NULL,          NULL,             MISC_CREATE_MENU_SEPARATOR,                         NULL,             NULL,                    NULL},
{"/file/print",         GTK_STOCK_PRINT,               "<control>P",  NULL,             MISC_CREATE_MENU_STOCK | MISC_CREATE_MENU_ELLIPSIS, command_print,    NULL,                    NULL},
{"/file/sep2",          NULL,                          NULL,          NULL,             MISC_CREATE_MENU_SEPARATOR,                         NULL,             NULL,                    NULL},
{"/file/properties",    GTK_STOCK_PROPERTIES,          "<alt>Return", NULL,             MISC_CREATE_MENU_STOCK | MISC_CREATE_MENU_ELLIPSIS, command_property, NULL,                    NULL},
{"/file/sep3",          NULL,                          NULL,          NULL,             MISC_CREATE_MENU_SEPARATOR,                         NULL,             NULL,                    NULL},
{"/file/exit",          GTK_STOCK_QUIT,                NULL,          NULL,             MISC_CREATE_MENU_STOCK,                             command_exit,     NULL,                    NULL},
#if GTK_CHECK_VERSION(2,6,0)
{"/edit",               GTK_STOCK_EDIT,                NULL,          NULL,             MISC_CREATE_MENU_STOCK | MISC_CREATE_MENU_NOIMAGE,  NULL,             NULL,                    NULL},
#else /* not GTK_CHECK_VERSION(2,6,0) */
{"/edit",               N_("_Edit"),                   NULL,          NULL,             MISC_CREATE_MENU_ITEM,                              NULL,             NULL,                    NULL},
#endif /* not GTK_CHECK_VERSION(2,6,0) */
{"/edit/",              NULL,                          NULL,          NULL,             MISC_CREATE_MENU_SHELL,                             NULL,             NULL,                    NULL},
{"/edit/undo",          GTK_STOCK_UNDO,                "<control>Z",  NULL,             MISC_CREATE_MENU_STOCK,                             command_undo,     NULL,                    NULL},
{"/edit/redo",          GTK_STOCK_REDO,                "F4",          NULL,             MISC_CREATE_MENU_STOCK,                             command_redo,     NULL,                    NULL},
{"/edit/sep1",          NULL,                          NULL,          NULL,             MISC_CREATE_MENU_SEPARATOR,                         NULL,             NULL,                    NULL},
{"/edit/cut",           GTK_STOCK_CUT,                 NULL,          NULL,             MISC_CREATE_MENU_STOCK,                             command_cut,      NULL,                    NULL},
{"/edit/copy",          GTK_STOCK_COPY,                NULL,          NULL,             MISC_CREATE_MENU_STOCK,                             command_copy,     NULL,                    NULL},
{"/edit/paste",         GTK_STOCK_PASTE,               NULL,          NULL,             MISC_CREATE_MENU_STOCK,                             command_paste,    NULL,                    NULL},
{"/edit/delete",        GTK_STOCK_DELETE,              NULL,          NULL,             MISC_CREATE_MENU_STOCK,                             command_delete,   NULL,                    NULL},
{"/edit/sep2",          NULL,                          NULL,          NULL,             MISC_CREATE_MENU_SEPARATOR,                         NULL,             NULL,                    NULL},
{"/edit/return",        N_("Return at Right _Margin"), NULL,          return16_xpm,     MISC_CREATE_MENU_ITEM,                              command_margin,   NULL,                    NULL},
{"/edit/tab",           N_("Convert Ta_b to Space"),   NULL,          tab16_xpm,        MISC_CREATE_MENU_ITEM,                              command_tab,      NULL,                    NULL},
{"/edit/sep3",          NULL,                          NULL,          NULL,             MISC_CREATE_MENU_SEPARATOR,                         NULL,             NULL,                    NULL},
{"/edit/character",     N_("Character by _Value"),     NULL,          character16_xpm,  MISC_CREATE_MENU_ITEM | MISC_CREATE_MENU_ELLIPSIS,  command_valchr,   NULL,                    NULL},
{"/edit/jump",          GTK_STOCK_JUMP_TO,             "<control>J",  NULL,             MISC_CREATE_MENU_STOCK,                             command_jump,     NULL,                    NULL},
{"/edit/sep4",          NULL,                          NULL,          NULL,             MISC_CREATE_MENU_SEPARATOR,                         NULL,             NULL,                    NULL},
#if GTK_CHECK_VERSION(2,10,0)
{"/edit/all",           GTK_STOCK_SELECT_ALL,          "<control>A",  NULL,             MISC_CREATE_MENU_STOCK,                             command_all,      NULL,                    NULL},
#else /* not GTK_CHECK_VERSION(2,10,0) */
{"/edit/all",           N_("_All"),                    "<control>A",  select_all16_xpm, MISC_CREATE_MENU_ITEM,                              command_all,      NULL,                    NULL},
#endif /* not GTK_CHECK_VERSION(2,10,0) */
{"/search",             N_("_Search"),                 NULL,          NULL,             MISC_CREATE_MENU_ITEM,                              NULL,             NULL,                    NULL},
{"/search/",            NULL,                          NULL,          NULL,             MISC_CREATE_MENU_SHELL,                             NULL,             NULL,                    NULL},
{"/search/find",        GTK_STOCK_FIND,                NULL,          NULL,             MISC_CREATE_MENU_STOCK | MISC_CREATE_MENU_ELLIPSIS, command_find,     NULL,                    NULL},
{"/search/next",        N_("_Next"),                   "F3",          next16_xpm,       MISC_CREATE_MENU_ITEM,                              command_next,     GINT_TO_POINTER (TRUE),  NULL},
{"/search/previous",    N_("_Previous"),               "<shift>F3",   previous16_xpm,   MISC_CREATE_MENU_ITEM,                              command_next,     GINT_TO_POINTER (FALSE), NULL},
{"/search/next",        GTK_STOCK_FIND_AND_REPLACE,    "<control>H",  NULL,             MISC_CREATE_MENU_STOCK | MISC_CREATE_MENU_ELLIPSIS, command_replace,  NULL,                    NULL},
{"/option",             N_("_Option"),                 NULL,          NULL,             MISC_CREATE_MENU_ITEM,                              NULL,             NULL,                    NULL},
{"/option/",            NULL,                          NULL,          NULL,             MISC_CREATE_MENU_SHELL,                             NULL,             NULL,                    NULL},
{"/option/preferences", GTK_STOCK_PREFERENCES,         NULL,          NULL,             MISC_CREATE_MENU_STOCK | MISC_CREATE_MENU_ELLIPSIS, command_pref,     NULL,                    NULL},
{"/option/file",        N_("File _Type"),              NULL,          filetype16_xpm,   MISC_CREATE_MENU_ITEM | MISC_CREATE_MENU_ELLIPSIS,  command_format,   NULL,                    NULL},
{"/window",             N_("_Window"),                 NULL,          NULL,             MISC_CREATE_MENU_ITEM,                              NULL,             NULL,                    NULL},
{"/window/",            NULL,                          NULL,          NULL,             MISC_CREATE_MENU_SHELL,                             NULL,             NULL,                    NULL},
{"/help",               GTK_STOCK_HELP,                NULL,          NULL,             MISC_CREATE_MENU_STOCK | MISC_CREATE_MENU_NOIMAGE,  NULL,             NULL,                    NULL},
{"/help/",              NULL,                          NULL,          NULL,             MISC_CREATE_MENU_SHELL,                             NULL,             NULL,                    NULL},
#if GTK_CHECK_VERSION(2,6,0)
{"/help/about",         GTK_STOCK_ABOUT,               NULL,          NULL,             MISC_CREATE_MENU_STOCK,                             command_version,  NULL,                    NULL},
#else /* not GTK_CHECK_VERSION(2,6,0) */
{"/help/about",         N_("_About"),                  NULL,          about16_xpm,      MISC_CREATE_MENU_ITEM,                              command_version,  NULL,                    NULL},
#endif /* not GTK_CHECK_VERSION(2,6,0) */
{NULL,                  NULL,                          NULL,          NULL,             MISC_CREATE_MENU_TERMINAL,                          NULL,             NULL,                    NULL}};
MiscCreateMenuEntry popup_entries[] = {
{"/",                   NULL,                          NULL,          NULL,             MISC_CREATE_MENU_SHELL,                             NULL,             NULL,                    NULL},
{"/undo",               GTK_STOCK_UNDO,                "<control>Z",  NULL,             MISC_CREATE_MENU_STOCK,                             command_undo,     NULL,                    NULL},
{"/sep1",               NULL,                          NULL,          NULL,             MISC_CREATE_MENU_SEPARATOR,                         NULL,             NULL,                    NULL},
{"/cut",                GTK_STOCK_CUT,                 NULL,          NULL,             MISC_CREATE_MENU_STOCK,                             command_cut,      NULL,                    NULL},
{"/copy",               GTK_STOCK_COPY,                NULL,          NULL,             MISC_CREATE_MENU_STOCK,                             command_copy,     NULL,                    NULL},
{"/paste",              GTK_STOCK_PASTE,               NULL,          NULL,             MISC_CREATE_MENU_STOCK,                             command_paste,    NULL,                    NULL},
{"/delete",             GTK_STOCK_DELETE,              NULL,          NULL,             MISC_CREATE_MENU_STOCK,                             command_delete,   NULL,                    NULL},
{"/sep2",               NULL,                          NULL,          NULL,             MISC_CREATE_MENU_SEPARATOR,                         NULL,             NULL,                    NULL},
#if GTK_CHECK_VERSION(2,10,0)
{"/all",                GTK_STOCK_SELECT_ALL,          "<control>A",  NULL,             MISC_CREATE_MENU_STOCK,                             command_all,      NULL,                    NULL},
#else /* not GTK_CHECK_VERSION(2,10,0) */
{"/all",                N_("_All"),                    "<control>A",  select_all16_xpm, MISC_CREATE_MENU_ITEM,                              command_all,      NULL,                    NULL},
#endif /* not GTK_CHECK_VERSION(2,10,0) */
{"/sep3",               NULL,                          NULL,          NULL,             MISC_CREATE_MENU_SEPARATOR,                         NULL,             NULL,                    NULL},
{"/input",              N_("_Input Methods"),          NULL,          NULL,             MISC_CREATE_MENU_ITEM,                              NULL,             NULL,                    NULL},
{NULL,                  NULL,                          NULL,          NULL,             MISC_CREATE_MENU_TERMINAL,                          NULL,             NULL,                    NULL}};
MiscCreateToolbarEntry toolbar_entries[] = {
{"new",        GTK_STOCK_NEW,              NULL, command_new,      NULL, NULL},
{"open",       GTK_STOCK_OPEN,             NULL, command_open,     NULL, NULL},
{"save",       GTK_STOCK_SAVE,             NULL, command_save,     NULL, NULL},
{"sep1",       NULL,                       NULL, NULL,             NULL, NULL},
{"properties", GTK_STOCK_PROPERTIES,       NULL, command_property, NULL, NULL},
{"sep2",       NULL,                       NULL, NULL,             NULL, NULL},
{"print",      GTK_STOCK_PRINT,            NULL, command_print,    NULL, NULL},
{"sep3",       NULL,                       NULL, NULL,             NULL, NULL},
{"cut",        GTK_STOCK_CUT,              NULL, command_cut,      NULL, NULL},
{"copy",       GTK_STOCK_COPY,             NULL, command_copy,     NULL, NULL},
{"paste",      GTK_STOCK_PASTE,            NULL, command_paste,    NULL, NULL},
{"delete",     GTK_STOCK_DELETE,           NULL, command_delete,   NULL, NULL},
{"sep4",       NULL,                       NULL, NULL,             NULL, NULL},
{"undo",       GTK_STOCK_UNDO,             NULL, command_undo,     NULL, NULL},
{"redo",       GTK_STOCK_REDO,             NULL, command_redo,     NULL, NULL},
{"sep5",       NULL,                       NULL, NULL,             NULL, NULL},
{"find",       GTK_STOCK_FIND,             NULL, command_find,     NULL, NULL},
{"replace",    GTK_STOCK_FIND_AND_REPLACE, NULL, command_replace,  NULL, NULL},
{NULL,         NULL,                       NULL, NULL,             NULL, NULL}};


/******************************************************************************
*                                                                             *
* ja:メイン関数                                                               *
*                                                                             *
******************************************************************************/
int
main (int   argc,
      char *argv[])
{
  gboolean *arg_files, success[12];
  const gchar *user_config;
  gsize length;
  gint i, count = 0, init_line = 0, license = -1;
  GKeyFile *key_file;
  GList *glist;
  GObject *accel, *winmenu;
  GtkAccelGroup *accel_group;
  GtkStyle *style;
  GtkWidget *handle, *label, *table, *vbox, *frame;
  /* en:Drag and Drop */
  GtkTargetEntry drag_types[1] = {{"text/uri-list", 0, 0}};
  /* ja:コマンドライン */
  gint array[2], def[2];
  ArgumentBool arg_n, arg_p, arg_x;
  ArgumentValue arg_h, arg_l;
  ArgumentArray arg_s;
  ArgumentExit arg_g, arg_v;
  ArgumentHelp arg_q;
  ArgumentList arg_opts[] = {
{"history", 'h',  "NUM",          N_("History"),                    ARGUMENT_TYPE_VALUE, &arg_h},
{"line",    'l',  "NUM",          N_("Caret Position"),             ARGUMENT_TYPE_VALUE, &arg_l},
{"new",     'n',  "ON/OFF",       N_("New File"),                   ARGUMENT_TYPE_BOOL,  &arg_n},
{"profile", 'p',  "ON/OFF",       N_("Profile"),                    ARGUMENT_TYPE_BOOL,  &arg_p},
{"size",    's',  "WIDTH,HEIGHT", N_("Window Size"),                ARGUMENT_TYPE_ARRAY, &arg_s},
{"second",  'x',  "ON/OFF",       N_("Open Window"),                ARGUMENT_TYPE_BOOL,  &arg_x},
{"license", 'g',  NULL,           N_("GNU GENERAL PUBLIC LICENSE"), ARGUMENT_TYPE_EXIT,  &arg_g},
{"version", 'v',  NULL,           N_("About"),                      ARGUMENT_TYPE_EXIT,  &arg_v},
{"help",    '?',  NULL,           N_("Help"),                       ARGUMENT_TYPE_HELP,  &arg_q},
{NULL,      '\0', NULL,           NULL,                             0,                   NULL}};

  /* ja:初期化 */
  setlocale (LC_ALL, "");
  gtk_set_locale ();
  gtk_init (&argc, &argv);
  bindtextdomain (PACKAGE, misc_spec_locale (argv[0]));
  bind_textdomain_codeset (PACKAGE, "UTF-8");
  textdomain (PACKAGE);

  def_width = gdk_screen_width () * 2 / 3;
  def_height = gdk_screen_height () * 2 / 3;

  /* ja:ファイルの履歴 */
  history = orz_history_new ();

  /* ja:キーファイルを求める */
  user_config = misc_spec_config ("maid.org", "tmaid", "main");

  /* ja:レジストリから読み出す */
  key_file = g_key_file_new ();
  if (g_key_file_load_from_file (key_file, user_config, G_KEY_FILE_NONE, NULL))
    {
      gchar *utf8str;

      license = g_key_file_get_integer (key_file, "license", "hash", NULL);
      orz_history_set_num (ORZ_HISTORY (history),
                g_key_file_get_integer (key_file, "pref", "history", NULL));
      newfile = g_key_file_get_boolean (key_file, "pref", "newfile", NULL);
      second = g_key_file_get_boolean (key_file, "pref", "second", NULL);
      def_width = g_key_file_get_integer (key_file, "pref", "width", NULL);
      def_height = g_key_file_get_integer (key_file, "pref", "height", NULL);
      n_pos = g_key_file_get_integer (key_file, "pref", "npos", NULL);
      ftnum = g_key_file_get_integer (key_file, "tmaid", "ftnum", NULL);
#if GTK_CHECK_VERSION(2,4,0) || defined (G_OS_WIN32)
      findex = g_key_file_get_integer (key_file, "tmaid", "findex", NULL);
#endif /* GTK_CHECK_VERSION(2,4,0) || defined (G_OS_WIN32) */
      utf8str = g_key_file_get_string (key_file, "tmaid", "openpath", NULL);
      open_path = utf8str
                ? g_filename_from_utf8 (utf8str, -1, NULL, NULL, NULL) : NULL;
      g_free (utf8str);
      utf8str = g_key_file_get_string (key_file, "tmaid", "savepath", NULL);
      save_path = utf8str
                ? g_filename_from_utf8 (utf8str, -1, NULL, NULL, NULL) : NULL;
      g_free (utf8str);
      /* ja:ファイルの履歴 */
      for (i = orz_history_get_num (ORZ_HISTORY (history)) - 1; i >= 0; i--)
        {
          gchar *key;

          key = g_strdup_printf ("file%d", i);
          utf8str = g_key_file_get_string (key_file, "history", key, NULL);
          g_free (key);
          if (utf8str)
            {
              gchar *file;

              file = g_filename_from_utf8 (utf8str, -1, NULL, NULL, NULL);
              g_free (utf8str);
              orz_history_add_file (ORZ_HISTORY (history), file);
              g_free (file);
            }
        }
      /* ja:検索 */
      find_arrow = g_key_file_get_boolean (key_file, "find", "arrow", NULL);
      find_ignorecase = g_key_file_get_boolean (key_file,
                                                "find", "ignorecase", NULL);
      find_num = g_key_file_get_integer (key_file, "find", "num", NULL);
      for (i = 0; i < find_num; i++)
        {
          gchar *key;

          key = g_strdup_printf ("text%02d", i);
          find_text[i] = g_key_file_get_string (key_file, "find", key, NULL);
          g_free (key);
          if (!find_text[i])
            break;
        }
      find_num = i;
      /* ja:置換 */
      replace_arrow = g_key_file_get_boolean (key_file,
                                                    "replace", "arrow", NULL);
      replace_ignorecase = g_key_file_get_boolean (key_file,
                                                "replace", "ignorecase", NULL);
      replace_num = g_key_file_get_integer (key_file, "replace", "num", NULL);
      for (i = 0; i < replace_num; i++)
        {
          gchar *key;

          key = g_strdup_printf ("text%02d", i);
          replace_text[i] = g_key_file_get_string (key_file, "replace", key,
                                                                        NULL);
          g_free (key);
          if (!replace_text[i])
            break;
        }
      replace_num = i;
      /* ja:印刷 */
#ifdef G_OS_UNIX
      printer.mode = g_key_file_get_boolean (key_file,
                                                    "printer", "mode", NULL);
      utf8str = g_key_file_get_string (key_file, "printer", "printer", NULL);
      if (utf8str)
        {
          printer.printer
                        = g_filename_from_utf8 (utf8str, -1, NULL, NULL, NULL);
          g_free (utf8str);
        }
      if (!printer.printer)
        printer.printer = g_strdup ("lpr");
#endif /* G_OS_UNIX */
#if defined (G_OS_UNIX) || defined (G_OS_WIN32)
      printer.copies = g_key_file_get_integer (key_file, "printer", "copies",
                                                                        NULL);
#endif /* defined (G_OS_UNIX) || defined (G_OS_WIN32) */
#ifndef G_OS_WIN32
      utf8str = g_key_file_get_string (key_file, "printer", "file", NULL);
      if (utf8str)
        {
          printer.file = g_filename_from_utf8 (utf8str, -1, NULL, NULL, NULL);
          g_free (utf8str);
        }
      if (!printer.file)
        printer.file = g_strdup ("tmaid.ps");
      printer.width = g_key_file_get_integer (key_file, "printer", "width",
                                                                        NULL);
      printer.height = g_key_file_get_integer (key_file, "printer", "height",
                                                                        NULL);
      printer.left = g_key_file_get_integer (key_file, "printer", "left",
                                                                        NULL);
      printer.top = g_key_file_get_integer (key_file, "printer", "top", NULL);
      printer.right = g_key_file_get_integer (key_file, "printer", "right",
                                                                        NULL);
      printer.bottom = g_key_file_get_integer (key_file, "printer", "bottom",
                                                                        NULL);
#endif /* not G_OS_WIN32 */
      /* ja:ファイルタイプ */
      if (ftnum > 0)
        ftype = g_malloc (ftnum * sizeof (FileType));
      for (i = 0; i < ftnum; i++)
        {
          gchar *group_name;
          gint *list;

          group_name = g_strdup_printf ("filetype%04d", i);
          ftype[i].ft_id = tcount++;
          ftype[i].text = g_key_file_get_string (key_file, group_name, "text",
                                                                        NULL);
          if (!ftype[i].text)
            ftype[i].text = g_strdup ("UNKNOW");
          ftype[i].ext = g_key_file_get_string (key_file, group_name, "ext",
                                                                        NULL);
          if (!ftype[i].ext)
            ftype[i].ext = g_strdup ("*");
          ftype[i].crlf = g_key_file_get_integer (key_file, group_name, "crlf",
                                                                        NULL);
          ftype[i].margin = g_key_file_get_integer (key_file,
                                                group_name, "margin", NULL);
          ftype[i].tab = g_key_file_get_integer (key_file, group_name, "tab",
                                                                        NULL);
          ftype[i].writemode = g_key_file_get_integer (key_file,
                                                group_name, "writemode", NULL);
          ftype[i].associate = g_key_file_get_boolean (key_file,
                                                group_name, "associate", NULL);
          ftype[i].autoindent = g_key_file_get_boolean (key_file,
                                            group_name, "autoindent", NULL);
          ftype[i].code = g_key_file_get_boolean (key_file, group_name, "code",
                                                                        NULL);
          ftype[i].eof = g_key_file_get_boolean (key_file, group_name, "eof",
                                                                        NULL);
          ftype[i].hexcase = g_key_file_get_boolean (key_file,
                                                group_name, "hexcase", NULL);
          ftype[i].html = g_key_file_get_boolean (key_file, group_name, "html",
                                                                        NULL);
          ftype[i].esc = g_key_file_get_boolean (key_file, group_name, "esc",
                                                                        NULL);
          ftype[i].limit = g_key_file_get_boolean (key_file, 
                                                    group_name, "limit", NULL);
          ftype[i].overwrite = g_key_file_get_boolean (key_file,
                                                group_name, "overwrite", NULL);
          ftype[i].space = g_key_file_get_boolean (key_file, 
                                                    group_name, "space", NULL);
          ftype[i].syscol = g_key_file_get_boolean (key_file,
                                                group_name, "syscol", NULL);
          ftype[i].tabconv = g_key_file_get_boolean (key_file,
                                                group_name, "tabconv", NULL);
          ftype[i].gline = g_key_file_get_boolean (key_file, 
                                                group_name, "gline", NULL);
          ftype[i].mline = g_key_file_get_boolean (key_file, 
                                                group_name, "mline", NULL);
          ftype[i].nline = g_key_file_get_boolean (key_file, 
                                                group_name, "nline", NULL);
          ftype[i].uline = g_key_file_get_boolean (key_file, 
                                                group_name, "uline", NULL);
          ftype[i].vline = g_key_file_get_boolean (key_file, 
                                                group_name, "vline", NULL);
          /* ja:キャラクターセット */
          ftype[i].charset = g_key_file_get_string (key_file,
                                                group_name, "charset", NULL);
          if (ftype[i].charset)
            {
              gchar *tmp;

              tmp = charuty_strip (ftype[i].charset);
              g_free (ftype[i].charset);
              ftype[i].charset = tmp;
            }
          if (!ftype[i].charset)
            {
              const gchar *charset;

              ftype[i].charset = g_get_charset (&charset)
                                        ? g_strdup_printf ("%s,UTF-8", charset)
                                                        : g_strdup ("UTF-8");
            }
          /* ja:フォント */
          ftype[i].font_name = g_key_file_get_string (key_file,
                                                    group_name, "font", NULL);
          /* ja:色 */
          list = g_key_file_get_integer_list (key_file, group_name, "color",
                                                                &length, NULL);
          if (list && length * sizeof (gint) >= sizeof (GdkColor) * 12)
            {
              g_memmove (ftype[i].color, list, sizeof (GdkColor) * 12);
              g_free (list);
            }
          else
            {
              g_memmove (ftype[i].color, system_color, sizeof (GdkColor) * 12);
            }
          gdk_colormap_alloc_colors (gdk_colormap_get_system (),
                                    ftype[i].color, 12, FALSE, TRUE, success);
          g_free (group_name);
        }
    }
  else
    {
#ifdef G_OS_UNIX
      printer.mode = TRUE;
      printer.printer = g_strdup ("lpr");
#endif /* G_OS_UNIX */
#ifndef G_OS_WIN32
      printer.file = g_strdup ("tmaid.ps");
#endif /* not G_OS_WIN32 */
#if defined (G_OS_UNIX) || defined (G_OS_WIN32)
      printer.copies = 1;
#endif /* defined (G_OS_UNIX) || defined (G_OS_WIN32) */
#ifndef G_OS_WIN32
      printer.width = 594;
      printer.height = 840;
      printer.left = printer.top = printer.right = printer.bottom = 20;
#endif /* not G_OS_WIN32 */
    }
  g_key_file_free (key_file);

  if (license_hash (gnugpl) != license)
    license = license_dialog (gnugpl);
  if (license < 0)
    return 0;

  /* ja:コマンドライン */
  /* ja:ファイルの履歴 */
  arg_h.val = orz_history_get_num (ORZ_HISTORY (history));
  arg_h.def = ORZ_HISTORY_NUM_DEF;
  /* ja:初期のキャレット位置 */
  arg_l.val = init_line;
  arg_l.def = 0;
  /* ja:新規ファイル */
  arg_n.flag = newfile;
  arg_n.def = FALSE;
  /* ja:レジストリ */
  arg_p.flag = TRUE;
  arg_p.def = TRUE;
  /* ja:ウインドウサイズ */
  array[0] = def_width;
  array[1] = def_height;
  def[0] = gdk_screen_width () * 2 / 3;
  def[1] = gdk_screen_height () * 2 / 3;
  arg_s.leng = 2;
  arg_s.array = array;
  arg_s.def = def;
  /* ja:新規ウインドウ */
  arg_x.flag = second;
  arg_x.def = FALSE;
  /* ja:ライセンス */
  arg_g.str = gnugpl;
  /* ja:バージョン */
  arg_v.str =
TMAID_VERSION_NOTICE"\n"
TMAID_COPYRIGHT_NOTICE"\n"
"This is free software; see the source for copying conditions.  There is NO\n"
"warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n";
  /* ja:ヘルプ */
  arg_q.str = TMAID_COPYRIGHT_NOTICE"\n\n";
  /* ja:コマンドラインの解析 */
  arg_files = arg_analyse (argv, arg_opts);
  /* ja:ファイルの履歴 */
  orz_history_set_num (ORZ_HISTORY (history), arg_h.val);
  /* ja:初期のキャレット位置 */
  if (arg_l.val >= 0)
    init_line = arg_l.val;
  /* ja:新規ファイル */
  newfile = arg_n.flag;
  /* ja:レジストリ */
  if (!arg_p.flag)
    user_config = NULL;
  /* ja:ウインドウサイズ */
  def_width = array[0];
  def_height = array[1];
  /* ja:新規ウインドウ */
  second = arg_x.flag;

  /* ja:ファイルタイプ */
  if (ftnum <= 0)
    {
      const gchar *charset;

      ftnum = 1;
      ftype = g_malloc0 (ftnum * sizeof (FileType));
      ftype[0].text = g_strdup (_("All Files"));
      ftype[0].ext = g_strdup ("*");
      ftype[0].crlf = 3;
      ftype[0].margin = 80;
      ftype[0].tab = 8;
      ftype[0].associate = TRUE;
      ftype[0].syscol = TRUE;
      /* ja:キャラクターセット */
      ftype[0].charset = !g_get_charset (&charset) && charset
                ? g_strconcat (charset, ",UTF-8", NULL) : g_strdup ("UTF-8");
      /* ja:色 */
      g_memmove (ftype[0].color, system_color, sizeof (GdkColor) * 12);
      gdk_colormap_alloc_colors (gdk_colormap_get_system (),
                                    ftype[0].color, 12, FALSE, TRUE, success);
    }

  /* ja:メインウインドウ */
  window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title (GTK_WINDOW (window), "Text maid");
  gtk_window_set_default_size (GTK_WINDOW (window), def_width, def_height);
  glist = g_list_append (NULL, gdk_pixbuf_new_from_xpm_data (tmaid16_xpm));
  glist = g_list_append (glist, gdk_pixbuf_new_from_xpm_data (tmaid24_xpm));
  glist = g_list_append (glist, gdk_pixbuf_new_from_xpm_data (tmaid32_xpm));
  glist = g_list_append (glist, gdk_pixbuf_new_from_xpm_data (tmaid48_xpm));
  gtk_window_set_default_icon_list (glist);
  g_object_unref (G_OBJECT (g_list_nth_data (glist, 0)));
  g_object_unref (G_OBJECT (g_list_nth_data (glist, 1)));
  g_object_unref (G_OBJECT (g_list_nth_data (glist, 2)));
  g_object_unref (G_OBJECT (g_list_nth_data (glist, 3)));
  g_list_free (glist);
  g_signal_connect (G_OBJECT(window), "delete-event",
                                            G_CALLBACK (signal_delete), NULL);
  g_signal_connect (G_OBJECT (window), "destroy",
                                            G_CALLBACK (gtk_main_quit), NULL);
  gtk_widget_realize (window);
  /* en:Drag and Drop */
  gtk_drag_dest_set (window, GTK_DEST_DEFAULT_ALL,
                    drag_types, G_N_ELEMENTS (drag_types), GDK_ACTION_COPY);
  g_signal_connect (G_OBJECT (window), "drag-data-received",
                                G_CALLBACK (signal_drag_data_received), NULL);
  /* ja:クリップボード */
  atom_targets = gdk_atom_intern ("TARGETS", FALSE);
  atom_compound = gdk_atom_intern ("COMPOUND_TEXT", FALSE);
  atom_text = gdk_atom_intern ("TEXT", FALSE);
  atom_utf8 = gdk_atom_intern ("UTF8_STRING", FALSE);
  gtk_selection_add_targets (window, GDK_SELECTION_PRIMARY,
                                            targets, G_N_ELEMENTS (targets));
  gtk_selection_add_targets (window, GDK_SELECTION_CLIPBOARD,
                                            targets, G_N_ELEMENTS (targets));
  g_signal_connect (G_OBJECT (window), "selection-clear-event",
                                    G_CALLBACK (signal_selection_clear), NULL);
  g_signal_connect (G_OBJECT (window), "selection-get",
                                    G_CALLBACK (signal_selection_get), NULL);
  g_signal_connect (G_OBJECT (window), "selection-received",
                                G_CALLBACK (signal_selection_received), NULL);
  /* en:MDI */
  mdi = orz_mdi_new ();
  gtk_notebook_set_scrollable (GTK_NOTEBOOK (mdi), TRUE);
  gtk_notebook_set_tab_pos (GTK_NOTEBOOK (mdi), n_pos);
  gtk_drag_dest_set (mdi, GTK_DEST_DEFAULT_ALL,
                    drag_types, G_N_ELEMENTS (drag_types), GDK_ACTION_COPY);
  g_signal_connect (G_OBJECT (mdi), "drag-data-received",
                                G_CALLBACK (signal_drag_data_received), NULL);
  g_signal_connect (G_OBJECT (mdi), "close-window",
                                G_CALLBACK (signal_close_window), NULL);
  g_signal_connect (G_OBJECT (mdi), "switch-window",
                                G_CALLBACK (signal_switch_window), NULL);
  g_signal_connect (GTK_OBJECT (mdi), "destroy",
                                G_CALLBACK (signal_destroy_mdi), NULL);
  /* ja:ポップアップ */
  accel_group = gtk_accel_group_new ();
  misc_create_menu (popup_entries, accel_group);
  /* ja:メニュー */
  accel_group = gtk_accel_group_new ();
  misc_create_menu (menu_entries, accel_group);
  /* ja:ファイルの履歴 */
  for (i = 0; menu_entries[i].path; i++)
#if GLIB_CHECK_VERSION(2,2,0)
    if (g_str_has_prefix (menu_entries[i].path, "/file/"))
#else /* not GLIB_CHECK_VERSION(2,2,0) */
    if (g_strncmp (menu_entries[i].path, "/file/", 6) == 0)
#endif /* not GLIB_CHECK_VERSION(2,2,0) */
      count++;
  orz_history_set_menu (ORZ_HISTORY (history),
                        misc_find_menu (menu_entries, "/file/"), count - 3);
  g_signal_connect (ORZ_HISTORY (history), "activate",
                            G_CALLBACK (signal_activate_menu_history), NULL);
  /* ja:ウインドウメニュー */
  winmenu = orz_winmenu_new ();
  orz_winmenu_set_mdi (ORZ_WINMENU (winmenu), ORZ_MDI (mdi));
  orz_winmenu_set_menu (ORZ_WINMENU (winmenu),
                                misc_find_menu (menu_entries, "/window/"), 0);
  /* ja:アクセルグループ */
  accel = orz_accel_new ();
  orz_accel_set_window (ORZ_ACCEL (accel), GTK_WINDOW (window));
  orz_accel_add_accel_group (ORZ_ACCEL (accel), accel_group);
  /* ja:テーブル */
  table = gtk_table_new (1, 5, FALSE);
  /* ja:ステータスバー */
  status = gtk_statusbar_new ();
  /* ja:カーソル座標 */
  label_cursor = gtk_label_new ("");
  gtk_label_set_justify (GTK_LABEL (label_cursor), GTK_JUSTIFY_CENTER);
  frame = gtk_frame_new (NULL);
  gtk_container_add( GTK_CONTAINER (frame), label_cursor);
  /* ja:テキスト */
  label = gtk_label_new ("");
  g_signal_connect (G_OBJECT (label), "style-set",
                                        G_CALLBACK (signal_style_set), NULL);
  /* ja:垂直ボックス/ハンドルボックス */
  vbox = gtk_vbox_new (FALSE, 0);
  handle = gtk_handle_box_new ();
  gtk_container_set_border_width (GTK_CONTAINER (handle), 2);
  gtk_container_add (GTK_CONTAINER (handle),
                                        misc_find_menu (menu_entries, "/"));
  gtk_box_pack_start (GTK_BOX (vbox), handle, FALSE, FALSE, 0);
  handle = gtk_handle_box_new ();
  gtk_container_set_border_width (GTK_CONTAINER (handle), 2);
  gtk_container_add (GTK_CONTAINER (handle),
                                        misc_create_toolbar (toolbar_entries));
  gtk_box_pack_start (GTK_BOX (vbox), handle, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), mdi, TRUE, TRUE, 0);
  gtk_table_attach (GTK_TABLE (table), frame, 0, 1, 0, 1,
            GTK_EXPAND | GTK_SHRINK | GTK_FILL, GTK_SHRINK | GTK_FILL, 0, 0);
  gtk_table_attach (GTK_TABLE (table), status, 1, 5, 0, 1,
            GTK_EXPAND | GTK_SHRINK | GTK_FILL, GTK_SHRINK | GTK_FILL, 0, 0);
  gtk_box_pack_end (GTK_BOX (vbox), table, FALSE, FALSE, 0);
  gtk_box_pack_end (GTK_BOX (vbox), label, FALSE, FALSE, 0);
  gtk_container_add (GTK_CONTAINER (window), vbox);
  /* ja:タイマ */
  timer_id = g_timeout_add (500, signal_timeout, NULL);

  /* ja:メニュー表示 */
  set_menu_bar (NULL);

  /* ja:既に起動しているかチェックする */
  if (!second)
    {
      GObject *comm;

      comm = orz_comm_new (window, "Text maid InterProcess communication");
      if (!orz_comm_is_first (ORZ_COMM (comm)))
        {
          orz_comm_send_with_files (ORZ_COMM (comm), argc, argv, arg_files);
          return 0;
        }
      g_signal_connect (G_OBJECT (comm), "recv",
                                    G_CALLBACK (signal_selection_recv), NULL);
    }

  /* ja:表示 */
  gtk_widget_show_all (window);
  gtk_widget_hide_all (label);

  /* ja:システム色 */
  style = gtk_widget_get_style (label);
  system_color[0] = style->text[0];
  system_color[1] = style->base[0];
  system_color[2] = style->dark[0];
  system_color[3] = style->dark[0];
  system_color[4] = style->dark[0];
  system_color[5] = style->dark[0];
  system_color[6] = style->dark[0];
  system_color[7] = style->mid[0];
  system_color[8] = style->fg[3];
  system_color[9] = style->bg[3];
  gdk_colormap_alloc_colors (gdk_colormap_get_system (),
                                    system_color, 12, FALSE, TRUE, success);
  /* ja:システムフォント */
  system_font = pango_font_description_copy
                                    (pango_context_get_font_description
                                    (gtk_widget_get_pango_context (label)));

  /* ja:新規ウインドウ */
  count = 0;
  for (i = 1; i < argc; i++)
    if (arg_files[i])
      {
        TmaidWindow *tmaid;

        tmaid = file_open_edit (argv[i]);
        if (init_line != 0)/* ja:指定行に移動 */
          jump_operation (tmaid, init_line);
        count++;
      }
  g_free (arg_files);
  if (newfile && count <= 0)
    file_open_edit (NULL);

  gtk_main ();

  /* ja:レジストリに書き込む */
  if (user_config)
    {
      gsize length;
      gchar *data, *group_name, *utf8str;

      key_file = g_key_file_new ();
      g_key_file_load_from_file (key_file, user_config, G_KEY_FILE_NONE, NULL);
      g_key_file_set_integer (key_file, "license", "hash", license);
      g_key_file_set_integer (key_file, "pref", "history",
                                orz_history_get_num (ORZ_HISTORY (history)));
      g_key_file_set_boolean (key_file, "pref", "newfile", newfile);
      g_key_file_set_boolean (key_file, "pref", "second", second);
      g_key_file_set_integer (key_file, "pref", "width", def_width);
      g_key_file_set_integer (key_file, "pref", "height", def_height);
      g_key_file_set_integer (key_file, "pref", "npos", n_pos);
      g_key_file_set_integer (key_file, "tmaid", "ftnum", ftnum);
#if GTK_CHECK_VERSION(2,4,0) || defined (G_OS_WIN32)
      for (i = ftnum - 1; i >= 0; i--)
        if (ftype[i].ft_id == findex)
          break;
      g_key_file_set_integer (key_file, "tmaid", "findex", i);
#endif /* GTK_CHECK_VERSION(2,4,0) || defined (G_OS_WIN32) */
      if (open_path)
        {
          utf8str = g_filename_to_utf8 (open_path, -1, NULL, NULL, NULL);
          if (utf8str)
            {
              g_key_file_set_string (key_file, "tmaid", "openpath", utf8str);
              g_free (utf8str);
            }
        }
      if (save_path)
        {
          utf8str = g_filename_to_utf8 (save_path, -1, NULL, NULL, NULL);
          if (utf8str)
            {
              g_key_file_set_string (key_file, "tmaid", "savepath", utf8str);
              g_free (utf8str);
            }
        }
      /* ja:ファイルの履歴 */
      glist = orz_history_get_file (ORZ_HISTORY (history));
      for (i = 0; i < g_list_length (glist); i++)
        {
          gchar *key;

          key = g_strdup_printf ("file%d", i);
          utf8str = g_filename_to_utf8 (g_list_nth_data (glist, i), -1,
                                                            NULL, NULL, NULL);
          if (utf8str)
            {
              g_key_file_set_string (key_file, "history", key, utf8str);
              g_free (utf8str);
            }
          g_free (key);
        }
      g_list_free (glist);
      while (i <= ORZ_HISTORY_NUM_MAX)
        {
          gchar *key;

          key = g_strdup_printf ("file%d", i++);
          g_key_file_remove_key (key_file, "history", key, NULL);
          g_free (key);
        }
      /* ja:検索 */
      g_key_file_set_boolean (key_file, "find", "arrow", find_arrow);
      g_key_file_set_boolean (key_file, "find", "ignorecase", find_ignorecase);
      g_key_file_set_integer (key_file, "find", "num", find_num);
      for (i = 0; i < find_num; i++)
        {
          gchar *key;

          key = g_strdup_printf ("text%02d", i);
          g_key_file_set_string (key_file, "find", key, find_text[i]);
          g_free (key);
        }
      while (i < 32)
        {
          gchar *key;

          key = g_strdup_printf ("text%02d", i++);
          g_key_file_set_string (key_file, "find", key, "");
          g_free (key);
        }
      /* ja:置換 */
      g_key_file_set_boolean (key_file, "replace", "arrow", replace_arrow);
      g_key_file_set_boolean (key_file, "replace", "ignorecase",
                                                        replace_ignorecase);
      g_key_file_set_integer (key_file, "replace", "num", replace_num);
      for (i = 0; i < replace_num; i++)
        {
          gchar *key;

          key = g_strdup_printf ("text%02d", i);
          g_key_file_set_string (key_file, "replace", key, replace_text[i]);
          g_free (key);
        }
      while (i < 32)
        {
          gchar *key;

          key = g_strdup_printf ("text%02d", i++);
          g_key_file_set_string (key_file, "replace", key, "");
          g_free (key);
        }
      /* ja:印刷 */
#ifdef G_OS_UNIX
      g_key_file_set_boolean (key_file, "printer", "mode", printer.mode);
      utf8str = g_filename_to_utf8 (printer.printer, -1, NULL, NULL, NULL);
      if (utf8str)
        {
          g_key_file_set_string (key_file, "printer", "printer", utf8str);
          g_free (utf8str);
        }
#endif /* not G_OS_UNIX */
#if defined (G_OS_UNIX) || defined (G_OS_WIN32)
      g_key_file_set_integer (key_file, "printer", "copies", printer.copies);
#endif /* defined (G_OS_UNIX) || defined (G_OS_WIN32) */
#ifndef G_OS_WIN32
      utf8str = g_filename_to_utf8 (printer.file, -1, NULL, NULL, NULL);
      if (utf8str)
        {
          g_key_file_set_string (key_file, "printer", "file", utf8str);
          g_free (utf8str);
        }
      g_key_file_set_integer (key_file, "printer", "width", printer.width);
      g_key_file_set_integer (key_file, "printer", "height", printer.height);
      g_key_file_set_integer (key_file, "printer", "left", printer.left);
      g_key_file_set_integer (key_file, "printer", "top", printer.top);
      g_key_file_set_integer (key_file, "printer", "right", printer.right);
      g_key_file_set_integer (key_file, "printer", "bottom", printer.bottom);
#endif /* not G_OS_WIN32 */
      /* ja:ファイルタイプ */
      for (i = 0; i < ftnum; i++)
        {
          gint list[(sizeof (GdkColor) * 12 + sizeof (gint) - 1)
                                                            / sizeof (gint)];

          group_name = g_strdup_printf ("filetype%04d", i);
          g_key_file_set_string (key_file, group_name, "text", ftype[i].text);
          g_key_file_set_string (key_file, group_name, "ext", ftype[i].ext);
          g_key_file_set_integer (key_file, group_name, "crlf", ftype[i].crlf);
          g_key_file_set_integer (key_file, group_name, "margin",
                                                            ftype[i].margin);
          g_key_file_set_integer (key_file, group_name, "tab", ftype[i].tab);
          g_key_file_set_integer (key_file, group_name, "writemode",
                                                        ftype[i].writemode);
          g_key_file_set_boolean (key_file, group_name, "associate",
                                                        ftype[i].associate);
          g_key_file_set_boolean (key_file, group_name, "autoindent",
                                                        ftype[i].autoindent);
          g_key_file_set_boolean (key_file, group_name, "code", ftype[i].code);
          g_key_file_set_boolean (key_file, group_name, "eof", ftype[i].eof);
          g_key_file_set_boolean (key_file, group_name, "hexcase",
                                                            ftype[i].hexcase);
          g_key_file_set_boolean (key_file, group_name, "html", ftype[i].html);
          g_key_file_set_boolean (key_file, group_name, "esc", ftype[i].esc);
          g_key_file_set_boolean (key_file, group_name, "limit",
                                                            ftype[i].limit);
          g_key_file_set_boolean (key_file, group_name, "overwrite",
                                                        ftype[i].overwrite);
          g_key_file_set_boolean (key_file, group_name, "space",
                                                            ftype[i].space);
          g_key_file_set_boolean (key_file, group_name, "syscol",
                                                            ftype[i].syscol);
          g_key_file_set_boolean (key_file, group_name, "tabconv",
                                                            ftype[i].tabconv);
          g_key_file_set_boolean (key_file, group_name, "gline",
                                                            ftype[i].gline);
          g_key_file_set_boolean (key_file, group_name, "mline",
                                                            ftype[i].mline);
          g_key_file_set_boolean (key_file, group_name, "nline",
                                                            ftype[i].nline);
          g_key_file_set_boolean (key_file, group_name, "uline",
                                                            ftype[i].uline);
          g_key_file_set_boolean (key_file, group_name, "vline",
                                                            ftype[i].vline);
          /* ja:キャラクターセット */
          g_key_file_set_string (key_file, group_name, "charset",
                                                            ftype[i].charset);
          /* ja:フォント */
          if (ftype[i].font_name)
            g_key_file_set_string (key_file, group_name, "font",
                                                        ftype[i].font_name);
          /* ja:色 */
          list[(sizeof (GdkColor) * 12 + sizeof (gint) - 1)
                                                    / sizeof (gint) - 1] = 0;
          g_memmove (list, ftype[i].color, sizeof (GdkColor) * 12);
          g_key_file_set_integer_list (key_file, group_name, "color", list,
                (sizeof (GdkColor) * 12 + sizeof (gint) - 1) / sizeof (gint));
          g_free (group_name);
        }
      while (group_name = g_strdup_printf ("filetype%04d", i++),
                                g_key_file_has_group (key_file, group_name))
        {
          g_key_file_remove_group (key_file, group_name, NULL);
          g_free (group_name);
        }
      g_free (group_name);
      data = g_key_file_to_data (key_file, &length, NULL);
      g_key_file_free (key_file);
      if (data)
        {
          fileio_save (user_config, data, length);
          g_free (data);
        }
    }

  pango_font_description_free (system_font);
  g_free (clipboard_text);
  g_free (open_path);
  g_free (save_path);
#ifdef G_OS_UNIX
  g_free (printer.printer);
#endif /* G_OS_UNIX */
#ifndef G_OS_WIN32
  g_free (printer.file);
#endif /* not G_OS_WIN32 */
  for (i = 0; i < find_num; i++)
    g_free (find_text[i]);
  for (i = 0; i < replace_num; i++)
    g_free (replace_text[i]);
  for (i = 0; i < ftnum; i++)
    {
      g_free (ftype[i].text);
      g_free (ftype[i].ext);
      g_free (ftype[i].font_name);
    }
  g_free (ftype);
  return 0;
}
