/*
    wcommon
    copyright (c) 1998-2018 Kazuki Iwamoto https://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "wcommon.h"
#include <shlwapi.h>
#include <shlobj.h>


/******************************************************************************
*                                                                             *
******************************************************************************/
typedef struct _ENUMNETRSRCW {
  int nPath, nRsrc;
  LPWSTR lpszPath, lpszRsrc;
} ENUMNETRSRCW, *LPENUMNETRSRCW;


static VOID WINAPI
EnumNetResourceW (LPNETRESOURCEW lpNetRsrc,
                  LPENUMNETRSRCW lpEnumNetRsrc)
{
  HANDLE hEnum;

  if (WNetOpenEnumW (RESOURCE_CONNECTED, RESOURCETYPE_ANY,
                                            0, lpNetRsrc, &hEnum) == NO_ERROR)
    {
      DWORD dwResult, dwBytes = 16 * 1024;

      do
        {
          DWORD dwCount;
          LPNETRESOURCEW lpNetRsrcLocal;

          dwCount = 1;
          lpNetRsrcLocal = MemoryAlloc (dwBytes);
          dwResult = WNetEnumResourceW (hEnum, &dwCount,
                                                    lpNetRsrcLocal, &dwBytes);
          switch (dwResult)
            {
              case ERROR_MORE_DATA:
                dwBytes *= 2;
                break;
              case NO_ERROR:
                if (lpNetRsrcLocal->lpRemoteName
                 && lpNetRsrcLocal->lpRemoteName[0] == '\\'
                 && lpNetRsrcLocal->lpRemoteName[1] == '\\')
                  {
                    int n;
                    LPWSTR pszEnd;

                    n = lstrlenW (lpNetRsrcLocal->lpRemoteName);
                    pszEnd = lpNetRsrcLocal->lpRemoteName + n;
                    while (pszEnd && n > lpEnumNetRsrc->nRsrc)
                      {
                        if (lpEnumNetRsrc->nPath > n
                                        && (lpEnumNetRsrc->lpszPath[n] == '\0'
                                         || lpEnumNetRsrc->lpszPath[n] == '\\')
                            && StrCmpNIW (lpEnumNetRsrc->lpszPath,
                                      lpNetRsrcLocal->lpRemoteName, n) == 0)
                          {
                            MemoryFree (lpEnumNetRsrc->lpszRsrc);
                            lpEnumNetRsrc->nRsrc = n;
                            lpEnumNetRsrc->lpszRsrc = StringDuplicateExW
                                            (lpNetRsrcLocal->lpRemoteName, n);
                            break;
                          }
                        pszEnd = StrRChrW (lpNetRsrcLocal->lpRemoteName,
                                                                pszEnd, '\\');
                        n = pszEnd - lpNetRsrcLocal->lpRemoteName;
                      }
                  }
                if (lpNetRsrcLocal->dwUsage & RESOURCEUSAGE_CONTAINER)
                  EnumNetResourceW (lpNetRsrcLocal, lpEnumNetRsrc);
            }
          MemoryFree (lpNetRsrcLocal);
        }
      while (dwResult == NO_ERROR || dwResult == ERROR_MORE_DATA);
      WNetCloseEnum (hEnum);
    }
}


/*  ja:ロングファイル名を取得する
    lpszShortName,ファイル名
              RET,ロングファイル名,NULL:エラー                              */
LPTSTR WINAPI
GetLongFileNewW (LPCWSTR lpszShortName)
{
  DWORD dwFile = MAX_PATH, dwSize;
  LPWSTR lpszFile = NULL, lpszLongName;

  do
    {
      LPWSTR p;

      dwSize = dwFile;
      lpszFile = MemoryReAlloc (lpszFile, dwSize * sizeof (WCHAR));
      dwFile = GetFullPathNameW (lpszShortName, dwSize, lpszFile, &p);
    }
  while (dwFile != 0 && dwFile >= dwSize);
  if (dwFile == 0)
    {
      lpszLongName = StringDuplicateW (lpszShortName);
    }
  else
    {
      int nIgnore = 0;
      LPCWSTR p;
      LPWSTR q;

      lpszLongName = MemoryAlloc ((dwFile + 1) * sizeof (WCHAR));
      p = lpszFile;
      q = lpszLongName;
      if (dwFile >= 2)
        {
          if (IsCharAlphaW (lpszFile[0]) && lpszFile[1] == ':')
            {
              *q++ = (WCHAR)CharUpper ((LPWSTR)*p++);
              *q++ = *p++;
            }
          else if (lpszFile[0] == '\\' && lpszFile[1] == '\\')
            {
              ENUMNETRSRCW EnumNetRsec;

              EnumNetRsec.nPath = lstrlenW (lpszFile);
              EnumNetRsec.lpszPath = lpszFile;
              EnumNetRsec.nRsrc = 2;
              EnumNetRsec.lpszRsrc = NULL;
              EnumNetResourceW (NULL, &EnumNetRsec);
              if (EnumNetRsec.nRsrc > 2)
                {
                  int i;
                  LPCTSTR r;

                  nIgnore = 3;
                  r = EnumNetRsec.lpszRsrc;
                  while (*r != '\0')
                    {
                      if (*r == '\\')
                        nIgnore--;
                      *q++ = *r++;
                    }
                  p += EnumNetRsec.nRsrc;
                }
              else
                {
                  nIgnore = 2;
                }
              MemoryFree (EnumNetRsec.lpszRsrc);
            }
        }
      while (*p == '\\')
        *q++ = *p++;
      while (*p != '\0')
        {
          LPWSTR r;

          r = q;
          while (*p != '\0' && *p != '*' && *p != '\?' && *p != '\\')
            *q++ = *p++;
          if (*p == '*' || *p == '\?')
            {
              while (*p != '\0')
                *q++ = *p++;
            }
          else
            {
              if (--nIgnore < 0)
                {
                  HANDLE hFind;
                  WIN32_FIND_DATAW rcf;

                  hFind = FindFirstFileW (lpszLongName, &rcf);
                  if (hFind != INVALID_HANDLE_VALUE)
                    {
                      FindClose (hFind);
                      if (lstrcmpW (rcf.cFileName, L".") != 0)
                        {
                          int delta;

                          delta = lstrlenW (rcf.cFileName) - (q - r);
                          *r = '\0';
                          if (delta != 0)
                            {
                              dwFile += delta;
                              lpszLongName = MemoryReAlloc (lpszLongName,
                                                (dwFile + 1) * sizeof (WCHAR));
                            }
                          lstrcatW (lpszLongName, rcf.cFileName);
                          q = lpszLongName + lstrlenW (lpszLongName);
                        }
                    }
                }
              if (*p == '\\')
                *q++ = *p++;
            }
        }
    }
  MemoryFree (lpszFile);
  return lpszLongName;
}


typedef struct _ENUMNETRSRCA {
  int nPath, nRsrc;
  LPSTR lpszPath, lpszRsrc;
} ENUMNETRSRCA, *LPENUMNETRSRCA;


static VOID WINAPI
EnumNetResourceA (LPNETRESOURCEA lpNetRsrc,
                  LPENUMNETRSRCA lpEnumNetRsrc)
{
  HANDLE hEnum;

  if (WNetOpenEnumA (RESOURCE_CONNECTED, RESOURCETYPE_ANY,
                                            0, lpNetRsrc, &hEnum) == NO_ERROR)
    {
      DWORD dwResult, dwBytes = 16 * 1024;

      do
        {
          DWORD dwCount;
          LPNETRESOURCEA lpNetRsrcLocal;

          dwCount = 1;
          lpNetRsrcLocal = MemoryAlloc (dwBytes);
          dwResult = WNetEnumResourceA (hEnum,
                                        &dwCount,
                                        lpNetRsrcLocal,
                                        &dwBytes);
          switch (dwResult)
            {
              case ERROR_MORE_DATA:
                dwBytes *= 2;
                break;
              case NO_ERROR:
                if (lpNetRsrcLocal->lpRemoteName
                 && lpNetRsrcLocal->lpRemoteName[0] == '\\'
                 && lpNetRsrcLocal->lpRemoteName[1] == '\\')
                  {
                    int n;
                    LPSTR pszEnd;

                    n = lstrlenA (lpNetRsrcLocal->lpRemoteName);
                    pszEnd = lpNetRsrcLocal->lpRemoteName + n;
                    while (pszEnd && n > lpEnumNetRsrc->nRsrc)
                      {
                        if (lpEnumNetRsrc->nPath > n
                                        && (lpEnumNetRsrc->lpszPath[n] == '\0'
                                         || lpEnumNetRsrc->lpszPath[n] == '\\')
                            && StrCmpNIA (lpEnumNetRsrc->lpszPath,
                                      lpNetRsrcLocal->lpRemoteName, n) == 0)
                          {
                            MemoryFree (lpEnumNetRsrc->lpszRsrc);
                            lpEnumNetRsrc->nRsrc = n;
                            lpEnumNetRsrc->lpszRsrc = StringDuplicateExA
                                            (lpNetRsrcLocal->lpRemoteName, n);
                            break;
                          }
                        pszEnd = StrRChrA (lpNetRsrcLocal->lpRemoteName,
                                                                pszEnd, '\\');
                        n = pszEnd - lpNetRsrcLocal->lpRemoteName;
                      }
                  }
                if (lpNetRsrcLocal->dwUsage & RESOURCEUSAGE_CONTAINER)
                  EnumNetResourceA (lpNetRsrcLocal, lpEnumNetRsrc);
            }
          MemoryFree (lpNetRsrcLocal);
        }
      while (dwResult == NO_ERROR || dwResult == ERROR_MORE_DATA);
      WNetCloseEnum (hEnum);
    }
}


/*  ja:ロングファイル名を取得する
    lpszShortName,ファイル名
              RET,ロングファイル名,NULL:エラー                              */
LPSTR WINAPI
GetLongFileNewA (LPCSTR lpszShortName)
{
  DWORD dwFile = MAX_PATH, dwSize;
  LPSTR lpszFile = NULL, lpszLongName;

  do
    {
      LPSTR p;

      dwSize = dwFile;
      lpszFile = MemoryReAlloc (lpszFile, dwSize * sizeof (CHAR));
      dwFile = GetFullPathNameA (lpszShortName, dwSize, lpszFile, &p);
    }
  while (dwFile != 0 && dwFile >= dwSize);
  if (dwFile == 0)
    {
      lpszLongName = StringDuplicateA (lpszShortName);
    }
  else
    {
      int nIgnore = 0;
      LPCSTR p;
      LPSTR q;

      lpszLongName = MemoryAlloc ((dwFile + 1) * sizeof (CHAR));
      p = lpszFile;
      q = lpszLongName;
      if (dwFile >= 2)
        {
          if (IsCharAlphaA (lpszFile[0]) && lpszFile[1] == ':')
            {
              *q++ = (CHAR)CharUpperA ((LPSTR)*p++);
              *q++ = *p++;
            }
          else if (lpszFile[0] == '\\' && lpszFile[1] == '\\')
            {
              ENUMNETRSRCA EnumNetRsec;

              EnumNetRsec.nPath = lstrlenA (lpszFile);
              EnumNetRsec.lpszPath = lpszFile;
              EnumNetRsec.nRsrc = 2;
              EnumNetRsec.lpszRsrc = NULL;
              EnumNetResourceA (NULL, &EnumNetRsec);
              if (EnumNetRsec.nRsrc > 2)
                {
                  int i;
                  LPCSTR r;

                  nIgnore = 3;
                  r = EnumNetRsec.lpszRsrc;
                  while (*r != '\0')
                    {
                      if (*r == '\\')
                        nIgnore--;
                      else if (IsDBCSLeadByteEx (CP_ACP, *r))
                        *q++ = *r++;
                      *q++ = *r++;
                    }
                  p += EnumNetRsec.nRsrc;
                }
              else
                {
                  nIgnore = 2;
                }
              MemoryFree (EnumNetRsec.lpszRsrc);
            }
        }
      while (*p == '\\')
        *q++ = *p++;
      while (*p != '\0')
        {
          LPSTR r;

          r = q;
          while (*p != '\0' && *p != '*' && *p != '\?' && *p != '\\')
            {
              if (IsDBCSLeadByteEx (CP_ACP, *p))
                *q++ = *p++;
              *q++ = *p++;
            }
          if (*p == '*' || *p == '\?')
            {
              while (*p != '\0')
                *q++ = *p++;
            }
          else
            {
              if (--nIgnore < 0)
                {
                  HANDLE hFind;
                  WIN32_FIND_DATAA rcf;

                  hFind = FindFirstFileA (lpszLongName, &rcf);
                  if (hFind != INVALID_HANDLE_VALUE)
                    {
                      FindClose (hFind);
                      if (lstrcmpA (rcf.cFileName, ".") != 0)
                        {
                          int delta;

                          delta = lstrlenA (rcf.cFileName) - (q - r);
                          *r = '\0';
                          if (delta != 0)
                            {
                              dwFile += delta;
                              lpszLongName = MemoryReAlloc (lpszLongName,
                                                (dwFile + 1) * sizeof (CHAR));
                            }
                          lstrcatA (lpszLongName, rcf.cFileName);
                          q = lpszLongName + lstrlenA (lpszLongName);
                        }
                    }
                }
              if (*p == '\\')
                *q++ = *p++;
            }
        }
    }
  MemoryFree (lpszFile);
  return lpszLongName;
}


/*  ja:特定のフォルダを取得する
    rfid,KNOWNFOLDERID
     RET,パス,NULL:エラー                                                   */
LPWSTR WINAPI
GetSpecificFolderPathW (REFKNOWNFOLDERID rfid)
{
  LPWSTR lpszResult, lpszPath = NULL;

  if (SUCCEEDED (SHGetKnownFolderPath (rfid, KF_FLAG_DEFAULT, NULL, &lpszPath)))
    {
      lpszResult = StringDuplicate (lpszPath);
      CoTaskMemFree (lpszPath);
    }
  else
    {
      lpszResult = NULL;
    }
  return lpszResult;
}


/*  ja:特定のフォルダを取得する
    rfid,KNOWNFOLDERID
     RET,パス,NULL:エラー                                                   */
LPSTR WINAPI
GetSpecificFolderPathA (REFKNOWNFOLDERID rfid)
{
  LPSTR lpszResult;
  LPWSTR lpszPath = NULL;

  if (SUCCEEDED (SHGetKnownFolderPath (rfid, KF_FLAG_DEFAULT, NULL, &lpszPath)))
    {
      lpszResult = StringWideCharToMultiByte (lpszPath);
      CoTaskMemFree (lpszPath);
    }
  else
    {
      lpszResult = NULL;
    }
  return lpszResult;
}


/*  ja:モジュールのファイル名を取得する
    hModule,モジュールのハンドル
        RET,ファイル名,NULL:エラー                                          */
LPWSTR WINAPI
GetModuleFileNameNewW (HMODULE hModule)
{
  int nResult;
  SSIZE_T nSize = 0;
  LPWSTR lpszModule = NULL;

  do
    {
      nSize += MAX_PATH;
      lpszModule = MemoryReAlloc (lpszModule, nSize * sizeof (WCHAR));
      nResult = GetModuleFileNameW (hModule, lpszModule, nSize);
    }
  while (nResult >= nSize - 1);
  if (nResult > 0)
    {
      lpszModule = MemoryReAlloc (lpszModule, (nResult + 1) * sizeof (WCHAR));
    }
  else
    {
      MemoryFree (lpszModule);
      lpszModule = NULL;
    }
  return lpszModule;
}


/*  ja:モジュールのファイル名を取得する
    hModule,モジュールのハンドル
        RET,ファイル名,NULL:エラー                                          */
LPSTR WINAPI
GetModuleFileNameNewA (HMODULE hModule)
{
  int nResult;
  SSIZE_T nSize = 0;
  LPSTR lpszModule = NULL;

  do
    {
      nSize += MAX_PATH;
      lpszModule = MemoryReAlloc (lpszModule, nSize * sizeof (CHAR));
      nResult = GetModuleFileNameA (hModule, lpszModule, nSize);
    }
  while (nResult >= nSize - 1);
  if (nResult > 0)
    {
      lpszModule = MemoryReAlloc (lpszModule, (nResult + 1) * sizeof (CHAR));
    }
  else
    {
      MemoryFree (lpszModule);
      lpszModule = NULL;
    }
  return lpszModule;
}


/*  ja:パスの拡張子を変更する
    lpszPath,パス
     lpszExt,新たな拡張子(NULL:拡張子削除)
         RET,パス,NULL:エラー                                               */
LPWSTR WINAPI
PathRenameExtensionNewW (LPCWSTR lpszPath,
                         LPCWSTR lpszExt)
{
  LPWSTR lpszResult;

  if (lpszPath)
    {
      int nLength;

      nLength = PathFindExtensionW (lpszPath) - lpszPath;
      lpszResult = MemoryAlloc ((nLength + lstrlenW (lpszExt) + 1)
                                                            * sizeof (WCHAR));
      MemoryCopy (lpszResult, lpszPath, nLength * sizeof (WCHAR));
      lstrcatW (lpszResult, lpszExt);
    }
  else
    {
      lpszResult = NULL;
    }
  return lpszResult;
}


/*  ja:パスの拡張子を変更する
    lpszPath,パス
     lpszExt,新たな拡張子(NULL:拡張子削除)
         RET,パス,NULL:エラー                                               */
LPSTR WINAPI
PathRenameExtensionNewA (LPCSTR lpszPath,
                         LPCSTR lpszExt)
{
  LPSTR lpszResult;

  if (lpszPath)
    {
      int nLength;

      nLength = PathFindExtensionA (lpszPath) - lpszPath;
      lpszResult = MemoryAlloc ((nLength + lstrlenA (lpszExt) + 1)
                                                            * sizeof (CHAR));
      MemoryCopy (lpszResult, lpszPath, nLength * sizeof (CHAR));
      lstrcatA (lpszResult, lpszExt);
    }
  else
    {
      lpszResult = NULL;
    }
  return lpszResult;
}


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  ファイルをメモリに読み込む
    lpszFile,ファイル名
    lpdwSize,ファイルサイズ
         RET,メモリ                                                         */
LPVOID WINAPI
LoadFileW (LPCWSTR lpszFile,
           LPDWORD lpdwSize)
{
  LPVOID lpBuffer = NULL;

  if (lpszFile)
    {
      BOOL fResult = FALSE;
      DWORD dwSize;
      HANDLE hFile;

      hFile = CreateFileW (lpszFile, GENERIC_READ, FILE_SHARE_READ, NULL,
                                OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
      if (hFile != INVALID_HANDLE_VALUE)
        {
          dwSize = GetFileSize (hFile, NULL);
          if (dwSize != INVALID_FILE_SIZE)
            {
              DWORD dwRead;

              lpBuffer = MemoryAlloc (((dwSize + 1) / sizeof (WCHAR) + 1)
                                                            * sizeof (WCHAR));
              if (lpBuffer)
                fResult = ReadFile (hFile, lpBuffer, dwSize, &dwRead, NULL)
                                                        && dwSize == dwRead;
            }
          if (!CloseHandle (hFile))
            fResult = FALSE;
        }
      if (!fResult)
        {
          MemoryFree (lpBuffer);
          lpBuffer = NULL;
        }
      else if (lpdwSize)
        {
          *lpdwSize = dwSize;
        }
    }
  return lpBuffer;
}


/*  ファイルをメモリに読み込む
    lpszFile,ファイル名
    lpdwSize,ファイルサイズ
         RET,メモリ                                                         */
LPVOID WINAPI
LoadFileA (LPCSTR  lpszFile,
           LPDWORD lpdwSize)
{
  LPVOID lpBuffer = NULL;

  if (lpszFile)
    {
      BOOL fResult = FALSE;
      DWORD dwSize;
      HANDLE hFile;

      hFile = CreateFileA (lpszFile, GENERIC_READ, FILE_SHARE_READ, NULL,
                                OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
      if (hFile != INVALID_HANDLE_VALUE)
        {
          dwSize = GetFileSize (hFile, NULL);
          if (dwSize != INVALID_FILE_SIZE)
            {
              DWORD dwRead;

              lpBuffer = MemoryAlloc (((dwSize + 1) / sizeof (WCHAR) + 1)
                                                            * sizeof (WCHAR));
              if (lpBuffer)
                fResult = ReadFile (hFile, lpBuffer, dwSize, &dwRead, NULL)
                                                        && dwSize == dwRead;
            }
          if (!CloseHandle (hFile))
            fResult = FALSE;
        }
      if (!fResult)
        {
          MemoryFree (lpBuffer);
          lpBuffer = NULL;
        }
      else if (lpdwSize)
        {
          *lpdwSize = dwSize;
        }
    }
  return lpBuffer;
}


/*  メモリをファイルに書き込む
    lpszFile,ファイル名
    lpBuffer,メモリ
      nSize,バイト数(負:NULL終端文字列)
         RET,TRUE:正常終了,FALSE:エラー                                     */
BOOL WINAPI
SaveFileW (LPCWSTR lpszFile,
           LPCVOID lpBuffer,
           SSIZE_T nSize)
{
  BOOL fResult = FALSE;

  if (lpszFile)
    {
      HANDLE hFile;
      WCHAR szDir[MAX_PATH];

      lstrcpyW (szDir, lpszFile);
      PathRemoveFileSpecW (szDir);
      SHCreateDirectoryExW (NULL, szDir, NULL);
      hFile = CreateFileW (lpszFile, GENERIC_WRITE, FILE_SHARE_READ, NULL,
                                CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);
      if (hFile != INVALID_HANDLE_VALUE)
        {
          DWORD dwSize, dwWrite;

          dwSize = nSize >= 0 ? nSize: lstrlenW (lpBuffer) * sizeof (WCHAR);
          fResult = WriteFile (hFile, lpBuffer, dwSize, &dwWrite, NULL)
                                                        && dwSize == dwWrite;
          if (!CloseHandle (hFile))
            fResult = FALSE;
        }
    }
  return fResult;
}


/*  メモリをファイルに書き込む
    lpszFile,ファイル名
    lpBuffer,メモリ
      nSize,バイト数(負:NULL終端文字列)
         RET,TRUE:正常終了,FALSE:エラー                                     */
BOOL WINAPI
SaveFileA (LPCSTR  lpszFile,
           LPCVOID lpBuffer,
           SSIZE_T nSize)
{
  BOOL fResult = FALSE;

  if (lpszFile)
    {
      HANDLE hFile;
      CHAR szDir[MAX_PATH];

      lstrcpyA (szDir, lpszFile);
      PathRemoveFileSpecA (szDir);
      SHCreateDirectoryExA (NULL, szDir, NULL);
      hFile = CreateFileA (lpszFile, GENERIC_WRITE, FILE_SHARE_READ, NULL,
                                CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);
      if (hFile != INVALID_HANDLE_VALUE)
        {
          DWORD dwSize, dwWrite;

          dwSize = nSize >= 0 ? nSize: lstrlenA (lpBuffer) * sizeof (CHAR);
          fResult = WriteFile (hFile, lpBuffer, dwSize, &dwWrite, NULL)
                                                        && dwSize == dwWrite;
          if (!CloseHandle (hFile))
            fResult = FALSE;
        }
    }
  return fResult;
}
