/*
 * pathmatch.c
 *
 * Expands pathname patterns.
 *
 * Copyright (C) 2005-2006  NTT DATA CORPORATION
 *
 * Version: 1.3     2006/11/11
 *
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dirent.h>

static int FileMatchesToPattern(const char *filename, const char *filename_end, const char *pattern, const char *pattern_end) {
	while (filename < filename_end && pattern < pattern_end) {
		if (*pattern != '\\') {
			if (*filename++ != *pattern++) return 0;
		} else {
			char c = *filename;
			pattern++;
			switch (*pattern) {
			case '?':
				if (c == '/') {
					return 0;
				} else if (c == '\\') {
					if ((c = filename[1]) == '\\') {
						filename++; /* safe because filename is \\ */
					} else if (c >= '0' && c <= '3' && (c = filename[2]) >= '0' && c <= '7' && (c = filename[3]) >= '0' && c <= '7') {
						filename += 3; /* safe because filename is \ooo */
					} else {
						return 0;
					}
				}
				break;
			case '\\':
				if (c != '\\') return 0;
				if (*++filename != '\\') return 0; /* safe because *filename != '\0' */
				break;
			case '+':
				if (c < '0' || c > '9') return 0;
				break;
			case 'x':
				if (!((c >= '0' && c <= '9') || (c >= 'A' && c <= 'F') || (c >= 'a' && c <= 'f'))) return 0;
				break;
			case 'a':
				if (!((c >= 'A' && c <= 'Z') || (c >= 'a' && c <= 'z'))) return 0;
				break;
			case '0':
			case '1':
			case '2':
			case '3':
				if (c == '\\' && (c = filename[1]) >= '0' && c <= '3' && c == *pattern
					&& (c = filename[2]) >= '0' && c <= '7' && c == pattern[1]
					&& (c = filename[3]) >= '0' && c <= '7' && c == pattern[2]) {
					filename += 3; /* safe because filename is \ooo */
					pattern += 2; /* safe because pattern is \ooo  */
					break;
				}
				return 0; /* Not matched. */
			case '*':
			case '@':
				{
					int i;
					for (i = 0; i <= filename_end - filename; i++) {
						if (FileMatchesToPattern(filename + i, filename_end, pattern + 1, pattern_end)) return 1;
						if ((c = filename[i]) == '.' && *pattern == '@') break;
						if (c == '\\') {
							if ((c = filename[i + 1]) == '\\') {
								i++; /* safe because filename is \\ */
							} else if (c >= '0' && c <= '3' && (c = filename[i + 2]) >= '0' && c <= '7' && (c = filename[i + 3]) >= '0' && c <= '7') {
								i += 3; /* safe because filename is \ooo */
							} else {
								break; /* Bad pattern. */
							}
						}
					}
					return 0; /* Not matched. */
				}
			default:
				{
					int i, j = 0;
					if ((c = *pattern) == '$') {
						while ((c = filename[j]) >= '0' && c <= '9') j++;
					} else if (c == 'X') {
						while (((c = filename[j]) >= '0' && c <= '9') || (c >= 'A' && c <= 'F') || (c >= 'a' && c <= 'f')) j++;
					} else if (c == 'A') {
						while (((c = filename[j]) >= 'A' && c <= 'Z') || (c >= 'a' && c <= 'z')) j++;
					}
					for (i = 1; i <= j; i++) {
						if (FileMatchesToPattern(filename + i, filename_end, pattern + 1, pattern_end)) return 1;
					}
				}
				return 0; /* Not matched or bad pattern. */
			}
			filename++; /* safe because *filename != '\0' */
			pattern++; /* safe because *pattern != '\0' */
		}
	}
	while (*pattern == '\\' && (*(pattern + 1) == '*' || *(pattern + 1) == '@')) pattern += 2;
	return (filename == filename_end && pattern == pattern_end);
}

static int need_separator = 0;

static void print(const char *pathname) {
	unsigned char c;
	if (need_separator) putchar(' ');
	need_separator = 0;
	while ((c = *pathname++) != '\0') {
		need_separator = 1;
		if (c == '\\') {
			putchar('\\'); putchar('\\');
		} else if (c > ' ' && c < 127) {
			putchar(c);
		} else {
			putchar('\\');
			putchar((c >> 6) + '0');
			putchar(((c >> 3) & 7) + '0');
			putchar((c & 7) + '0');
		}
	}
}

static char *current_part = NULL;

static int filter(const struct dirent *buf) {
	char buffer[1024];
	char c;
	char *dp = buffer;
	const char *cp = buf->d_name;
	if (buf->d_type == DT_DIR) {
		if (strcmp(cp, ".") == 0 || strcmp(cp, "..") == 0) return 0;
	}
	if (strlen(cp) > 255) return 0;
	while ((c = *cp++) != '\0') {
		if (c == '\\') {
			*dp++ = '\\';
			*dp++ = '\\';
		} else if (c > ' ' && c < 127) {
			*dp++ = c;
		} else {
			*dp++ = '\\';
			*dp++ = ((c >> 6) + '0');
			*dp++ = (((c >> 3) & 7) + '0');
			*dp++ = ((c & 7) + '0');
		}
	}
	*dp = '\0';
	//printf("Compare: %s %s\n", buffer, current_part);
	if (FileMatchesToPattern(buffer, dp, current_part, strchr(current_part, '\0'))) return 1;
	return 0;
}

static int target_is_dir = 0; 
static int target_depth = 0;
static char **target_part = NULL;

static void ScanDir(const char *path, int depth) {
	struct dirent **namelist;
	int i, n;
	current_part = target_part[depth];
	//printf("Scan: %d %s\n", depth, current_part);
	if ((n = scandir(path, &namelist, filter, 0)) >= 0) {	
		char buffer[4096];
		memset(buffer, 0, sizeof(buffer));
		for (i = 0; i < n; i++) {
			const char *cp = namelist[i]->d_name;
			const unsigned char type = namelist[i]->d_type;
			if (depth < target_depth - 1) {
				if (type == DT_DIR) {
					snprintf(buffer, sizeof(buffer) - 1, "%s%s/", path, cp);
					//printf("Check: %s\n", buffer);
					ScanDir(buffer, depth + 1);
				}
			} else if (target_is_dir) {
				if (type == DT_DIR) {
					snprintf(buffer, sizeof(buffer) - 1, "%s%s/", path, cp);
					print(buffer);
				}
			} else if (type != DT_DIR) {
				snprintf(buffer, sizeof(buffer) - 1, "%s%s", path, cp);
				print(buffer);
			}
			free((void *) namelist[i]);
		}
		free((void *) namelist);
	}
}

static void do_main(char *target) {
	if (strcmp(target, "/") == 0) {
		printf("/\n");
	} else if (target[0] != '/') {
		putchar('\n');
	} else {
		char *cp;
		int i;
		target_is_dir = (*(strchr(target, '\0') - 1) == '/');
		target_depth = 0;
		cp = target + 1;
		for (i = 1; ; i++) {
			char c = target[i];
			if (c == '/' || c == '\0') {
				target[i] = '\0';
				target_part = (char **) realloc(target_part, (target_depth + 1) * sizeof(char *));
				if (target + i != cp) target_part[target_depth++] = cp;
				cp = target + i + 1;
				if (!c) break;
			}
		}
		//for (i = 0; i < target_depth; i++) printf("%d %s\n", i, target_part[i]);
		need_separator = 0;
		ScanDir("/", 0);
		putchar('\n');
	}
}

int main(int argc, char *argv[]) {
	if (argc > 1) {
		int i;
		for (i = 1; i < argc; i++) do_main(argv[i]);
	} else {
		char buffer[4096];
		while (memset(buffer, 0, sizeof(buffer)), fgets(buffer, sizeof(buffer) - 1, stdin)) {
			char *cp = strchr(buffer, '\n');
			if (!cp) break;
			*cp = '\0';
			do_main(buffer);
		}
	}
	return 0;
}
