/*******************************************************************************
 * Copyright (c) 2007  NTT DATA CORPORATION
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Version: 1.0.0 - 2007/06/15
 *          initial API and implementation
 *******************************************************************************/
package jp.sourceforge.tomoyo.core;

import java.lang.reflect.InvocationTargetException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.LinkedList;
import java.util.NoSuchElementException;

import jp.sourceforge.tomoyo.core.extensions.IConnectionAdapter;
import jp.sourceforge.tomoyo.core.local.parser.PolicyParserManager;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.resource.ImageRegistry;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.plugin.AbstractUIPlugin;
import org.osgi.framework.BundleContext;

/**
 * This is the top-level class of the property sheet example.
 *
 * @see AbstractUIPlugin for additional information on UI plugins
 */
public class TomoyoCorePlugin extends AbstractUIPlugin {
	
	// Default instance of the receiver
	private static TomoyoCorePlugin inst;

    /**
     * Create the PropertySheet plugin and cache its default instance
     */
    public TomoyoCorePlugin() {
        if (inst == null)
            inst = this;
    }
    
    /**
     * Returns the plugin singleton.
     *
     * @return the default PropertySheetPlugin instance
     */
    static public TomoyoCorePlugin getDefault() {
    	return inst;
    }
    	
	public static String getPluginId() {
		return getDefault().getBundle().getSymbolicName();
	}

	public void installNature(IProject project) {
		try {
			IProjectDescription description = project.getDescription();
			if (!description.hasNature(TomoyoCoreNature.ID)) {
				String[] ids = description.getNatureIds();
				String[] newIds = new String[ids.length + 1];
				System.arraycopy(ids, 0, newIds, 0, ids.length);
				newIds[ids.length] = TomoyoCoreNature.ID;
				description.setNatureIds(newIds);
				project.setDescription(description, null);
			}
		} catch (CoreException e) {
			logException(e);
		}
	}
	
	public IProject[] getTomoyoCoreProjects() {
		ArrayList<IProject> projectList = new ArrayList<IProject>();
		IProject[] projects = ResourcesPlugin.getWorkspace().getRoot().getProjects();
		for (int cnt = 0; cnt < projects.length; cnt++) {
			IProject project = projects[cnt];
			try {
				IProjectDescription description = projects[cnt].getDescription();
				if (description.hasNature(TomoyoCoreNature.ID)) {
					projectList.add(project);
				}
			} catch (CoreException e) {
			}
		}
		return (IProject[])projectList.toArray(new IProject[projectList.size()]);
	}
	
	public IProject findProject(String projectName) {
		IProject[] projects = getTomoyoCoreProjects();
		for (int cnt = 0; cnt < projects.length; cnt++) {
			IProject project = projects[cnt];
			if (project.getName().equals(projectName))
				return project;
		}
		return null;
	}
	
	public static boolean isTomoyoCoreProject(IProject project) {
		if (project == null)
			return false;
		try {
			return project.getDescription().hasNature(TomoyoCoreNature.ID);
		} catch (CoreException e) {
			logException(e);
			return false;
		}
	}
	
	public void start(BundleContext context) throws Exception {
		super.start(context);
		
//		ResourcesPlugin.getWorkspace().addResourceChangeListener(
//				new PolicyUpdateListener(), IResourceChangeEvent.POST_CHANGE);
		ResourcesPlugin.getWorkspace().addResourceChangeListener(
				new TomoyoCoreProjectPostOpenListener(), IResourceChangeEvent.POST_CHANGE);
//		ResourcesPlugin.getWorkspace().addResourceChangeListener(
//				new TomoyoCoreProjectPreCloseListener(), IResourceChangeEvent.PRE_CLOSE);
//		ResourcesPlugin.getWorkspace().addResourceChangeListener(
//				new TomoyoCoreProjectPreDeleteListener(), IResourceChangeEvent.PRE_DELETE);

		IProject[] projects = getTomoyoCoreProjects();
		for (int cnt = 0; cnt < projects.length; cnt++) {
			IProject project = projects[cnt];
			if (project.isOpen()) {
				handleOpenProjectEvent(project);
			}
		}
		
		findConnectionAdapters();
	}
	
	private Hashtable<String, IConnectionAdapter> connectAdapters = new Hashtable<String, IConnectionAdapter>();
	
	private void findConnectionAdapters() {
		IExtensionRegistry registry = Platform.getExtensionRegistry();
		IExtensionPoint point =
			registry.getExtensionPoint(
			"jp.sourceforge.tomoyo.core.registerConnectionAdapter"); //$NON-NLS-1$
		IExtension[] extensions = point.getExtensions();
		for (int i = 0; i < extensions.length; i++) {
			IConfigurationElement[] elements =
				extensions[i].getConfigurationElements();
			for (int j = 0; j < elements.length; j++) {
				IConnectionAdapter adapter = null;
				if ("register".equals(elements[j].getName())) {	 //$NON-NLS-1$
					try {
						adapter = (IConnectionAdapter)elements[j].createExecutableExtension("class"); //$NON-NLS-1$
						connectAdapters.put(adapter.getID(), adapter);
					} catch (CoreException e) {
					}
				}
			}
		}
	}

	public IConnectionAdapter getConnectionAdapter(String adapterID) {
		if (adapterID == null)
			return null;
		IConnectionAdapter adapter = connectAdapters.get(adapterID);
		return adapter;
	}
	
	public void stop(BundleContext context) throws Exception {
		super.stop(context);
	}
	
	/*
	private class PolicyUpdateListener implements IResourceChangeListener {
		public void resourceChanged(IResourceChangeEvent event) {
			IResourceDelta delta = event.getDelta();
			IResourceDelta[] children = delta.getAffectedChildren();
			for (int i = 0; i < children.length; i++) {
				IResource resource = children[i].getResource();
				if (resource instanceof IProject) {
				}
			}
		}
	}
	*/
	private class TomoyoCoreProjectPostOpenListener implements IResourceChangeListener {
		public void resourceChanged(IResourceChangeEvent event) {
			IResourceDelta delta = event.getDelta();
			IResourceDelta[] children = delta.getAffectedChildren();
			for (int i = 0; i < children.length; i++) {
				IResource resource = children[i].getResource();
				if (resource instanceof IProject) {
					IProject project = (IProject)resource;
					if (children[i].getFlags() == IResourceDelta.OPEN) {
						if (project.isOpen()) {
							handleOpenProjectEvent(project);
						}
					}
				}
			}
		}
	}

	private void handleOpenProjectEvent(IProject project) {
    	PolicyParserManager.getInstance().preload(project);
	}

	/**
	 * Returns the standard display to be used. The method first checks, if
	 * the thread calling this method has an associated disaply. If so, this
	 * display is returned. Otherwise the method returns the default display.
	 */
	public static Display getStandardDisplay() {
		Display display;
		display = Display.getCurrent();
		if (display == null)
			display = Display.getDefault();
		return display;
	}
	
    public static Image getImage(String name) {
    	ImageRegistry registry = getDefault().getImageRegistry();
    	Image image = registry.get(name);
    	if (image == null) {
        	try {
            	URL url = getDefault().getBundle().getEntry("/"); //$NON-NLS-1$
    			image = ImageDescriptor.createFromURL(new URL(url, "icons/" + name)).createImage(); //$NON-NLS-1$
    			registry.put(name, image);
        	} catch (MalformedURLException e) {
    			image = ImageDescriptor.getMissingImageDescriptor().createImage();
    		}
    	}
    	return image;
    }
    
    public static ImageDescriptor getImageDescriptor(String name) {
    	ImageRegistry registry = getDefault().getImageRegistry();
    	ImageDescriptor imageDescriptor = registry.getDescriptor(name);
    	if (imageDescriptor == null) {
        	try {
            	URL url = getDefault().getBundle().getEntry("/"); //$NON-NLS-1$
            	imageDescriptor = ImageDescriptor.createFromURL(new URL(url, "icons/" + name)); //$NON-NLS-1$
    			registry.put(name, imageDescriptor);
        	} catch (MalformedURLException e) {
        		imageDescriptor = ImageDescriptor.getMissingImageDescriptor();
    		}
    	}
    	return imageDescriptor;
    }
    
	public static void log(IStatus status) {
		ResourcesPlugin.getPlugin().getLog().log(status);
	}
	
	public static void logErrorMessage(String message) {
		log(new Status(IStatus.ERROR, getPluginId(), IStatus.ERROR, message, null));
	}
	
	public static void logException(
		Throwable e,
		final String title,
		String message) {
		if (e instanceof InvocationTargetException) {
			e = ((InvocationTargetException) e).getTargetException();
		}
		IStatus status = null;
		if (e instanceof CoreException)
			status = ((CoreException) e).getStatus();
		else {
			if (message == null)
				message = e.getMessage();
			if (message == null)
				message = e.toString();
			status = new Status(IStatus.ERROR, getPluginId(), IStatus.OK, message, e);
		}
		ResourcesPlugin.getPlugin().getLog().log(status);
		Display display = getStandardDisplay();
		final IStatus fstatus = status;
		display.asyncExec(new Runnable() {
			public void run() {
				ErrorDialog.openError(null, title, null, fstatus);
			}
		});
	}
	
	public static void logException(Throwable e) {
		logException(e, null, null);
	}
	
	public static void log(Throwable e) {
		if (e instanceof InvocationTargetException)
			e = ((InvocationTargetException) e).getTargetException();
		IStatus status = null;
		if (e instanceof CoreException)
			status = ((CoreException) e).getStatus();
		else
			status =
				new Status(IStatus.ERROR, getPluginId(), IStatus.OK, e.getMessage(), e);
		log(status);
	}
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.views.markers.internal.TableView#getDialogSettings()
	 */
	public IDialogSettings getDialogSettings(String sectionName) {
		IDialogSettings workbenchSettings = TomoyoCorePlugin.getDefault().getDialogSettings();
		IDialogSettings settings = workbenchSettings.getSection(sectionName);
		if (settings == null) {
			settings = workbenchSettings.addNewSection(sectionName);
		}
		return settings;
	}

	
	
	
	
	
	
	
	
	
	public void setStatusMessage(String message) {
		setStatusMessage(message, false);
	}
	
	public void setStatusMessage(String message, boolean isPriotiryLow) {
		setStatusMessage(message, getImage("info.gif"), isPriotiryLow); //$NON-NLS-1$
	}

	public void setStatusMessage(String message, Image image) {
		setStatusMessage(message, image, false); //$NON-NLS-1$
	}
	
	public void setStatusMessage(String message, Image image, boolean isPriotiryLow) {
		setStatusMessage(true, message, image, isPriotiryLow);
	}

	public void setStatusErrorMessage(String message) {
		setStatusErrorMessage(message, getImage("error.gif")); //$NON-NLS-1$
	}
	
	public void setStatusErrorMessage(String message, Image image) {
		setStatusMessage(false, message, image, false);
	}

	private LinkedList<StatusMessageRunnable> msgQueue = new LinkedList<StatusMessageRunnable>();
	
	public void setStatusMessage(boolean normal, String message, Image image, boolean isPriotiryLow) {
		msgQueue.add(new StatusMessageRunnable(normal, message, image, isPriotiryLow));
		
		refreshStatusMessage();
	}

	private void refreshStatusMessage() {
		if (msgQueue.size() == 0)
			return;
		StatusMessageRunnable runnable = msgQueue.peek();
		if (runnable.isRunning())
			return;
		Display.getDefault().syncExec(runnable);
	}
	
	private IEditorSite getActiveEditorSite() {
		IWorkbench workbench = PlatformUI.getWorkbench();
		if (workbench.isClosing())
			return null;
		IWorkbenchWindow workbenchWindow = workbench.getActiveWorkbenchWindow();
		IWorkbenchPage workbenchPage = workbenchWindow.getActivePage();
		if (workbenchPage == null)
			return null;
		IEditorPart editorPart = workbenchPage.getActiveEditor();
		if (editorPart == null)
			return null;
		IEditorSite editorSite = editorPart.getEditorSite();
		return editorSite;
	}
	
	private class StatusMessageRunnable implements Runnable {
		private boolean normal;
		private Image image;
		private String message;
		private boolean isRunning = false;
		private boolean isPriotiryLow = false;
		public StatusMessageRunnable(boolean normal, String message, Image image, boolean isPriotiryLow) {
			this.normal = normal;
			this.message = message;
			this.image = image;
			this.isPriotiryLow = isPriotiryLow;
		}
		public void run() {
			isRunning = true;
			IEditorSite editorSite = getActiveEditorSite();
			if (editorSite == null) {
				isRunning = false;
				return;
			}
			IActionBars actionBars = editorSite.getActionBars();
			IStatusLineManager manager = actionBars.getStatusLineManager();
			if (normal) {
				manager.setErrorMessage(null);
				if (image != null)
					manager.setMessage(image, getDecotatedMessage());
				else
					manager.setMessage(getDecotatedMessage());
			} else {
				if (image != null)
					manager.setErrorMessage(image, getDecotatedMessage());
				else
					manager.setErrorMessage(getDecotatedMessage());
			}
			manager.update(true);
			
			new Thread(new ClearStatusMessageRunnable(this)).start();
		}
		private String getDecotatedMessage() {
			if (msgQueue.size() == 1)
				return message;
			else
				return message;
//				return "(1/" + msgQueue.size() + ") " + message;
		}
		public boolean isRunning() {
			return isRunning;
		}
		public boolean isLowPriority() {
			return isPriotiryLow;
		}
	}
	
	private class ClearStatusMessageRunnable implements Runnable {
		public ClearStatusMessageRunnable(StatusMessageRunnable runnable) {
		}
		public void run() {
			try {
				StatusMessageRunnable runnable = msgQueue.peek();
				for (int cnt = 0; cnt < 50; cnt++) {
					Thread.sleep(100);
					if (runnable != null && runnable.isLowPriority() && msgQueue.size() > 1) {
						if (cnt < 30)
							Thread.sleep(1900);
						break;
					}
				}
				Display.getDefault().asyncExec(new Runnable() {
					public void run() {
						IEditorSite editorSite = getActiveEditorSite();
						if (editorSite == null)
							return;
						IActionBars actionBars = editorSite.getActionBars();
						IStatusLineManager manager = actionBars.getStatusLineManager();
						manager.setMessage(null);
						manager.setErrorMessage(null);
					}
				});
			} catch (InterruptedException e) {
			} finally {
				try {
					msgQueue.remove();				
				} catch (NoSuchElementException e) {
				}
				refreshStatusMessage();
			}
		}
	}
	
}
