/*******************************************************************************
 * Copyright (c) 2007  NTT DATA CORPORATION
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Version: 1.0.0 - 2007/06/15
 *          initial API and implementation
 *******************************************************************************/
package jp.sourceforge.tomoyo.core.local.model;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.IProject;

import jp.sourceforge.tomoyo.core.local.model.domain.AccessPermission;
import jp.sourceforge.tomoyo.core.local.model.domain.Domain;
import jp.sourceforge.tomoyo.core.local.model.domain.IAccessPermission;
import jp.sourceforge.tomoyo.core.local.model.domain.Profile;
import jp.sourceforge.tomoyo.core.local.model.status.ProfileGroup;

public class PolicyCacheManager {

	private static PolicyCacheManager instance;
	
	public static PolicyCacheManager getInstance() {
		if (instance == null)
			instance = new PolicyCacheManager();
		return instance;
	}

	public void clear(IProject project) {
		domainTransitionCache.clear(project);
		initializerCache.clear(project);
		
		projectCache.clear(project);
	}

	protected class IndividualProjectCache {
		
		private Hashtable projects = new Hashtable();
		
		protected Hashtable getCache(IProject project) {
			Hashtable cache = null;
			if (projects.get(project) == null) {
				cache = new Hashtable();
				projects.put(project, cache);
			} else {
				cache = (Hashtable)projects.get(project);
			}
			return cache;
		}
		
		public void clear(IProject project) {
			if (projects.get(project) == null)
				return;
			Hashtable cache = (Hashtable)projects.get(project);
			cache.clear();
		}

		public int count(IProject project) {
			if (projects.get(project) == null)
				return 0;
			Hashtable cache = (Hashtable)projects.get(project);
			return cache.size();
		}
		
	}
	
	//-----------------------------------------------------------------------------------------
	// Domain transition cache
	//-----------------------------------------------------------------------------------------
	
	private DomainTransitionCache domainTransitionCache = new DomainTransitionCache();
	
	private class DomainTransitionCache extends IndividualProjectCache {

		private Hashtable<String, Domain> workDomainCache = new Hashtable<String, Domain>();
		
		public void clear(IProject project) {
			super.clear(project);
			workDomainCache.clear();
		}

		public void createDomainTranstion(IProject project) {
			clear(project);

			List domainList = findElementList(project, Domain.class);
			
			Hashtable transitionCache = getCache(project);
			for (int cnt = 0; cnt < domainList.size(); cnt++) {
				Domain domain = (Domain)domainList.get(cnt);
				if (domain.isDeleted())
					continue;
				if (domain.isKernel()) {
					;
				} else {
					if (isParentDomainCached(transitionCache, domain)) {
						String strParentDomain = domain.getText2();
						Domain parentDomain = (Domain)workDomainCache.get(strParentDomain);
						if (parentDomain != null)
							if (parentDomain.isDeleted())
								parentDomain.setSkelton(true);
					} else {
						for (int depth = 0; depth < domain.getDepth() - 1; depth++) {
							String strDomainPath = domain.getText2(depth + 1);
							if (isDomainCached(transitionCache, strDomainPath)) {
								continue;
							} else {
								Domain skeltonDomain = new Domain(strDomainPath, 0, 0);
								skeltonDomain.setCreated(false);
								skeltonDomain.setChecked(true);
								skeltonDomain.setSkelton(true);
								cacheDomain(transitionCache, skeltonDomain);
								workDomainCache.put(skeltonDomain.getText(), skeltonDomain);
							}
						}
					}
				}
				cacheDomain(transitionCache, domain);
				workDomainCache.put(domain.getText(), domain);
			}
		}
		
		private void cacheDomain(Hashtable transitionCache, Domain domain) {
			String strDomain = domain.getText();
			transitionCache.put(strDomain, new ArrayList(0));
			String strParentDomain = domain.getText2();
			if (strParentDomain.length() == 0)
				return;
			ArrayList list = (ArrayList)transitionCache.get(strParentDomain);
			list.add(domain);
		}

		private boolean isParentDomainCached(Hashtable transitionCache, Domain domain) {
			String parentDomainText = domain.getText2();
			return transitionCache.get(parentDomainText) != null;
		}

		private boolean isDomainCached(Hashtable transitionCache, String domainText) {
			return transitionCache.get(domainText) != null;
		}
		
		public Collection listDomains(IProject project) {
			Hashtable domainCache = getCache(project);
			return domainCache.values();
		}
		
		public List<Domain> listChildDomains(IProject project, Domain domain) {
			Hashtable domainCache = getCache(project);
			List retlist = (List)domainCache.get(domain.getText());
			if (retlist == null)
				return new ArrayList<Domain>(0);
			else
				return retlist;
		}

		public List<Domain> listDecendantDomains(IProject project, Domain domain) {
			ArrayList<Domain> childList = new ArrayList<Domain>();
			_listDecendantDomains(project, childList, domain);
			return childList;
		}
		
		public void _listDecendantDomains(IProject project, List<Domain> domainList, Domain domain) {
			Hashtable domainCache = getCache(project);
			ArrayList list = (ArrayList)domainCache.get(domain.getText());
			if (list == null)
				list = new ArrayList<Domain>();
			for (int cnt = 0; cnt < list.size(); cnt++) {
				Domain child = (Domain)list.get(cnt);
				domainList.add(child);
				_listDecendantDomains(project, domainList, child);
			}
		}

		public Domain getParentDomain(IProject project, Domain domain) {
			String parentDomainText = domain.getText2();
			Domain parentDomain = (Domain)workDomainCache.get(parentDomainText);
			return parentDomain;
		}
		
	}

	public void createDomainTranstion(IProject project) {
		domainTransitionCache.createDomainTranstion(project);
	}

	public int countDomains(IProject project, boolean includeSkelton) {
		if (includeSkelton)
			return domainTransitionCache.count(project);
		else
			return findElementList(project, Domain.class).size();
	}

	public List<Domain> listChildDomains(IProject project, Domain domain) {
		return domainTransitionCache.listChildDomains(project, domain);
	}

	public List<Domain> listDecendantDomains(IProject project, Domain domain) {
		return domainTransitionCache.listDecendantDomains(project, domain);
	}
	
	public int listDecendantDomainCount(IProject project, Domain domain) {
		return listDecendantDomains(project, domain).size();
	}

	public Domain getParentDomain(IProject project, Domain domain) {
		return domainTransitionCache.getParentDomain(project, domain);
	}

	//-----------------------------------------------------------------------------------------
	// Initializer cache
	//-----------------------------------------------------------------------------------------
	
	private InitializerCache initializerCache = new InitializerCache();
	
	private class InitializerCache extends IndividualProjectCache {
		
		public void addInitializerTarget(IProject project, Domain domain) {
			Hashtable cache = getCache(project);
			cache.put(domain.getProcessName(), domain);
		}

		public Domain getInitializerTarget(IProject project, String executableName) {
			Hashtable cache = (Hashtable)getCache(project);
			return (Domain)cache.get(executableName);
		}

	}

	public void addInitializerTarget(IProject project, Domain domain) {
		initializerCache.addInitializerTarget(project, domain);
	}
	
	public Domain getInitializerTarget(IProject project, String executableName) {
		return initializerCache.getInitializerTarget(project, executableName);
	}
	
	public int countInitializerTarget(IProject project) {
		return initializerCache.count(project);
	}
	
	
	
	
	
	
	
	
	
	//-----------------------------------------------------------------------------------------
	// Element cache
	//-----------------------------------------------------------------------------------------
	
	public void addElement(IProject project, PolicyElement cacheElement) {
		projectCache.addElement(project, cacheElement);
	}

	public boolean exists(IProject project, PolicyElement policyElement) {
		return projectCache.exists(project, policyElement);
	}

	public boolean exists(IProject project, Domain domain, AccessPermission accessPermission) {
		return projectCache.exists(project, domain, accessPermission);
	}
	
	public PolicyElement findElement(IProject project, Class cacheClass, String text) {
		return projectCache.findElement(project, cacheClass, text, false);
	}
	
	public PolicyElement findElement(IProject project, PolicyElement element) {
		return projectCache.findElement(project, element, false);
	}

	public PolicyElement[] findElements(IProject project, Class cacheClass) {
		return projectCache.findElements(project, cacheClass);
	}

	public PolicyElement[] findElements(IProject project, Class[] cacheClasses) {
		return projectCache.findElements(project, cacheClasses);
	}
	
	public List<PolicyElement> findElementList(IProject project, Class cacheClass) {
		return projectCache.findElementList(project, cacheClass);
	}

	public Domain findDomain(IProject project, String domain) {
		PolicyElement[] elements = projectCache.findElements(project, Domain.class);
		if (elements == null)
			return null;
		for (int cnt = 0; cnt < elements.length; cnt++) {
			if (elements[cnt].getText().equals(domain)) {
				return (Domain)elements[cnt];
			}
		}
		return null;
	}
	
	public AccessPermission findAccessPermission(IProject project, Domain domain, String strAccessPermission) {
		Domain foundDomain = findDomain(project, domain.getText());
		if (foundDomain == null)
			return null;
		for (int cnt = 0; cnt < foundDomain.getChildrenCount(); cnt++) {
			PolicyElement element = foundDomain.getChild(cnt);
			if (element instanceof IAccessPermission) {
				AccessPermission permission = (AccessPermission)element;
				if (permission.getText().equals(strAccessPermission)) {
					return permission;
				}
			}
		}
		return null;
	}

	public AccessPermission findAccessPermission(IProject project, String strDomain, String strPermission, String strPath) {
		Domain foundDomain = findDomain(project, strDomain);
		if (foundDomain == null)
			return null;
		for (int cnt = 0; cnt < foundDomain.getChildrenCount(); cnt++) {
			PolicyElement element = foundDomain.getChild(cnt);
			if (element instanceof IAccessPermission) {
				AccessPermission permission = (AccessPermission)element;
				if (permission.getDeclaredPermission().equals(strPermission)) {
					if (permission.getPermission().equals(strPath)) {
						return permission;
					}
				}
			}
		}
		return null;
	}

	public int countDomain(IProject project, ProfileGroup group) {
		int count = 0;
		PolicyElement[] elements = findElements(project, Domain.class);
		for (int cnt = 0; cnt < elements.length; cnt++) {
			Domain domain = (Domain)elements[cnt];
			if (domain.isDeleted())
				continue;
			Profile useProfile = domain.getProfile();
			if (useProfile != null) {
				if (useProfile.getNo().equals(String.valueOf(group.getNo())))
					count++;
			}
		}
		return count;
	}
	
	public int count(IProject project, Class cacheClass, boolean includeDeleted) {
		if (includeDeleted)
			return findElements(project, cacheClass).length;
		else {
			PolicyElement[] elements = findElements(project, cacheClass);
			if (elements == null)
				return 0;
			int count = 0;
			for (int cnt = 0; cnt < elements.length; cnt++) {
				if (!elements[cnt].isDeleted())
//				if (elements[cnt].isChecked())
					count++;
			}
			return count;
		}
	}
	
	public int count(IProject project, Domain domain, Class cacheClass, boolean includeDeleted) {
		PolicyElement[] elements = findElements(project, cacheClass);
		int count = 0;
		for (int cnt = 0; cnt < elements.length; cnt++) {
			if (elements[cnt] instanceof AccessPermission) {
				AccessPermission access = (AccessPermission)elements[cnt];
				if (access.isDeleted() && !includeDeleted)
					continue;
				Domain parentDomain = (Domain)access.getParent();
				if (parentDomain.getText() == null || domain.getText() == null)
					continue;
				if (parentDomain.getText().equals(domain.getText()))
					count++;
			}
		}
		return count;
	}
	
	public void clear(IProject project, Class cacheClass) {
		projectCache.clear(project, cacheClass);
	}
	
	private ProjectCache projectCache = new ProjectCache();
	
	private class ProjectCache {
		
		private Hashtable projectTable = new Hashtable();

		public void addElement(IProject project, PolicyElement cacheData) {
			ElementCache elementCache = null;
			if (projectTable.get(project) == null) {
				elementCache = new ElementCache();
				projectTable.put(project, elementCache);
			} else {
				elementCache = (ElementCache)projectTable.get(project);
			}
			elementCache.add(cacheData);
		}

		public boolean exists(IProject project, PolicyElement policyElement) {
			PolicyElement foundElement = findElement(project, policyElement, false);
			if (foundElement == null)
				return false;
			if (foundElement.isDeleted())
				return false;
			return true;
		}

		public boolean exists(IProject project, Domain domain, AccessPermission accessPermission) {
			Domain foundDomain = (Domain)findElement(project, Domain.class, domain.getText(), false);
			if (foundDomain == null)
				return false;
			for (int cnt = 0; cnt < foundDomain.getChildrenCount(); cnt++) {
				if (foundDomain.getChild(cnt) instanceof AccessPermission) {
					AccessPermission foundAccess = (AccessPermission)foundDomain.getChild(cnt);
					if (foundAccess.isDeleted())
						continue;
					if (foundAccess.getText().equals(accessPermission.getText()))
						return true;
				}
			}
			return false;
		}

		public PolicyElement[] findElements(IProject project, Class cacheClass) {
			if (projectTable.get(project) == null) {
				return null;
			}
			ElementCache elementCache = (ElementCache)projectTable.get(project);
			return elementCache.get(cacheClass);
		}
		
		public PolicyElement[] findElements(IProject project, Class[] cacheClasses) {
			if (projectTable.get(project) == null)
				return null;
			ElementCache elementCache = (ElementCache)projectTable.get(project);
			ArrayList elementList = new ArrayList();
			for (int cnt = 0; cnt < cacheClasses.length; cnt++) {
				elementList.addAll(elementCache.getList(cacheClasses[cnt]));
			}
			return (PolicyElement[])elementList.toArray(new PolicyElement[elementList.size()]);
		}

		public List<PolicyElement> findElementList(IProject project, Class cacheClass) {
			if (projectTable.get(project) == null)
				return null;
			ElementCache elementCache = (ElementCache)projectTable.get(project);
			return elementCache.getList(cacheClass);
		}

		public PolicyElement findElement(IProject project, Class cacheClass, String text, boolean includeDeleted) {
			if (projectTable.get(project) == null)
				return null;
			ElementCache elementCache = (ElementCache)projectTable.get(project);
			PolicyElement[] elements = elementCache.get(cacheClass);
			for (int cnt = 0; cnt < elements.length; cnt++) {
				PolicyElement finding = elements[cnt];
				if (finding.getText().equals(text)) {
					if (finding.isDeleted()) {
						if (includeDeleted)
							return finding;
					} else {
						return finding;
					}
					return null;
				}
			}
			return null;
		}

		public PolicyElement findElement(IProject project, PolicyElement element, boolean includeDeleted) {
			return findElement(project, element.getClass(), element.getText(), includeDeleted);
		}

		public void clear(IProject project) {
			if (projectTable.get(project) == null)
				return;
			ElementCache elementCache = (ElementCache)projectTable.get(project);
			elementCache.clear();
		}
		
		public void clear(IProject project, Class cacheClass) {
			if (projectTable.get(project) == null)
				return;
			ElementCache elementCache = (ElementCache)projectTable.get(project);
			elementCache.clear(cacheClass);
		}

		public void clearSubclass(IProject project, Class cacheClass) {
			if (projectTable.get(project) == null)
				return;
			ElementCache elementCache = (ElementCache)projectTable.get(project);
			elementCache.clearSubclass(cacheClass);
		}

		private class ElementCache {
			
			private Hashtable<Class<?>, ArrayList<PolicyElement>> elementTable = new Hashtable<Class<?>, ArrayList<PolicyElement>>();

			public void add(PolicyElement cacheData) {
				Class cacheClass = cacheData.getClass();
				add(cacheClass, cacheData);
				
				if (cacheData instanceof AccessPermission)
					add(AccessPermission.class, cacheData);
				
				if (cacheData instanceof AbstractDomainPolicyElement)
					add(DomainPolicyModel.class, cacheData);
				if (cacheData instanceof AbstractExceptPolicyElement)
					add(ExceptPolicyModel.class, cacheData);
				if (cacheData instanceof AbstractSystemPolicyElement)
					add(SystemPolicyModel.class, cacheData);
			}

			private void add(Class cacheClass, PolicyElement cacheData) {
				ArrayList elementList = null;
				if (elementTable.get(cacheClass) == null) {
					elementList = new ArrayList();
				} else {
					elementList = (ArrayList)elementTable.get(cacheClass);
				}
				elementList.add(cacheData);
				elementTable.put(cacheClass, elementList);
			}

			public PolicyElement[] get(Class cacheClass) {
				List list = getList(cacheClass);
				return (PolicyElement[])list.toArray(new PolicyElement[list.size()]);
			}

			public List<PolicyElement> getList(Class cacheClass) {
				if (elementTable.get(cacheClass) == null)
					return new ArrayList<PolicyElement>(0);
				ArrayList<PolicyElement> elementList = (ArrayList<PolicyElement>)elementTable.get(cacheClass);
				return elementList;
			}

			public void clear() {
				elementTable.clear();
			}
			
			public void clear(Class cacheClass) {
				if (elementTable.get(cacheClass) == null)
					return;
				ArrayList elementList = (ArrayList)elementTable.get(cacheClass);
				elementList.clear();
			}

			public void clearSubclass(Class cacheClass) {
				ArrayList elementList = (ArrayList)elementTable.get(cacheClass);
				if (elementList != null)
					elementList.clear();
				
				Iterator it = elementTable.keySet().iterator();
				while (it.hasNext()) {
					Class key = (Class)it.next();
					if (key.getSuperclass().equals(cacheClass)) {
						elementList = (ArrayList)elementTable.get(key);
						if (elementList != null)
							elementList.clear();
					}
					if (key.getSuperclass().getSuperclass().equals(cacheClass)) {
						elementList = (ArrayList)elementTable.get(key);
						if (elementList != null)
							elementList.clear();
					}
				}
			}

		}
		
	}

}
